/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "LibrariesManager.h"

#include <map>

// LLVM
#include <llvm/System/Path.h>

#include "Debug.h"
#include "Library.h"

using namespace OpenShiva;

struct LibrariesManager::Private {
  std::map<GTLCore::String, Library*> libraries;
  std::list<GTLCore::String> directories;
  static LibrariesManager* s_instance;
  
};

LibrariesManager* LibrariesManager::Private::s_instance = 0;

LibrariesManager::LibrariesManager() : d(new Private)
{
  addDirectory( "." );
  addDirectory( _OPENSHIVA_SHIVA_STD_LIB_SRC_DIR_ );
  addDirectory( _OPENSHIVA_SHIVA_LIBRARIES_DIR_ );
}

void LibrariesManager::addDirectory(const GTLCore::String& directory)
{
  d->directories.push_back( directory );
}

Library* LibrariesManager::library(const GTLCore::String& name, int _channelsNb)
{
  std::map<GTLCore::String, Library*>::iterator it = d->libraries.find(name);
  if( it == d->libraries.end())
  {
    return 0;
  }
  return it->second;
}


Library* LibrariesManager::loadLibrary( const GTLCore::String& name, int _channelsNb)
{
  Library* l = library( name, _channelsNb );
  if(not l )
  {
    GTLCore::String sourceName = name + ".shiva";
    for( std::list<GTLCore::String>::iterator it = d->directories.begin();
        it != d->directories.end(); ++it )
    {
      llvm::sys::Path path( *it );
      path.appendComponent( sourceName);
      SHIVA_DEBUG("try " << path.c_str() );
      if(path.exists() and path.canRead())
      {
        l = new Library( false, _channelsNb );
        registerLibrary( name, l );
        l->loadFromFile( path.c_str());
        break;
      }
    }
  }
  return l;
}

void LibrariesManager::registerLibrary(const GTLCore::String& name, Library* l)
{
  d->libraries[ name ] = l;
}

LibrariesManager* LibrariesManager::instance()
{
  if( not Private::s_instance )
  {
    Private::s_instance = new LibrariesManager;
  }
  return Private::s_instance;
}
