/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifndef _GTLCORE_Visitor_H_
#define _GTLCORE_Visitor_H_

#include <list>
#include <vector>
#include <GTLCore/Macros.h>

class VisitorsFactory;

namespace llvm {
  class BasicBlock;
  class Constant;
  class ErrorMessage;
  class Module;
  class Type;
  class Value;
}

namespace GTLCore {
  class ExpressionResult;
  class GenerationContext;
  class Type;
  /**
   * @internal
   * @brief This is an interface that defines functions use to access the member of a variable.
   * 
   * This class is used internally by \ref VariableNG to access the exact value of a varible.
   *
   * @ingroup GTLCore
   */
  class Visitor {
    public:
      /**
       * @param constant set it to true if the variable accessible by this Visitor is constant
       */
      Visitor();
      virtual ~Visitor();
      /**
       * @return the type that will be returned by a call to \ref pointerToIndex
       */
      virtual const Type* pointerToIndexType( const Type* _type ) const = 0;
      /**
       * Allow to access to the element of an object through an index, used for vectors,
       * arrays and some stucture (like pixels).
       */
      virtual llvm::Value* pointerToIndex(GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer, const Type* _type, llvm::Value* _index) const = 0;
      /**
       * Allow to get the value
       * @param _errMsg a pointer to a pointer (usually initialised to 0), the pointer will be
       *                set if an error occurs with the content of the error
       * @param _currentBlock the llvm block that will contains the instructions generated
       *                      by calling this function
       * @param _pointer a pointer toward the data
       * @param _ma
       * @return the value
       */
      virtual ExpressionResult get(GenerationContext& _generationContext, llvm::BasicBlock* currentBlock, llvm::Value* pointer, const Type* _pointerType) const = 0;
      /**
       * Allow to set the value
       * @param _errMsg a pointer to a pointer (usually initialised to 0), the pointer will be
       *                set if an error occurs with the content of the error
       * @param _currentBlock the llvm block that will contains the instructions generated
       *                      by calling this function
       * @param _pointer a pointer toward the data
       * @param _value the new value
       * @param _ma
       */
      virtual llvm::BasicBlock* set(GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer, const Type* _pointerType, llvm::Value* _value, const Type* _valueType, bool _allocatedInMemory) const = 0;
      /**
       * initialise a the data which is pointed by this Visitor.
       * @param _errMsg a pointer to a pointer (usually initialised to 0), the pointer will be
       *                set if an error occurs with the content of the error
       * @param _currentBlock the llvm block that will contains the instructions generated
       *                      by calling this function
       * @param _pointer a pointer toward the data
       * @param _size the size of the array, if applicable
       */
      virtual llvm::BasicBlock* initialise(GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer, const Type* _pointerType, const std::list< llvm::Value*>& _sizes, bool _allocatedInMemory) const = 0;
      virtual llvm::BasicBlock* cleanUp( GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer, const Type* _pointerType, llvm::Value* _donttouch, bool _allocatedInMemory, bool _ignoreCount, bool _deletePointer ) const = 0;
      virtual llvm::BasicBlock* mark( GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer, const Type* _pointerType, llvm::Value* increment ) const = 0;
      /**
       * Create a static variable. (ie static float v;)
       */
      virtual llvm::Constant* createStaticVariable( llvm::Module*, const Type* type, const std::list< int>& _sizes ) const = 0;
    public:
      /**
       * This function create the Visitor for the type given in argument
       */
      static const Visitor* getVisitorFor(const Type* _type);
  };
  /**
   * @internal
   * @brief Give access to a primitive type (float, integer, bool).
   */
  class PrimitiveVisitor : public Visitor {
      friend class ::VisitorsFactory;
    private:
      PrimitiveVisitor();
      virtual ~PrimitiveVisitor();
    public:
      virtual const Type* pointerToIndexType( const Type* _type ) const;
      virtual llvm::Value* pointerToIndex(GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer, const Type* _type, llvm::Value* _index) const;
      virtual ExpressionResult get( GenerationContext& _generationContext, llvm::BasicBlock* currentBlock,
                                llvm::Value* pointer, const Type* _pointerType) const;
      virtual llvm::BasicBlock* set(GenerationContext& _generationContext, llvm::BasicBlock* currentBlock, llvm::Value* pointer, const Type* _pointerType, llvm::Value*, const Type* _valueType, bool _allocatedInMemory) const;
      virtual llvm::BasicBlock* initialise(GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer, const Type* _pointerType, const std::list< llvm::Value*>& _sizes, bool _allocatedInMemory) const;
      virtual llvm::BasicBlock* cleanUp( GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer, const Type* _pointerType, llvm::Value* _donttouch, bool _allocatedInMemory, bool _ignoreCount, bool _deletePointer ) const;
      virtual llvm::BasicBlock* mark( GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer, const Type* _pointerType, llvm::Value* increment ) const;
      virtual llvm::Constant* createStaticVariable( llvm::Module*, const Type* type, const std::list< int>& _sizes ) const;
  };
  /**
   * @internal
   * @brief Give access to the elements of an array
   */
  class ArrayVisitor : public Visitor {
      friend class ::VisitorsFactory;
    private:
      ArrayVisitor();
      virtual ~ArrayVisitor();
    public:
      virtual const Type* pointerToIndexType( const Type* _type ) const;
      virtual llvm::Value* pointerToIndex(GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer, const Type* _type, llvm::Value* _index) const;
      virtual ExpressionResult get( GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer, const Type* _pointerType) const;
      virtual llvm::BasicBlock* set(GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer, const Type* _pointerType, llvm::Value*, const Type* _valueType, bool _allocatedInMemory) const;
      virtual llvm::BasicBlock* initialise( GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer, const Type* _pointerType, const std::list< llvm::Value*>& _sizes, bool _allocatedInMemory) const;
      virtual llvm::BasicBlock* cleanUp( GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer, const Type* _pointerType, llvm::Value* _donttouch, bool _allocatedInMemory, bool _ignoreCount, bool _deletePointer ) const;
      virtual llvm::BasicBlock* mark( GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer, const Type* _pointerType, llvm::Value* increment ) const;
      virtual llvm::Constant* createStaticVariable( llvm::Module*, const Type* type, const std::list< int>& _sizes ) const;
    private:
      /**
       * Allow to access the size of an array.
       * @param _errMsg a pointer to a pointer (usually initialised to 0), the pointer will be
       *                set if an error occurs with the content of the error
       * @param _currentBlock the llvm block that will contains the instructions generated
       *                      by calling this function
       * @param _pointer a pointer toward the data
       * @return the size if an array, or generate an error message
       */
      llvm::Value* getSize(GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer ) const;
      /**
       * Allow to set the size of an array
       * @param _errMsg a pointer to a pointer (usually initialised to 0), the pointer will be
       *                set if an error occurs with the content of the error
       * @param _currentBlock the llvm block that will contains the instructions generated
       *                      by calling this function
       * @param _pointer a pointer toward the data
       * @param _size the new size of the array
       */
      llvm::BasicBlock* setSize(GenerationContext& _generationContext, llvm::BasicBlock* currentBlock, llvm::Value* _pointer, const Type* _pointerType, llvm::Value* _size, bool _allocatedInMemory) const;
  };

  class VectorVisitor : public Visitor {
      friend class ::VisitorsFactory;
      VectorVisitor( );
      virtual ~VectorVisitor();
    public:
      virtual const Type* pointerToIndexType( const Type* _type ) const;
      virtual llvm::Value* pointerToIndex(GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer, const Type* _type, llvm::Value* _index) const;
      virtual ExpressionResult get( GenerationContext& _generationContext, llvm::BasicBlock* currentBlock, llvm::Value* pointer, const Type* _pointerType) const;
      virtual llvm::BasicBlock* set(GenerationContext& _generationContext, llvm::BasicBlock* currentBlock, llvm::Value* pointer, const Type* _pointerType, llvm::Value*, const Type* _valueType, bool _allocatedInMemory) const;
      virtual llvm::BasicBlock* initialise( GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer, const Type* _pointerType, const std::list< llvm::Value*>& _sizes, bool _allocatedInMemory) const;
      virtual llvm::BasicBlock* cleanUp( GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer, const Type* _pointerType, llvm::Value* _donttouch, bool _allocatedInMemory, bool _ignoreCount, bool _deletePointer ) const;
      virtual llvm::BasicBlock* mark( GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer, const Type* _pointerType, llvm::Value* increment ) const;
      virtual llvm::Constant* createStaticVariable( llvm::Module*, const Type* type, const std::list< int>& _sizes ) const;
  };

  /**
   * @internal
   * @brief Give access to members of a structure
   */
  class StructureVisitor : public Visitor {
      friend class ::VisitorsFactory;
      StructureVisitor( );
      virtual ~StructureVisitor();
    public:
      virtual const Type* pointerToIndexType( const Type* _type ) const;
      virtual llvm::Value* pointerToIndex(GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer, const Type* _type, llvm::Value* _index) const;
      virtual ExpressionResult get( GenerationContext& _generationContext, llvm::BasicBlock* currentBlock, llvm::Value* pointer, const Type* _pointerType) const;
      virtual llvm::BasicBlock* set(GenerationContext& _generationContext, llvm::BasicBlock* currentBlock, llvm::Value* pointer, const Type* _pointerType, llvm::Value*, const Type* _valueType, bool _allocatedInMemory) const;
      virtual llvm::BasicBlock* initialise( GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer, const Type* _pointerType, const std::list< llvm::Value*>& _sizes, bool _allocatedInMemory) const;
      virtual llvm::BasicBlock* cleanUp( GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer, const Type* _pointerType, llvm::Value* _donttouch, bool _allocatedInMemory, bool _ignoreCount, bool _deletePointer ) const;
      virtual llvm::BasicBlock* mark( GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer, const Type* _pointerType, llvm::Value* increment ) const;
      virtual llvm::Constant* createStaticVariable( llvm::Module*, const Type* type, const std::list< int>& _sizes ) const;
    private:
      llvm::Value* pointerToValue(GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer, int index ) const;
  };

}


#endif
