/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "Type_p.h"

#include "Debug.h"
#include "Function.h"
#include "Parameter.h"
#include "ScopedName.h"
#include "Visitor_p.h"
#include "wrappers/StructWrap.h"

using namespace GTLCore;

#include <llvm/Type.h>
#include <llvm/DerivedTypes.h>

struct Type::StructFunctionMember::Private
{
  GTLCore::Function* function; // TODO make it shareable, and free memory
};

Type::StructFunctionMember::StructFunctionMember(Function* _function) : d(new Private)
{
  d->function = _function;
}

Type::StructFunctionMember::StructFunctionMember(const StructFunctionMember& _rhs) : d(new Private)
{
  *d = *_rhs.d;
}

Type::StructFunctionMember::StructFunctionMember& Type::StructFunctionMember::operator=(const StructFunctionMember& _rhs)
{
  *d = *_rhs.d;
  return *this;
}

Type::StructFunctionMember::~StructFunctionMember()
{
//   delete d->function; // FIXME don't leak functions
  delete d;
}

const GTLCore::String& Type::StructFunctionMember::name() const
{
  return d->function->name().name();
}

const Type* Type::StructFunctionMember::returnType() const
{
  return d->function->returnType();
}

const std::vector<Parameter>& Type::StructFunctionMember::parameters() const
{
  return d->function->parameters();
}

const Function* Type::StructFunctionMember::function() const
{
  return d->function;
}

Type::Private::Private( Type::DataType _dataType) : dataType(_dataType), structDataMembers(0), structFunctionMembers(0), structPrivateFunctionMembers(0), m_visitor(0)
{
  switch( _dataType )
  {
    case Type::UNDEFINED:
      m_type = 0;
      break;
    case Type::BOOLEAN:
      m_type = llvm::IntegerType::get(1);
      break;
    case Type::INTEGER8:
    case Type::UNSIGNED_INTEGER8:
      m_type = llvm::Type::Int8Ty;
      break;
    case Type::INTEGER16:
    case Type::UNSIGNED_INTEGER16:
      m_type = llvm::Type::Int16Ty;
      break;
    case Type::INTEGER32:
    case Type::UNSIGNED_INTEGER32:
      m_type = llvm::Type::Int32Ty;
      break;
    case Type::HALF: // HALF are stored in unsigned int16
      m_type = llvm::Type::Int16Ty;
      break;
    case Type::DOUBLE:
      m_type = llvm::Type::DoubleTy;
      break;
    case Type::FLOAT:
      m_type = llvm::Type::FloatTy;
      break;
    case Type::VOID:
      m_type = llvm::Type::VoidTy;
      break;
    case Type::POINTER:
      m_type = llvm::PointerType::get(llvm::Type::Int8Ty, 0);
      break;
    default:
      m_type = 0;
      GTL_ABORT("This isn't a primitive type.");
      break;
  }
}

Type::Private::~Private()
{
  delete structDataMembers;
  delete structFunctionMembers;
  delete structPrivateFunctionMembers;
}

const llvm::Type * Type::Private::pointerType() const
{
  return llvm::PointerType::get( type(), 0);
}

const llvm::Type * Type::Private::asArgumentType() const
{
  if( dataType == Type::ARRAY or dataType == Type::STRUCTURE )
  {
    return pointerType();
  } else {
    return type();
  }
}


void Type::Private::setType( const llvm::Type* _type)
{
  GTL_ASSERT(m_type == 0);
  m_type = _type;
}

void Type::Private::addFunctionMember( const StructFunctionMember& sfm)
{
  GTL_ASSERT(  dataType == Type::STRUCTURE );
  if( not structFunctionMembers ) structFunctionMembers = new std::vector<Type::StructFunctionMember>();
  structFunctionMembers->push_back( sfm );
}

void Type::Private::addPrivateFunctionMember( const StructFunctionMember& sfm)
{
  GTL_ASSERT(  dataType == Type::STRUCTURE );
  if( not structPrivateFunctionMembers ) structPrivateFunctionMembers = new std::vector<Type::StructFunctionMember>();
  structPrivateFunctionMembers->push_back( sfm );
}

int Type::Private::memberToIndex(const GTLCore::String& name)
{
  GTL_ASSERT(structDataMembers);
  int count = 0;
  for( std::vector<GTLCore::Type::StructDataMember>::const_iterator it = structDataMembers->begin();
        it != structDataMembers->end(); ++it, ++count)
  {
    if( it->name() == name)
    {
      return count - STRUCT_FIRST_ELEMENT;
    }
  }
  return -1;
}

const Type::StructFunctionMember* Type::Private::functionMember( const GTLCore::String& _name) const
{
  if(not structFunctionMembers) return 0;
  for( std::vector<GTLCore::Type::StructFunctionMember>::const_iterator it = structFunctionMembers->begin();
        it != structFunctionMembers->end(); ++it)
  {
    if( it->name() == _name)
    {
      return &*it;
    }
  }
  return 0;
}

const Type::StructFunctionMember* Type::Private::privateFunctionMember( const GTLCore::String& _name) const
{
  if(not structPrivateFunctionMembers) return 0;
  for( std::vector<GTLCore::Type::StructFunctionMember>::const_iterator it = structPrivateFunctionMembers->begin();
        it != structPrivateFunctionMembers->end(); ++it)
  {
    if( it->name() == _name)
    {
      return &*it;
    }
  }
  return 0;
}

const Visitor* Type::Private::visitor() const
{
  return m_visitor;
}

void Type::Private::setVisitor( const Visitor* _visitor )
{
  GTL_ASSERT( m_visitor == 0 );
  m_visitor = _visitor;
}

Type* Type::Private::createArbitraryType( const llvm::Type* _type)
{
  GTLCore::Type* type = new GTLCore::Type();
  type->d->setType( _type );
  return type;
}

bool Type::Private::isComplexStructure()
{
  if( dataType != Type::STRUCTURE )
  {
    return false;
  }
  GTL_ASSERT( structDataMembers );
  for( std::vector<StructDataMember>::iterator it = structDataMembers->begin();
       it != structDataMembers->end(); ++it )
  {
    if( it->type()->dataType() == Type::STRUCTURE or it->type()->dataType() == Type::ARRAY )
    {
      return true;
    }
  }
  return false;
}

bool Type::Private::isNestedArray()
{
  GTL_ASSERT( dataType != Type::ARRAY or arrayType );
  return dataType == Type::ARRAY and (arrayType->dataType() == Type::ARRAY or arrayType->dataType() == Type::STRUCTURE);
}


const Type* Type::Private::selectType(const Type* type1, const Type* type2)
{
  GTL_DEBUG("Select type : "<< *type1 << " " << *type2 );
  GTL_ASSERT( (type1->isNumber() or type1->dataType() == Type::VECTOR ) and (type2->isNumber() or type2->dataType() == Type::VECTOR ) );
  if( type1 == type2 )
  {
    return type1;
  } else if( type1->dataType() == Type::VECTOR and type2->dataType() == Type::VECTOR )
  {
    const Type* best = selectType( type1->embeddedType(), type2->embeddedType() );
    if( type1->embeddedType() == best) return type1;
    else return type2;
  } else if( type1->dataType() == Type::VECTOR ) {
    return type1;
  } else if( type2->dataType() == Type::VECTOR ) {
    return type2;
  } else if( type1 == Type::Float or type2 == Type::Float )
  {
    return Type::Float;
  } else if( type1 == Type::Integer32 or type2 == Type::Integer32 )
  {
    return Type::Integer32;
  } else if( type1 == Type::UnsignedInteger32 or type2 == Type::UnsignedInteger32 )
  {
    return Type::UnsignedInteger32;
  } else if( type1 == Type::Boolean or type2 == Type::Boolean )
  {
    return Type::Boolean;
  }
  GTL_DEBUG("Select type failed : "<< *type1 << " " << *type2 );
  return 0;
}

const GTLCore::Type* Type::Private::subtypeAt( unsigned int _index )
{
  switch( dataType )
  {
    case Type::VECTOR:
    case Type::ARRAY:
      return arrayType;
    case Type::STRUCTURE:
      return (*structDataMembers)[_index + STRUCT_FIRST_ELEMENT].type();
    default:
      GTL_ABORT("No subtype");
      return 0;
  }
}
