/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "Optimiser_p.h"

#include <llvm/PassManager.h>
#include <llvm/Analysis/LoopPass.h>
#include <llvm/Analysis/Verifier.h>
#include "llvm/Assembly/Parser.h"
#include "llvm/Transforms/IPO.h"
#include "llvm/Transforms/Scalar.h"
#include "llvm/Target/TargetData.h"

#include "Debug.h"

using namespace GTLCore;

struct PassInfo {
    String passName;
    String passDescription;
    int level;
    llvm::Pass* (*creator)();
};

static const PassInfo infos[] = {
    { "CFGSimplification", "Clean up disgusting code", 2, (llvm::Pass* (*)())(llvm::createCFGSimplificationPass) },
    { "GlobalDCE", "Remove unused globals", 2, (llvm::Pass* (*)())(llvm::createGlobalDCEPass) },
    { "IPConstantPropagation", "IP Constant Propagation", 2, (llvm::Pass* (*)())(llvm::createIPConstantPropagationPass) },
    { "InstructionCombining", "Clean up after IPCP", 2, (llvm::Pass* (*)())(llvm::createInstructionCombiningPass) },
    { "CFGSimplification", "Clean up after IPCP", 2, (llvm::Pass* (*)())(llvm::createCFGSimplificationPass) },
    { "FunctionInlining", "Inline small definitions (functions)", 2, (llvm::Pass* (*)())(llvm::createFunctionInliningPass) },
    { "TailDuplication", "Simplify cfg by copying code", 2, (llvm::Pass* (*)())(llvm::createTailDuplicationPass) },
    { "CFGSimplification", "Merge & remove BBs", 3, (llvm::Pass* (*)())(llvm::createCFGSimplificationPass) },
    { "InstructionCombining", "Compile silly sequences", 3, (llvm::Pass* (*)())(llvm::createInstructionCombiningPass) },
    { "Reassociate", "Reassociate expressions", 3, (llvm::Pass* (*)())(llvm::createReassociatePass) },
    { "InstructionCombining", "Combine silly sequences", 3, (llvm::Pass* (*)())(llvm::createInstructionCombiningPass) },
    { "TailCallElimination", "Eliminate tail calls", 3, (llvm::Pass* (*)())(llvm::createTailCallEliminationPass) },
    { "CFGSimplification", "Merge & remove BBs", 3, (llvm::Pass* (*)())(llvm::createCFGSimplificationPass) },
    { "LICM", "Hoist loop invariants", 3, (llvm::Pass* (*)())(llvm::createLICMPass) },
    { "InstructionCombining", "Clean up after the unroller", 3, (llvm::Pass* (*)())(llvm::createInstructionCombiningPass) },
    { "IndVarSimplify", "Canonicalize indvars", 3, (llvm::Pass* (*)())(llvm::createIndVarSimplifyPass) },
    { "LoopUnroll", "Unroll small loops", 3, (llvm::Pass* (*)())(llvm::createLoopUnrollPass) },
    { "InstructionCombining", "Clean up after the unroller", 3, (llvm::Pass* (*)())(llvm::createInstructionCombiningPass) },
    { "SCCP", "Constant prop with SCCP", 3, (llvm::Pass* (*)())(llvm::createSCCPPass) },
    { "InstructionCombining", "Run instcombine again after redundancy elimination", 4, (llvm::Pass* (*)())(llvm::createInstructionCombiningPass) },
    { "DeadStoreElimination", "Delete dead stores", 4, (llvm::Pass* (*)())(llvm::createDeadStoreEliminationPass) },
    { "AggressiveDCE", "SSA based 'Aggressive DCE'", 4, (llvm::Pass* (*)())(llvm::createAggressiveDCEPass) },
    { "CFGSimplification", "Merge & remove BBs", 4, (llvm::Pass* (*)())(llvm::createCFGSimplificationPass) },
    { "ConstantMerge", "Merge dup global constants", 4, (llvm::Pass* (*)())(llvm::createConstantMergePass) },
    { "CFGSimplification", "Merge & remove BBs", 1, (llvm::Pass* (*)())(llvm::createCFGSimplificationPass) },
    { "PromoteMemoryToRegister", "Memory To Register", 1, (llvm::Pass* (*)())(llvm::createPromoteMemoryToRegisterPass) },
    { "InstructionCombining", "Compile silly sequences", 1, (llvm::Pass* (*)())(llvm::createInstructionCombiningPass) },
    { "", "", -1, 0 } };
    
Optimiser* Optimiser::Private::s_instance = new Optimiser;

Optimiser::Private::Private() : m_passManager(0)
{
  setLevel( 2 );
}

Optimiser::Private::~Private()
{
  delete m_passManager;
}

llvm::PassManager* Optimiser::Private::passManager()
{
  if(not m_passManager)
  {
    m_passManager = new llvm::PassManager;
    m_passManager->add( new llvm::TargetData("") );
    m_passManager->add( llvm::createVerifierPass() );
    for(int i = 0; infos[i].passName != ""; ++i)
    {
      GTL_DEBUG( infos[i].passName );
      if( isActive( infos[i].passName ) )
      {
        m_passManager->add( infos[i].creator() );
      }
    }
    m_passManager->add( llvm::createVerifierPass() );
  }
  GTL_DEBUG("Finished pass manager");
  return m_passManager;
}

void Optimiser::Private::setLevel( int _level )
{
  for(int i = 0; infos[i].passName != ""; ++i)
  {
    GTL_DEBUG( i << " " << infos[i].passName );
    setActive( infos[i].passName, infos[i].level <= _level );
  }
}


bool Optimiser::Private::isActive(const String& _passName ) const
{
  std::map<String, bool>::const_iterator it = m_passActivity.find( _passName );
  if( it == m_passActivity.end() )
  {
    return false;;
  }
  return it->second;
}

void Optimiser::Private::setActive(const String& _passName, bool _active )
{
  delete m_passManager;
  m_passManager = 0;
  m_passActivity[_passName] = _active;
}
