# -*- coding: utf-8 -*-

# Copyright (C) 2004-2006 Johan Svedberg <johan@svedberg.com>

# This file is part of ontv.

# ontv is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.

# ontv is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with ontv; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA

import os.path
from gettext import gettext as _

import gtk
import gnomeapplet

import GUI
from Configuration import Configuration
from XMLTVFile import XMLTVFile
from Reminders import Reminders
from KeyBinder import KeyBinder
from ProgramWindow import ProgramWindow
from PreferencesDialog import PreferencesDialog
from SearchDialog import SearchDialog
from AboutDialog import AboutDialog
from ontv import NAME, VERSION, UI_DIR, LOCALE_DIR

class OnTVApplet(object):
    """Singleton representing the applet"""

    instance = None

    def __new__(type, *args):
        if OnTVApplet.instance is None:
            OnTVApplet.instance = object.__new__(type)
            OnTVApplet.instance.__init(*args)
        return OnTVApplet.instance

    def __init(self, *args):
        self.applet = args[0]
        self.config = Configuration(args[1], args[2])
        self.xmltvfile = XMLTVFile(self.config)
        self.xmltvfile.connect("loading", self.__xmltvfile_loading)
        self.xmltvfile.connect("loaded", self.__xmltvfile_loaded)

        self.reminders = Reminders(self.config)
        if os.path.exists(self.reminders.file):
            self.reminders = self.reminders.load()

        self.kb = KeyBinder(self.config)
        self.kb.connect("activated", self.__kb_activated)

        self.pw = ProgramWindow(self.xmltvfile, self.applet,
                                self.get_docking_data)
        self.sd = SearchDialog(self.xmltvfile, self.reminders)
        self.pd = PreferencesDialog(self)
        self.tooltips = gtk.Tooltips()

        self.applet.set_applet_flags(gnomeapplet.EXPAND_MINOR)

        ui_file = os.path.join(UI_DIR, "GNOME_OnTVApplet.xml")
        verb_list = [("Reload XMLTV file", self.xmltvfile.load),
                     ("Preferences", self.pd.show),
                     ("Search Program", self.sd.show),
                     ("About", self.__run_ad)]
        self.applet.setup_menu_from_file(None, ui_file, NAME, verb_list)

        self.applet.connect("button-press-event", self.__button_press)

        image = gtk.Image()
        scaled_icon = GUI.icon.scale_simple(24, 24, gtk.gdk.INTERP_BILINEAR)
        image.set_from_pixbuf(scaled_icon)

        self.applet.add(image)

        self.applet.show_all()

    def __run_ad(self, uicomponent=None, verb=None):
        self.ad = AboutDialog()
        self.ad.run()

    def run(self):
        if self.config.xmltv_file == '':
            self.pd.show()
            self.pd.notebook.set_current_page(0)
        else:
            self.xmltvfile.props.path = self.config.xmltv_file

    def __button_press(self, button, event):
        if event.type == gtk.gdk.BUTTON_PRESS and event.button == 1:
            self.__press_button()

    def __press_button(self):
        if not self.pw.props.visible:
            self.applet.set_state(gtk.STATE_SELECTED)
            self.pw.position_window()
            self.pw.stick()
            self.pw.show_all()
            self.pw.grab_focus()
        else:
            self.applet.set_state(gtk.STATE_NORMAL)
            self.pw.hide()

        if not self.config.display_current_programs:
            self.pw.cpt.hide()
        if not self.config.display_upcoming_programs:
            self.pw.upt.hide()

    def set_tooltip(self, text):
        self.tooltips.set_tip(self.applet, text)

    def __xmltvfile_loading(self, xmltvfile):
        self.set_tooltip(_("Loading XMLTV file..."))

    def __xmltvfile_loaded(self, xmltvfile, listings):
        self.set_tooltip(None)

    def __kb_activated(self, kb, name, time):
        if name == "show-window-hotkey":
            self.__press_button()
        elif name == "show-search-program-hotkey":
            self.sd.present(time)

    def get_docking_data(self, middle, w=0, h=0):
        self.applet.realize()
        (x, y) = self.applet.window.get_origin()

        button_w = self.applet.allocation.width
        button_h = self.applet.allocation.height

        screen = self.applet.get_screen()

        found_monitor = False
        n = screen.get_n_monitors()
        for i in range(0, n):
            monitor = screen.get_monitor_geometry(i)
            if (x >= monitor.x and x <= monitor.x + monitor.width and
                y >= monitor.y and y <= monitor.y + monitor.height):
                    found_monitor = True
                    break

        if not found_monitor:
            screen_width = screen.get_width()
            monitor = gtk.gdk.Rectangle(0, 0, screen_width, screen_width)

        orient = self.applet.get_orient()

        if orient == gnomeapplet.ORIENT_RIGHT:
            x += button_w

            if ((y + h) > monitor.y + monitor.height):
                y -= (y + h) - (monitor.y + monitor.height)

            if middle:
                y += button_h/2

            if ((y + h) > (monitor.height / 2)):
                gravity = gtk.gdk.GRAVITY_SOUTH_WEST
            else:
                gravity = gtk.gdk.GRAVITY_NORTH_WEST
        elif orient == gnomeapplet.ORIENT_LEFT:
            x -= w

            if ((y + h) > monitor.y + monitor.height):
                y -= (y + h) - (monitor.y + monitor.height)

            if middle:
                y += button_h/2

            if ((y + h) > (monitor.height / 2)):
                gravity = gtk.gdk.GRAVITY_SOUTH_EAST
            else:
                gravity = gtk.gdk.GRAVITY_NORTH_EAST
        elif orient == gnomeapplet.ORIENT_DOWN or self.config.standalone:
            y += button_h

            if ((x + w) > monitor.x + monitor.width):
                x -= (x + w) - (monitor.x + monitor.width)

            if middle:
                x += button_w/2

            gravity = gtk.gdk.GRAVITY_NORTH_WEST
        elif orient == gnomeapplet.ORIENT_UP:
            y -= h

            if ((x + w) > monitor.x + monitor.width):
                x -= (x + w) - (monitor.x + monitor.width)

            if middle:
                x += button_w/2

            gravity = gtk.gdk.GRAVITY_SOUTH_WEST

        return (x, y, gravity)

# vim: set sw=4 et sts=4 tw=79 fo+=l:
