/* -*- Mode: C; tab-width: 2; indent-tabs-mode: t; c-basic-offset: 2; coding: utf-8 -*- 
 *
 * Copyright (C) 2007, 2008, 2009 John P. Swensen
 *
 * This file is as a part of OctaveDE.
 *
 * Octave is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * Octave is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Octave; see the file COPYING.  If not, write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 * */

#include "OctaveUI.h"

#include "MEditorPanel.h"
#include "GtkSourceViewUtilities.h"

#if defined(__APPLE__)
#include <sys/param.h>
#include <mach-o/dyld.h>
#endif

#include <sys/stat.h>



GtkSourceLanguage *get_language_for_mime_type (const gchar *mime);
//static gboolean gtk_source_buffer_save_file (GtkSourceBuffer *source_buffer, const gchar *filename, GError **error);



MEditorPanel::MEditorPanel(Dock& dock, string filename)
: m_toolbar(),
	DockItem(dock,filename,filename,"FilenameIcon",DockItem::DOCKED_STATE,(GdlDockItemBehavior)(GDL_DOCK_ITEM_BEH_CANT_ICONIFY))
{
	lm = gtk_source_language_manager_get_default ();
	
	// Create the whole viewer widget
	GtkSourceLanguage *language = get_language_for_mime_type ("text/x-octave");
	buffer = create_source_buffer (lm, language);
	mmBuffer = Glib::wrap ( (GtkTextBuffer*)buffer);
	
	view = (Gtk::TextView*)Glib::wrap( gtk_source_view_new_with_buffer(buffer) );
	gtk_source_view_set_show_line_numbers(GTK_SOURCE_VIEW(view->gobj()), true);
	gtk_source_view_set_show_line_marks(GTK_SOURCE_VIEW(view->gobj()), true);
	MEditor::loadDebugPixmaps (view);	
	mmBuffer->signal_modified_changed ().connect (sigc::mem_fun (*this, &MEditorPanel::onModifiedChanged), false);
	view->signal_button_press_event().connect( sigc::mem_fun( *this, &MEditorPanel::onMouseClicked), false );
  
	// Set the font
	PangoFontDescription *font_desc = pango_font_description_from_string ("monospace");
	if (font_desc != NULL)
  {
		gtk_widget_modify_font ((GtkWidget*)view->gobj(), font_desc);
		pango_font_description_free (font_desc);
	}

	// Add the editor TextView to the scrollview
	scrollView.add(*view);

	// Add the toolbar and scroll view to the dockitem (this)
	this->get_vbox() ->pack_start(m_toolbar, Gtk::PACK_SHRINK);
	this->get_vbox() ->pack_start(scrollView);

	// Register events for the toolbar
  m_toolbar.m_tool_new.signal_clicked ().connect( sigc::mem_fun( *this, &MEditorPanel::onNew) );
  m_toolbar.m_tool_open.signal_clicked ().connect( sigc::mem_fun( *this, &MEditorPanel::onOpen) );	
  m_toolbar.m_tool_save.signal_clicked ().connect( sigc::mem_fun( *this, &MEditorPanel::onSave) );	


	m_toolbar.m_tool_undo.signal_clicked ().connect( sigc::mem_fun( *this, &MEditorPanel::onUndo) );
	m_toolbar.m_tool_redo.signal_clicked ().connect( sigc::mem_fun( *this, &MEditorPanel::onRedo) );

	m_toolbar.m_tool_cut.signal_clicked ().connect( sigc::mem_fun( *this, &MEditorPanel::onCut) );
	m_toolbar.m_tool_copy.signal_clicked ().connect( sigc::mem_fun( *this, &MEditorPanel::onCopy) );
	m_toolbar.m_tool_paste.signal_clicked ().connect( sigc::mem_fun( *this, &MEditorPanel::onPaste) );

	m_toolbar.m_tool_stepover.signal_clicked ().connect (sigc::mem_fun( *this, &MEditorPanel::onStepOver) );
	
	MEditor* meditor = MEditor::GetInstance(false);
  if (meditor == NULL)
	{
		std::cout << "The editor singleton must be initialized before creating an editor buffer" << std::endl;
	}
	else
	{
		signal_on_new = sigc::mem_fun(*meditor, &MEditor::onNew);
		signal_on_open = sigc::mem_fun(*meditor, &MEditor::onOpen);
	}

	if (filename.compare(0,8,"Untitled")!=0)
	{
		open_file (buffer, filename.c_str());
	}

}

////////////////////////////////////////////////////////////////////////////////////
static void remove_all_marks (GtkSourceBuffer *buffer)
{
	GtkTextIter s, e;
	gtk_text_buffer_get_bounds (GTK_TEXT_BUFFER (buffer), &s, &e);		 
	gtk_source_buffer_remove_source_marks (buffer, &s, &e, NULL);
}

bool MEditorPanel::has_breakpoint_marker (bp_info_t bp)
{
	const gchar *mark_type = MARK_TYPE_1;
	GSList *mark_list = NULL;

	if (bp.filename.compare (getFuncName()) != 0)
		return false;

	mark_list = gtk_source_buffer_get_source_marks_at_line (buffer,
																													bp.line_number-1,
																													mark_type);
	
	if (mark_list != NULL)
		return true;
	else
		return false;
}

////////////////////////////////////////////////////////////////////////////////////
bool MEditorPanel::onMouseClicked(GdkEventButton* ev)
{
	if (ev->window == gtk_text_view_get_window (GTK_TEXT_VIEW (view->gobj()), GTK_TEXT_WINDOW_LEFT))
	{
		gint y_buf;
		GtkTextIter line_start;
		GSList *mark_list;
		const gchar *mark_type;
		
		if (ev->button == 1)
			mark_type = MARK_TYPE_1;
		else
			mark_type = MARK_TYPE_2;
		
		gtk_text_view_window_to_buffer_coords (GTK_TEXT_VIEW (view->gobj()),
																					 GTK_TEXT_WINDOW_LEFT,
																					 ev->x, ev->y,
																					 NULL, &y_buf);
		
		/* get line bounds */
		gtk_text_view_get_line_at_y (GTK_TEXT_VIEW (view->gobj()),
																 &line_start,
																 y_buf,
																 NULL);
		
		/* get the marks already in the line */
		mark_list = gtk_source_buffer_get_source_marks_at_line (buffer,
																														gtk_text_iter_get_line (&line_start),
																														mark_type);
		
		// TODO: I need to add a bunch more stuff here to set the breakpoint in Octave, then recheck to find out
		// where it was placed before actually adding it to the buffer.
		
		if (mark_list != NULL)
		{
			// just take the first and delete it
			//gtk_text_buffer_delete_mark (GTK_TEXT_BUFFER (buffer),
			//														 GTK_TEXT_MARK (mark_list->data));

			Glib::ustring filename = this->get_title();
			std::string shortName = Glib::path_get_basename(filename);
			std::string funcName;
			int dot = shortName.find_first_of(".");
			if (dot!=string::npos)
			{
				funcName = shortName.substr (0,dot);
			}
			bp_info_t bp;
			bp.filename = funcName;
			bp.line_number = gtk_text_iter_get_line (&line_start)+1;
			oct_octave_server.remove_breakpoint(bp);

			/*
			bp_table::intmap lines;
			lines[0] = gtk_text_iter_get_line (&line_start)+1;
			bp_table::remove_breakpoint (funcName,lines);
			*/
		}
		else
		{
			Glib::ustring filename = this->get_title();
			std::string shortName = Glib::path_get_basename(filename);
			std::string funcName;
			int dot = shortName.find_first_of(".");
			if (dot!=string::npos)
			{
				funcName = shortName.substr (0,dot);
			}
			bp_info_t bp;
			bp.filename = funcName;
			bp.line_number = gtk_text_iter_get_line (&line_start)+1;
			oct_octave_server.add_breakpoint(bp);
			
			/*
			usleep(50000);

			remove_all_marks (buffer);
			std::vector<bp_info_t> bps = oct_octave_server.get_breakpoint_list();
			for (int i = 0 ; i < bps.size() ; i++)
			{
				gtk_text_iter_set_line (&line_start, bps[i].line_number-1);
				gtk_source_buffer_create_source_mark (buffer,
																							NULL,
																							mark_type,
																							&line_start);
			}
			*/
		}
		g_slist_free (mark_list);
	}

	return false;
}

////////////////////////////////////////////////////////////////////////////////////
void MEditorPanel::onModifiedChanged ()
{
	//cout << "Modified changed" << endl;
	
	// In this function, we need to determine which breakpoints have been invalidated
	// as replace them if they have changed
	// If I remember correctly, all breakpoints are reset when a file is saved.  I think this
	// is kindof dumb and breakpoints should try to follow where they are at.
}

void MEditorPanel::remove_all_markers (bp_marker_type_t type)
{
	const gchar *mark_type;
	if (type == BP_MARKER_TYPE_LOC)
		mark_type = MARK_TYPE_1;
	else
		mark_type = MARK_TYPE_2;

	GtkTextIter s, e;
	gtk_text_buffer_get_bounds (GTK_TEXT_BUFFER (buffer), &s, &e);		 
	gtk_source_buffer_remove_source_marks (buffer, &s, &e, mark_type);

	//remove_all_marks (buffer);
}

void MEditorPanel::add_breakpoint_marker (int line, bp_marker_type_t type)
{
	gint y_buf;
	GtkTextIter line_start;
	GSList *mark_list;
	const gchar *mark_type;

	GtkTextIter s, e;
	gtk_text_buffer_get_bounds (GTK_TEXT_BUFFER (buffer), &s, &e);
	
	if (type == BP_MARKER_TYPE_LOC)
		mark_type = MARK_TYPE_1;
	else
		mark_type = MARK_TYPE_2;

	gtk_text_iter_set_line (&s, line-1);
	gtk_source_buffer_create_source_mark (buffer,
																				NULL,
																				mark_type,
																				&s);	

	/*
	gtk_text_iter_set_line (&line_start, line-1);
	gtk_source_buffer_create_source_mark (buffer,
																				NULL,
																				mark_type,
																				&line_start);	
	*/
}

////////////////////////////////////////////////////////////////////////////////////
void MEditorPanel::onNew(void)
{
	// Need to create a new panel and pass it back to the MEditor singleton
	signal_on_new();
}


////////////////////////////////////////////////////////////////////////////////////
void MEditorPanel::onOpen(void)
{
	// Need to pass an event back to the MEditor singleton
	signal_on_open();
}

////////////////////////////////////////////////////////////////////////////////////
void MEditorPanel::onSave(void)
{
	Glib::ustring filename = this->get_title();

	if (filename.compare(0,8,"Untitled")==0)
	{
		int status = MEditor::saveAsFileDialog( filename );
		if (!status)
		{
			gboolean saveSuccess = gtk_source_buffer_save_file (buffer, 
																													filename.c_str(), 
																													NULL);
			// If we saved successfully, we need to rename the dock panel
			if (saveSuccess)
			{
				std::string shortName = Glib::path_get_basename(filename);
				
				this->hide();
				this->set_title(filename);
				this->set_tablabel(shortName);
				this->shortName = shortName;
				this->show_all();
				this->present();
				this->grab_focus();
			}
		}
	}
	else
	{
		gboolean saveSuccess = gtk_source_buffer_save_file (buffer, 
																												filename.c_str(), 
																												NULL);
	}
}

////////////////////////////////////////////////////////////////////////////////////
void MEditorPanel::onSaveAs(void)
{
	Glib::ustring filename;// = viewSrcDockMap[bufferW]->get_title();

	int status = MEditor::saveAsFileDialog( filename );
	if (!status)
	{
		gboolean saveSuccess = gtk_source_buffer_save_file (buffer, 
																													filename.c_str(), 
																													NULL);
		// If we saved successfully, we need to rename the dock panel
		if (saveSuccess)
		{
			std::string shortName = Glib::path_get_basename(filename);
			this->hide();
			this->set_title(filename);
			this->set_tablabel(shortName);
			this->shortName = shortName;
			this->show_all();
			this->present();
			this->grab_focus();

		}
	}

}

////////////////////////////////////////////////////////////////////////////////////
void MEditorPanel::onUndo(void)
{
	if (gtk_source_buffer_can_undo (buffer))
		gtk_source_buffer_undo (buffer);
}


////////////////////////////////////////////////////////////////////////////////////
void MEditorPanel::onRedo(void)
{
	if (gtk_source_buffer_can_redo (buffer))
		gtk_source_buffer_redo (buffer);
}


////////////////////////////////////////////////////////////////////////////////////
void MEditorPanel::onCut(void)
{
	GtkClipboard *gtkclip = gtk_clipboard_get( GDK_NONE );
	gtk_text_buffer_cut_clipboard( GTK_TEXT_BUFFER(buffer),
																 gtkclip,
																 true );
}

////////////////////////////////////////////////////////////////////////////////////
void MEditorPanel::onCopy(void)
{
	GtkClipboard *gtkclip = gtk_clipboard_get( GDK_NONE );
	gtk_text_buffer_copy_clipboard( GTK_TEXT_BUFFER(buffer),
																	gtkclip );
}
  

////////////////////////////////////////////////////////////////////////////////////
void MEditorPanel::onPaste(void)
{
	GtkClipboard *gtkclip = gtk_clipboard_get( GDK_NONE );
	gtk_text_buffer_paste_clipboard( GTK_TEXT_BUFFER(buffer),
                                   gtkclip,
																	 NULL,
                                   true );
}

////////////////////////////////////////////////////////////////////////////////////
void MEditorPanel::onStepOver(void)
{
	oct_octave_server.set_breakpoint_action (BP_ACTION_STEP_OVER);
}


std::vector<bp_info_t> MEditorPanel::get_breakpoint_markers ()
{
	std::vector<bp_info_t> retval;
	GtkTextIter s, e;
	gtk_text_buffer_get_bounds (GTK_TEXT_BUFFER (buffer), &s, &e);
	
	gboolean updated = true;
	while (updated)
  {
		updated = gtk_source_buffer_forward_iter_to_source_mark (buffer,
																														 &s,
																														 MARK_TYPE_1);
		if (updated)
		{
			bp_info_t tmp;
			tmp.filename = getFuncName();
			tmp.line_number = gtk_text_iter_get_line(&s)+1;
			retval.push_back (tmp);
		}
	}

	return retval;
}
