/* -*- Mode: C; tab-width: 2; indent-tabs-mode: t; c-basic-offset: 2; coding: utf-8 -*- 
 *
 * Copyright (C) 2007, 2008, 2009 John P. Swensen
 *
 * This file is as a part of OctaveDE.
 *
 * Octave is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * Octave is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Octave; see the file COPYING.  If not, write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 * */
#include "GtkSourceViewUtilities.h"

////////////////////////////////////////////////////////////////////////////////////
GtkSourceBuffer* create_source_buffer (GtkSourceLanguageManager *manager, 
				       GtkSourceLanguage* language )
{
  GtkSourceBuffer *buffer;

  if (language == NULL)
    buffer = GTK_SOURCE_BUFFER (gtk_source_buffer_new (NULL));
  else
    buffer = GTK_SOURCE_BUFFER (gtk_source_buffer_new_with_language (language));

  g_object_ref (manager);
  g_object_set_data_full (G_OBJECT (buffer), "languages-manager",
                          manager, (GDestroyNotify) g_object_unref);
  gtk_source_buffer_set_highlight_syntax( buffer, true );
  return buffer;
}


////////////////////////////////////////////////////////////////////////////////////
void error_dialog (GtkWindow *parent, const gchar *msg, ...)
{
  va_list ap;
  gchar *tmp;
  GtkWidget *dialog;

  va_start (ap, msg);
  tmp = g_strdup_vprintf (msg, ap);
  va_end (ap);

  dialog = gtk_message_dialog_new (parent,
                                   GTK_DIALOG_DESTROY_WITH_PARENT,
                                   GTK_MESSAGE_ERROR,
                                   GTK_BUTTONS_OK,
                                   tmp);
  g_free (tmp);

  gtk_dialog_run (GTK_DIALOG (dialog));
  gtk_widget_destroy (dialog);
}


////////////////////////////////////////////////////////////////////////////////////
gboolean gtk_source_buffer_load_with_encoding (GtkSourceBuffer *source_buffer,
					       const gchar     *filename,
					       const gchar     *encoding,
					       GError         **error)
{
  GIOChannel *io;
  GtkTextIter iter;
  gchar *buffer;
  gboolean reading;

  g_return_val_if_fail (source_buffer != NULL, FALSE);
  g_return_val_if_fail (filename != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_SOURCE_BUFFER (source_buffer), FALSE);

  *error = NULL;

  io = g_io_channel_new_file (filename, "r", error);
  if (!io)
  {
    error_dialog (NULL, "%s\nFile %s", (*error)->message, filename);
    return FALSE;
  }

  if (g_io_channel_set_encoding (io, encoding, error) != G_IO_STATUS_NORMAL)
  {
    error_dialog (NULL, "Failed to set encoding:\n%s\n%s",
                  filename, (*error)->message);
    return FALSE;
  }

  gtk_source_buffer_begin_not_undoable_action (source_buffer);

  gtk_text_buffer_set_text (GTK_TEXT_BUFFER (source_buffer), "", 0);
  buffer = (gchar*)g_malloc (READ_BUFFER_SIZE);
  reading = TRUE;
  while (reading)
  {
    gsize bytes_read;
    GIOStatus status;

    status = g_io_channel_read_chars (io, buffer,
                                      READ_BUFFER_SIZE, &bytes_read,
                                      error);
    switch (status)
    {
      case G_IO_STATUS_EOF:
        reading = FALSE;
        /* fall through */

      case G_IO_STATUS_NORMAL:
        if (bytes_read == 0)
        {
          continue;
        }

        gtk_text_buffer_get_end_iter (
                                     GTK_TEXT_BUFFER (source_buffer), &iter);
        gtk_text_buffer_insert (GTK_TEXT_BUFFER (source_buffer),
                                &iter, buffer, bytes_read);
        break;

      case G_IO_STATUS_AGAIN:
        continue;

      case G_IO_STATUS_ERROR:
      default:
        error_dialog (NULL, "%s\nFile %s", (*error)->message, filename);

        /* because of error in input we clear already loaded text */
        gtk_text_buffer_set_text (GTK_TEXT_BUFFER (source_buffer), "", 0);

        reading = FALSE;
        break;
    }
  }
  g_free (buffer);

  gtk_source_buffer_end_not_undoable_action (source_buffer);

  g_io_channel_unref (io);

  if (*error)
    return FALSE;

  gtk_text_buffer_set_modified (GTK_TEXT_BUFFER (source_buffer), FALSE);

  /* move cursor to the beginning */
  gtk_text_buffer_get_start_iter (GTK_TEXT_BUFFER (source_buffer), &iter);
  gtk_text_buffer_place_cursor (GTK_TEXT_BUFFER (source_buffer), &iter);

  return TRUE;
}


////////////////////////////////////////////////////////////////////////////////////
void remove_all_markers (GtkSourceBuffer *buffer)
{
  GtkTextIter s, e;
  gtk_text_buffer_get_bounds (GTK_TEXT_BUFFER (buffer), &s, &e);
  gtk_source_buffer_remove_source_marks (buffer, &s, &e, NULL);	
}


////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////
// BEGIN NEW GTKSOURCEVIEW-2.0 CODE
////////////////////////////////////////////////////////////////
/* Note this is wrong for several reasons, e.g. g_pattern_match is broken
 * for glob matching. */
GtkSourceLanguage* get_language_for_filename (const gchar *filename)
{
  const gchar * const *languages;
  gchar *filename_utf8;
  GtkSourceLanguageManager *manager;

  filename_utf8 = g_filename_to_utf8 (filename, -1, NULL, NULL, NULL);
  g_return_val_if_fail (filename_utf8 != NULL, NULL);

  manager = gtk_source_language_manager_get_default ();
  languages = gtk_source_language_manager_get_language_ids (manager);

  while (*languages != NULL)
  {
    GtkSourceLanguage *lang;
    gchar **globs, **p;
    
    lang = gtk_source_language_manager_get_language (manager,
						     *languages);
    g_return_val_if_fail (GTK_IS_SOURCE_LANGUAGE (lang), NULL);
    ++languages;

    globs = gtk_source_language_get_globs (lang);
    if (globs == NULL)
      continue;

    for (p = globs; *p != NULL; p++)
    {
      if (g_pattern_match_simple (*p, filename_utf8))
      {
	g_strfreev (globs);
	g_free (filename_utf8);
	
	return lang;
      }
    }
    
    g_strfreev (globs);
  }

  g_free (filename_utf8);
  return NULL;
}

////////////////////////////////////////////////////////////////////////////////////
/* Note this is wrong, because it ignores mime parent types and subtypes.
 * It's fine to use in a simplish program like this, but is unacceptable
 * in a serious text editor. */
GtkSourceLanguage* get_language_for_mime_type (const gchar *mime)
{
  const gchar * const *languages;
  GtkSourceLanguageManager *manager;

  manager = gtk_source_language_manager_get_default ();
  languages = gtk_source_language_manager_get_language_ids (manager);
  
  while (*languages != NULL)
  {
    GtkSourceLanguage *lang;
    gchar **mimetypes, **p;
    
    lang = gtk_source_language_manager_get_language (manager,
						     *languages);
    g_return_val_if_fail (GTK_IS_SOURCE_LANGUAGE (lang), NULL);
    ++languages;
    
    mimetypes = gtk_source_language_get_mime_types (lang);
    
    if (mimetypes == NULL)
      continue;

    for (p = mimetypes; *p != NULL; p++)
    {
      if (strcmp (*p, mime) == 0)
      {
	g_strfreev (mimetypes);
	return lang;
      }
    }
    
    g_strfreev (mimetypes);
  }
  
  return NULL;
}


////////////////////////////////////////////////////////////////////////////////////
GtkSourceLanguage* get_language_for_file (const gchar *filename)
{
  GtkSourceLanguage *language = NULL;

#if defined(USE_GNOME_VFS)
  gchar *mime_type;
  gchar *uri;

  /* I hate this! */
  if (g_path_is_absolute (filename))
  {
    uri = gnome_vfs_get_uri_from_local_path (filename);
  }
  else
  {
    gchar *curdir, *path;
    curdir = g_get_current_dir ();
    path = g_strconcat (curdir, "/", filename, NULL);
    g_free (curdir);
    uri = gnome_vfs_get_uri_from_local_path (path);
    g_free (path);
  }

  if ((mime_type = gnome_vfs_get_mime_type (uri)))
    language = get_language_for_mime_type (mime_type);

  g_free (mime_type);
  g_free (uri);
#endif
	
  // Explicitly get the Octave mime type
  language = get_language_for_mime_type ("text/x-octave");
  
  if (!language)
    language = get_language_for_filename (filename);
  
  return language;
}


////////////////////////////////////////////////////////////////////////////////////
GtkSourceLanguage* get_language_by_id (const gchar *id)
{
  GtkSourceLanguageManager *manager;
  manager = gtk_source_language_manager_get_default ();
  return gtk_source_language_manager_get_language (manager, id);
}

////////////////////////////////////////////////////////////////////////////////////
GtkSourceLanguage* get_language (GtkTextBuffer *buffer, const gchar *filename)
{
  GtkSourceLanguage *language = NULL;
  GtkTextIter start, end;
  gchar *text;
  const gchar *lang_string;
  
  gtk_text_buffer_get_start_iter (buffer, &start);
  end = start;
  gtk_text_iter_forward_line (&end);

#define LANG_STRING "gtk-source-lang:"
  text = gtk_text_iter_get_slice (&start, &end);
  lang_string = strstr (text, LANG_STRING);
  if (lang_string != NULL)
  {
    gchar **tokens;
    
    lang_string += strlen (LANG_STRING);
    g_strchug ( const_cast<gchar*>(lang_string));
    
    tokens = g_strsplit_set (lang_string, " \t\n", 2);
    
    if (tokens != NULL && tokens[0] != NULL)
      language = get_language_by_id (tokens[0]);

    g_strfreev (tokens);
  }
  
  if (!language)
    language = get_language_for_file (filename);

  g_free (text);
  return language;
}

////////////////////////////////////////////////////////////////////////////////////
/*
void error_dialog (GtkWindow *parent, const gchar *msg, ...)
{
  va_list ap;
  gchar *tmp;
  GtkWidget *dialog;
  
  va_start (ap, msg);
  tmp = g_strdup_vprintf (msg, ap);
  va_end (ap);
  
  dialog = gtk_message_dialog_new (parent,
				   GTK_DIALOG_DESTROY_WITH_PARENT,
				   GTK_MESSAGE_ERROR,
				   GTK_BUTTONS_OK,
				   tmp);
  g_free (tmp);
  
  gtk_dialog_run (GTK_DIALOG (dialog));
  gtk_widget_destroy (dialog);
}
*/

////////////////////////////////////////////////////////////////////////////////////
gboolean gtk_source_buffer_save_file (GtkSourceBuffer *source_buffer, 
				      const gchar *filename, 
				      GError **error)
{
  GError *error_here = NULL;
  GtkTextIter start, end;
  
  g_return_val_if_fail (GTK_IS_SOURCE_BUFFER (source_buffer), FALSE);
  g_return_val_if_fail (filename != NULL, FALSE);
  
  gtk_text_buffer_get_bounds (GTK_TEXT_BUFFER (source_buffer), &start, &end);
  gchar* buffer = gtk_text_buffer_get_text (GTK_TEXT_BUFFER (source_buffer),&start,&end,true);
  if (!g_file_set_contents (filename, buffer, -1, &error_here))
  {
    //		error_dialog (NULL, "%s\nFile %s", error_here->message, filename);
    //    g_propagate_error (error, error_here);
    return FALSE;
  }
}

////////////////////////////////////////////////////////////////////////////////////
gboolean gtk_source_buffer_load_file (GtkSourceBuffer *source_buffer,
				      const gchar     *filename,
				      GError         **error)
{
  GtkTextIter iter;
  gchar *buffer;
  GError *error_here = NULL;

  g_return_val_if_fail (GTK_IS_SOURCE_BUFFER (source_buffer), FALSE);
  g_return_val_if_fail (filename != NULL, FALSE);
  
  if (!g_file_get_contents (filename, &buffer, NULL, &error_here))
  {
    error_dialog (NULL, "%s\nFile %s", error_here->message, filename);
    g_propagate_error (error, error_here);
    return FALSE;
  }

  Glib::ustring buffer2 = Glib::locale_to_utf8(std::string(buffer));

  gtk_source_buffer_begin_not_undoable_action (source_buffer);
  gtk_text_buffer_set_text (GTK_TEXT_BUFFER (source_buffer), buffer2.c_str(), -1);
  gtk_source_buffer_end_not_undoable_action (source_buffer);
  gtk_text_buffer_set_modified (GTK_TEXT_BUFFER (source_buffer), FALSE);

  /* move cursor to the beginning */
  gtk_text_buffer_get_start_iter (GTK_TEXT_BUFFER (source_buffer), &iter);
  gtk_text_buffer_place_cursor (GTK_TEXT_BUFFER (source_buffer), &iter);

  {
    GtkTextIter start, end;
    char *text;
    gtk_text_buffer_get_bounds (GTK_TEXT_BUFFER (source_buffer), &start, &end);
    text = gtk_text_buffer_get_text (GTK_TEXT_BUFFER (source_buffer), &start, &end, TRUE);
    //g_assert (!strcmp (text, buffer));
    g_free (text);
  }

  g_free (buffer);
  return TRUE;
}

////////////////////////////////////////////////////////////////////////////////////
gboolean open_file (GtkSourceBuffer *buffer, const gchar *filename)
{
  GtkSourceLanguage *language = NULL;
  gchar *freeme = NULL;
  gboolean success = FALSE;
  
  if (!g_path_is_absolute (filename))
  {
    gchar *curdir = g_get_current_dir ();
    freeme = g_build_filename (curdir, filename, NULL);
    filename = freeme;
    g_free (curdir);
  }

  //	remove_all_marks (buffer);

  success = gtk_source_buffer_load_file (buffer, filename, NULL);

  if (!success)
    goto out;

  language = get_language (GTK_TEXT_BUFFER (buffer), filename);
  
  if (language == NULL)
    g_print ("No language found for file `%s'\n", filename);

  gtk_source_buffer_set_language (buffer, language);
  g_object_set_data_full (G_OBJECT (buffer),
			  "filename", g_strdup (filename),
			  (GDestroyNotify) g_free);

  if (language != NULL)
  {
    gchar **styles;
    
    styles = gtk_source_language_get_style_ids (language);
    
    if (styles == NULL)
    {
      //g_print ("No styles in language '%s'\n", gtk_source_language_get_name (language));
    }
    else
    {
      gchar **ids;
      //g_print ("Styles in in language '%s':\n", gtk_source_language_get_name (language));

      ids = styles;
      
      while (*ids != NULL)
      {
	const gchar *name;
	
	name = gtk_source_language_get_style_name (language, *ids);
	
	//g_print ("- %s (name: '%s')\n", *ids, name);

	++ids;
      }
      
      g_strfreev (styles);
    }

    //g_print("\n");
  }
 out:
  g_free (freeme);
  return success;
}
