/* OpenCP Module Player
 * copyright (c) '94-'05 Niklas Beisert <nbeisert@physik.tu-muenchen.de>
 *
 * Archive DataBase
 *
 * revision history: (please note changes here)
 *  -ss040823   Stian Skjelstad <stian@nixia.no
 *    -first release (splited out from pfilesel.c)
 *  -ss040915   Stian Skjelstad <stian@nixia.no>
 *    -Make sure that the adb_ReadHandle does not survive a fork
 *  -ss040918   Stian Skjelstad <stian@nixia.no>
 *    -Add "Scanning archive" message to the console
 *  -ss050124   Stian Skjelstad <stian@nixia.no>
 *    -fnmatch into place
 */

#include "config.h"
#include <fcntl.h>
#include <fnmatch.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>
#include "types.h"
#include "adb.h"
#include "boot/plinkman.h"
#include "stuff/poutput.h"
#include "boot/psetting.h"
#include "modlist.h"
#include "mdb.h"
#include "stuff/compat.h"

static struct adbregstruct *adbPackers = 0;
static char adbDirty;
static struct arcentry *adbData;
static uint32_t adbNum;
static uint32_t adbFindArc;
static uint32_t adbFindPos;

static void adbRegister(struct adbregstruct *r)
{
	r->next=adbPackers;
	adbPackers=r;
}

char adbInit(void)
{
	char path[PATH_MAX+1];
	int f;
	char sig[20];

	const char *regs;
	char regname[50];
	int old=0;

	regs=_lnkReadInfoReg("arcs");
	while (cfGetSpaceListEntry(regname, &regs, 49))
	{
		void *reg=_lnkGetSymbol(regname);
		if (reg)
			adbRegister((struct adbregstruct *)reg);
	}

      	adbDirty=0;
	adbData=0;
	adbNum=0;
	if ((strlen(cfConfigDir)+10)>=PATH_MAX)
		return 1; /* path is too long */
	strcpy(path, cfConfigDir);
      	strcat(path, "CPARCS.DAT");

	if ((f=open(path, O_RDONLY))<0)
		return 1;

	fprintf(stderr, "Loading %s .. ", path);

	if (read(f, sig, 20)!=20)
	{
		fprintf(stderr, "No header\n");
		close(f);
		return 1;
	}

	if (!memcmp(sig, "CPArchiveCache\x1B\x00", 16))
	{
		old=1;
		fprintf(stderr, "(Old format)  ");
	} else if (memcmp(sig, "CPArchiveCache\x1B\x01", 16))
	{
		fprintf(stderr, "Invalid header\n");
		close(f);
		return 1;
	}

	adbNum=*(uint32_t *)(sig+16);
	if (!adbNum)
	{
		fprintf(stderr, "Cache empty\n");
		close(f);
		return 1;
	}
	adbData=malloc(sizeof(struct arcentry)*adbNum);
	if (!adbData)
		return 0;
	if (old)
	{
		#define OLD_ARC_PATH_MAX 63
		struct __attribute__((packed)) oldarcentry
		{
			uint8_t flags;
			uint32_t parent;
			char name[OLD_ARC_PATH_MAX+1]; /* some stupid archives needs full path, which can be long */
			uint32_t size;
		} oldentry;
		int i;
		for (i=0;i<adbNum;i++)
		{
			if (read(f, &oldentry, sizeof(oldentry))!=(sizeof(oldentry)))
			{
				fprintf(stderr, "EOF\n");
				/* File is broken / premature EOF */
				free(adbData);
				adbData=0;
				adbNum=0;
				close(f);
				return 1;
			}
			adbData[i].flags=oldentry.flags;
			adbData[i].parent=oldentry.parent;
			strncpy(adbData[i].name, oldentry.name, sizeof(adbData[i].name));
			adbData[i].name[sizeof(adbData[i].name)-1]=0;
			adbData[i].size=oldentry.size;
		}
	} else {
		if (read(f, adbData, adbNum*sizeof(*adbData))!=(adbNum*sizeof(*adbData)))
		{
			fprintf(stderr, "EOF\n");
			/* File is broken / premature EOF */
			free(adbData);
			adbData=0;
			adbNum=0;
			close(f);
			return 1;
		}
	}
	close(f);
	fprintf(stderr, "Done\n");
	return 1;
}

void adbUpdate(void)
{
	char path[PATH_MAX+1];
	int f;
	uint32_t i, j;

	if (!adbDirty)
		return;
	adbDirty=0;

	if ((strlen(cfConfigDir)+10)>=PATH_MAX)
		return; /* path is too long */
	strcpy(path, cfConfigDir);
      	strcat(path, "CPARCS.DAT");

	if ((f=open(path, O_WRONLY|O_CREAT, S_IREAD|S_IWRITE))<0)
	{
		perror("open(CPARCS.DAT");
		return;
	}

	lseek(f, 0, SEEK_SET);
	write(f, "CPArchiveCache\x1B\x01", 16);
	write(f, &adbNum, 4);

	i=0;

	while (i<adbNum)
	{
		if (!(adbData[i].flags&ADB_DIRTY))
		{
			i++;
			continue;
		}
		for (j=i; j<adbNum; j++)
			if (adbData[j].flags&ADB_DIRTY)
				adbData[j].flags&=~ADB_DIRTY;
			else
				break;
		lseek(f, 20+i*sizeof(*adbData), SEEK_SET);
		write(f, adbData+i, (j-i)*sizeof(*adbData));

		i=j;
	}
	lseek(f, 0, SEEK_END);
	close(f);
}

void adbClose(void)
{
	adbUpdate();
	free(adbData);
	adbPackers=0;
}

int adbAdd(const struct arcentry *a)
{
	uint32_t i;

	fprintf(stderr, "adbAdd: %s %d - ", a->name, a->size);
	
	for (i=0; i<adbNum; i++)
		if (!(adbData[i].flags&ADB_USED))
			break;

	if (i==adbNum)
	{
		void *t;
		uint32_t j;
		
		adbNum+=256;
		if (!(t=realloc(adbData, adbNum*sizeof(*adbData))))
			return 0;
		adbData=(struct arcentry*)t;
		memset(adbData+i, 0, (adbNum-i)*sizeof(*adbData));
		for (j=i; j<adbNum; j++)
			adbData[j].flags|=ADB_DIRTY;
	}
	fprintf(stderr, " i=%d\n", i);
	memcpy(&adbData[i], a, sizeof(struct arcentry));
	adbData[i].flags|=ADB_USED|ADB_DIRTY;
	if (a->flags&ADB_ARC)
		adbData[i].parent=i;
	adbDirty=1;
	return 1;
}

uint32_t adbFind(const char *arcname)
{
	uint32_t i;
	size_t len=strlen(arcname)+1;
	for (i=0; i<adbNum; i++)
		if ((adbData[i].flags&(ADB_USED|ADB_ARC))==(ADB_USED|ADB_ARC))
			if (!memcmp(adbData[i].name, arcname, len))
				return i;
	return 0xFFFFFFFF;
}

#if 0
     	DIRTY NASTY CREEPY system
int adbCallArc(const char *cmd, const char *arc, const char *name, const char *dir)
{
	char cmdline[200];
	char *cp=cmdline;
	while (*cmd)
	{
		if (*cmd=='%')
		{
			if ((cmd[1]=='a')||(cmd[1]=='A'))
			{
				strcpy(cp, arc);
				cp+=strlen(cp);
			}
			if ((cmd[1]=='n')||(cmd[1]=='N'))
			{
				strcpy(cp, name);
				cp+=strlen(cp);
			}
			if ((cmd[1]=='d')||(cmd[1]=='D'))
			{
				strcpy(cp, dir);
				cp+=strlen(cp);
			}
			if (cmd[1]=='%')
				*cp++='%';
			cmd+=cmd[1]?2:1;
		}
		else
			*cp++=*cmd++;
	}
	*cp=0;
#ifdef DOS32
	return plSystem(cmdline);
#else
	return system(cmdline);
#endif
}
#endif

static unsigned char isarchive(const char *ext)
{
	struct adbregstruct *packers;
	for (packers=adbPackers; packers; packers=packers->next)
		if (!strcasecmp(ext, packers->ext))
			return 1;
	return 0;
}

char isarchivepath(const char *p)
{
	char path[PATH_MAX+1];
	char ext[NAME_MAX+1];
/*	struct stat st;*/

	strcpy(path, p);
	if (*p)
		if (path[strlen(path)-1]=='/')
			path[strlen(path)-1]=0;
	
/*	if (stat(path, &st))
		return 0;
	if (!S_ISREG(st.st_mode))
		return 0;*/

	_splitpath(path, 0, 0, 0, ext);

	return isarchive(ext);
}

static signed char adbFindNext(char *findname, uint32_t *findlen, uint32_t *adb_ref)
{
	uint32_t i;
	for (i=adbFindPos; i<adbNum; i++)
		if ((adbData[i].flags&(ADB_USED|ADB_ARC))==ADB_USED)
			if (adbData[i].parent==adbFindArc)
			{
				strcpy(findname, adbData[i].name);
				*findlen=adbData[i].size;
				adbFindPos=i+1;
				*adb_ref=i;
				return 0;
			}
	return 1;
}

static signed char adbFindFirst(const char *path, unsigned long arclen, char *findname, uint32_t *findlen, uint32_t *adb_ref)
{
	char ext[NAME_MAX+1]; /* makes life safe */
	char name[NAME_MAX+1]; /* makes life safe */
	char arcname[ARC_PATH_MAX+1]; /* makes life safe */
	uint32_t ar, i;

	_splitpath(path, 0, 0, name, ext);
	if ((strlen(name)+strlen(ext))>ARC_PATH_MAX)
		return -1;

	strcpy(arcname, name);
	strcat(arcname, ext);

	ar=adbFind(arcname);
	
	if ((ar==0xFFFFFFFF)||(arclen!=(ar!=0xffffffff?adbData[ar].size:0)))
	{
		if (ar!=0xFFFFFFFF)
			for (i=0; i<adbNum; i++)
				if (adbData[i].parent==ar)
					adbData[i].flags=(adbData[i].flags|ADB_DIRTY)&~ADB_USED;
		adbDirty=1;
		{
			struct adbregstruct *packers;
			for (packers=adbPackers; packers; packers=packers->next)
				if (!strcasecmp(ext, packers->ext))
				{
					conRestore();
					fprintf(stderr, "Scaning archive %s\n", path);
					if (!packers->Scan(path))
						return -1;
					else
						break;
				}
			if (!packers)
				return 1;
		}
		ar=adbFind(arcname);
	}
	adbFindArc=ar;
	adbFindPos=0;
	return adbFindNext(findname, findlen, adb_ref);
}




static FILE *adb_ReadHandle(struct modlistentry *entry)
{
	char drive[NAME_MAX+1];
	char dir[PATH_MAX+1];
	char name[NAME_MAX+1];
	char ext[NAME_MAX+1];
	char path[PATH_MAX+1];

	FILE *retval;

	struct arcentry *this/*, *parent*/;
	this=&adbData[entry->adb_ref];
/*	parent=&adbData[this->parent];*/
/*	fprintf(stderr, "TODO: adb_ReadHandle\n");
	fprintf(stderr, "entry->name=\"%s\"\n", entry->name);
	fprintf(stderr, "entry->fullname=\"%s\"\n", entry->fullname);
	fprintf(stderr, "entry->adb_ref=%d\n", entry->adb_ref);
	fprintf(stderr, "entry->modinfo.gen.shortname=\"%s\"\n", entry->modinfo.gen.shortname);
	fprintf(stderr, "arcEntry->flags=%d\n", this->flags);
	fprintf(stderr, "arcEntry->parent=%d\n", this->parent);
	fprintf(stderr, "arcEntry->name=\"%s\"\n", this->name);
	fprintf(stderr, "arcEntry->size=%d\n", this->size);*/

	_splitpath(entry->fullname, drive, dir, name, ext);
	if ((strlen(name)+strlen(ext))<=NAME_MAX)
		strcat(name, ext);
	_makepath(path, 0, dir, 0, 0);

	path[strlen(path)-1]=0;

/*	fprintf(stderr, "Validating %s ", path);*/
	
	if (!isarchivepath(path))
	{
/*		fprintf(stderr, "failed\n");*/
		return 0;
	}
/*	fprintf(stderr, "Ok\n");*/

	if ((strlen(cfTempDir)+strlen(name))>PATH_MAX)
	{
/*		fprintf(stderr, "But tempname is too long\n");*/
		return 0;
	}
	
	_splitpath(path, 0, 0, 0, ext);

	{
		struct adbregstruct *packers;
/*		fprintf(stderr, "Trying to find handler for %s\n", ext);*/
		for (packers=adbPackers; packers; packers=packers->next)
			if (!strcasecmp(ext, packers->ext))
				packers->Call(adbCallGet, path, this->name, name, cfTempDir);
	}
	strcpy(path, cfTempDir);
	strcat(path, name);
	strcpy(entry->fullname, path);

	if ((retval = fopen(path, "r")))
		fcntl(fileno(retval), F_SETFD, 1<<FD_CLOEXEC);
	/*unlink(path); TODO, schedule.. adplug needs file-access */
	return retval;
}

static int adb_ReadHeader(struct modlistentry *entry, char *mem, size_t *size)
{
	FILE *f=adb_ReadHandle(entry);
	int res;
	if (!f)
		return -1;
	res=fread(mem, 1, *size, f);
	*size=res;
	fclose(f);
	return 0;
}

static int adb_Read(struct modlistentry *entry, char **mem, size_t *size)
{
	FILE *f=adb_ReadHandle(entry);
	int res;
	if (!f)
		return -1;

	*mem=malloc(*size=(1024*1024*128));
	res=fread(mem, 1, *size, f);
	if (!(*size=res))
	{
		free(*mem);
		*mem=NULL;
	} else {
		*mem=realloc(*mem, *size);
	}
	
	fclose(f);
	return 0;
}

static int arcReadDir(struct modlist *ml, const char *drive, const char *path, const char *mask, unsigned long opt)
{
	char _path[PATH_MAX+1];

	struct modlistentry m;

	if (strcmp(drive, "file:"))
		return 1;
	
	strcpy(_path, path);
	if (_path[strlen(_path)-1]=='/')
		_path[strlen(_path)-1]=0;
	
/*	fprintf(stderr, "arcReadDir drive=%s path=%s mask=%s opt=%ld\n", drive, path, mask, opt);*/
	
/*	dmGetPath(path, dirref);*/
      	if (isarchivepath(_path))
      	{
/*		dmGetPath(path, dirref);
	    	path[strlen(path)-1]=0;*/

		size_t flength;
		int tf;

		uint32_t size, adb_ref;
		signed char done;

		char newfilepath[ARC_PATH_MAX+1];
	
		if ((tf=open(_path, O_RDONLY))<0)
		if (tf<0)
			return 1;
		flength=filelength(tf);
		close(tf);

	    	for (done=adbFindFirst(_path, flength, newfilepath, &size, &adb_ref); !done; done=adbFindNext(newfilepath, &size, &adb_ref))
		{
			char name[NAME_MAX+1];
			char ext[NAME_MAX+1];

			char *tmp=rindex(newfilepath, '/');
			if (!tmp)
				continue;
			if (fnmatch(mask, tmp+1, FNM_CASEFOLD))
				continue;

			_splitpath(newfilepath, 0, 0, name, ext);
			strcpy(m.name, newfilepath);

			m.drive=drive;
			_makepath(m.fullname, 0, _path, name, ext);
			if ((strlen(name)+strlen(ext))<NAME_MAX)
				strcat(name, ext);
			m.flags=MODLIST_FLAG_FILE|MODLIST_FLAG_VIRTUAL;
			m.Read=adb_Read;
			m.ReadHeader=adb_ReadHeader;
			m.ReadHandle=adb_ReadHandle;
			fs12name(m.shortname, name);
			m.fileref=mdbGetModuleReference(m.shortname, size);
			m.adb_ref=adb_ref;
#if 0
			m.modinfo.gen.size=size;
			m.modinfo.gen.modtype=mtUnRead; /* TODO mdb etc. */
#endif

			ml->append(ml, &m);
		}
		if (done==-1)
			return 0;
	}
	return 1;
}

struct mdbreaddirregstruct adbReadDirReg = {arcReadDir};
