/* OpenCP Module Player
 * copyright (c) '94-'05 Niklas Beisert <nbeisert@physik.tu-muenchen.de>
 *
 * Link manager - contains high-level routines to load and handle
 *                external DLLs
 *
 * revision history: (please note changes here)
 *  -nb980510   Niklas Beisert <nbeisert@physik.tu-muenchen.de>
 *    -first release
 *  -kb980717   Tammo Hinrichs <opencp@gmx.net>
 *    -added lnkReadInfoReg() to read out _dllinfo entries
 *  -fd981014   Felix Domke    <tmbinc@gmx.net>
 *    -increased the dllinfo-buffersize from 256 to 1024 chars in parseinfo
 *  -fd981206   Felix Domke    <tmbinc@gmx.net>
 *    -edited for new binfile
 *  -ryg981206  Fabian Giesen  <fabian@jdcs.su.nw.schule.de>
 *    -added DLL autoloader (DOS only)
 *  -ss040613   Stian Skjelstad  <stian@nixia.no>
 *    -rewritten for unix
 *  -ss040831   Stian Skjelstad  <stian@nixia.no>
 *    -made lnkDoLoad more strict, and work correct when LD_DEBUG is not defined
 *  -doj040907  Dirk Jagdmann  <doj@cubic.org>
 *    -better error message of dllextinfo is not found
 */

#include "config.h"
#include <dlfcn.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include "types.h"

#include "psetting.h"
#include "plinkman.h"


static int handlecounter;
struct dll_handle loadlist[MAXDLLLIST];
int loadlist_n;

static char reglist[1024];
static void parseinfo (char *pi, const char *key)
{
	char s[1024];
	char *dip;
	char keyok;
	char kstate;

	strcpy (s,pi);

	s[strlen(s)+1]=0;

	dip=s;
	keyok=0;
	kstate=0;
	while (*dip)
	{
		char *d2p=dip;
		while (*dip)
		{
			char t;

			d2p++;
			t=*d2p;
			if (t==';' || t==' ' || !t)
			{
				*d2p=0;
				if (!kstate)
				{
					keyok = !strcmp(dip,key);
					kstate= 1;
				} else if (keyok)
				{
					strcat(reglist,dip);
					strcat(reglist," ");
				}

				if (t==';')
					kstate=keyok=0;

				do
					dip=++d2p;
				while (*dip && (*dip==' ' || *dip==';'));
			}
		}
	}
}

char *_lnkReadInfoReg(const char *key)
{
	int i;
	char **pi;
	*reglist=0;
/*	for (i=loadlist_n-1;i>=0;i--) */
	for (i=0;i<loadlist_n;i++)
		if ((pi=dlsym(loadlist[i].handle, "dllinfo")))
			parseinfo(*pi, key);
	if (*reglist)
		reglist[strlen(reglist)-1]=0;
	return reglist;
}


/*char *lnkReadInfoReg(const char *name, const char *key)*/
char *lnkReadInfoReg(const int id, const char *key)
{
	char **pi=0;
	int i;

	*reglist=0;

	for (i=loadlist_n-1;i>=0;i--)
/*		if (strcasecmp(loadlist[i].dllname, name))*/
		if (loadlist[i].id==id)
			if ((pi=dlsym(loadlist[i].handle, "dllinfo")))
				parseinfo(*pi, key);
/*	if (pi)
		parseinfo(*pi,key);*/

	if (*reglist)
		reglist[strlen(reglist)-1]=0;

	return reglist;
}

static int lnkDoLoad(const char *file)
{
	char buffer[PATH_MAX+1];

	char *handle;

	if (loadlist_n>=MAXDLLLIST)
	{
		fprintf(stderr, "Too many open shared objects\n");
		return -1;
	}

#ifdef LD_DEBUG
	fprintf(stderr, "Request to load %s\n", file);
#endif
	if ((strlen(cfProgramDir)+strlen(file)+3)>PATH_MAX)
	{
#ifdef LD_DEBUG
		fprintf(stderr, "File path to long\n");
#endif
		return -1;
	}
	strcat(strcat(strcpy(buffer, cfProgramDir), file), LIB_SUFFIX);
#ifdef LD_DEBUG
	fprintf(stderr, "Attempting to load %s\n", buffer);
#endif

	if (!(handle=dlopen(buffer, RTLD_NOW|RTLD_GLOBAL)))
	{
		fprintf(stderr, "%s\n", dlerror());
		return -1;
	}
	strncpy(loadlist[loadlist_n].dllname, file, 15);
	loadlist[loadlist_n].handle=handle;
	loadlist[loadlist_n].id=++handlecounter;
	if (!(loadlist[loadlist_n].info=(struct linkinfostruct *)dlsym(handle, "dllextinfo")))
	{
		fprintf(stderr, "lnkDoLoad(%s): dlsym(dllextinfo): %s\n", file, dlerror());
		return -1;
	}

#if 0
	loadlist[loadlist_n].info->size=(char *)loadlist[loadlist_n].info->stop-(char *)loadlist[loadlist_n].info->start;
#else
	{
		struct stat st;
		if (stat(buffer, &st))
			st.st_size=0;
		loadlist[loadlist_n].info->size=st.st_size;
	}
#endif
	loadlist_n++;
	return handlecounter;
}

int lnkLink(const char *files)
{
	int retval=0;
	char *tmp=strdup(files);
	char *tmp2=tmp;
	char *tmp3;
	while ((tmp3=strtok(tmp2, " ")))
	{
		tmp2=NULL;
		tmp3[strlen(tmp3)]=0;
		if (strlen(tmp3))
		{
			if ((retval=lnkDoLoad(tmp3))<0)
				break;
		}
	}
	free(tmp);
	return retval;
}

void lnkFree(const int id)
{
	int i;

	if (!id)
	{
		for (i=loadlist_n-1;i>=0;i--)
		{
#ifndef HAVE_DUMA
#ifndef HAVE_EFENCE
			dlclose(loadlist[i].handle);
#endif
#endif
		}
		loadlist_n=0;
	} else {
		for (i=loadlist_n-1;i>=0;i--)
			if (loadlist[i].id==id)
			{
#ifndef HAVE_DUMA
#ifndef HAVE_EFENCE
				dlclose(loadlist[i].handle);
#endif
#endif
				memmove(&loadlist[i], &loadlist[i+1], (MAXDLLLIST-i-1)*sizeof(loadlist[0]));
				loadlist_n--;
				return;
			}
	}
}

void lnkInit(void)
{
	loadlist_n=0;
	handlecounter=0;
	memset(loadlist, 0, sizeof(loadlist));
}

void *lnkGetSymbol(const int id, const char *name)
{
	int i;
	if (!id)
	{
		void *retval;
		for (i=loadlist_n-1;i>=0;i--)
			if ((retval=dlsym(loadlist[i].handle, name)))
				return retval;
	} else
		for (i=loadlist_n-1;i>=0;i--)
			if (loadlist[i].id==id)
				return dlsym(loadlist[i].handle, name);
	return NULL;
}

int lnkCountLinks(void)
{
	return loadlist_n;
}

int lnkGetLinkInfo(struct linkinfostruct *l, int index)
{
	if (index<0)
		return 0;
	if (index>=loadlist_n)
		return 0;
	if (!loadlist[index].info)
		return 0;
	memcpy(l, loadlist[index].info, sizeof(struct linkinfostruct));
	return 1;
}

