/** OpenCP Module Player
 * copyright (c) '94-'05 Niklas Beisert <nbeisert@physik.tu-muenchen.de>
 *
 * ALSA Player device
 *
 *  revision history: (please note changes here)
 */

#include "config.h"
#define ALSA_PCM_NEW_HW_PARAMS_API
#define ALSA_PCM_NEW_SW_PARAMS_API
#include <alsa/asoundlib.h>
#include <alsa/mixer.h>
#include <alsa/pcm.h>
#include <alsa/pcm_plugin.h>
#include <string.h>
#include "types.h"
#include "boot/plinkman.h"
#include "cpiface/vol.h"
#include "dev/imsdev.h"
#include "dev/player.h"
#include "filesel/mdb.h"
#include "filesel/modlist.h"
#include "stuff/imsrtns.h"

static snd_pcm_t *alsa_pcm = NULL;
static snd_mixer_t *mixer = NULL;

struct sounddevice plrAlsa;
/*static struct deviceinfo currentcard;*/
static struct mdbreaddirregstruct readdirAlsa;
static void alsaOpenDevice(void);

static int defaultcard, defaultdevice;


/* stolen from devposs */
#define MAX_ALSA_MIXER 256
static char *playbuf;
static int buflen;
volatile static int kernpos, cachepos, bufpos;
volatile static int cachelen, kernlen; /* to make life easier */
static struct ocpvolstruct mixer_entries[MAX_ALSA_MIXER];
static int alsa_mixers_n=0;

/*  playbuf     kernpos  cachepos   bufpos      buflen
 *    |           | kernlen | cachelen |          |
 *
 *  on flush, we update all variables> *  on getbufpos we return kernpos-(1 sample) as safe point to buffer up to
 *  on getplaypos we use last known kernpos if locked, else update kernpos
 */

volatile static long playpos; /* how many samples have we done totally */
static int stereo;
static int bit16;

volatile static int busy=0;

/* stolen from devposs */
static int getbufpos(void)
{
	int retval;
	
	if (busy++)
	{
		/* can't escape if already set, and shouldn't happen */
	}

	if (kernpos==bufpos)
	{
		if (cachelen|kernlen)
		{
			retval=kernpos;
			busy--;
			return retval;
		}
	}
/*	if ((!cachelen)&&(!kernlen))
		retval=(kernpos+buflen-(0<<(bit16+stereo)))%buflen;
	else*/
		retval=(kernpos+buflen-(1<<(bit16+stereo)))%buflen;
	busy--;
	return retval;
}
/* more or less stolen from devposs */
static int getplaypos(void)
{
	int retval;

	if (busy++)
	{
	} else {
		snd_pcm_sframes_t tmp;	
		snd_pcm_status_t *alsa_pcm_status=NULL;
		int err;
		
		snd_pcm_status_alloca(&alsa_pcm_status);
		if ((err=snd_pcm_status(alsa_pcm, alsa_pcm_status))<0)
		{
			fprintf(stderr, "ALSA: snd_pcm_status() failed: %s\n", snd_strerror(-err));
		} else {
			tmp=snd_pcm_status_get_delay(alsa_pcm_status);
			tmp*=1<<(bit16+stereo);
	
			if (tmp<0) /* we ignore buffer-underruns */
				tmp=0;
		
			if (tmp>kernlen)
			{
			} else {
				kernlen=tmp;
			}
			kernpos=(cachepos-kernlen+buflen)%buflen;
		}
	}
	retval=kernpos;
	busy--;
	return retval;
}
/* more or less stolen from devposs */
static void flush(void)
{
	int result, n, odelay;
	snd_pcm_status_t *alsa_pcm_status=NULL;
	int err;

	if (busy++)
	{
		busy--;
		return;		
	}

		
	snd_pcm_status_alloca(&alsa_pcm_status);
	if ((err=snd_pcm_status(alsa_pcm, alsa_pcm_status))<0)
	{
		fprintf(stderr, "ALSA: snd_pcm_status() failed: %s\n", snd_strerror(-err));
		busy--;
		return;
	}
	odelay=snd_pcm_status_get_delay(alsa_pcm_status);
	odelay*=1<<(bit16+stereo);	
	if (odelay<0) /* we ignore buffer-underruns */
		odelay=0;

	if (odelay>kernlen)
	{
		odelay=kernlen;
	} else if ((odelay<kernlen))
	{
		kernlen=odelay;
		kernpos=(cachepos-kernlen+buflen)%buflen;
	}
	
	if (!cachelen)
	{
		busy--;
		return;
	}
	
	if (bufpos<=cachepos)
		n=buflen-cachepos;
	else
		n=bufpos-cachepos;

	/* TODO, check kernel-size
	if (n>info.bytes)
		n=info.bytes;
	*/
	
	if (n<=0)
	{
		busy--;
		return;
	}
	
	result=snd_pcm_writei(alsa_pcm, playbuf+cachepos, n>>(bit16+stereo));
	if (result<0)
	{
		if (result==-EPIPE)
		{
			fprintf(stderr, "ALSA: Machine is too slow, calling snd_pcm_prepare()\n");
			snd_pcm_prepare(alsa_pcm);
		}
		busy--;
		return;
	}
	result<<=(bit16+stereo);
	cachepos=(cachepos+result+buflen)%buflen;
	playpos+=result;
	cachelen-=result;
	kernlen+=result;

	busy--;
}
/* stolen from devposs */
static void advance(int pos)
{
	if (busy++)
	{
	}

	cachelen+=(pos-bufpos+buflen)%buflen;
	bufpos=pos;

	busy--;
}
/* stolen from devposs */
static long gettimer(void)
{
	long tmp=playpos;
	int odelay;

	if (busy++)
	{
		odelay=kernlen;
	} else {
		snd_pcm_status_t *alsa_pcm_status=NULL;
		int err;
		
		snd_pcm_status_alloca(&alsa_pcm_status);
		if ((err=snd_pcm_status(alsa_pcm, alsa_pcm_status))<0)
		{
			fprintf(stderr, "ALSA: snd_pcm_status() failed: %s\n", snd_strerror(-err));
			odelay=kernlen;
		} else {
			odelay=snd_pcm_status_get_delay(alsa_pcm_status);
			odelay*=1<<(bit16+stereo);	
			if (odelay<0) /* we ignore buffer-underruns */
				odelay=0;

			if (odelay>kernlen)
			{
				odelay=kernlen;
			} else if ((odelay<kernlen))
			{
				kernlen=odelay;
				kernpos=(cachepos-kernlen+buflen)%buflen;
			}
		}
	}
	
	tmp-=odelay;
	busy--;
	return imuldiv(tmp, 65536>>(stereo+bit16), plrRate);
}

static FILE *alsaSelectPcmOut(struct modlistentry *entry)
{
	char *t;
	int card, device;
	if (!(t=strchr(entry->name, ':')))
		return NULL;
	card=atoi(t+1);
	if (!(t=strchr(entry->name, ',')))
		return NULL;
	device=atoi(t+1);
/* #ifdef ALSA_DEBUG */
	fprintf(stderr, "ALSA: Selected card hw:%d,%d\n", card, device);
/* #endif */
	defaultcard=card;
	defaultdevice=device;
	alsaOpenDevice();
	return NULL;
}

static int list_devices_for_card(int card, struct modlist *ml, const char *drive, const char *mask, unsigned long opt)
{
	char dev[64];
	char *card_name;
	int err;
	int pcm_device=-1;
	snd_ctl_t *ctl;
	snd_pcm_info_t *pcm_info;
	
	snprintf(dev, sizeof(dev), "hw:%i", card);
	if ((err=snd_ctl_open(&ctl, dev, 0))<0)
	{
#ifdef ALSA_DEBUG
		fprintf(stderr, "ALSA: snd_ctl_open() failed: %s\n", snd_strerror(-err));
#endif
		return -1;
	}
	if ((err=snd_card_get_name(card, &card_name))!=0)
	{
#ifdef ALSA_DEBUG
		fprintf(stderr, "ALSA: snd_card_get_name() failed: %s\n", snd_strerror(-err));
#endif
		card_name="Unknown card";
	}
#ifdef ALSA_DEBUG
	fprintf(stderr, "ALSA: %s name: %s\n", dev, card_name);
#endif
	snd_pcm_info_alloca(&pcm_info);
	
	while(1)
	{
		if ((err=snd_ctl_pcm_next_device(ctl, &pcm_device))<0)
		{
#ifdef ALSA_DEBUG
			fprintf(stderr, "ALSA: snd_ctl_pcm_next_device() failed: %s\n", snd_strerror(-err));
#endif
			pcm_device=-1;
		}
		if (pcm_device<0)
			break;
		snd_pcm_info_set_device(pcm_info, pcm_device);
		snd_pcm_info_set_subdevice(pcm_info, 0);
		snd_pcm_info_set_stream(pcm_info, SND_PCM_STREAM_PLAYBACK);
		
		if ((err=snd_ctl_pcm_info(ctl, pcm_info))<0)
		{
			if (err!=-ENOENT)
				fprintf(stderr, "ALSA: snd_device_form_card(): snd_ctl_pcm_info(%d:%d) failed: %s\n", card, pcm_device, snd_strerror(-err));
			continue;
		}

		if ((defaultcard==card)&&(defaultdevice<0))
			defaultdevice=pcm_device;
#ifdef ALSA_DEBUG	
		fprintf(stderr, "ALSA: hw:%d,%d: name %s\n", card, pcm_device, snd_pcm_info_get_name(pcm_info));
#endif
		if (ml)
		{
			struct modlistentry entry;
			memset(&entry, 0, sizeof(entry));

			snprintf(entry.shortname, sizeof(entry.shortname), "hw:%d,%d.dev", card, pcm_device);
			strcpy(entry.name, entry.shortname);
			entry.drive=drive;
			snprintf(entry.fullname, sizeof(entry.fullname), "/ALSA/PCM.OUT/%s", entry.name);
			entry.flags=MODLIST_FLAG_FILE|MODLIST_FLAG_VIRTUAL;
			entry.fileref=mdbGetModuleReference(entry.name, 0);
			if (entry.fileref!=0xffffffff)
			{
				struct moduleinfostruct mi;
				mdbGetModuleInfo(&mi, entry.fileref);
				mi.flags1&=~MDB_VIRTUAL;
				mi.channels=2;
				snprintf(mi.modname, sizeof(mi.modname), snd_pcm_info_get_name(pcm_info));
				mi.modtype=mtUnRead;
				mdbWriteModuleInfo(entry.fileref, &mi);
			}
			entry.adb_ref=0xffffffff;
			entry.Read=0; entry.ReadHeader=0; entry.ReadHandle=alsaSelectPcmOut;
			ml->append(ml, &entry);
		}
	}
	snd_ctl_close(ctl);
	return 0;
}

static int list_cards(struct modlist *ml, const char *mask, const char *drive, unsigned long opt)
{
	int card=-1;

        if (snd_card_next(&card)!=0)
        {
#ifdef ALSA_DEBUG
                fprintf(stderr, "ALSA: snd_card_next() failed\n");
#endif
                return -1;
        }
	defaultcard=card;
	defaultdevice=-1;
        while (card>-1)
        {
                list_devices_for_card(card, ml, drive, mask, opt);
                if (snd_card_next(&card))
                {
#ifdef ALSA_DEBUG
			fprintf(stderr, "ALSA: snd_card_next() failed\n");
#endif
			return -1;
		}
	}
	return 0;		
}

/* plr API start */

static void SetOptions(int rate, int opt)
{
	int err;
	snd_pcm_format_t format;
	snd_pcm_hw_params_t *hwparams;
	unsigned int channels;
	/* start with setting default values, if we bail out */
	plrRate=rate;
	plrOpt=opt;

	alsaOpenDevice();

	snd_pcm_hw_params_alloca(&hwparams);

	if ((err=snd_pcm_hw_params_any(alsa_pcm, hwparams))<0)
	{
		fprintf(stderr, "ALSA: snd_pcm_hw_params_any() failed: %s\n", snd_strerror(-err));
		return;
	}

	if ((err=snd_pcm_hw_params_set_access(alsa_pcm, hwparams, SND_PCM_ACCESS_RW_INTERLEAVED))<0)
	{
		fprintf(stderr, "ALSA: snd_pcm_hw_params_set_access() failed: %s\n", snd_strerror(-err));
		return;
	}
	if (opt&PLR_16BIT)
		if (opt&PLR_SIGNEDOUT)
			format=SND_PCM_FORMAT_S16;
		else
			format=SND_PCM_FORMAT_U16;
	else
		if (opt&PLR_SIGNEDOUT)
			format=SND_PCM_FORMAT_S8;
		else
			format=SND_PCM_FORMAT_U8;
	if ((err=snd_pcm_hw_params_set_format(alsa_pcm, hwparams, format))<0)
	{
		if ((err=snd_pcm_hw_params_set_format(alsa_pcm, hwparams, SND_PCM_FORMAT_S16))>=0)
		{
			opt|=PLR_16BIT|PLR_SIGNEDOUT;
		} else if ((err=snd_pcm_hw_params_set_format(alsa_pcm, hwparams, SND_PCM_FORMAT_U16))>=0)
		{
			opt&=~(PLR_16BIT|PLR_SIGNEDOUT);
			opt|=PLR_16BIT;
		} else if ((err=snd_pcm_hw_params_set_format(alsa_pcm, hwparams, SND_PCM_FORMAT_S8))>=0)
		{
			opt&=~(PLR_16BIT|PLR_SIGNEDOUT);
			opt|=PLR_SIGNEDOUT;
		} else if ((err=snd_pcm_hw_params_set_format(alsa_pcm, hwparams, SND_PCM_FORMAT_U8))>=0)
		{
			opt&=~(PLR_16BIT|PLR_SIGNEDOUT);
		} else {
			fprintf(stderr, "ALSA: snd_pcm_hw_params_set_format() failed: %s\n", snd_strerror(-err));
			return;
		}
	}
	bit16=!!(opt&PLR_16BIT);
	if (opt&PLR_STEREO)
		channels=2;
	else
		channels=1;
	if ((err=snd_pcm_hw_params_set_channels_near(alsa_pcm, hwparams, &channels))<0)
	{
		fprintf(stderr, "ALSA: snd_pcm_hw_params_set_channels_near() failed: %s\n", snd_strerror(-err));
		return;
	}
	if (channels==1)
	{
		stereo=0;
		opt&=~PLR_STEREO;
	} else if (channels==2)
	{
		stereo=1;
		opt|=PLR_STEREO;
	} else {
		fprintf(stderr, "ALSA: snd_pcm_hw_params_set_channels_near() gave us %d channels\n", channels);
		return;
	}
	if ((err=snd_pcm_hw_params_set_rate_near(alsa_pcm, hwparams, &rate, 0))<0)
	{
		fprintf(stderr, "ALSA: snd_pcm_hw_params_set_rate_near() failed: %s\n", snd_strerror(-err));
		return;
	}
	if (rate==0)
	{
		fprintf(stderr, "ALSA: No usable samplerate available.\n");
		return;
	}
	if ((err=snd_pcm_hw_params(alsa_pcm, hwparams))<0)
	{
		fprintf(stderr, "ALSA: snd_pcm_hw_params() failed: %s\n", snd_strerror(-err));
		return;
	}
	plrRate=rate;
	plrOpt=opt;
}

static int alsaPlay(void **buf, int *len)
{
	if ((*len)<(plrRate&~3))
		*len=plrRate&~3;
	if ((*len)>(plrRate*4))
		*len=plrRate*4;
	playbuf=*buf=malloc(*len);

	memsetd(*buf, (plrOpt&PLR_SIGNEDOUT)?0:(plrOpt&PLR_16BIT)?0x80008000:0x80808080, (*len)>>2);

	buflen=*len;
	bufpos=0;
	cachepos=0;
	cachelen=0;
	playpos=0;
	kernpos=0;
	kernlen=0;

	plrGetBufPos=getbufpos;
	plrGetPlayPos=getplaypos;
	plrIdle=flush;
	plrAdvanceTo=advance;
	plrGetTimer=gettimer;

	return 1;
}

static void alsaStop(void)
{
	free(playbuf);
}

/* plr API stop */


/* driver front API start */

static void alsaOpenDevice(void)
{
	int err;
	char device[64];
	snd_mixer_elem_t *current;

	alsa_mixers_n=0;

	if (alsa_pcm)
	{
		snd_pcm_close(alsa_pcm);
		alsa_pcm=NULL;
	}

	if (mixer)
	{
		snd_mixer_close(mixer);
		mixer=NULL;
	}
	
	snprintf(device, sizeof(device), "hw:%d,%d", defaultcard, defaultdevice);
	if ((err=snd_pcm_open(&alsa_pcm, device, SND_PCM_STREAM_PLAYBACK, SND_PCM_NONBLOCK))<0)
	{
/* #if ALSA_DEBUG */
		fprintf(stderr, "ALSA: failed to open pcm device (%s): %s\n", device, snd_strerror(-err));
/* #endif */
		alsa_pcm=NULL;
		return;
	}

	snprintf(device, sizeof(device), "hw:%d", defaultcard);
	if ((err=snd_mixer_open(&mixer, 0))<0)
	{
		fprintf(stderr, "ALSA: snd_mixer_open() failed: %s\n", snd_strerror(-err));
		return;
	}
	if ((err=snd_mixer_attach(mixer, device))<0)
	{
		 fprintf(stderr, "ALSA: snd_mixer_attach() failed: %s\n", snd_strerror(-err));
		 snd_mixer_close(mixer);
		 mixer=NULL;
		 return;
	}
	if ((err=snd_mixer_selem_register(mixer, NULL, NULL))<0)
	{
		fprintf(stderr, "ALSA: snd_mixer_selem_register() failed: %s\n", snd_strerror(-err));
		snd_mixer_close(mixer);
		mixer=NULL;
		return;
	}
	if ((err=snd_mixer_load(mixer))<0)
	{
		fprintf(stderr, "ALSA: snd_mixer_load() failed: %s\n", snd_strerror(-err));
		snd_mixer_close(mixer);
		mixer=NULL;
		return;
	}
	current = snd_mixer_first_elem(mixer);
	while (current)
	{
		if (snd_mixer_selem_is_active(current) &&
			snd_mixer_selem_has_playback_volume(current) &&
			(alsa_mixers_n<MAX_ALSA_MIXER))
		{
			long int a, b;
			long min, max;
			snd_mixer_selem_get_playback_volume(current, SND_MIXER_SCHN_FRONT_LEFT, &a);
			snd_mixer_selem_get_playback_volume(current, SND_MIXER_SCHN_FRONT_RIGHT, &b);
			mixer_entries[alsa_mixers_n].val=(a+b)>>1;
			snd_mixer_selem_get_playback_volume_range(current, &min, &max);
			mixer_entries[alsa_mixers_n].min=min;
			mixer_entries[alsa_mixers_n].max=max;
			mixer_entries[alsa_mixers_n].step=1;
			mixer_entries[alsa_mixers_n].log=0;
			mixer_entries[alsa_mixers_n].name=snd_mixer_selem_get_name(current);
			alsa_mixers_n++;
		}
		current = snd_mixer_elem_next(current);
	}
}

static int volalsaGetNumVolume(void)
{
	return alsa_mixers_n;
}

static int volalsaGetVolume(struct ocpvolstruct *v, int n)
{
	if (n<alsa_mixers_n)
	{
		memcpy(v, &mixer_entries[n], sizeof(mixer_entries[n]));
		return 1;
	}
	return 0;
}

static int volalsaSetVolume(struct ocpvolstruct *v, int n)
{
	snd_mixer_elem_t *current;
	current = snd_mixer_first_elem(mixer);
	int count=0;
	while (current)
	{
		if (snd_mixer_selem_is_active(current) &&
			snd_mixer_selem_has_playback_volume(current))
		{
			if (count==n)
			{
				snd_mixer_selem_set_playback_volume(current, SND_MIXER_SCHN_FRONT_LEFT, v->val);
				snd_mixer_selem_set_playback_volume(current, SND_MIXER_SCHN_FRONT_RIGHT, v->val);
				mixer_entries[n].val=v->val;
				return 1;
			}
			count++;
		}
		current = snd_mixer_elem_next(current);
	}
	return 0;
}
static int alsaInit(const struct deviceinfo *c)
{
/*	memcpy(&currentcard, card, sizeof(struct deviceinfo));*/
	plrSetOptions=SetOptions;
	plrPlay=alsaPlay;
	plrStop=alsaStop;

	alsaOpenDevice();

	SetOptions(44100, PLR_16BIT|PLR_STEREO);
	return 1;
}

static int alsaDetect(struct deviceinfo *card)
{
	list_cards(NULL, NULL, NULL, 0);
	card->dev=&plrAlsa;
	card->port=defaultcard;
	card->port2=defaultdevice;
/*	card->irq=-1;
	card->irq2=-1;
	card->dma=-1;
	card->dma2=-1;*/
	card->subtype=-1;
	card->mem=0;
	card->chan=2;

	return defaultdevice!=-1;
}

static void alsaClose(void)
{
	plrPlay=0;
}

/* driver font API stop */

static int alsaReadDir(struct modlist *ml, const char *drive, const char *path, const char *mask, unsigned long opt)
{
	struct modlistentry entry;
	memset(&entry, 0, sizeof(entry));

	if (strcmp(drive, "setup:"))
		return 1;
	if (!strcmp(path, "/"))
	{
		strcpy(entry.shortname, "ALSA");
		strcpy(entry.name, "ALSA");
		entry.drive=drive;
		strcpy(entry.fullname, "/ALSA/");
		entry.flags=MODLIST_FLAG_DIR;
		entry.fileref=0xffffffff;
		entry.adb_ref=0xffffffff;
		entry.Read=0; entry.ReadHeader=0; entry.ReadHandle=0;
		ml->append(ml, &entry);
		return 1;
	}
	if (!strcmp(path, "/ALSA/"))
	{
		strcpy(entry.shortname, "PCM.OUT");
		strcpy(entry.name, "PCM.OUT");
		entry.drive=drive;
		strcpy(entry.fullname, "/ALSA/PCM.OUT/");
		entry.flags=MODLIST_FLAG_DIR;
		entry.fileref=0xffffffff;
		entry.adb_ref=0xffffffff;
		entry.Read=0; entry.ReadHeader=0; entry.ReadHandle=0;
		ml->append(ml, &entry);
		return 1;
	}
	if (!strcmp(path, "/ALSA/PCM.OUT/"))
	{
		list_cards(ml, drive, mask, opt);
		return 1;
	}

	return 1;
}

static void __attribute__((constructor))init(void)
{
	mdbRegisterReadDir(&readdirAlsa);
}

static void __attribute__((destructor))fini(void)
{
	mdbUnregisterReadDir(&readdirAlsa);
	if (alsa_pcm)
	{
		snd_pcm_close(alsa_pcm);
		alsa_pcm=NULL;
	}
	if (mixer)
	{
		snd_mixer_close(mixer);
		mixer=NULL;
	}
}

struct sounddevice plrAlsa={SS_PLAYER, "ALSA device driver", alsaDetect, alsaInit, alsaClose};
struct ocpvolregstruct volalsa={volalsaGetNumVolume, volalsaGetVolume, volalsaSetVolume};
static struct mdbreaddirregstruct readdirAlsa = {alsaReadDir};
char *dllinfo="driver plrAlsa; volregs volalsa";
struct linkinfostruct dllextinfo = {"devpalsa", "OpenCP Player Device: ALSA (c) 2005 Stian Skjelstad", DLLVERSION, 0};
