(* This module is a version of Str with a thread-safe interface
 * implemented using Pcre. 
 *
 * This modules processes PCRE-style regular expressions.
 *)

type regexp = Pcre.regexp;;
type split_result = Pcre.split_result = 
  | Text of string
  | Delim of string
  | Group of int * string
  | NoGroup
;;

type result;;
  (* The type of matching results *)

val regexp: string -> regexp
val regexp_case_fold: string -> regexp
val quote: string -> string
val regexp_string: string -> regexp
val regexp_string_case_fold: string -> regexp

val string_match: 
      ?groups:int -> regexp -> string -> int -> result option
val search_forward: 
      ?groups:int -> regexp -> string -> int -> (int * result)
val search_backward:
      ?groups:int -> regexp -> string -> int -> (int * result)
(*
  string_partial_match: not available
*)

(* The ~groups option is ignored *)

val matched_string : result -> string -> string
val match_beginning : result -> int
val match_end : result -> int
val matched_group : result -> int -> string -> string
val group_beginning : result -> int -> int
val group_end : result -> int -> int

val global_replace: regexp -> string -> string -> string
val replace_first: regexp -> string -> string -> string
  (* In [templ] one can refer to matched groups by the dollar notation:
   * $1 for the first group, $2 for the second etc.
   * $& is the whole match. $$ is a dollar character.
   *)

val global_substitute:
       ?groups:int -> 
       regexp -> (result -> string -> string) -> string -> string
val substitute_first:
       ?groups:int ->
       regexp -> (result -> string -> string) -> string -> string

(* replace_matched: not available *)

val split: regexp -> string -> string list
val bounded_split: regexp -> string -> int -> string list
val split_delim: regexp -> string -> string list
val bounded_split_delim: regexp -> string -> int -> string list
val full_split: regexp -> string -> split_result list
val bounded_full_split: regexp -> string -> int -> split_result list

val string_before: string -> int -> string
val string_after: string -> int -> string
val first_chars: string -> int -> string
val last_chars: string -> int -> string

