(* ocamlgsl - OCaml interface to GSL                        *)
(* Copyright (©) 2002-2005 - Olivier Andrieu                *)
(* distributed under the terms of the GPL version 2         *)

open Gsl_vectmat

type symm_ws

external _symm_alloc : int -> symm_ws
    = "ml_gsl_eigen_symm_alloc"
external _symm_free : symm_ws -> unit
    = "ml_gsl_eigen_symm_free"

external _symm : mat -> vec -> symm_ws -> unit
    = "ml_gsl_eigen_symm"

let symm ?protect a =
  let a' = Gsl_vectmat.mat_convert ?protect a in
  let (n, _) = Gsl_vectmat.dims a' in
  let v = Gsl_vector.create n in
  let ws = _symm_alloc n in
  _symm a' (`V v) ws ;
  _symm_free ws ;
  v

type symmv_ws
external _symmv_alloc_v : int -> symmv_ws
    = "ml_gsl_eigen_symmv_alloc"
external _symmv_free_v : symmv_ws -> unit
    = "ml_gsl_eigen_symmv_free"

external _symmv : mat -> vec -> mat -> symmv_ws -> unit
    = "ml_gsl_eigen_symmv"

let symmv ?protect a =
  let a' = Gsl_vectmat.mat_convert ?protect a in
  let (n, _) = Gsl_vectmat.dims a' in
  let v = Gsl_vector.create n in
  let evec = Gsl_matrix.create n n in
  let ws = _symmv_alloc_v n in
  _symmv a' (`V v) (`M evec) ws ;
  _symmv_free_v ws ;
  (v, evec)

type sort =
  | VAL_ASC
  | VAL_DESC
  | ABS_ASC
  | ABS_DESC

external symmv_sort : Gsl_vector.vector * Gsl_matrix.matrix -> sort -> unit    = "ml_gsl_eigen_symmv_sort"



(* Complex Hermitian Matrices *)
type herm_ws

external _herm_alloc : int -> herm_ws
    = "ml_gsl_eigen_herm_alloc"
external _herm_free : herm_ws -> unit
    = "ml_gsl_eigen_herm_free"

external _herm : cmat -> vec -> herm_ws -> unit
    = "ml_gsl_eigen_herm"

let herm ?protect a =
  let a' = Gsl_vectmat.cmat_convert ?protect a in
  let (n, _) = Gsl_vectmat.dims a' in
  let v = Gsl_vector.create n in
  let ws = _herm_alloc n in
  _herm a' (`V v) ws ;
  _herm_free ws ;
  v

type hermv_ws
external _hermv_alloc_v : int -> hermv_ws
    = "ml_gsl_eigen_hermv_alloc"
external _hermv_free_v : hermv_ws -> unit
    = "ml_gsl_eigen_hermv_free"

external _hermv : cmat -> vec -> cmat -> hermv_ws -> unit
    = "ml_gsl_eigen_hermv"

let hermv ?protect a =
  let a' = Gsl_vectmat.cmat_convert ?protect a in
  let (n, _) = Gsl_vectmat.dims a' in
  let v = Gsl_vector.create n in
  let evec = Gsl_matrix_complex.create n n in
  let ws = _hermv_alloc_v n in
  _hermv a' (`V v) (`CM evec) ws ;
  _hermv_free_v ws ;
  (v, evec)

external hermv_sort : Gsl_vector.vector * Gsl_matrix_complex.matrix -> sort -> unit    = "ml_gsl_eigen_hermv_sort"

