(* Example DBI program.
 * Copyright (C) 2003 Merjis Ltd.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: dbi_example.ml,v 1.6 2004/04/10 09:16:01 rwmj Exp $
 *
 * See examples/dbi/README
 *)

open Printf

(* XXX hack for SQLite:
 * Do we allow typeless (all `String) query results?
 * Default false, but true on SQLite. *)
let typeless = ref false

let msg s = print_string("* " ^ s); flush stdout

let create_tables dbh =
  msg "Creating some temporary tables... ";
  let sth = dbh#prepare_cached
	      "CREATE TEMPORARY TABLE users(userid INT4,
                                            username TEXT NOT NULL,
                                            age INT2 NOT NULL,
                                            last_login DATE,
                                            UNIQUE (userid),
                                            UNIQUE (username))" in
  sth#execute [];
  let sth = dbh#prepare_cached
	      "CREATE TEMPORARY TABLE aliases(userid INT4
	                                      REFERENCES users(userid),
	                                      alias TEXT NOT NULL)" in
  sth#execute [];
  print_endline "OK"

let insert_data dbh =
  msg "Inserting some data into the temporary tables... ";
  let sth = dbh#prepare_cached
	      "INSERT INTO users(userid, username, age) VALUES (?, ?, ?)" in
  sth#execute [`Int 1; `String "rich"; `Int 31];
  sth#execute [`Int 2; `String "mark"; `Int 30];
  sth#execute [`Int 3; `String "yoko"; `Int 30];
  sth#execute [`Int 4; `String "dan"; `Int 25];

  let sth = dbh#prepare_cached
	      "INSERT INTO aliases(userid, alias) VALUES (?, ?)" in
  sth#execute [`Int 1; `String "richard"];
  sth#execute [`Int 1; `String "richie"];
  sth#execute [`Int 3; `String "monkey"];
  sth#execute [`Int 4; `String "daniel"];
  print_endline "OK"

let select_data (dbh : Dbi.connection) =
  msg "Making some queries on the data and checking their results... ";
  let sth = dbh#prepare_cached "SELECT username FROM users
                                WHERE age >= ?
                                ORDER BY 1" in

  (* All users over 30. *)
  sth#execute [`Int 30];
  let res = sth#map (function [`String name] -> name | _ -> assert false) in
  assert (res = [ "mark"; "rich"; "yoko" ]);

  (* All users over 20. *)
  sth#execute [`Int 20];
  let res = sth#map (function [`String name] -> name | _ -> assert false) in
  assert (res = [ "dan"; "mark"; "rich"; "yoko" ]);

  (* All ages, in order. *)
  let sth = dbh#prepare_cached "SELECT age FROM users ORDER by 1" in
  sth#execute [];
  let res =
    if !typeless then
      sth#map (function [`String age] -> int_of_string age | _ -> assert false)
    else
      sth#map (function [`Int age] -> age | _ -> assert false) in
  assert (res = [ 25; 30; 30; 31 ]);

  (* All rich's aliases. *)
  let sth = dbh#prepare_cached "SELECT a.alias FROM users u, aliases a
                                WHERE u.username = 'rich' AND
                                      u.userid = a.userid
                                ORDER BY 1" in
  sth#execute [];
  let res = sth#map (function [`String alias] -> alias | _ -> assert false) in
  assert (res = ["richard"; "richie"]);
  print_endline "OK"

let do_test dbh =
  create_tables dbh;
  insert_data dbh;
  select_data dbh;
  dbh#rollback ()


(* Get the connection information from the command line.  If the
   driver or the connection strings are not given, display usage. *)
let drivers = Dbi.Factory.database_types()
let driver = ref(match drivers with
                 | d :: _ -> d
                 | _ -> "NONE")
let host = ref None
let port = ref None
let user = ref None
let passwd = ref None
let db = ref ""
let debug = ref false
let opts =
  let set v s = v := Some s in
  [("--driver", Arg.Set_string driver,
    sprintf "Database driver (default: %s; available: %s)" !driver
      (String.concat ", " drivers));
   ("--host", Arg.String (set host), "Host running the database");
   ("--port", Arg.String (set port), "Port of the database server");
   ("--user", Arg.String (set user), "Username under which to connect");
   ("--passwd", Arg.String (set passwd),
    "Password used to access the database");
   ("--debug", Arg.Set debug, "Enable database debugging output");
  ]
let usage = sprintf "%s [options] dbname" (Filename.basename Sys.argv.(0))

let () =
  let nb_anon = ref 0 in
  Arg.parse opts (fun t -> db := t; incr nb_anon) usage;
  if !nb_anon = 0 then (Arg.usage opts usage; exit 1)

let () =
  msg "Trying to connect to the database... ";
  let dbh = Dbi.Factory.connect !driver ?host:!host ?port:!port
              ?user:!user ?password:!passwd !db in
  print_endline "CONNECTED.";

  if !debug then dbh#set_debug true;

  msg "Pinging the database... ";
  let b = dbh#ping() in
  printf "the database is %s.\n" (if b then "UP" else "DOWN");
  msg(sprintf "Database type is '%s'.\n" dbh#database_type);
  if dbh#database_type = "sqlite" then typeless := true;

  do_test dbh;

  msg "Closing connection to the database... ";
  dbh#close ();
  print_endline "CLOSED.  Exiting."
