open Schemaparser;;
  (** Object oriented interface to LDAP *)

(* Just like modatts, but the mod_op is not needed *)
type op = string * string list
type op_lst = op list
type changetype = ADD | DELETE | MODIFY | MODDN | MODRDN

class type ldapentry_t =
object
  method add : op_lst -> unit
  method delete : op_lst -> unit
  method replace : op_lst -> unit
  method attributes : string list
  method get_value : string -> string list
  method exists : string -> bool
  method changes : (Ocamldap.mod_op * string * string list) list
  method changetype : changetype
  method set_changetype : changetype -> unit
  method flush_changes : unit
  method dn : string
  method set_dn : string -> unit
  method print : unit
end;;

(********************************************************************************
********************************************************************************
********************************************************************************)
(* basic stuff. ldapentry, a local object representing an object in the directory,
   ldapcon, an object representing a connection to the directory. *)

(** An ldap entry, returned from a server *)
class ldapentry :
object
  (** Add attributes and objectclasses to an entry locally *)
  method add : op_lst -> unit

  (** Delete attributes and objectclasses from an entry locally *)
  method delete : op_lst -> unit

  (** Replace attributes and objectclasses in an entry locally *)
  method replace : op_lst -> unit

  (** Get a list of attributes on the entry *)
  method attributes : string list

  (** check to see whether the attribute exists *)
  method exists : string -> bool

  (** Get a list of values for an entry *)
  method get_value : string -> string list

  (** Get a list of changes made to the entry locally in the
    form of a modattr list suitable for feeding directly to
    modify_s *)
  method changes : (Ocamldap.mod_op * string * string list) list

  (** Get the changetype. Controls whether or not local changes to the
    entry are recorded or not. *)
  method changetype : changetype

  (** Set the changetype. changetype MODIFY will record changes, all others
    will ignore them for now. This only applies to the changes list, 
    locally stored attributs will always be changed *)
  method set_changetype : changetype -> unit

  (** Get rid of recorded changes. Note, this will not change the
    the locally recorded value of any attributes on the entry, it
    will only prevent changes from being propigated to the server when
    passing this entry to update_entry *)
  method flush_changes : unit

  (** Get the distinguished name of the entry *)
  method dn : string
    
  (** Set the destinguished name of the entry *)
  method set_dn : string -> unit

  (** depricated, print a textual representation of the entry *)
  method print : unit
end

(** A connection to an ldap server. *)
class ldapcon :
  ?version:int -> 
  ?async:bool ->
  ?port:int ->
  string ->
object
  (** Add an entry to the server *)
  method add : ldapentry -> unit

  (** Bind to the server *)
  method bind :
    ?cred:string -> ?meth:Ocamldap.auth_method -> string -> unit

  method unbind : unit

  (** Delete an entry from the server *)
  method delete : string -> unit

  (** Modify an entry in the server *)
  method modify : string -> Ocamldap.modattr list -> unit

  (** change the rdn of an entry modrdn dn ?(deleteoldrdn=true) newrdn *)
  method modrdn : string -> ?deleteoldrdn:bool -> string -> unit

  (** Find entries, and return them as ldapentry objects *)
  method search :
    ?scope:Ocamldap.search_scope ->
    ?attrs:string list ->
    ?attrsonly:bool -> ?base:string -> string -> ldapentry list

  (** interrogate an ldapv3 directory server for it's schema definitions,
    raise Not_found if no schema can be located, or of bound with v2 **)
  method schema : schema

  (** grab the raw (text) schema **)
  method rawschema : ldapentry

  (** Sync changes made locally with the server *)
  method update_entry : ldapentry -> unit
end

(********************************************************************************)
(********************************************************************************)
(********************************************************************************)
(* A schema checking entry:
   An entry which validates its validity against the server's
   schema *)

type scflavor = Optimistic (* attempt to find objectclasses which make illegal
			      attributes legal, delete them if no objectclass can
			      be found *)
		| Pessimistic (* delete any illegal attributes, do not add *)

class scldapentry: Schemaparser.schema ->
object
  inherit ldapentry
  method of_entry: ldapentry -> unit
  method is_allowed : string -> bool
  method is_missing : string -> bool
  method list_allowed : Schemaparser.Oid.t list
  method list_present : Schemaparser.Oid.t list
  method list_missing : Schemaparser.Oid.t list
end;;

(********************************************************************************)
(********************************************************************************)
(********************************************************************************)
(* a high level interface for accounts, and services in the directory *)
type generator = {gen_name:string;
		  required:string list;
		  genfun:(ldapentry_t -> string list)};;

type service = {svc_name: string;
		static_attrs: (string * (string list)) list;
		generate_attrs: string list;
		depends: string list};;

type generation_error = Missing_required of string list
			| Generator_error of string

exception No_generator of string;;
exception Generation_failed of generation_error;;
exception No_service of string;;
exception Service_dep_unsatisfiable of string;;
exception Generator_dep_unsatisfiable of string * string;;

(* circular dependancies detected among the attributes of string list *)
exception Cannot_sort_dependancies of (string list);;

class ldapaccount: 
  Schemaparser.schema ->
  (string, generator) Hashtbl.t ->
  (string, service) Hashtbl.t ->
object
  inherit scldapentry
  method add_generate: string -> unit
  method delete_generate : string -> unit
  method generate : unit
  method add_service : string -> unit
  method delete_service : string -> unit
  method adapt_service : service -> service
end;;
