/* $Id: libocamldap.c,v 1.9 2003/12/24 04:09:45 eric Exp $ */
#include <assert.h>
#include <caml/alloc.h>
#include <caml/callback.h>
#include <caml/fail.h>
#include <caml/memory.h>
#include <caml/mlvalues.h>
#include <ldap.h>
#include <stdio.h>
#include <string.h>

typedef struct { value key; int data; } lookup_info;
#include "ocamldap_tags.h"
#include "ocamldap_tags.c"

#define UNWRAP_LD               LDAP *ld = (LDAP *)ldap
#define CHECK_LDAP_RESULT(x) \
    if (result != LDAP_SUCCESS) { \
        raise_ldap_exception(result); \
    } \
    CAMLreturn(Val_unit)

/*
    Translate from c constant to variant value.
    Borrowed from lablGTK.
*/
static value ml_lookup_from_c (lookup_info *table, int data)
{
    int i;
    for (i = table[0].data; i > 0; i--)
        if (table[i].data == data) return table[i].key;
    invalid_argument ("ml_lookup_from_c");
}
    
/*
    Translate from variant value to c constant.
    Borrowed from lablGTK.
*/
static int ml_lookup_to_c (lookup_info *table, value key)
{
    int first = 1, last = table[0].data, current;

    while (first < last) {
        current = (first+last)/2;
        if (table[current].key >= key) last = current;
        else first = current + 1;
    }
    if (table[first].key == key) return table[first].data;
    invalid_argument ("ml_lookup_to_c");
}

/*
    Retrieves the string error message for the given 
    error variant.
*/
value err2string(value error_code)
{
    CAMLparam1(error_code);
    int errno;
    char* errstr;

    errno = ml_lookup_to_c(ml_table_error_code, error_code);
    errstr = ldap_err2string(errno);
    CAMLreturn(copy_string(errstr));
}

/*
    Builds a dynamically allocated LDAPMod array from the given
    ocaml values.
*/
LDAPMod **mod_array_new(value mods)
{
    CAMLparam1(mods);
    int attr_count;
    int i,j;
    LDAPMod **ma = NULL;
    CAMLlocal2(a, vals);

    attr_count = Wosize_val(mods);
    ma = calloc(attr_count + 1, sizeof(LDAPMod *));
    for (i = 0; i < attr_count; i++) {
        int val_count;
        a = Field(mods, i);
        vals = Field(a, 2);
        val_count = Wosize_val(Field(a, 2));
        ma[i] = malloc(sizeof(LDAPMod));
        ma[i]->mod_op = ml_lookup_to_c(
            ml_table_mod_op,
            Field(a, 0));
        ma[i]->mod_type = String_val(Field(a, 1));
        ma[i]->mod_values = calloc(val_count + 1, sizeof(char *));
        for (j = 0; j < val_count; j++) {
            ma[i]->mod_values[j] = String_val(Field(vals, j));
        }
    }
    CAMLreturn(ma);
}

/*
    Deletes a dynamically allocated LDAPMod array.
*/
void mod_array_delete(LDAPMod **ma)
{
    LDAPMod **next = NULL;
    assert(ma);

    for (next = ma; *next != NULL; next++) {
        free((*next)->mod_values);
        free(*next);
    }
    free(ma);
}

/*
    Raises an exception encoding the given ldap error.
*/
static void raise_ldap_exception(int error_code)
{
    static value *exn = NULL;
    if (exn == NULL) {
        exn = caml_named_value("LDAP_Failure");
    }

    raise_with_arg(*exn, ml_lookup_from_c(ml_table_error_code, error_code));
}


/*
    Deletes the value with the matching dn.
*/
value ocamldap_delete_s(value ldap, value dn)
{
    CAMLparam2(ldap, dn);
    UNWRAP_LD;
    int result;

    result = ldap_delete_s(ld, String_val(dn));
    CHECK_LDAP_RESULT(result);
}

/*
    Initializes the client library and returns an ldap handle/
*/
value ocamldap_init(value host, value port, value version)
{
    LDAP * ld = ldap_init(String_val(host), Int_val(port));
    int ver = Int_val(version);
    ldap_set_option(ld, LDAP_OPT_PROTOCOL_VERSION, (void *) &ver);
    return (value) ld;
}

/*
    Adds the given value to the ldap database.
*/
value ocamldap_add_s(value ldap, value dn, value ocaml_mods)
{
    CAMLparam3(ldap, dn, ocaml_mods);
    UNWRAP_LD;
    LDAPMod **mods = NULL;
    int result;

    mods = mod_array_new(ocaml_mods);
    result = ldap_add_s(ld, String_val(dn), mods);
    mod_array_delete(mods);
    CHECK_LDAP_RESULT(result);
}

/*
    Bind to the server with the given authentication method.
*/
value ocamldap_bind_s(value ldap, value who, value cred, value method)
{
    CAMLparam4(ldap, who, cred, method);
    UNWRAP_LD;
    int method_c;
    int result;

    method_c = ml_lookup_to_c(ml_table_auth_method, method);
    result = ldap_bind_s(ld, String_val(who), String_val(cred), method_c);
    CHECK_LDAP_RESULT(result);
}

/*
    Bind using an already-established kerberos ticket.
*/

/*
kerberos support is not always avaliable, so put it in an ifdef
 */
#ifdef HAVE_KERBEROS

value ocamldap_kerberos_bind_s(value ldap, value who)
{
    CAMLparam2(ldap, who);
    UNWRAP_LD;
    int result;

    result = ldap_kerberos_bind_s(ld, String_val(who));
    CHECK_LDAP_RESULT(result);
}

value ocamldap_kerberos_bind1_s(value ldap, value who)
{
    CAMLparam2(ldap, who);
    UNWRAP_LD;
    int result;

    result = ldap_kerberos_bind1_s(ld, String_val(who));
    CHECK_LDAP_RESULT(result);
}

value ocamldap_kerberos_bind2_s(value ldap, value who)
{
    CAMLparam2(ldap, who);
    UNWRAP_LD;
    int result;

    result = ldap_kerberos_bind2_s(ld, String_val(who));
    CHECK_LDAP_RESULT(result);
}

#endif
/*
    Makes the given modifications to the given ldap entry.
*/
value ocamldap_modify_s(value ldap, value dn, value ocaml_mods)
{
    CAMLparam3(ldap, dn, ocaml_mods);
    UNWRAP_LD;
    LDAPMod **mods = NULL;
    int result;
  
    mods = mod_array_new(ocaml_mods);
    result = ldap_modify_s(ld, String_val(dn), mods);
    mod_array_delete(mods);
    CHECK_LDAP_RESULT(result);
}

/*
    Renames the given ldap entry.
*/
value ocamldap_modrdn_s(value ldap, value dn, value newrdn)
{
    CAMLparam3(ldap, dn, newrdn);
    UNWRAP_LD;
    int result;

    result = ldap_modrdn_s(ld, String_val(dn), String_val(newrdn));
    CHECK_LDAP_RESULT(result); 
}

/*
    Renames the given ldap entry, and, optionally deletes the old
    entry.
*/
value ocamldap_modrdn2_s(value ldap, value dn, value newrdn, value deleteoldrdn)
{
    CAMLparam4(ldap, dn, newrdn, deleteoldrdn);
    UNWRAP_LD;
    int result;
    int delete;

    delete = deleteoldrdn == Val_true ? 1 : 0;

    result = ldap_modrdn2_s(ld, String_val(dn), String_val(newrdn), delete);
    CHECK_LDAP_RESULT(result); 
}

/*
    Searches the ldap database for matching values.
*/
value ocamldap_search_s_native(
    value ldap, 
    value base, 
    value scope, 
    value filter,
    value attrs,
    value attrsonly)
{
    CAMLparam5(ldap, base, scope, filter, attrs);
    CAMLxparam1(attrsonly);
    UNWRAP_LD;

    int curr_attr;
    int curr_entry;
    char **desired_attrs = NULL;
    int desired_attrs_count;
    int entry_count;
    int i;
    LDAPMessage *res = NULL;
    LDAPMessage *entry = NULL;
    int ret;
    CAMLlocal5(result, caml_entry, caml_attrs, caml_attr, caml_values);

    /* copy the attrs argument into a c string array */
    desired_attrs_count = Wosize_val(attrs);
    if (desired_attrs_count) {
        desired_attrs = malloc((desired_attrs_count + 1) * sizeof(char *));
        desired_attrs[desired_attrs_count] = NULL;
        for (i = 0; i < desired_attrs_count; i++) {
            desired_attrs[i] = String_val(Field(attrs, i));
        }
    }

    /* do the search */
    ret = ldap_search_s(
            ld,
            String_val(base),
            ml_lookup_to_c(ml_table_search_scope, scope),
            String_val(filter),
            desired_attrs,
            Int_val(attrsonly),
            &res);
    if (desired_attrs != NULL) {
        free(desired_attrs);
    }
    if (ret != LDAP_SUCCESS) {
        raise_ldap_exception(ret);
    }

    /* allocate an array to hold the results */
    entry_count = ldap_count_entries(ld, res);
    if (entry_count < 1) {
        CAMLreturn(alloc(0, 0));
    }
    result = alloc(entry_count, 0);
    /*    printf("allocated for %d entries\n", entry_count); */

    /* copy the search results into the result array */
    entry = ldap_first_entry(ld, res);
    curr_entry = 0;
    while (entry != NULL) {
        char *dn = NULL;
        BerElement *ber = NULL;
        char *attr = NULL;
        int attr_count;
        
        /* create the record and add the dn */
        dn = ldap_get_dn(ld, entry);
        caml_entry = alloc(2, 0);
        Store_field(caml_entry, 0, copy_string(dn));
        ldap_memfree(dn);

        /* count the attributes.  yuck */
        attr_count = 0;
        attr = ldap_first_attribute(ld, entry, &ber);
	ldap_memfree(attr);
        while (attr) {
            attr_count++;
            attr = ldap_next_attribute(ld, entry, ber);
	    ldap_memfree(attr);
        }
        ber_free(ber, 0);
        ber = NULL;

        /* copy the attributes */
        curr_attr = 0;
        caml_attrs = alloc(attr_count, 0);
        attr = ldap_first_attribute(ld, entry, &ber);
        while (attr != NULL) {
            char **values = ldap_get_values(ld, entry, attr);
            /* cast to keep the compiler happy */
            caml_values = copy_string_array((char const **)values);
            caml_attr = alloc(2, 0);
            Store_field(caml_attr, 0, copy_string(attr));
            Store_field(caml_attr, 1, caml_values);
            Store_field(caml_attrs, curr_attr, caml_attr);

            ldap_value_free(values);
	    ldap_memfree(attr);
            attr = ldap_next_attribute(ld, entry, ber);
            curr_attr++;
        }
        Store_field(caml_entry, 1, caml_attrs);
        ber_free(ber, 0);

        /* add the entry to the result array and move to the next entry */
        Store_field(result, curr_entry, caml_entry);
        entry = ldap_next_entry(ld, entry);
        curr_entry++;
    }
    ldap_msgfree(res);
    CAMLreturn(result);
}

/*
    Arg-destructuring wrapper for native-code implementation above.
*/
value ocamldap_search_s_bytecode(value *values, int val_count)
{
    return ocamldap_search_s_native(
            values[0],
            values[1],
            values[2],
            values[3],
            values[4],
            values[5]);
}

/*
    Binds to the ldap server with the given id and password.  Binds
    anonymously if who and password are null.
*/
value ocamldap_simple_bind_s(value ldap, value who, value password)
{
    CAMLparam3(ldap, who, password);
    UNWRAP_LD;
    int result;

    result = ldap_simple_bind_s(ld, String_val(who), String_val(password));
    CHECK_LDAP_RESULT(result);
}

/*
    Releases the ldap handle and frees associated resources.
*/
value ocamldap_unbind(value ldap)
{
    CAMLparam1(ldap);
    UNWRAP_LD;
    int result;

    result = ldap_unbind(ld);
    CHECK_LDAP_RESULT(result);
}
