
  (**************************************************************************)
  (*  ocaml-dtools                                                          *)
  (*  Copyright (C) 2003-2006  The Savonet Team                             *)
  (**************************************************************************)
  (*  This program is free software; you can redistribute it and/or modify  *)
  (*  it under the terms of the GNU General Public License as published by  *)
  (*  the Free Software Foundation; either version 2 of the License, or     *)
  (*  any later version.                                                    *)
  (**************************************************************************)
  (*  Contact: stephane.gimenez@ens-lyon.fr                                 *)
  (**************************************************************************)

(* $Id: dtools.ml 2912 2007-03-02 13:51:00Z metamorph68 $ *)

(**
  ocaml-dtools
  @author Stephane Gimenez
*)

module Conf =
struct

  type t = string

  let cons t1 t2 = t1 ^ "." ^ t2

  let hash_size = 25

  let conf_table_int : (t, int ref) Hashtbl.t =
    Hashtbl.create hash_size
  let conf_table_float : (t, float ref) Hashtbl.t =
    Hashtbl.create hash_size
  let conf_table_bool : (t, bool ref) Hashtbl.t =
    Hashtbl.create hash_size
  let conf_table_string : (t, string ref) Hashtbl.t =
    Hashtbl.create hash_size
  let conf_table_list : (t, string list ref) Hashtbl.t =
    Hashtbl.create hash_size

  exception Wrong_Conf of t * string
  exception File_Wrong_Conf of t * int * string
  exception Undefined of t * string

  let solve root key =
    begin match root with
      | None -> key
      | Some k -> cons k key
    end

  let ref t table ?root ?default key =
    let k = solve root key in
    begin try Hashtbl.find table k with
      | Not_found ->
	  let default_val =
	    begin match default with
	      | None ->
		  let k = solve root key in
		  raise (Undefined (k, t))
	      | Some d -> d
	    end
	  in
	  let r = ref default_val in
	  Hashtbl.add table k r; r
    end

  let get t table ?root ?default key =
    !(ref t table ?root ?default key)

  let set table ?root key v =
    let k = solve root key in
    let r =
      begin try Hashtbl.find table k with
	| Not_found ->
	    let r = Pervasives.ref v in
	    Hashtbl.add table k r; r
      end
    in
    r := v

  let ref_int = ref "int" conf_table_int
  let ref_float = ref "float" conf_table_float
  let ref_bool = ref "bool" conf_table_bool
  let ref_string = ref "string" conf_table_string
  let ref_list = ref "list" conf_table_list

  let get_int = get "int" conf_table_int
  let get_float = get "float" conf_table_float
  let get_bool = get "bool" conf_table_bool
  let get_string = get "string" conf_table_string
  let get_list = get "list" conf_table_list

  let set_int = set conf_table_int
  let set_float = set conf_table_float
  let set_bool = set conf_table_bool
  let set_string = set conf_table_string
  let set_list = set conf_table_list

  let conf s =
    let re =
      Str.regexp
	"^[ \t]*\\([a-zA-Z]+\\)[ \t]+\\([a-zA-Z0-9._-]+\\)[ \t]*:\\(.*\\)$"
    in
    if Str.string_match re s 0
    then
      let val0 = Str.matched_group 1 s in
      let val1 = Str.matched_group 2 s in
      let val2 = Str.matched_group 3 s in
      begin match val0 with
	| "int" ->
	    let i =
	      begin try int_of_string val2  with
		| Failure "int_of_string" ->
		    raise (Wrong_Conf (s, "integer expected"))
	      end
	    in
	    set_int val1 i
	| "float" ->
	    let f =
	      begin try float_of_string val2 with
		| Failure "float_of_string" ->
		    raise (Wrong_Conf (s, "float expected"))
	      end
	    in
	    set_float val1 f
	| "string" ->
	    let s = val2 in
	    set_string val1 s
	| "list" ->
	    let l = Str.split (Str.regexp ":") val2 in
	    set_list val1 l
	| "bool" ->
	    let b =
	      begin try bool_of_string val2 with
		| Failure "bool_of_string" ->
		    raise (Wrong_Conf (s, "boolean expected"))
	      end
	    in
	    set_bool val1 b
	| _ -> raise (Wrong_Conf (s, "unknown type"))
      end
    else raise (Wrong_Conf (s, "syntax error"))

  let read_file s =
    let nb = Pervasives.ref 0 in
    let f = open_in s in
    begin try
      while true do
	nb := !nb + 1;
	let l = input_line f in
	let re = Str.regexp "^[ ]*\\(#.*\\)?$" in
	if Str.string_match re l 0
	then ()
	else
	  begin try conf l with
	    | Wrong_Conf (x,y) ->
		raise (File_Wrong_Conf (s,!nb,y))
	  end
      done
    with
      | End_of_file -> ()
    end

end

module Var =
struct
  (* Enforce some typing of the settings.
   * Registration associates a unique type to a setting name.
   * Get is unchanged, you get a specific type.
   * Set checks that the type corresponds to the declared type. *)

  type kind = Bool | Int | Float | String | List

  exception Duplicate_definition
  exception Type_error of string*kind

  let vars : (string,kind) Hashtbl.t = Hashtbl.create 100

  let register key kind =
    if Hashtbl.mem vars key then raise Duplicate_definition ;
    Hashtbl.add vars key kind

  let check k t =
    try
      let t' = Hashtbl.find vars k in
      if t'<>t then raise (Type_error (k,t'))
    with
      | Not_found -> ()

  let set kind setter = fun k v -> check k kind ; setter k v

  let set_bool   = set Bool Conf.set_bool
  let set_int    = set Int Conf.set_int
  let set_float  = set Float Conf.set_float
  let set_string = set String Conf.set_string
  let set_list   = set List Conf.set_list

  let _ =
    register "log.file" String ;
    register "daemon.pidfile" String ;
    register "log.stdout" Bool ;
    register "log.level" Int ;
    register "daemon" Bool

end

module Init =
struct

  let log = ref (fun s -> ())

  type t =
    {
      name: string;
      mutable launched: bool;
      mutable depends: t list;
      mutable triggers: t list;
      mutable mutex: Mutex.t;
      f: unit -> unit;
    }

  let make ?(name="") ?(depends=[]) ?(triggers=[]) ?(after=[]) ?(before=[]) f =
    let na =
      {
	name = name;
	launched = false;
	depends = depends;
	triggers = triggers;
	mutex = Mutex.create ();
	f = f;
      }
    in
    List.iter (fun a -> a.triggers <- na :: a.triggers) after;
    List.iter (fun a -> a.depends <- na :: a.depends) before;
    na

  let start = make ~name:"init-start" flush_all

  let stop = make ~name:"init-stop" flush_all

  let at_start ?name ?depends ?triggers ?after ?before f =
    let a = make ?name ?depends ?triggers ?after ?before f in
    start.triggers <- a :: start.triggers;
    a

  let at_stop ?name ?depends ?triggers ?after ?before f =
    let a = make ?name ?depends ?triggers ?after ?before f in
    stop.depends <- a :: stop.depends;
    a

  let trace = Conf.ref_bool ~default:false "init.trace"
  let concurrent = Conf.ref_bool ~default:false "init.concurrent"
  let pidfile = Conf.ref_string ~default:"" "daemon.pidfile"
  let daemon = Conf.ref_bool ~default:false "daemon"
  let catch_exn = Conf.ref_bool ~default:true "daemon.catch_exn"

  let rec exec a =
    let log =
      if !trace then
	begin fun s ->
	  let id = Thread.id (Thread.self ()) in
	  Printf.printf "init(%i):%-35s@%s\n%!" id a.name s
	end
      else
	begin fun s -> () end
    in
    let rec exec a =
      log "called";
      Mutex.lock a.mutex;
      try
        if not a.launched
        then begin
          a.launched <- true;
	  log "start";
	  log "start-depends";
	  mult_exec a.depends;
	  log "stop-depends";
	  log "start-atom";
	  a.f ();
	  log "stop-atom";
	  log "start-triggers";
	  mult_exec a.triggers;
	  log "stop-triggers";
	  log "stop";
	end;
        Mutex.unlock a.mutex;
        log "return"
      with e -> Mutex.unlock a.mutex; raise e
    and mult_exec l =
      begin match !concurrent with
	| true ->
	    let ask x =
	      log (Printf.sprintf "exec %s" x.name);
	      Thread.create exec x
	    in
	    let threads = List.map ask l in
	    List.iter Thread.join threads
	| false ->
	    List.iter exec l
      end
    in
    exec a

  let rec wait_signal () =
    begin try
      ignore (Thread.wait_signal [Sys.sigterm; Sys.sigint]);
    with
      | Unix.Unix_error (Unix.EINTR,_,_) -> ()
      | Sys_error("Thread.wait_signal: Interrupted system call") ->
          wait_signal ()
    end

  exception StartError of exn
  exception StopError of exn

  let main f () =
    begin try exec start with e -> raise (StartError e) end;
    let quit pid = Unix.kill pid Sys.sigterm in
    let thread pid =
      begin try f (); quit pid with
	| e ->
	    let se = Printexc.to_string e in
	    Printf.eprintf
	      "init: exception encountered during main phase:\n  %s\n%!" se;
	    !log (Printf.sprintf "exception: %s" se);
	    if !catch_exn then quit pid else raise e
      end
    in
    ignore (Thread.create thread (Unix.getpid ()));
    wait_signal ();
    begin try exec stop with e -> raise (StopError e) end

  let catch f clean =
    begin try
	f ()
      with
	| StartError (e) ->
	    Printf.eprintf
	      "init: exception encountered during start phase:\n  %s\n%!"
	      (Printexc.to_string e);
	    clean ()
	| StopError (e) ->
	    Printf.eprintf
	      "init: exception encountered during stop phase:\n  %s\n%!"
	      (Printexc.to_string e);
	    clean ()
    end

  let daemonize f =
    close_in stdin;
    flush_all ();
    begin match Unix.fork () with
      | 0 ->
	  let _ = Unix.setsid () in
	  begin match !pidfile with
	    | "" -> ()
	    | filename ->
		let f = open_out filename in
		let pid = Unix.getpid () in
		output_string f (string_of_int pid);
		output_char f '\n';
		close_out f
	  end;
	  catch f (fun () -> Unix.unlink !pidfile; exit (-1));
	  Unix.unlink !pidfile;
          exit 0
      | _ -> exit 0
    end

  let exit_when_root () =
    let security s = Printf.eprintf "init: security exit, %s\n%!" s in
    if Unix.geteuid () = 0 then
      begin security "root euid."; exit (-1) end;
    if Unix.getegid () = 0 then
      begin security "root egid."; exit (-1) end

  let init ?(prohibit_root=false) f =
    if prohibit_root then exit_when_root ();
    let signal_h i = () in
    Sys.set_signal Sys.sigterm (Sys.Signal_handle signal_h);
    Sys.set_signal Sys.sigint (Sys.Signal_handle signal_h);
    if !daemon
    then daemonize (main f)
    else catch (main f) (fun () -> exit (-1))

end

module Log =
struct

  let log_ch = ref None

  (* Mutex to avoid interlacing logs *)
  let log_mutex = Mutex.create ()

  let log_default = Conf.ref_int ~default:3 "log.level"
  let log_timestamps_raw = Conf.ref_bool ~default:false "log.timestamps.raw"
  let log_file = Conf.ref_string ~default:"" "log.file"
  let log_append = Conf.ref_bool ~default:true "log.append"
  let log_perms = Conf.ref_int ~default:0o600 "log.perms"
  let log_stdout = Conf.ref_bool ~default:false "log.stdout"

  let logged label level =
    let conf_level =
      Conf.get_int ~default:!log_default ~root:"log.level" label
    in
    conf_level >= level

  let print str =
    begin match !log_ch with
      | None -> ()
      | Some ch ->
          let time =
            if !log_timestamps_raw then
              Printf.sprintf "%f" (Unix.gettimeofday ())
            else
              let date = Unix.localtime (Unix.time ()) in
              Printf.sprintf "%d/%02d/%02d %02d:%02d:%02d"
                (date.Unix.tm_year+1900)
                (date.Unix.tm_mon+1)
                date.Unix.tm_mday
                date.Unix.tm_hour
                date.Unix.tm_min
                date.Unix.tm_sec
          in
	  Mutex.lock log_mutex;
	  try
            Printf.fprintf ch "%s %s\n%!" time str;
	    if !log_stdout then
	      Printf.printf "%s %s\n%!" time str;
	    Mutex.unlock log_mutex
          with e -> Mutex.unlock log_mutex; raise e
    end

  let log ?(label="default") level s =
    if logged label level
    then
      print ("[" ^ label ^ ":" ^ (string_of_int level) ^ "] " ^ s)

  let logl ?(label="default") level sl =
    if logged label level
    then
      let s = Lazy.force sl in
      print ("[" ^ label ^ ":" ^ (string_of_int level) ^ "] " ^ s)

  let init () =
    let opts =
      if !log_append
      then  [Open_wronly; Open_creat; Open_append;]
      else  [Open_wronly; Open_creat; Open_trunc;]
    in
    (* Re-open log file on SIGUSR1 -- for logrotate *)
    Sys.set_signal Sys.sigusr1
      (Sys.Signal_handle
        begin fun _ ->
          begin match !log_ch with
            | None -> ()
            | Some ch -> log_ch := None; close_out ch;
          end;
          if !log_file <> "" then
            log_ch := Some (open_out_gen opts !log_perms !log_file)
	end
      );
    if !log_file <> "" then
      begin
	log_ch := Some (open_out_gen opts !log_perms !log_file);
	print ">>> LOG START";
	Init.log := log ~label:"init" 0
      end

  let start = Init.make ~name:"init-log-start" ~before:[Init.start] init

  let close () =
    begin match !log_ch with
      | None -> ()
      | Some ch ->
	  Init.log := (fun s -> ());
	  print ">>> LOG END";
	  log_ch := None;
	  close_out ch;
    end

  let stop = Init.make ~name:"init-log-stop" ~after:[Init.stop] close

  let f fmt = Printf.sprintf fmt


end

module Options =
struct

  let list =
    [
      "--config-file", Arg.String (Conf.read_file),
	"configuration file";
      "-C"           , Arg.String (Conf.read_file),
	"configuration file";
      "--config"     , Arg.String (Conf.conf),
	"configuration assignation";
      "-c"           , Arg.String (Conf.conf),
	"configuration assignation";
      "-d"           , Arg.Unit (fun () -> Conf.set_bool "daemon" true),
	"daemon";
      "--stdout"     , Arg.Unit (fun () -> Conf.set_bool "log.stdout" true),
        "logging also to stdout";
      "-v"           , Arg.Unit (fun () -> Conf.set_bool "log.stdout" true),
        "logging also to stdout";
      "--logfile"    , Arg.String (Conf.set_string "log.file"),
        "log file";
      "-l"           , Arg.String (Conf.set_string "log.file"),
        "log file";
    ]

end
