#include "macro-dock.hpp"
#include "macro.hpp"
#include "utility.hpp"

#include <QLayout>

namespace advss {

MacroDock::MacroDock(Macro *m, QWidget *parent, const QString &runButtonText,
		     const QString &pauseButtonText,
		     const QString &unpauseButtonText)
	: OBSDock(parent),
	  _pauseButtonText(pauseButtonText),
	  _unpauseButtonText(unpauseButtonText),
	  _run(new QPushButton(runButtonText)),
	  _pauseToggle(new QPushButton()),
	  _macro(m)
{
	if (_macro) {
		setWindowTitle(QString::fromStdString(_macro->Name()));
		_run->setVisible(_macro->DockHasRunButton());
		_pauseToggle->setVisible(_macro->DockHasPauseButton());
	} else {
		setWindowTitle("<deleted macro>");
	}

	setFeatures(DockWidgetClosable | DockWidgetMovable |
		    DockWidgetFloatable);

	QWidget::connect(_run, SIGNAL(clicked()), this, SLOT(RunClicked()));
	QWidget::connect(_pauseToggle, SIGNAL(clicked()), this,
			 SLOT(PauseToggleClicked()));

	QVBoxLayout *layout = new QVBoxLayout;
	layout->addWidget(_run);
	layout->addWidget(_pauseToggle);

	QWidget::connect(&_timer, SIGNAL(timeout()), this,
			 SLOT(UpdatePauseText()));
	_timer.start(1000);

	UpdatePauseText();

	// QFrame wrapper is necessary to avoid dock being partially
	// transparent
	QFrame *wrapper = new QFrame;
	wrapper->setFrameShape(QFrame::StyledPanel);
	wrapper->setFrameShadow(QFrame::Sunken);
	wrapper->setLayout(layout);
	setWidget(wrapper);

	setFloating(true);
	hide();
}

void MacroDock::SetName(const QString &name)
{
	setWindowTitle(name);
}

void MacroDock::ShowRunButton(bool value)
{
	_run->setVisible(value);
}

void MacroDock::SetRunButtonText(const QString &text)
{
	_run->setText(text);
}

void MacroDock::ShowPauseButton(bool value)
{
	_pauseToggle->setVisible(value);
}

void MacroDock::SetPauseButtonText(const QString &text)
{
	_pauseButtonText = text;
	UpdatePauseText();
}

void MacroDock::SetUnpauseButtonText(const QString &text)
{
	_unpauseButtonText = text;
	UpdatePauseText();
}

void MacroDock::RunClicked()
{
	if (!_macro) {
		return;
	}

	auto ret = _macro->PerformActions();
	if (!ret) {
		QString err =
			obs_module_text("AdvSceneSwitcher.macroTab.runFail");
		DisplayMessage(err.arg(QString::fromStdString(_macro->Name())));
	}
}

void MacroDock::PauseToggleClicked()
{
	if (!_macro) {
		return;
	}

	_macro->SetPaused(!_macro->Paused());
	UpdatePauseText();
}

void MacroDock::UpdatePauseText()
{
	if (!_macro) {
		return;
	}

	_pauseToggle->setText(_macro->Paused() ? _unpauseButtonText
					       : _pauseButtonText);
}

} // namespace advss
