
import os

import gobject
import gst

gobject.threads_init()

class WakePipe():
    def __init__(self):
        (self._read_fd, self._write_fd) = os.pipe()

    def sleep(self):
        return os.read(self._read_fd, 1)

    def wake(self):
        os.write(self._write_fd, '\0')

class RunPipeline(object):
    def __init__(self, pipeline):
        self.pipeline = pipeline
        self.errors = []
        self.eos = False
        self.wake = WakePipe()

        bus = pipeline.get_bus()
        bus.set_sync_handler(self._message)

    def _message(self, bus, msg):
        if msg.type == gst.MESSAGE_ERROR:
            self.errors.append(msg)
            self.wake.wake()
        elif msg.type == gst.MESSAGE_EOS:
            self.eos = True
            self.wake.wake()

        return gst.BUS_PASS

    def start(self):
        self.pipeline.set_state(gst.STATE_PLAYING)

    def finish(self):
        ok = self.pipeline.set_state(gst.STATE_NULL)

        if ok != gst.STATE_CHANGE_SUCCESS:
            raise RuntimeError()

        state = self.pipeline.get_state()
        assert state[1] == gst.STATE_NULL, state

    def wait(self):
        self.wake.sleep()

    def __iter__(self):
        try:
            self.start()

            while not (self.eos or self.errors):
                self.wake.sleep()

                yield
        finally:
            self.finish()

        if self.errors:
            raise RuntimeError(self.errors[0])

    def run(self):
        for _ in self:
            pass

class Reader(RunPipeline):
    def __init__(self, pipeline, appsink, cb):
        RunPipeline.__init__(self, pipeline)
        self.appsink = appsink
        self.cb = cb

        appsink.props.emit_signals = True
        appsink.props.max_buffers = 10
        appsink.props.sync = False
        appsink.connect('new-buffer', self._new_buffer)

    def _new_buffer(self, _appsink):
        buf = self.appsink.emit('pull-buffer')
        self.wake.wake()

        if buf is None:
            return

        v = self._process_buffer(buf)

        if v is not None:
            self.cb(v)

def _run_appsrc_pipeline(pipeline, appsrc, get_chunk):
    position = [0]

    def need_data(src, length):
        try:
            (delta_p, a) = get_chunk(position[0], length)
        except IndexError:
            src.emit('end-of-stream')
            return

        if len(a) == 0:
            src.emit('end-of-stream')
            return

        src.emit('push-buffer', gst.Buffer(a.data))
        position[0] += delta_p

    appsrc.props.emit_signals = True
    appsrc.connect('need-data', need_data)
    run = RunPipeline(pipeline)
    run.run()
