(* file kernel/pascal/p/xlong.p: Pascal extensible integer definitions
 +-----------------------------------------------------------------------+
 |  Copyright 2005, Michel Quercia (michel.quercia@prepas.org)           |
 |                                                                       |
 |  This file is part of Numerix. Numerix is free software; you can      |
 |  redistribute it and/or modify it under the terms of the GNU Lesser   |
 |  General Public License as published by the Free Software Foundation; |
 |  either version 2.1 of the License, or (at your option) any later     |
 |  version.                                                             |
 |                                                                       |
 |  The Numerix Library is distributed in the hope that it will be       |
 |  useful, but WITHOUT ANY WARRANTY; without even the implied warranty  |
 |  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU  |
 |  Lesser General Public License for more details.                      |
 |                                                                       |
 |  You should have received a copy of the GNU Lesser General Public     |
 |  License along with the GNU MP Library; see the file COPYING. If not, |
 |  write to the Free Software Foundation, Inc., 59 Temple Place -       |
 |  Suite 330, Boston, MA 02111-1307, USA.                               |
 +-----------------------------------------------------------------------+
 |                                                                       |
 |                 Dfinitions pour l'interface Pascal                   |
 |                                                                       |
 +-----------------------------------------------------------------------*)

unit _name_;
interface

(* Numerix Pascal-library, public interface

   Pascal extensible integers are defined as pointers to structures containing
   two fields of long type ...

   THIS IS NOT THE REALITY

   ... but it will be enough to make the Pascal compiler happy without having
   to include all the low-level Numerix headers.

   Please,

     1. never make an xint of your own by directing some pointer towards
        a double-long structure. Use only the functions described below
        to build xint values.

     2. never use an xint created for one mode (clong/dlong/slong) with
        functions dealing with another mode.

     3. never use pointer arithmetic on xints.

*)
type _xint = record lmax : longint; hd : longint; end; (* white lie *)
type xint  = ^_xint;

(* creation/destruction *)
function xnew : xint;
procedure xfree(var x : xint);

(* addition/soustraction *)
procedure add  (var c:xint; a:xint; b:xint   );
procedure sub  (var c:xint; a:xint; b:xint   );
procedure add_1(var c:xint; a:xint; b:longint);
procedure sub_1(var c:xint; a:xint; b:longint);

function f_add  (a:xint; b:xint   ):xint;
function f_sub  (a:xint; b:xint   ):xint;
function f_add_1(a:xint; b:longint):xint;
function f_sub_1(a:xint; b:longint):xint;

(* comparaison *)
function cmp    (a:xint; b:xint   ):longint; cdecl; external 'numerix-c' name 'xx(cmp)';
function cmp_1  (a:xint; b:longint):longint; cdecl; external 'numerix-c' name 'xx(cmp_1)';
function sgn    (a:xint           ):longint;
function eq     (a:xint; b:xint   ):boolean;
function neq    (a:xint; b:xint   ):boolean;
function inf    (a:xint; b:xint   ):boolean;
function infeq  (a:xint; b:xint   ):boolean;
function sup    (a:xint; b:xint   ):boolean;
function supeq  (a:xint; b:xint   ):boolean;
function eq_1   (a:xint; b:longint):boolean;
function neq_1  (a:xint; b:longint):boolean;
function inf_1  (a:xint; b:longint):boolean;
function infeq_1(a:xint; b:longint):boolean;
function sup_1  (a:xint; b:longint):boolean;
function supeq_1(a:xint; b:longint):boolean;

(* conversions *)
procedure copy_int(var b:xint; a:longint); cdecl; external 'numerix-c' name 'xx(copy_int)';
function of_int(a:longint):xint;

function int_of  (a:xint           ):longint; cdecl; external 'numerix-c' name 'xx(int_of)';
function nbits   (a:xint           ):longint; cdecl; external 'numerix-c' name 'xx(nbits)';
function lowbits (a:xint           ):longint; cdecl; external 'numerix-c' name 'xx(lowbits)';
function highbits(a:xint           ):longint; cdecl; external 'numerix-c' name 'xx(highbits)';
function nth_word(a:xint; n:longint):longint; cdecl; external 'numerix-c' name 'xx(nth_word)';
function nth_bit (a:xint; n:longint):boolean; cdecl; external 'numerix-c' name 'xx(nth_bit)';

(* copie *)
procedure copy (var b:xint; a:xint);
procedure abs  (var b:xint; a:xint);
procedure neg  (var b:xint; a:xint);

function f_copy(a:xint):xint;
function f_abs (a:xint):xint;
function f_neg (a:xint):xint;

(* division *)
procedure quomod   (var c,d:xint; a:xint; b:xint   );     
procedure quo      (var c  :xint; a:xint; b:xint   );
procedure modulo   (var d  :xint; a:xint; b:xint   );
procedure quomod_1 (var c  :xint; a:xint; b:longint);
procedure quo_1    (var c  :xint; a:xint; b:longint);
function  mod_1                  (a:xint; b:longint):longint;
procedure gquomod  (var c,d:xint; a:xint; b:xint;    mode:longint);
procedure gquo     (var c  :xint; a:xint; b:xint;    mode:longint);
procedure gmod     (var d  :xint; a:xint; b:xint;    mode:longint);
function  gquomod_1(var c  :xint; a:xint; b:longint; mode:longint):longint;
procedure gquo_1   (var c  :xint; a:xint; b:longint; mode:longint);
function  gmod_1                 (a:xint; b:longint; mode:longint):longint;

function f_quo     (a:xint; b:xint   ):xint;
function f_mod     (a:xint; b:xint   ):xint;
function f_quo_1   (a:xint; b:longint):xint;
function f_mod_1   (a:xint; b:longint):longint;
function f_gquo    (a:xint; b:xint;    mode:longint):xint;
function f_gmod    (a:xint; b:xint;    mode:longint):xint;
function f_gquo_1  (a:xint; b:longint; mode:longint):xint;
function f_gmod_1  (a:xint; b:longint; mode:longint):longint;

(* affichage hexadcimal *)
procedure dump(a:xint); cdecl; external 'numerix-c' name 'xx(dump)';

(* factorielle *)
procedure fact(var a:xint; n:longint); cdecl; external 'numerix-c' name 'xx(fact)';
function f_fact(n:longint):xint;

(* pgcd *)
procedure gcd   (var d:xint;         a,b:xint);
procedure gcd_ex(var d,u,v:xint;     a,b:xint);
procedure cfrac (var d,u,v,p,q:xint; a,b:xint);

function f_gcd(a,b:xint):xint;

(* multiplication *)
procedure mul  (var c:xint; a:xint; b:xint   ); cdecl; external 'numerix-c' name 'xx(mul)';
procedure mul_1(var c:xint; a:xint; b:longint); cdecl; external 'numerix-c' name 'xx(mul_1)';
procedure sqr  (var b:xint; a:xint);            cdecl; external 'numerix-c' name 'xx(sqr)';

function f_mul  (a:xint; b:xint   ):xint;
function f_mul_1(a:xint; b:longint):xint;
function f_sqr  (a:xint):xint;           

(* exponentiation *)
procedure pow    (var b:xint; a:xint;    p:longint); cdecl; external 'numerix-c' name 'xx(pow)';
procedure pow_1  (var b:xint; a:longint; p:longint); cdecl; external 'numerix-c' name 'xx(pow_1)';
procedure powmod (var d:xint; a:xint;    b:xint; c:xint);             
procedure gpowmod(var d:xint; a:xint;    b:xint; c:xint; mode:longint);

function f_pow    (a:xint;    p:longint):xint;
function f_pow_1  (a:longint; p:longint):xint;
function f_powmod (a:xint;    b:xint; c:xint):xint;              
function f_gpowmod(a:xint;    b:xint; c:xint; mode:longint):xint;

(* nombres alatoires *)
procedure random_init(n:longint); cdecl; external 'numerix-c' name 'xx(random_init)';

procedure nrandom (var a:xint; n:longint);
procedure zrandom (var a:xint; n:longint);
procedure nrandom1(var a:xint; n:longint);
procedure zrandom1(var a:xint; n:longint);

function f_nrandom (n:longint):xint;
function f_zrandom (n:longint):xint;
function f_nrandom1(n:longint):xint;
function f_zrandom1(n:longint):xint;

(* racines *)
procedure sqrt (var b:xint; a:xint);                         
procedure gsqrt(var b:xint; a:xint;            mode:longint);
procedure root (var b:xint; a:xint; p:longint);              
procedure groot(var b:xint; a:xint; p:longint; mode:longint);

function f_sqrt (a:xint                ):xint;
function f_gsqrt(a:xint; mode:  longint):xint;
function f_root (a:xint; p:     longint):xint;
function f_groot(a:xint; p,mode:longint):xint;

(* dcalages *)
procedure shiftl(var b:xint; a:xint; n:longint);
procedure shiftr(var b:xint; a:xint; n:longint);

function f_shl(a:xint; n:longint):xint;
function f_shr(a:xint; n:longint):xint;

procedure split(var b,c:xint; a:xint; n:longint);      cdecl; external 'numerix-c' name 'xx(split)';
procedure join(var c:xint; a:xint; b:xint; n:longint); cdecl; external 'numerix-c' name 'xx(join)';

function f_join(a:xint; b:xint; n:longint):xint;

(* conversion en chaine *)
procedure copy_string(var a:xint; s:pchar); cdecl; external 'numerix-c' name 'xx(copy_string)';
function of_string(s:pchar):xint;

function string_of (a:xint):ansistring;
function hstring_of(a:xint):ansistring;
function ostring_of(a:xint):ansistring;
function bstring_of(a:xint):ansistring;

(* chronomtrage *)
procedure chrono(msg:pchar); cdecl; external 'numerix-c' name 'chrono';

(*----------------------------- end of interface ---------------------------- *)
implementation

type pxint  = ^xint;

(* creation/destruction *)
function  xnew : xint;
begin
   xnew := NIL;
end;

procedure free(x : pointer); cdecl; external 'c';
procedure xfree(var x: xint);
begin
   if x <> NIL then begin free(x); x := NIL; end;
end;

(* addition/soustraction *)
procedure private_add  (var x:xint; a,b:xint; s:longint); cdecl; external 'numerix-c' name 'xx(private_add)';
procedure private_add_1(var x:xint; a:xint; b,s:longint); cdecl; external 'numerix-c' name 'xx(private_add_1)';

procedure add  (var c:xint; a:xint; b:xint   ); begin private_add  (c,a,b,0); end;
procedure sub  (var c:xint; a:xint; b:xint   ); begin private_add  (c,a,b,1); end;
procedure add_1(var c:xint; a:xint; b:longint); begin private_add_1(c,a,b,0); end;
procedure sub_1(var c:xint; a:xint; b:longint); begin private_add_1(c,a,b,1); end;

function f_add  (a:xint; b:xint   ):xint; var c:xint; begin c:=xnew(); private_add  (c,a,b,0); f_add  :=c; end;
function f_sub  (a:xint; b:xint   ):xint; var c:xint; begin c:=xnew(); private_add  (c,a,b,1); f_sub  :=c; end;
function f_add_1(a:xint; b:longint):xint; var c:xint; begin c:=xnew(); private_add_1(c,a,b,0); f_add_1:=c; end;
function f_sub_1(a:xint; b:longint):xint; var c:xint; begin c:=xnew(); private_add_1(c,a,b,1); f_sub_1:=c; end;

(* comparaison *)
function sgn    (a:xint           ):longint; begin sgn     := cmp_1(a,0);      end;
function eq     (a:xint; b:xint   ):boolean; begin eq      := cmp  (a,b) =  0; end;
function neq    (a:xint; b:xint   ):boolean; begin neq     := cmp  (a,b) <> 0; end;
function inf    (a:xint; b:xint   ):boolean; begin inf     := cmp  (a,b) <  0; end;
function infeq  (a:xint; b:xint   ):boolean; begin infeq   := cmp  (a,b) <= 0; end;
function sup    (a:xint; b:xint   ):boolean; begin sup     := cmp  (a,b) >  0; end;
function supeq  (a:xint; b:xint   ):boolean; begin supeq   := cmp  (a,b) >= 0; end;
function eq_1   (a:xint; b:longint):boolean; begin eq_1    := cmp_1(a,b) =  0; end;
function neq_1  (a:xint; b:longint):boolean; begin neq_1   := cmp_1(a,b) <> 0; end;
function inf_1  (a:xint; b:longint):boolean; begin inf_1   := cmp_1(a,b) <  0; end;
function infeq_1(a:xint; b:longint):boolean; begin infeq_1 := cmp_1(a,b) <= 0; end;
function sup_1  (a:xint; b:longint):boolean; begin sup_1   := cmp_1(a,b) >  0; end;
function supeq_1(a:xint; b:longint):boolean; begin supeq_1 := cmp_1(a,b) >= 0; end;

(* conversions *)
function of_int(a:longint):xint; var b:xint; begin b:=xnew(); copy_int(b,a); of_int:=b; end;

(* copie *)
procedure private_copy(var b:xint; a:xint; s:longint); cdecl; external 'numerix-c' name 'xx(private_copy)';

procedure copy (var b:xint; a:xint); begin private_copy(b,a,0); end;
procedure abs  (var b:xint; a:xint); begin private_copy(b,a,1); end;
procedure neg  (var b:xint; a:xint); begin private_copy(b,a,2); end;

function f_copy(a:xint):xint; var b:xint; begin b:=xnew(); private_copy(b,a,0); f_copy:=b; end;
function f_abs (a:xint):xint; var b:xint; begin b:=xnew(); private_copy(b,a,1); f_abs :=b; end;
function f_neg (a:xint):xint; var b:xint; begin b:=xnew(); private_copy(b,a,2); f_neg :=b; end;

(* division *)
procedure private_quomod  (c,d:pxint; a,b:xint; mode:longint);         cdecl; external 'numerix-c' name 'xx(private_quomod)';
function  private_quomod_1(c:  pxint; a:xint; b,mode:longint):longint; cdecl; external 'numerix-c' name 'xx(private_quomod_1)';

procedure quomod   (var c,d:xint; a:xint; b:xint);                          begin            private_quomod  (@c, @d, a,b, 0 or 0);            end;
procedure quo      (var c  :xint; a:xint; b:xint);                          begin            private_quomod  (@c,NIL, a,b, 4 or 0);            end;
procedure modulo   (var d  :xint; a:xint; b:xint);                          begin            private_quomod  (NIL,@d, a,b, 8 or 0);            end;
procedure quomod_1 (var c  :xint; a:xint; b:longint);                       begin            private_quomod_1(@c,     a,b, 8 or 0);            end;
procedure quo_1    (var c  :xint; a:xint; b:longint);                       begin            private_quomod_1(@c,     a,b, 4 or 0);            end;
function  mod_1                  (a:xint; b:longint):longint;               begin mod_1    :=private_quomod_1(NIL,    a,b, 8 or 0);            end;
procedure gquomod  (var c,d:xint; a:xint; b:xint;    mode:longint);         begin            private_quomod  (@c,@d,  a,b, 0 or (mode and 3)); end;
procedure gquo     (var c:xint;   a:xint; b:xint;    mode:longint);         begin            private_quomod  (@c, NIL,a,b, 4 or (mode and 3)); end;
procedure gmod     (var d:xint;   a:xint; b:xint;    mode:longint);         begin            private_quomod  (NIL,@d, a,b, 8 or (mode and 3)); end;
function  gquomod_1(var c:xint;   a:xint; b:longint; mode:longint):longint; begin gquomod_1:=private_quomod_1(@c,     a,b, 8 or (mode and 3)); end;
procedure gquo_1   (var c:xint;   a:xint; b:longint; mode:longint);         begin            private_quomod_1(@c,     a,b, 4 or (mode and 3)); end;
function  gmod_1                 (a:xint; b:longint; mode:longint):longint; begin gmod_1   :=private_quomod_1(NIL,    a,b, 8 or (mode and 3)); end;

function f_quo   (a:xint; b:xint                 ):xint;    var c:xint;  begin c:=xnew(); private_quomod  (@c,NIL,a,b, 4 or 0);            f_quo   :=c; end;
function f_mod   (a:xint; b:xint                 ):xint;    var d:xint;  begin d:=xnew(); private_quomod  (NIL,@d,a,b, 8 or 0);            f_mod   :=d; end;
function f_quo_1 (a:xint; b:longint              ):xint;    var c:xint;  begin c:=xnew(); private_quomod_1(@c,    a,b, 4 or 0);            f_quo_1 :=c; end;
function f_mod_1 (a:xint; b:longint              ):longint;              begin f_mod_1:=  private_quomod_1(NIL,   a,b, 8 or 0);                         end;
function f_gquo  (a:xint; b:xint;    mode:longint):xint;     var c:xint; begin c:=xnew(); private_quomod  (@c,NIL,a,b, 4 or (mode and 3)); f_gquo  :=c; end;
function f_gmod  (a:xint; b:xint;    mode:longint):xint;     var d:xint; begin d:=xnew(); private_quomod  (NIL,@d,a,b, 8 or (mode and 3)); f_gmod  :=d; end;
function f_gquo_1(a:xint; b:longint; mode:longint):xint;     var c:xint; begin c:=xnew(); private_quomod_1(@c,    a,b, 4 or (mode and 3)); f_gquo_1:=c; end;
function f_gmod_1(a:xint; b:longint; mode:longint):longint;              begin f_gmod_1:= private_quomod_1(NIL,   a,b, 8 or (mode and 3));              end;

(* factorielle *)
function f_fact(n:longint):xint; var a:xint; begin a:=xnew(); fact(a,n); f_fact:=a; end;

(* pgcd *)
procedure private_cfrac(d,u,v,p,q:pxint; a,b:xint; mode:longint); cdecl; external 'numerix-c' name 'xx(private_cfrac)';

procedure gcd   (var d:xint;         a,b:xint); begin private_cfrac(@d,NIL,NIL,NIL,NIL,a,b,1); end;
procedure gcd_ex(var d,u,v:xint;     a,b:xint); begin private_cfrac(@d,@u, @v, NIL,NIL,a,b,0); end;
procedure cfrac (var d,u,v,p,q:xint; a,b:xint); begin private_cfrac(@d,@u, @v, @p, @q, a,b,0); end;

function f_gcd(a,b:xint):xint; var d:xint; begin d:=xnew(); private_cfrac(@d,NIL,NIL,NIL,NIL,a,b,1); f_gcd:=d; end;

(* multiplication *)
function f_mul  (a:xint; b:xint   ):xint; var c:xint; begin c:=xnew(); mul  (c,a,b); f_mul  :=c; end;
function f_mul_1(a:xint; b:longint):xint; var c:xint; begin c:=xnew(); mul_1(c,a,b); f_mul_1:=c; end;
function f_sqr  (a:xint):xint;            var c:xint; begin c:=xnew(); sqr  (c,a);   f_sqr  :=c; end;

(* exponentiation *)
function f_pow  (a:xint;    p:longint):xint; var b:xint; begin b:=xnew(); pow  (b,a,p); f_pow  :=b; end;
function f_pow_1(a:longint; p:longint):xint; var b:xint; begin b:=xnew(); pow_1(b,a,p); f_pow_1:=b; end;

procedure private_powmod(var d:xint; a:xint; b:xint; c:xint; mode:longint); cdecl; external 'numerix-c' name 'xx(private_powmod)';

procedure powmod (var d:xint; a:xint; b:xint; c:xint);               begin private_powmod(d,a,b,c,0);   end;
procedure gpowmod(var d:xint; a:xint; b:xint; c:xint; mode:longint); begin private_powmod(d,a,b,c,mode); end;

function f_powmod (a:xint; b:xint; c:xint):xint;               var d:xint; begin d:=xnew(); private_powmod(d,a,b,c,0);    f_powmod :=d; end;
function f_gpowmod(a:xint; b:xint; c:xint; mode:longint):xint; var d:xint; begin d:=xnew(); private_powmod(d,a,b,c,mode); f_gpowmod:=d; end;

(* nombres alatoires *)
procedure private_random(var a:xint; n:longint; mode:longint); cdecl; external 'numerix-c' name 'xx(private_random)';

procedure nrandom (var a:xint; n:longint); begin private_random(a,n,0); end;
procedure zrandom (var a:xint; n:longint); begin private_random(a,n,1); end;
procedure nrandom1(var a:xint; n:longint); begin private_random(a,n,2); end;
procedure zrandom1(var a:xint; n:longint); begin private_random(a,n,3); end;

function f_nrandom (n:longint):xint; var a:xint; begin a:=xnew(); private_random(a,n,0); f_nrandom :=a; end;
function f_zrandom (n:longint):xint; var a:xint; begin a:=xnew(); private_random(a,n,1); f_zrandom :=a; end;
function f_nrandom1(n:longint):xint; var a:xint; begin a:=xnew(); private_random(a,n,2); f_nrandom1:=a; end;
function f_zrandom1(n:longint):xint; var a:xint; begin a:=xnew(); private_random(a,n,3); f_zrandom1:=a; end;

(* racines *)
procedure private_sqrt(var b:xint; a:xint;            mode:longint); cdecl; external 'numerix-c' name 'xx(private_sqrt)';
procedure private_root(var b:xint; a:xint; p:longint; mode:longint); cdecl; external 'numerix-c' name 'xx(private_root)';

procedure sqrt (var b:xint; a:xint);                          begin private_sqrt(b, a,0);      end;
procedure gsqrt(var b:xint; a:xint;            mode:longint); begin private_sqrt(b, a,mode);   end;
procedure root (var b:xint; a:xint; p:longint);               begin private_root(b, a,p,0);    end;
procedure groot(var b:xint; a:xint; p:longint; mode:longint); begin private_root(b, a,p,mode); end;

function f_sqrt (a:xint                ):xint; var b:xint; begin b:=xnew(); private_sqrt(b,a,0);     f_sqrt :=b; end;
function f_gsqrt(a:xint; mode:  longint):xint; var b:xint; begin b:=xnew(); private_sqrt(b,a,mode);  f_gsqrt:=b; end;
function f_root (a:xint; p:     longint):xint; var b:xint; begin b:=xnew(); private_root(b,a,p,0);   f_root :=b; end;
function f_groot(a:xint; p,mode:longint):xint; var b:xint; begin b:=xnew(); private_root(b,a,p,mode);f_groot:=b; end;

(* dcalages *)
procedure private_shift(var b:xint; a:xint; n:longint; sens:longint); cdecl; external 'numerix-c' name 'xx(private_shift)';

procedure shiftl(var b:xint; a:xint; n:longint); begin if n >= 0 then private_shift(b,a, n,1) else private_shift(b,a,-n,0); end;
procedure shiftr(var b:xint; a:xint; n:longint); begin if n < 0  then private_shift(b,a,-n,1) else private_shift(b,a, n,0); end;

function f_shl(a:xint; n:longint):xint; var b:xint; begin b:=xnew(); if n >= 0 then private_shift(b,a, n,1) else private_shift(b,a,-n,0); f_shl:=b; end;
function f_shr(a:xint; n:longint):xint; var b:xint; begin b:=xnew(); if n < 0  then private_shift(b,a,-n,1) else private_shift(b,a, n,0); f_shr:=b; end;

function f_join(a:xint; b:xint; n:longint):xint; var c:xint; begin c:=xnew(); join(c,a,b,n); f_join:=c; end;

(* conversion en chaine *)
function of_string(s:pchar):xint; var a:xint; begin a:=xnew(); copy_string(a,s); of_string:=a; end;

function xstring_of (x:xint):pchar; cdecl; external 'numerix-c' name 'xx(string_of)';
function xhstring_of(x:xint):pchar; cdecl; external 'numerix-c' name 'xx(hstring_of)';
function xostring_of(x:xint):pchar; cdecl; external 'numerix-c' name 'xx(ostring_of)';
function xbstring_of(x:xint):pchar; cdecl; external 'numerix-c' name 'xx(bstring_of)';

function string_of (a:xint):ansistring; var p:pchar; s:ansistring; begin p:=xstring_of (a); s:=p; free(p);  string_of:=s; end;
function hstring_of(a:xint):ansistring; var p:pchar; s:ansistring; begin p:=xhstring_of(a); s:=p; free(p); hstring_of:=s; end;
function ostring_of(a:xint):ansistring; var p:pchar; s:ansistring; begin p:=xostring_of(a); s:=p; free(p); ostring_of:=s; end;
function bstring_of(a:xint):ansistring; var p:pchar; s:ansistring; begin p:=xbstring_of(a); s:=p; free(p); bstring_of:=s; end;

end.  
