// file kernel/n/h/numerix.h: natural integer definitions
/*-----------------------------------------------------------------------+
 |  Copyright 2005, Michel Quercia (michel.quercia@prepas.org)           |
 |                                                                       |
 |  This file is part of Numerix. Numerix is free software; you can      |
 |  redistribute it and/or modify it under the terms of the GNU Lesser   |
 |  General Public License as published by the Free Software Foundation; |
 |  either version 2.1 of the License, or (at your option) any later     |
 |  version.                                                             |
 |                                                                       |
 |  The Numerix Library is distributed in the hope that it will be       |
 |  useful, but WITHOUT ANY WARRANTY; without even the implied warranty  |
 |  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU  |
 |  Lesser General Public License for more details.                      |
 |                                                                       |
 |  You should have received a copy of the GNU Lesser General Public     |
 |  License along with the GNU MP Library; see the file COPYING. If not, |
 |  write to the Free Software Foundation, Inc., 59 Temple Place -       |
 |  Suite 330, Boston, MA 02111-1307, USA.                               |
 +-----------------------------------------------------------------------+
 |                                                                       |
 |                    Dfinitions pour la couche n                       |
 |                                                                       |
 +-----------------------------------------------------------------------*/

/* fichier de configuration spcifique  la machine cible */
#include "../../config.h"

/* options de compilation :

   bits_32     -> architecture 32 bits
   bits_64     -> architecture 64 bits

   use_clong   -> arithmtique en C, un chiffre = 1/2 mot
   use_dlong   -> arithmtique en C, un chiffre = 1 mot
   use_slong   -> arithmtique en C et assembleur, un chiffre = 1 mot
*/

                        /* +-------------------------+
                           |  Contrle de cohrence  |
                           +-------------------------+ */

#if   defined(bits_32) + defined(bits_64) < 1
#error  "missing bits_xx definition"
#elif defined(bits_32) + defined(bits_64) > 1
#error  "multiple bits_xx definitions"
#endif

/* mode = clong ou bien dlong ou bien slong */
#if   defined(use_clong) + defined(use_dlong) + defined(use_slong) < 1
#error  "missing use_xlong definition"
#elif defined(use_clong) + defined(use_dlong) + defined(use_slong) > 1
#error  "multiple use_xlong definitions"
#endif /* use_xlong */

#ifdef  bits_32

                        /* +------------------------+
                           |  Architecture 32 bits  |
                           +------------------------+ */

#define SIGN_m   0x80000000L            /* masque bit de signe     */
#define LONG_m   0x7fffffffL            /* masque mot de longueur  */

#ifdef  use_clong
#define chiffres_per_long 2             /* conversion de longueur  */
#define HW       16                     /* bits par chiffre        */
#define hw       4                      /* log_2(HW)               */
#define BASE     0x10000L               /* base de numration      */
#define LMAX     0x20000000L            /* longueur maximale       */
#define chiffre  unsigned short         /* un chiffre              */
#define ndouble  unsigned long          /* entier double non sign */
#define zdouble           long          /* entier double sign     */
#endif  /* use_clong */

#ifdef  use_dlong
#define chiffres_per_long 1             /* conversion de longueur  */
#define HW       32                     /* bits par chiffre        */
#define hw       5                      /* log_2(HW)               */
#define BASE     0x100000000LL          /* base de numration      */
#define LMAX     0x10000000L            /* longueur maximale       */
#define chiffre  unsigned long          /* un chiffre              */
#define ndouble  unsigned long long     /* entier double non sign */
#define zdouble           long long     /* entier double sign     */
#endif  /* use_dlong */

#ifdef  use_slong
#define chiffres_per_long 1             /* conversion de longueur  */
#define HW       32                     /* bits par chiffre        */
#define hw       5                      /* log_2(HW)               */
#define BASE     0x100000000LL          /* base de numration      */
#define LMAX     0x10000000L            /* longueur maximale       */
#define chiffre  unsigned long          /* un chiffre              */
#define ndouble  unsigned long long     /* entier double non sign */
#define zdouble           long long     /* entier double sign     */
#endif  /* use_slong */

#endif  /* bits_32 */

#ifdef  bits_64
                        /* +------------------------+
                           |  Architecture 64 bits  |
                           +------------------------+ */

#define SIGN_m   0x8000000000000000L    /* masque bit de signe     */
#define LONG_m   0x7fffffffffffffffL    /* masque mot de longueur  */

#ifdef  use_clong
#define chiffres_per_long 2             /* conversion de longueur  */
#define HW       32                     /* bits par chiffre        */
#define hw       5                      /* log_2(HW)               */
#define BASE     0x100000000L           /* base de numration      */
#define LMAX     0x1000000000000000L    /* longueur maximale       */
#define chiffre  unsigned int           /* un chiffre              */
#define ndouble  unsigned long          /* entier double non sign */
#define zdouble           long          /* entier double sign     */
#else   /* use_clong */
#error  "use_dlong and use_slong modes are not supported on 64 bits processors"
#endif  /* use_clong */


#endif /* bits_64 */


                   /* +-----------------------------------+
                      |  prfixe des fonctions publiques  |
                      +-----------------------------------+ */

#if defined(use_clong)
#define xn(nom)  cn_##nom
#define xx(nom)  cx_##nom
#elif defined(use_dlong)
#define xn(nom)  dn_##nom
#define xx(nom)  dx_##nom
#elif defined(use_slong)
#define xn(nom)  sn_##nom
#define xx(nom)  sx_##nom
#endif

/* include standards */
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

                        /* +-----------------------+
                           |  Liste des fonctions  |
                           +-----------------------+ */


/*
  dump.h doit tre lu en premier pour avoir le prototype de xn(internal_error)
  Les autres fichiers peuvent tre chargs dans un ordre quelconque, ici
  ils sont regroups par catgorie d'oprations
*/
#include "dump.h"

/* allocation mmoire */
#include "alloc.h"

/* addition/soustraction */
#include "add.h"

/* multiplication/carr */
#include "mul_n2.h"
#include "karatsuba.h"
#include "toom.h"

/* oprations modulo BASE^n +/- 1 */
#include "mmod.h"
#include "smod.h"

/* transformation de Fourier */
#include "fft.h"
#include "fftmul.h"

/* division/racine carre */
#include "div_n2.h"
#include "sqrt_n2.h"
#include "burnikel.h"
#include "zimmermann.h"
#include "moddiv.h"
#include "karp.h"

/* exponentiation */
#include "pow.h"
#include "powmod.h"
#include "montgomery.h"

/* pgcd */
#include "gcd.h"

/* divers */
#include "cmp.h"
#include "shift.h"
#include "random.h"


       /* +-----------------------------------------------------------+
          |  Aiguillage des oprations selon la taille des oprandes  |
          +-----------------------------------------------------------+ */

extern inline void xn(mul)(chiffre *a, long la, chiffre *b, long lb, chiffre *c) {
    (lb <= karamul_lim) ? xn(mul_n2)(a,la,b,lb,c) : xn(fftmul)(a,la,b,lb,c);
}

extern inline void xn(sqr)(chiffre *a, long la, chiffre *c) {
    (la <= karasqr_lim) ?  xn(sqr_n2)(a,la,c) : xn(fftsqr)(a,la,c);
}

extern inline void xn(div)(chiffre *a, long la, chiffre *b, long lb, chiffre *c) {
    (lb <= burnidiv_lim) ? xn(div_n2) (a,la,b,lb,c) : xn(karpdiv)(a,la,b,lb,c,1);
}

