(* +------------------------------------------------------------------------+
   |                                                                        |
   |              Recherche du k-me nombre de Hamming                      |
   |                                                                        |
   |            Mthode volumtrique avec balayage oblique                  |
   |                                                                        |
   +------------------------------------------------------------------------+ *)

(* M. Quercia, le 06/02/2001 *)
(* Rmq : algorithme en O(n^1/3*ln(n)^2, il y a un autre programme
   en O(ln(n)^3 (estim) disponible sur http://pauillac.inria.fr/~quercia *)


open Printf
open Numerix

(* +------------------------------------------------------------------------+
   |                      Recherche du k-me entier de Hamming              |
   +------------------------------------------------------------------------+ *)

module Main(E:Int_type) = struct
  module Infixes = Infixes(E)
  open E
  open Infixes
  let zero  = of_int 0
  let un    = of_int 1
  let deux  = of_int 2
  let trois = of_int 3
  let quatre= of_int 4
  let six   = of_int 6
  let dix   = of_int 10
  let douze = of_int 12

  let debug = ref false

             (* +-------------------------------------+
                |  Dcodage d un nombre avec suffixe  |
                +-------------------------------------+ *)

  let of_string(s) =

    if String.length s =! 0 then zero
    else begin
      let a,negatif = match s.[0] with
	| '+' -> 1,false
	| '-' -> 1,true
	| _   -> 0,false
      in
      let b,exp = match s.[String.length(s) -! 1] with
	| 'K' -> 2,3
	| 'M' -> 2,6
	| 'G' -> 2,9
	| 'T' -> 2,12
	| 'P' -> 2,15
	| 'E' -> 2,18
	| 'Z' -> 2,21
	| 'Y' -> 2,24
	| _   -> 1,0
      in
      
      let n = ref(zero) in
      for i=a to String.length(s) -! b do
	n := !n*dix + of_int(Char.code(s.[i]) -! 48)
      done;
      for i=1 to exp do n := !n*dix done;
      if negatif then zero - !n else !n
    end


                       (* +--------------------------+
                          |  Coefficients de Bezout  |
                          +--------------------------+ *)

  (* retourne u,v tels que u*a + v*b = pgcd(a,b) *)
  let rec bezout a b =
    if a = zero then (zero,un)
    else let (q,r) = b/%a       in
	 let (v,u) = bezout r a in (u-q*v,v)


          (* +----------------------------------------------------+
             |  Calcul de surf(a,b,n) par dcoupage de triangles  |
             +----------------------------------------------------+ *)

  (* prcalcule les lments constants pour chaque tape de la rcursion :
     a, b, q = b/a, qa <= b ?, q=1, q=2 ?  *)
  type surf2cas = Pos_1 | Pos_2 | Pos | Neg_2 | Neg

  let rec prpare_surf2 a b =
    if a=zero then [] else
      let q,r   = b/%a in
      if a < r+r
      then
	let q = q+un and r = a-r                in
        let cas = if q=deux then Neg_2 else Neg in
	(cas,a,b,q)::prpare_surf2 r a
      else
	let cas = if q=un then Pos_1 else if q=deux then Pos_2 else Pos in
	(cas,a,b,q)::prpare_surf2 r a

  (* Retourne 2*surf(a,b,n) pour viter des divisions continuelles par 2 *)
  let surf2 a b =

    let liste = prpare_surf2 a b in

    (* addition des lments variables *)
    let rec add liste s sgn n =
      if n < zero then s else match liste with
	| [] -> failwith "cas impossible"
	| (Pos_1,a,b,_)::suite ->
	    let x = n/b + un in
	    let s = if sgn then s - x*(x+un) else s + x*(x+un) in
	    add suite s sgn (n - a*x)
	| (Pos_2,a,b,_)::suite ->
	    let y,z = n/%b in
	    let u = y + un in
	    if z+z < b
	    then
	      let uv = u*u      in
	      let s = if sgn then s - uv - uv else s + uv + uv in
	      add suite s sgn (n - a*(u+y))
	    else
	      let uv = u*(u+un) in
	      let s = if sgn then s - uv - uv else s + uv + uv in
	      add suite s sgn (n - a*(u+u))
	| (Pos,a,b,q)::suite ->
	    let x   = (q*n)/b in
	    let y,z = x/%q    in
	    let s = if sgn then s - (y+un)*(x+z+deux) else s + (y+un)*(x+z+deux) in
	    add suite s sgn (n - a*(x+un))
	| (Neg_2,a,b,_)::suite ->
	    let y,z = n/%b in
	    let u = y + un in
	    if z+z < b
	    then
	      let uv = u*u in
	      let s = if sgn then s - uv - uv else s + uv + uv in
	      add suite s (not sgn) (a*(y+y) - n - un)
	    else
	      let uv = u*(u+un)   in
	      let s = if sgn then s - uv - uv else s + uv + uv in
	      add suite s (not sgn) (a*(u+y) - n - un)
	| (Neg,a,b,q)::suite ->
	    let x   = (q*n)/b in
	    let y,z = x/%q    in
	    let s = if sgn then s - (y+un)*(x+z+deux) else s + (y+un)*(x+z+deux) in
	    add suite s (not sgn) (a*x - n - un)

    (* retourne la fonction n -> surf(a,b,n) *)
    in function n -> add liste zero false n

      (* +---------------------------------------------------+
         |  Calcul de vol(a,b,c,n) par addition de surfaces  |
         +---------------------------------------------------+ *)

  let vol2_func a b c n =
    let surf = surf2 a b in
    let rec add s n = if n < zero then s else add (s+surf(n)) (n-c) in
    (add zero n)/deux

  let vol2_surplace a b c = let liste = prpare_surf2 a b in

    (* pr-alloue les variables de travail *)
    let x = make_ref(of_int 0)
    and y = make_ref(of_int 0)
    and z = make_ref(of_int 0)
    and n = make_ref(of_int 0)
    and p = make_ref(of_int 0)
    and s = make_ref(of_int 0)
    and t = make_ref(of_int 0)
    and u = make_ref(of_int 0)
    and sg = ref(true) 
    and l = ref(liste)    in

    (* retourne la fonction n -> vol(a,b,c,n) *)
    function nn ->
      copy_in p nn;
      copy_in s zero;
      while ~~p >= zero do
	l := liste;
	copy_in n ~~p;
	sg := false;
	while ~~n >= zero do match !l with
	  | [] -> failwith "cas impossible"
	  | (Pos_1,a,b,_)::suite ->
	      quomod_in x z ~~n   b;
	      add_in    x    ~~x  un;
	      add_in    y    ~~x  un;
	      mul_in    z    ~~x  ~~y;
	      (if !sg then sub_in else add_in) s ~~s ~~z;
	      mul_in    t    a    ~~x;
	      sub_in    n    ~~n  ~~t;
	      l := suite
	  | (Pos_2,a,b,_)::suite ->
	      quomod_in y z ~~n  b;
	      add_in    u    ~~y  un;
	      add_in    z    ~~z  ~~z;
	      if ~~z < b then begin
		mul_in  t    ~~u  ~~u;
		add_in  t    ~~t  ~~t;
		add_in  x    ~~u  ~~y;
	      end else begin
		add_in  x    ~~u  un;
		mul_in  t    ~~u  ~~x;
		add_in  t    ~~t  ~~t;
		add_in  x    ~~u  ~~u;
	      end;
	      (if !sg then sub_in else add_in) s ~~s ~~t;
	      mul_in    t    a    ~~x;
	      sub_in    n    ~~n  ~~t;
	      l := suite
	  | (Pos,a,b,q)::suite ->
	      mul_in    y    q    ~~n;
	      quomod_in x z ~~y  b;
	      quomod_in y z ~~x  q;
	      add_in    y    ~~y  un;
	      add_in    z    ~~z  ~~x;
	      add_in    z    ~~z  deux;
	      mul_in    t    ~~y  ~~z;
	      (if !sg then sub_in else add_in) s ~~s ~~t;
	      add_in    x    ~~x  un;
	      mul_in    t    a    ~~x;
	      sub_in    n    ~~n  ~~t;
	      l := suite
	  | (Neg_2,a,b,_)::suite ->
	      quomod_in y z ~~n  b;
	      add_in    u    ~~y  un;
	      add_in    z    ~~z  ~~z;
	      if ~~z < b then begin
		mul_in  t    ~~u  ~~u;
		add_in  t    ~~t  ~~t;
		add_in  x    ~~y  ~~y;
	      end else begin
		add_in  x    ~~u  un;
		mul_in  t    ~~u  ~~x;
		add_in  t    ~~t  ~~t;
		add_in  x    ~~u  ~~y;
	      end;
	      (if !sg then sub_in else add_in) s ~~s ~~t;
	      mul_in    t    a    ~~x;
	      sub_in    n    ~~t  ~~n;
	      sub_in    n    ~~n  un;  
	      sg := not !sg;
	      l := suite
	  | (Neg,a,b,q)::suite ->
	      mul_in    y    q    ~~n;
	      quomod_in x z ~~y  b;
	      quomod_in y z ~~x  q;
	      add_in    y    ~~y  un;
	      add_in    z    ~~z  ~~x;
	      add_in    z    ~~z  deux;
	      mul_in    t    ~~y  ~~z;
	      (if !sg then sub_in else add_in) s ~~s ~~t;
	      mul_in    t    a    ~~x;
	      sub_in    t    ~~t  un;
	      sub_in    n    ~~t  ~~n;
	      sg := not !sg;
	      l := suite
	done;
	sub_in p ~~p c
      done;
      ~~s/deux

                         (* +-----------------------+
                            |  Calcul de mu(a,b,n)  |
                            +-----------------------+ *)

  (* pour viter les calculs en rationnels on ne calcule que les numrateurs *)
  (* rho a b c n -> dnominateur = 12abc                                     *)
  (* mu a b    n -> dnominateur = 12a                                       *) 
  let rho a b c =
    let a' = a-un and b' = b-un and c' = c-un  in
    let s = a' + b' + c' + deux                in
    let p = s*(s-deux) + a'*b' + a'*c' + b'*c' in
    let d = a*b*c                              in
    function n -> six*(n+un)*(n+s) + p

  let rec mu a b =
    if a = un then (function n -> zero)
    else let r = b%a           in
	 let rho1 = rho un a r in
	 let mu1  = mu r a     in
	 function n -> let k = n%a
	 in douze*a*(k/r + un) - (rho1(k) + a*mu1(k))/r


                   (* +-----------------------------------+
                      |  Calcul de surf(a,b,c,n) avec mu  |
                      +-----------------------------------+ *)

  let surf3 a b c =
    let (a',_) = bezout a b
    and	(b',_) = bezout b c
    and	(c',_) = bezout c a  in
    let rho1   = rho a b c
    and	mua    = mu a (b*c')
    and	mub    = mu b (c*a')
    and	muc    = mu c (a*b')
    and	ab     = a*b
    and ac     = a*c
    and bc     = b*c
    and d      = douze*a*b*c in
    function n -> (rho1(n) + bc*mua(n*c') + ac*mub(n*a') + ab*muc(n*b'))/d


                       (* +---------------------------+
                          |  Rsoud ax + by + cz = n  |
                          +---------------------------+ *)

  let search3 a b c =
    let surf = surf3 a b c in
    function n ->
      let rec dicho u v =
	if u = v then u
	else let z = (u+v+un)/deux in
	if surf(n-c*z) = zero then dicho u (z-un) else dicho z v in
      let z = dicho zero (n/c) in
      let (a',b') = bezout a b and n1 = n-c*z in ((n1*a')%b, (n1*b')%a, z)


                  (* +------------------------------------+
                     |  Dcoupage rcursif de ttradres  |
                     +------------------------------------+ *)

  type noeud =
    | Nul
    | Clos of E.t * E.t                 (* q,n     *)
    | Sum  of E.t * E.t * E.t * E.t     (* a,b,c,n *)
    | Sym  of noeud * E.t * E.t * E.t   (* 1/2-tt, 2a,c,n *)
    | Asym of noeud * noeud * noeud     (* clos, tt+, tt- *)

  (* arbre servant  calculer vol(a,b,c,n)
   
     Nul          -> v = 0 (cas n < 0)
     Clos(q,n)    -> formule close
     Sum(a,b,c,n) -> additionner les tranches
     Sym(t,u,v,n) -> 2*vol(t) + surf2(u,v,n)
     Asym(t,u,v)  -> vol(t) + vol(u) - vol(v)
  *)

  let pmax = of_int 9 (* profondeur maxi de dcoupage *)

  (* Dveloppe l'arbre de calcul jusqu' la profondeur p en minimisant      *)
  (* le nombre de tranches  additionner (compte une tranche supplmentaire *)
  (* pour chaque tape de division). Retourne l arbre et son cot.          *)

  let rec prpare p a b c n =

    (* classe les paramtres *)
    if b < a      then prpare p b a c n
    else if c < b then prpare p a c b n

    (* volume nul *)
    else if n < zero then (Nul,zero)

    (* autres cas : h = hauteur = cot si on additionne par tranches *)
    else
      let h = n/c in
      let suite,cout =

	(* cas symtrique *)
	if a=b then begin
	  let d = deux*a in
	  let (t,cout) = prpare p a d c (n-a) in
	  (if d < c then Sym(t,d,c,n) else Sym(t,c,d,n)), cout+un
	end

	else if zero < p then let q = (c+b-un)/b in

	(* cas asymtrique emboit *)
	if c < q*a then begin
	  let q = q-un    in
	  let k = (q*n)/c in
	  let (u,cu) = prpare (p-un) a (b-a) (c-q*a) (n-a*k-a) in
	  let (v,cv) = prpare (p-un) (b-a) b (c-q*b) (n+a-b*(k+deux)) in
	  (Asym(Clos(q,k),u,v), cu+cv+un)
	end
	
	(* cas asymetrique crois *)
	else begin
	  let k = (q*n)/c in
	  let (u,cu) = prpare (p-un) a (b-a) (c-q*a) (n-a*k-a) in
	  let (v,cv) = prpare (p-un) (b-a) b (q*b-c) (b*k-n-un) in
	  (Asym(Clos(q,k),u,v), cu+cv+un)
	end
	
	(* profondeur maximale atteinte *)
	else (Sum(a,b,c,n),h)
	
      in
      (* compare le cout de la recursion a celui des additions *)
      if cout < h then (suite,cout) else (Sum(a,b,c,n),h)


         (* +------------------------------------------------------+
            |  Calcul de vol(a,b,c,n) par dcoupage de ttradres  |
            +------------------------------------------------------+ *)

  let rec eval vol2 = function
    | Nul          -> zero
    | Sym(t,u,v,n) -> deux*(eval vol2 t) + (surf2 u v n)/deux
    | Asym(t,u,v)  -> (eval vol2 t) + (eval vol2 u) - (eval vol2 v)
    | Sum(a,b,c,n) -> vol2 a b c n
    | Clos(q,n)    -> let k = n/q in
                      (k+un)*(  k*(deux*k+un)*q*q
				- trois*k*q*(deux*n+trois)
				+ six*(n+un)*(n+deux)
			     )/douze

  let vol3 vol2 p a b c n = let t,_ = prpare p a b c n in eval vol2 t

	   
                          (* +---------------------+
                             |  Formule approche  |
                             +---------------------+ *)

  (* retourne v(a,b,c,n) et v'(a,b,c,n) (numrateur et dnominateur spars) *)
  let approx a b c n =
    let a' = a-un and b' = b-un and c' = c-un in
    let s = a' + b' + c' and p = a'*b' + a'*c' + b'*c' and d = douze*a*b*c in
    let v  = (s+p)*(s+deux)
	     + (n+un)*(deux*(s*(s+deux)+p)
		       + (n+deux)*(six*s + quatre*(n+trois)))
    and v' = (s+un)*(s+quatre)+p + six*(n+un)*(n+s+trois)
    in (v/(deux*d), v',d)



                          (* +--------------------+
                             |  Balayage oblique  |
                             +--------------------+ *)

(* Parcourt les tranches ax+by+cz = n+1, n+2, n+3, ... (monte)
                        ax+by+cz = n, n-1, n-2, ...   (descend)
 et incrmente ou dcremente t chaque fois qu une tranche contient un
 point. Note dans x = (n0,n1,n2) les tranches faisant passer t  0,-1,1 *)

  let testeq t n (n0,n1,n2) =
    let test t nom p =
      if t=zero then begin
	if !debug then chrono(sprintf "%s         %s" nom (string_of n));
	Some(n)
      end
      else p
    in (test t "n0" n0), (test (t+un) "n1" n1), (test (t-un) "n2" n2)


  let rec monte surf t n x =
    if t < un then begin
      let n = n+un    in
      let s = surf(n) in
      let t = t+s     in
      if s = zero    then monte surf t n x
      else if s = un then monte surf t n (testeq t n x)
      else failwith "points multiples"
    end
    else x


  let rec descend surf t n x = match x with
    |  _,None,_ -> let s = surf(n) in
                   if s = zero    then descend surf t      (n-un) x
	           else if s = un then descend surf (t-un) (n-un) (testeq t n x)
	           else failwith "points multiples"
    | (Some n0),(Some n1),(Some n2) -> (n0,n1,n2)
    | _ -> failwith "balayge incomplet"


                (* +----------------------------------------+
                   |  Recherche du k-me entier de Hamming  |
                   +----------------------------------------+ *)

  let ham vol2 a0 b0 c0 a1 b1 c1 k =

    (* n = racine cubique approche de 6kabc *)
    let p =  six*a0*b0*c0*k         in
    let n =  shl un (nbits(p) /! 3) in
    let n = (n*.2 + p/(n*n))/.3     in
    let n = (n*.2 + p/(n*n))/.3     in
    let n = (n*.2 + p/(n*n))/.3     in

    (* Rsolution de approx(a,b,c,n) = k+1 par la mthode de Newton *)
    (* arrte  +/- 2                                               *)
    let rec newton n =
      let v,p,q = approx a0 b0 c0 n in
      if !debug then chrono(sprintf "Approx     %s -> %s" (string_of n) (string_of v));
      let t = v-k-un      in
      let n = n - (t*q)/p in
      if abs(t) < trois then n else newton n
    in
    let n = newton n in

    (* Calcule le volume exact *)
    let v = vol3 vol2 pmax a0 b0 c0 n in
    if !debug then chrono(sprintf "Exact      %s -> %s" (string_of n) (string_of v));

    (* Cherche les triangles contenant les points de rangs k-1, k, k+1 *)
    let surf       = surf3 a0 b0 c0 in
    let x          = monte   surf (v-k-un) n (None,None,None) in
    let (n0,n1,n2) = descend surf (v-k-un) n x                in
    
    (* Dtermine les exposants *)
    let x,y,z = search3 a0 b0 c0 n0 in
    if !debug then chrono(sprintf "Dichotomie %s %s %s" (string_of x) (string_of y) (string_of z));

    (* Validation *)
    let nu = a1*x + b1*y + c1*z in
    if (n1*a0*b1 < n0*a1*b0) && (n1*a0*c1 < n0*a1*c0) && (nu*a0 < n2*a1)
    then (x,y,z) else failwith "Contrle ngatif"


                (* +-------------------------------+
                   |  Rang d un entier de Hamming  |
                   +-------------------------------+ *)

  let rang vol2 a0 b0 c0 a1 b1 c1 x y z =

    (* calcule le volume associ  a*x+b*y+c*z, k=v-1 *)
    let n  = a0*x + b0*y + c0*z in
    let v = vol3 vol2 pmax a0 b0 c0 n in
    if !debug then chrono(sprintf "Exact      %s -> %s" (string_of n) (string_of v));

    (* Cherche les triangles contenant les points de rangs k-1, k, k+1 *)
    let surf       = surf3 a0 b0 c0 in
    let u          = monte   surf zero n (None,None,None) in
    let (n0,n1,n2) = descend surf zero n u                in

    (* Validation *)
    let nu = a1*x + b1*y + c1*z in
    if (n1*a0*b1 < n0*a1*b0) && (n1*a0*c1 < n0*a1*c0) && (nu*a0 < n2*a1)
    then v-un else failwith "Contrle ngatif"


(*+-------------------------------------------------------------------------+
  |  approximations de ln(3)/ln(2) et ln(5)/ln(2) avec l algorithme LLLint  |
  +-------------------------------------------------------------------------+*)
 
(*                                                                  
   *----*    MuPAD 1.4.1 -- Multi Processing Algebra Data Tool
  /|   /|                                                         
 *----* |    Copyright (c)  1997 - 98 by SciFace Software GmbH    
 | *--|-*                   All rights reserved.                  
 |/   |/                                                          
 *----*      Licensed to:   Michel Quercia
                                                                
>> compute := proc(n,signe)
&> begin
&> 
&>   DIGITS := n+5;
&>   l3 := ln(3)/ln(2);
&>   l5 := ln(5)/ln(2);
&>   r  := FAIL;
&>   rand := random(10^n..10^(n+1));
&> 
&>   while r = FAIL do
&> 
&>     a  := rand();
&>     b  := round(a*l3);
&>     c  := round(a*l5);
&>     l  := lllint([[1,0,0],[b,-a,0],[c,0,-a]]);
&> 
&>     p1 := l[1][1]; q1 := l[2][1]; r1 := l[3][1];
&>     p2 := l[1][2]; q2 := l[2][2]; r2 := l[3][2];
&>     p3 := l[1][3]; q3 := l[2][3]; r3 := l[3][3];
&>     d1 := igcd(p1,q1) + igcd(p1,r1) + igcd(q1,r1);
&>     d2 := igcd(p2,q2) + igcd(p2,r2) + igcd(q2,r2);
&>     d3 := igcd(p3,q3) + igcd(p3,r3) + igcd(q3,r3);
&> 
&>     s1 := signe*float(q1/p1-l3) < 0;
&>     s2 := signe*float(q2/p2-l3) < 0;
&>     s3 := signe*float(q3/p3-l3) < 0;
&> 
&>     if   s1 and (d1 = 3) then r := [abs(p1), abs(q1), abs(r1)]
&>     elif s2 and (d2 = 3) then r := [abs(p2), abs(q2), abs(r2)]
&>     elif s3 and (d3 = 3) then r := [abs(p3), abs(q3), abs(r3)]
&>     end_if;
&> 
&>   end_while;
&>   r
&> 
&> end_proc:
>> 
>> for i from 1 to 30 do print([i,compute(i,-1),compute(i,1)]) end_for;
[1, [3, 5, 7],                                             	     	      [2, 3, 5]]
[2, [22, 35, 51],                                          	     	      [65, 103, 151]]
[3, [494, 783, 1147],                                      	     	      [171, 271, 397]]
[4, [1171, 1856, 2719],                                    	     	      [1901, 3013, 4414]]
[5, [1171, 1856, 2719],                                    	     	      [6197, 9822, 14389]]
 ------ 6 -> plante (?), on saute et on prendra la valeur pour 7 ...
[7, [103169, 163519, 239551],                              	     	      [52841, 83751, 122693]]
[8, [562607, 891711, 1306333],                             	     	      [151714, 240461, 352269]]
[9, [5314780, 8423727, 12340537],                          	     	      [1481483, 2348095, 3439897]]
[10, [5314780, 8423727, 12340537],                                            [1481483, 2348095, 3439897]]
[11, [128977283, 204424157, 299475977],                                       [38044331, 60298838, 88336201]]
[12, [173817877, 275494817, 403592612],                                       [218658471, 346565477, 507709247]]
[13, [3004037238, 4761286373, 6975158461],                                    [218658471, 346565477, 507709247]]
[14, [12081378335, 19148531618, 28052091781],                                 [14699735488, 23298529519, 34131728817]]
[15, [12081378335, 19148531618, 28052091781],         	       	     	      [38862492158, 61595592755, 90235912379]]
[16, [102891352153, 163078934811, 238906321285],      	       	     	      [38862492158, 61595592755, 90235912379]]
[17, [745624491379, 1181786858455, 1731286454769],    	       	     	      [38862492158, 61595592755, 90235912379]]
[18, [2585021157573, 4097161598324, 6002233251647],   	       	     	      [6314758360454, 10008655202435, 14662414849563]]
[19, [2585021157573, 4097161598324, 6002233251647],   	       	     	      [14159628195824, 22442479714535, 32877638521043]]
[20, [18313623320471, 29026406215279, 42522916506986],         	     	      [69100498157237, 109521698360372, 160446388042001]]
[21, [87414121477708, 138548104575651, 202969304548987],       	     	      [86629634494171, 137304722124441, 201147782181839]]
[22, [87414121477708, 138548104575651, 202969304548987],       	     	      [1343132146905340, 2128814086358063, 3118656267045889]]
[23, [2599634659316509, 4120323450591685, 6036164751909939],   	     	      [4658039940413373, 7382818632416605, 10815633804753266]]
[24, [42376959207445831, 67165891238391769, 98396252159664167],      	      [27146336873794543, 43025925976908332, 63031842260540319]]
[25, [42376959207445831, 67165891238391769, 98396252159664167],      	      [92726866148813068, 146968605655258685, 215305115661788977]]
[26, [227830691505071967, 361103102548909139, 529006483483242121],   	      [92726866148813068, 146968605655258685, 215305115661788977]]
[27, [4909778451825201523, 7781814732991718049, 11400152526965513528],        [92726866148813068, 146968605655258685, 215305115661788977]]
[28, [9220818747550078807, 14614651940813492219, 21410078107800629027],       [21327370787119314453, 33803082936559961711, 49520621420692335374]]
[29, [9220818747550078807, 14614651940813492219, 21410078107800629027],       [112971939391127494233, 179056287568680323501, 262312720006171228577]]
[30, [265712887064474460533, 421144961955547593151, 616966217648636050582],   [213331315452734241649, 338122135222099492316, 495339974868982141755]]
*)


  let llltab = [|
    (("3", "5", "7"),                                                             ("2", "3", "5"));
    (("22", "35", "51"),                                                          ("65", "103", "151"));
    (("494", "783", "1147"),                                                      ("171", "271", "397"));
    (("1171", "1856", "2719"),                                                    ("1901", "3013", "4414"));
    (("1171", "1856", "2719"),                                                    ("6197", "9822", "14389"));
    (("103169", "163519", "239551"),                                              ("52841", "83751", "122693"));
    (("103169", "163519", "239551"),                                              ("52841", "83751", "122693"));
    (("562607", "891711", "1306333"),                                             ("151714", "240461", "352269"));
    (("5314780", "8423727", "12340537"),                                          ("1481483", "2348095", "3439897"));
    (("5314780",  "8423727",  "12340537"),                                        ("1481483",  "2348095",  "3439897"));
    (("128977283",  "204424157",  "299475977"),                                   ("38044331",  "60298838",  "88336201"));
    (("173817877",  "275494817",  "403592612"),                                   ("218658471",  "346565477",  "507709247"));
    (("3004037238",  "4761286373",  "6975158461"),                                ("218658471",  "346565477",  "507709247"));
    (("12081378335",  "19148531618",  "28052091781"),                             ("14699735488",  "23298529519",  "34131728817"));
    (("12081378335", "19148531618", "28052091781"),                               ("38862492158", "61595592755", "90235912379"));
    (("102891352153", "163078934811", "238906321285"),                            ("38862492158", "61595592755", "90235912379"));
    (("745624491379", "1181786858455", "1731286454769"),                          ("38862492158", "61595592755", "90235912379"));
    (("2585021157573", "4097161598324", "6002233251647"),                         ("6314758360454", "10008655202435", "14662414849563"));
    (("2585021157573", "4097161598324", "6002233251647"),                         ("14159628195824", "22442479714535", "32877638521043"));
    (("18313623320471", "29026406215279", "42522916506986"),                  	  ("69100498157237", "109521698360372", "160446388042001"));
    (("87414121477708", "138548104575651", "202969304548987"),                	  ("86629634494171", "137304722124441", "201147782181839"));
    (("87414121477708", "138548104575651", "202969304548987"),                	  ("1343132146905340", "2128814086358063", "3118656267045889"));
    (("2599634659316509", "4120323450591685", "6036164751909939"),            	  ("4658039940413373", "7382818632416605", "10815633804753266"));
    (("42376959207445831", "67165891238391769", "98396252159664167"),             ("27146336873794543", "43025925976908332", "63031842260540319"));
    (("42376959207445831", "67165891238391769", "98396252159664167"),             ("92726866148813068", "146968605655258685", "215305115661788977"));
    (("227830691505071967", "361103102548909139", "529006483483242121"),          ("92726866148813068", "146968605655258685", "215305115661788977"));
    (("4909778451825201523", "7781814732991718049", "11400152526965513528"),      ("92726866148813068", "146968605655258685", "215305115661788977"));
    (("9220818747550078807", "14614651940813492219", "21410078107800629027"),     ("21327370787119314453", "33803082936559961711", "49520621420692335374"));
    (("9220818747550078807", "14614651940813492219", "21410078107800629027"),     ("112971939391127494233", "179056287568680323501", "262312720006171228577"));
    (("265712887064474460533", "421144961955547593151", "616966217648636050582"), ("213331315452734241649", "338122135222099492316", "495339974868982141755"))
  |]

                     (* +-------------------------------+
                        |  interface ligne de commande  |
                        +-------------------------------+ *)

  type data  = {volume : t -> t -> t -> t -> t;
		vname  : string;
		args   : t list}
  type query = Exposants of t | Rang of t*t*t

  let rec parse data = function
    | "-v"::nom::s ->
	let v = match nom with
	  | "func"     -> vol2_func
	  | "surplace" -> vol2_surplace
	  | _          -> failwith ("volume inconnu : " ^ nom)
	in parse {volume = v; vname = nom; args = data.args} s
    | "-v"::_    -> failwith "volume non spcifi"
    | "-d"::s    -> debug := true;  parse data s
    | "-test"::s -> debug := false; parse {volume = data.volume;
                                          vname  = data.vname;
                                          args   = [of_string "1G"]} s
    | n::s    -> parse {volume = data.volume;
                        vname  = data.vname;
                        args   = (of_string n)::data.args} s
    | []      -> match data.args with
	| [k]     -> data.volume, data.vname, Exposants(k)
	| [z;y;x] -> data.volume, data.vname, Rang(x,y,z)
	| _       -> failwith "nombre d\039arguments incorrect"

    
  let main arglist =

    let vol2,vname,query = try
      parse {volume=vol2_func; vname="func"; args=[]} arglist
    with ex ->
      fprintf stderr "syntaxe : hamming (k | x y z) [-d] [-v vol2]";
      fprintf stderr " [-e entiers] [-count]\n";
      flush stderr;
      exit(1)
    in

    let lg(x) = String.length(string_of x) in
    let l = match query with
      | Exposants(k) -> lg(k)
      | Rang(x,y,z)  -> lg(x) +! lg(y) +! lg(z) +! 4
    in

    (* choisit a0,..,c1 en fonction de la longueur des donnes *)
    let (a0,b0,c0),(a1,b1,c1) = llltab.(Pervasives.min l (Array.length(llltab) -! 1)) in
    let a0 = of_string a0 and a1 = of_string a1
    and b0 = of_string b0 and b1 = of_string b1
    and c0 = of_string c0 and c1 = of_string c1 in

    if !debug then begin
      chrono(sprintf "entiers    %s" (name()));
      chrono(sprintf "volume     %s" vname);
      chrono(sprintf "Paramtres %s  %s" (string_of a0) (string_of a1));
      chrono(sprintf "           %s  %s" (string_of b0) (string_of b1));
      chrono(sprintf "           %s  %s" (string_of c0) (string_of c1));
    end;

    match query with
      | Exposants(k) ->
	  let x,y,z = ham vol2 a0 b0 c0 a1 b1 c1 k in
	  printf "x = %s  y = %s  z = %s\n" (string_of x) (string_of y) (string_of z);
	  flush stdout
      | Rang(x,y,z) ->
	  let r = rang vol2 a0 b0 c0 a1 b1 c1 x y z in
	  printf "r = %s\n" (string_of r);
	  flush stdout

end

module S = Numcount.Start(Main)
let _ = S.start()

