/*
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Initial Developer of this code is David Baum.
 * Portions created by David Baum are Copyright (C) 1998 David Baum.
 * All Rights Reserved.
 */

#include <windows.h>
#include "RCX_Pipe.h"

/*
 * This file provides USB tower support by thunking over to the
 * \\.\LEGOTOWER1 device which is almost like a serial port (just
 * can't control speed, parity, etc.).
 *
 */


class RCX_USBTowerPipe_win : public RCX_Pipe
{
public:
						RCX_USBTowerPipe_win() : fFile(INVALID_HANDLE_VALUE) {}
	virtual				~RCX_USBTowerPipe_win() { Close(); }
						
	virtual RCX_Result	Open(const char *name, int mode);
	virtual void		Close();

	virtual int			GetCapabilities() const;
	virtual RCX_Result	SetMode(int mode);

	virtual long		Read(void *ptr, long count, long timeout_ms);
	virtual long		Write(const void *ptr, long count);

private:
	void		SetTimeout(long ms);
	HANDLE	fFile;
};


RCX_Pipe* RCX_NewUSBTowerPipe()
{
	return new RCX_USBTowerPipe_win();
}

#define DEFAULT_TOWER_NAME "\\\\.\\LEGOTOWER1"

RCX_Result RCX_USBTowerPipe_win::Open(const char *name, int mode)
{
	if (name == 0 || *name==0)
	{
		name = DEFAULT_TOWER_NAME;
	}

	if (mode != kNormalIrMode) return kRCX_PipeModeError;
	
	fFile = CreateFile(name, GENERIC_READ | GENERIC_WRITE,
		0, 0, OPEN_EXISTING, 0, 0);

	if (fFile == INVALID_HANDLE_VALUE) return kRCX_OpenSerialError;

	return kRCX_OK;
}


void RCX_USBTowerPipe_win::Close()
{
	if (fFile == INVALID_HANDLE_VALUE) return;

	CloseHandle(fFile);
	fFile = INVALID_HANDLE_VALUE;
}


long RCX_USBTowerPipe_win::Write(const void *ptr, long count)
{
	DWORD actual;

	if (!WriteFile(fFile, ptr, (unsigned long)count, &actual, (struct _OVERLAPPED *)NULL))
		return -1;

	FlushFileBuffers(fFile);

	return actual;
}


long RCX_USBTowerPipe_win::Read(void *ptr, long count, long timeout)
{
	DWORD actual;

	SetTimeout(timeout);
	
	if (!ReadFile(fFile, ptr, count, &actual, (struct _OVERLAPPED *)NULL))
	{
		DWORD lstError = GetLastError();
		DWORD Errors;
		COMSTAT cstat;
		ClearCommError( fFile, &Errors, &cstat );
		
		return 0;
	}

	return actual;
}


int RCX_USBTowerPipe_win::GetCapabilities() const
{
	// only normal IR mode is supported, and
	// the ABSORB_0x55_FLAG is needed to tell the
	// transport that initial 0x55 bytes don't make
	// it through the USB/driver shim
	return kNormalIrMode + kAbsorb55Flag;
}


RCX_Result RCX_USBTowerPipe_win::SetMode(int mode)
{
	switch(mode)
	{
		case kNormalIrMode:
			return kRCX_OK;
		default:
			return kRCX_PipeModeError;
	}
}


void RCX_USBTowerPipe_win::SetTimeout(long timeout_ms)
{
	COMMTIMEOUTS timeouts;

	timeouts.ReadIntervalTimeout = 0;
	timeouts.ReadTotalTimeoutMultiplier = 0;
	timeouts.ReadTotalTimeoutConstant = 0;
	timeouts.WriteTotalTimeoutMultiplier = 0; 
	timeouts.WriteTotalTimeoutConstant = 0;

	switch(timeout_ms)
	{
		case 0:
			timeouts.ReadIntervalTimeout = MAXDWORD;
			break;
		default:
			timeouts.ReadTotalTimeoutConstant = timeout_ms;
			break;
	}
	SetCommTimeouts(fFile, &timeouts);
}
