# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-
# Copyright 2013 Canonical
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.

"""Tests for Notes app"""

from __future__ import absolute_import

from testtools.matchers import Equals
from autopilot.matchers import Eventually
from autopilot.platform import model

from notes_app.tests import NotesTestCaseBaseWithHTTPServer, DatabaseMixin

import sqlite3


class TestFocus(NotesTestCaseBaseWithHTTPServer, DatabaseMixin):
    """Tests focusing notes"""

    """ This is needed to wait for the application to start.
        In the testfarm, the application may take some time to show up."""
    def setUp(self):
        # Setup the database before starting the app
        self.setup_db()
        super(TestFocus, self).setUp()

    def setup_db(self):
        note_data = """{ "elements" : [
            {"content":"foobar","type":"text"},
            {"content":"http://localhost:8129/image.png","type":"image"}
        ]}"""

        path = self.find_db()
        conn = sqlite3.connect(path)
        cursor = conn.cursor()
        cursor.execute("DELETE FROM notes")
        cursor.execute("INSERT INTO notes (date, note) VALUES ('2013-04-07', '" + note_data + "')")
        conn.commit()
        conn.close()

    def focus_note_for_editing(self, index):
        notes = self.main_window.get_notes()
        note = notes[index]

        self.pointing_device.click_object(note)
        self.assertThat(note.isExpanded, Eventually(Equals(True)))
        self.pointing_device.click()

        return note

    def wait_note_expanded(self, note):
        note.height.wait_for(note.actualExpandedHeight)

    def get_loader_item(self, loader, obj_type):
        self.assertThat(loader.progress, Eventually(Equals(1)))
        children = loader.get_children_by_type(obj_type)
        self.assertThat(len(children), Equals(1))
        return children[0]

    def test_parts_focus(self):
        note = self.main_window.get_notes()[0]

        # Expand the note
        self.pointing_device.click_object(note)
        self.wait_note_expanded(note)

        # Verify that the note we inserted has the right parts
        parts = self.main_window.get_note_parts(note)
        self.assertThat(len(parts), Equals(2))

        text = self.get_loader_item(parts[0], "TextDelegate")
        image = self.get_loader_item(parts[1], "ImageDelegate")

        # Clicking on the upper half of the image should focus the first
        # text area
        part_x, part_y, part_w, part_h = image.globalRect
        self.pointing_device.move(part_x + part_w / 2, part_y + 1)
        self.pointing_device.click()
        self.assert_osk_eventually_shown()
        self.assertThat(text.activeFocus, Eventually(Equals(True)))

        # Clicking on the lower half of the image should create a new text
        # area below the image and focus it
        self.pointing_device.move(part_x + part_w / 2, part_y + part_h - 1)
        self.pointing_device.click()
        self.assert_osk_eventually_hidden()
        # On the phone input focus is lost when the OSK appears so
        # we click the input box to gain focus again. LP: #1204084
        if model() != 'Desktop':
            self.pointing_device.click()
        self.assertThat(text.activeFocus, Eventually(Equals(False)))

        parts = self.main_window.get_note_parts(note)
        self.assertThat(len(parts), Equals(3))

        new_text = self.get_loader_item(parts[2], "TextDelegate")
        self.assertThat(new_text.activeFocus, Eventually(Equals(True)))

    def test_parts_delete_when_empty(self):
        note = self.main_window.get_notes()[0]

        # Expand the note
        self.pointing_device.click_object(note)
        self.wait_note_expanded(note)

        # Click on the lower half of the image to create the new text part
        parts = self.main_window.get_note_parts(note)
        image = self.get_loader_item(parts[1], "ImageDelegate")
        part_x, part_y, part_w, part_h = image.globalRect
        self.pointing_device.move(part_x + part_w / 2, part_y + part_h - 1)
        self.pointing_device.click()
        self.assertThat(
            lambda: len(self.main_window.get_note_parts(note)),
            Eventually(Equals(3)))

        # Now verify that clicking on the upper part of the image will cause
        # the new text to be deleted because it was empty
        self.pointing_device.move(part_x + part_w / 2, part_y + 1)
        self.pointing_device.click()
        self.assertThat(
            lambda: len(self.main_window.get_note_parts(note)),
            Eventually(Equals(2)))

    def test_parts_delete_empty_header(self):
        header = self.main_window.get_header()
        note = self.main_window.get_notes()[0]

        # Expand the note
        self.pointing_device.click_object(note)
        self.wait_note_expanded(note)

        # Click on the lower half of the image to create the new text part
        parts = self.main_window.get_note_parts(note)
        image = self.get_loader_item(parts[1], "ImageDelegate")
        part_x, part_y, part_w, part_h = image.globalRect
        self.pointing_device.move(part_x + part_w / 2, part_y + part_h - 1)
        self.pointing_device.click()
        self.assertThat(
            lambda: len(self.main_window.get_note_parts(note)),
            Eventually(Equals(3)))

        # Now verify that clicking on the header will will still cause the
        # new part to disappear
        self.pointing_device.click_object(header)
        self.assertThat(
            lambda: len(self.main_window.get_note_parts(note)),
            Eventually(Equals(2)))

    def test_parts_no_delete_with_text(self):
        header = self.main_window.get_header()
        note = self.main_window.get_notes()[0]

        # Expand the note
        self.pointing_device.click_object(note)
        self.wait_note_expanded(note)

        # Click on the lower half of the image to create the new text part
        parts = self.main_window.get_note_parts(note)
        image = self.get_loader_item(parts[1], "ImageDelegate")
        part_x, part_y, part_w, part_h = image.globalRect
        self.pointing_device.move(part_x + part_w / 2, part_y + part_h - 1)
        self.pointing_device.click()
        self.assertThat(
            lambda: len(self.main_window.get_note_parts(note)),
            Eventually(Equals(3)))

        # Verify that after typing something in the note will cause it to stay
        # when unfocusing
        self.keyboard.type("Some text", delay=self.TYPING_DELAY)
        self.pointing_device.click_object(header)
        note.isExpanded.wait_for(False)
        self.assertThat(
            lambda: len(self.main_window.get_note_parts(note)),
            Eventually(Equals(3)))
