////////////////////////////////////////////////////////////////////////////
// NoteCase notes manager project <http://notecase.sf.net>
//
// This code is licensed under BSD license.See "license.txt" for more details.
//
// File: Defines atomic action performed on NoteCase document 
//		 (base for Undo/Redo framework)
////////////////////////////////////////////////////////////////////////////

#include "DocAction.h"
#include "callbacks.h"
#include "support.h"
#include "lib/NoteDocument.h"
#include "lib/DocumentIterator.h"
#include "TextView.h"
#include "lib/debug.h"

extern NoteDocument g_doc;
extern GtkWidget *window1;
extern int g_nActiveNodeIdx;
extern TextView g_text;
void set_title_bar(const char *szText);
void rebuild_gui_tree (int nPID = -1);

DocAction::DocAction()
{
	m_nType = ACT_UNDEFINED;
	m_pDoc  = NULL;
	m_bSelected = false;
	m_nNodeSibling = 0;
}

DocAction::~DocAction()
{
}

void DocAction::Redo()
{
	switch(m_nType){
		case ACT_TEXT_INSERT:
			DoTextInsert();
			break;
		case ACT_TEXT_DELETE:
			DoTextDelete();
			break;
		case ACT_NODE_INSERT:
			DoTreeInsert();
			break;
		case ACT_NODE_RENAME:
			DoNodeRename();
			break;
		case ACT_TREE_DELETE:
			DoTreeDelete();
			break;
		case ACT_TREE_MOVE:
			DoTreeMove();
			break;	
		case ACT_TREE_IMPORT:
			DoTreeImport();
			break;
		case ACT_TREE_DND:
			DoTreeDND(true);
			break;
	}
}

void DocAction::Undo()
{
	switch(m_nType){
		case ACT_TEXT_INSERT:
			DoTextDelete();
			break;
		case ACT_TEXT_DELETE:
			DoTextInsert();
			break;
		case ACT_NODE_INSERT:
			DoTreeDelete();
			break;
		case ACT_NODE_RENAME:
			DoNodeUnrename();
			break;
		case ACT_TREE_DELETE:
			DoTreeInsert();
			break;
		case ACT_TREE_MOVE:
			DoTreeUnmove();
			break;	
		case ACT_TREE_IMPORT:
			DoTreeUnimport();
			break;
		case ACT_TREE_DND:
			DoTreeDND(false);
			break;
	}
}

void DocAction::DoTextInsert()
{
	InsertNodeText(m_nNodeIndex, m_nTextStartPos, m_strNodeText.c_str());
	if(m_bSelected){
		g_text.SelectRange(m_nTextStartPos, m_nTextStartPos+m_strNodeText.size());

		GtkWidget *textview = lookup_widget(window1, "textview1");
		gtk_window_set_focus(GTK_WINDOW(window1), textview);
	}
}

void DocAction::DoTextDelete()
{
	DeleteNodeText(m_nNodeIndex, m_nTextStartPos, m_strNodeText.size());
}
	
void DocAction::DoTreeInsert()
{
	TRACE("UNDO/REDO: insert %d nodes under PID=%d\n", m_objSubTree.GetNodeCount(), m_nNodePID);

	int nCount = g_doc.GetNodeCount();

	TRACE("subtree dump:\n");
	m_objSubTree.Dump();
	TRACE("UNDO/REDO insert: %d nodes before\n", g_doc.GetNodeCount());	
	TRACE("UNDO/REDO insert: PID=%d, Sibling=%d\n", m_nNodePID, m_nNodeSibling);	
	g_doc.Merge(m_objSubTree, m_nNodePID, m_nNodeSibling, true);
	TRACE("UNDO/REDO insert: %d nodes after\n", g_doc.GetNodeCount());
	g_doc.Dump();
	
	//rebuild affected branch in the gui tree
	rebuild_gui_tree (m_nNodePID);

	g_nActiveNodeIdx = -1;	//no selection

	//select the node in the GUI tree
	SelectNodeByIdx(nCount);  //select subtree 'root' node in new tree
}

void DocAction::DoTreeDelete()
{
	TRACE("UNDO/REDO: delete node ID=%d\n", m_nNodeID);

	TRACE("subtree dump:\n");
	m_objSubTree.Dump();

	GtkTreeIter  iter;
	TreeIterFromID(iter, m_nNodeID);

	g_doc.NodeDelete(m_nNodeID);
	
	g_nActiveNodeIdx = -1;	//no selection

	//refresh GUI
	GtkTreeView *treeview = (GtkTreeView *)lookup_widget(window1, "treeview1");
	GtkTreeModel *model = gtk_tree_view_get_model((GtkTreeView *)treeview);

	//remove GUI tree node
	gtk_tree_store_remove(GTK_TREE_STORE(model), &iter); 

	//clear edit belonging to selected node
	g_text.Clear();  // TOFIX only if current node is one of being deleted
	
	set_title_bar(_("Untitled"));	//update node title label
	RefreshMainTitle();
}

void DocAction::DoNodeRename()
{
	SetNodeTitle(m_nNodeIndex, m_strNodeNameNew.c_str());	//TOFIX recursive index ?
}

void DocAction::DoNodeUnrename()
{
	SetNodeTitle(m_nNodeIndex, m_strNodeNameOld.c_str());	//TOFIX recursive index ?
}

void DocAction::DoTreeMove()
{
	SelectNodeByIdx(m_nNodeIndex);//select proper node

	switch(m_nMoveDirection){
		case MOVE_LEFT:
			do_node_move_left(false);
			break;
		case MOVE_RIGHT:
			do_node_move_right(false);
			break;
		case MOVE_UP:
			do_node_move_up(false);
			break;
		case MOVE_DOWN:
			do_node_move_down(false);
			break;
	}
}

//reverse the moving operation
void DocAction::DoTreeUnmove()
{
	int i;

	SelectNodeByIdx(m_nNodeIndex);//select proper node

	switch(m_nMoveDirection){
		case MOVE_LEFT:
			do_node_move_right(false);
			//restore sibling index position when moving back from left to right
			for(i=0; i<m_nNodeSibling; i++)	
				do_node_move_down(false);
			break;
		case MOVE_RIGHT:
			do_node_move_left(false);
			break;
		case MOVE_UP:
			do_node_move_down(false);
			break;
		case MOVE_DOWN:
			do_node_move_up(false);
			break;
	}
}

void DocAction::DoTreeImport()
{
	DocumentIterator it(g_doc);
	int nCnt = it.GetChildCount(-1);

	//TOFIX ask user for import details and merge with current document
	g_doc.Merge(m_objSubTree);

	//refresh tree starting from new content
	add_child_nodes(NULL, -1, nCnt);
}

void DocAction::DoTreeUnimport()
{
	DocumentIterator it(g_doc);
	int nCount = it.GetNodeCount();
	int i;

	g_nActiveNodeIdx = -1;	//no selection

	GtkTreeView *treeview = (GtkTreeView *)lookup_widget(window1, "treeview1");
	GtkTreeModel *model = gtk_tree_view_get_model((GtkTreeView *)treeview);

	//delete all nodes after idx = m_nNodeIndex 
	//delete in backward direction
	for(i=nCount-1; i>=m_nNodeIndex; i--)
	{
		int nID = g_doc.GetNodeByIdx(i).m_nID;

		GtkTreeIter  iter;
		TreeIterFromID(iter, nID);

		//delete in memory
		g_doc.NodeDelete(nID);	

		//remove GUI tree node
		gtk_tree_store_remove(GTK_TREE_STORE(model), &iter); 

		//clear edit belonging to selected node
		g_text.Clear();  // TOFIX only if current node is one of being deleted

		//update node title label
		set_title_bar(_("Untitled"));
		RefreshMainTitle();
	}
}

void DocAction::DoTreeDND(bool bForward)
{
	GtkTreeView *treeview = (GtkTreeView *)lookup_widget(window1, "treeview1");
	GtkTreeModel *model = gtk_tree_view_get_model((GtkTreeView *)treeview);

	if(bForward)
	{
		//redo - move node branch from original position to new one
		int nID = m_nNodeID;

		GtkTreeIter iter;
		TreeIterFromID(iter, nID);

		TRACE("REDO dnd: tree dump on start\n");
		#ifdef _DEBUG
			g_doc.Dump();
		#endif

		TRACE("REDO dnd: delete node ID=%d\n", nID);

		//delete in memory
		g_doc.NodeDelete(nID);	

		TRACE("REDO dnd: tree dump after delete\n");
		#ifdef _DEBUG
			g_doc.Dump();
		#endif

		//remove GUI tree node
		gtk_tree_store_remove(GTK_TREE_STORE(model), &iter); 

		//clear edit belonging to selected node
		g_text.Clear();  // TOFIX only if current node is one of being deleted

		TRACE("REDO dnd: merge subtree under PID=%d, SIB=%d\n", m_nNodeNewPID, m_nNodeNewSibling);

		//insert node branch into new position
		g_doc.Merge(m_objSubTree, m_nNodeNewPID, m_nNodeNewSibling, true);  //keep IDs

		TRACE("REDO dnd: tree dump after merge\n");
		#ifdef _DEBUG
			g_doc.Dump();
		#endif

		//rebuild GUI
		rebuild_gui_tree (m_nNodeNewPID);
	}
	else
	{
		//undo - move new node branch back to original position
		int nID = m_nNewNodeID;

		GtkTreeIter iter;
		TreeIterFromID(iter, nID);

		TRACE("UNDO dnd: tree dump on start\n");
		#ifdef _DEBUG
			g_doc.Dump();
		#endif

		TRACE("UNDO dnd: delete node ID=%d\n", nID);

		//delete in memory
		g_doc.NodeDelete(nID);	

		TRACE("UNDO dnd: tree dump after delete\n");
		#ifdef _DEBUG
			g_doc.Dump();
		#endif

		//remove GUI tree node
		gtk_tree_store_remove(GTK_TREE_STORE(model), &iter); 

		//clear edit belonging to selected node
		g_text.Clear();  // TOFIX only if current node is one of being deleted

		TRACE("UNDO dnd: merge subtree under PID=%d\n", m_nNodePID);

		//insert node branch into original position
		g_doc.Merge(m_objSubTree, m_nNodePID, m_nNodeSibling, true);	//keep IDs

		TRACE("UNDO dnd: tree dump after merge\n");
		#ifdef _DEBUG
			g_doc.Dump();
		#endif

		//rebuild GUI
		rebuild_gui_tree (m_nNodePID);
	}
}

void rebuild_gui_tree (int nPID)
{
	GtkTreeView *treeview = (GtkTreeView *)lookup_widget(window1, "treeview1");
	GtkTreeModel *model = gtk_tree_view_get_model((GtkTreeView *)treeview);

	if(nPID == -1)
	{
		//clear all nodes
		gtk_tree_store_clear(GTK_TREE_STORE(model));

		//rebuild parent node children
		add_child_nodes(NULL, nPID, 0);
	}
	else
	{
		GtkTreeIter iter;
		TreeIterFromID(iter, nPID);

		//remove all parent node children
		GtkTreeIter iterChild;
		while(gtk_tree_model_iter_children(model, &iterChild, &iter))
			gtk_tree_store_remove(GTK_TREE_STORE(model), &iterChild); //delete all children

		//rebuild parent node children
		add_child_nodes(&iter, nPID, 0);
	}
}

