/**
 * Copyright (c) Facebook, Inc. and its affiliates. All Rights Reserved.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 */

import * as path from 'path';
import pluginTester from 'babel-plugin-tester';
import {format as formatCode, resolveConfig} from 'prettier';
import babelPluginJestHoist from '..';

const prettierOptions = {
  ...resolveConfig.sync(__filename),
  filepath: __filename,
};

const formatResult = (code: string) => formatCode(code, prettierOptions);

pluginTester({
  plugin: babelPluginJestHoist,
  pluginName: 'babel-plugin-jest-hoist',
  tests: {
    /* eslint-disable sort-keys */
    'automatic react runtime': {
      babelOptions: {
        babelrc: false,
        configFile: false,
        filename: path.resolve(__dirname, '../file.js'),
        presets: [
          [
            require.resolve('@babel/preset-react'),
            {development: true, runtime: 'automatic'},
          ],
        ],
      },
      code: formatResult(`
        jest.mock('./App', () => () => <div>Hello world</div>);
      `),
      formatResult(code) {
        // replace the filename with something that will be the same across OSes and machine
        const codeWithoutSystemPath = code.replace(
          /var _jsxFileName = ".*";/,
          'var _jsxFileName = "/root/project/src/file.js";',
        );

        return formatResult(codeWithoutSystemPath);
      },
      snapshot: true,
    },
    'top level mocking': {
      code: formatResult(`
        require('x');

        jest.enableAutomock();
        jest.disableAutomock();
      `),
      formatResult,
      snapshot: true,
    },
    'within a block': {
      code: formatResult(`
        beforeEach(() => {
          require('x')
          jest.mock('someNode')
        })
      `),
      formatResult,
      snapshot: true,
    },
    'within a block with no siblings': {
      code: formatResult(`
        beforeEach(() => {
          jest.mock('someNode')
        })
      `),
      formatResult,
      snapshot: true,
    },

    'required `jest` within `jest`': {
      code: formatResult(`
        const {jest} = require('@jest/globals');

        jest.mock('some-module', () => {
          jest.requireActual('some-module');
        });
      `),
      formatResult,
      snapshot: true,
    },
  },
  /* eslint-enable */
});
