/* A rowview in a workspace ... not a widget, part of column
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

/* 
#define DEBUG
 */

#include "ip.h"

/* Draw LEDs rather than recolouring tally buttons.
 */
#define CALC_DISPLAY_LED (watch_bool_get( "CALC_DISPLAY_LED", FALSE ))

static ModelClass *parent_class = NULL;

enum {
	ROWVIEW_TARGET_STRING,
};

static GtkTargetEntry rowview_target_table[] = {
	{ "STRING", 0, ROWVIEW_TARGET_STRING },
	{ "text/plain", 0, ROWVIEW_TARGET_STRING }
};

/* Just one popup for all tally buttons.
 */
static GtkWidget *rowview_popup_menu = NULL;

static void 
rowview_destroy( GtkObject *object )
{
	Rowview *rview;

	g_return_if_fail( object != NULL );
	g_return_if_fail( IS_ROWVIEW( object ) );

	rview = ROWVIEW( object );

#ifdef DEBUG
	printf( "rowview_destroy: " );
	row_name_print( ROW( VIEW( rview )->model ) );
	printf( "\n" );
#endif /*DEBUG*/

	FREEFI( gtk_timeout_remove, rview->set_name_tid );
	FREE( rview->last_tooltip );

	/* Kill children ... must do this ourselves, since we are not a
	 * self-contained widget.
	 */
	FREEW( rview->but );
	FREEW( rview->spin );
	FREEW( rview->led );

	GTK_OBJECT_CLASS( parent_class )->destroy( object );
}

static void
rowview_attach( Rowview *rview, GtkWidget *child, int x, 
	GtkAttachOptions xoptions, GtkAttachOptions yoptions )
{
	Subcolumnview *sview = rview->sview;

	gtk_widget_ref( child );

	if( child->parent )
		gtk_container_remove( GTK_CONTAINER( sview->table ), child );
	gtk_table_attach( GTK_TABLE( sview->table ), child,
		x, x + 1, rview->rnum, rview->rnum + 1, 
		xoptions, yoptions, 0, 0 );

	gtk_widget_unref( child );
}

static gint
rowview_set_name_cb( Rowview *rview )
{
	rview->set_name_tid = 0;

#ifdef DEBUG
	printf( "rowview_set_name_cb: " );
	row_name_print( ROW( VIEW( rview )->model ) );
	printf( ".name = %s\n", rview->to_set_name );
#endif /*DEBUG*/

	set_name( rview->but, rview->to_set_name );
	rview->set_name = rview->to_set_name;
	rview->to_set_name = NULL;

	return( FALSE );
}

static void
rowview_set_name( Rowview *rview, const char *set_name )
{
	gboolean changed = FALSE;

	/* If there's a timeout currently set, test against the name we're
	 * planning to set. Pointer equality is mostly good enough .. we are 
	 * setting with static strings.
	 */
	if( rview->set_name_tid && set_name != rview->to_set_name ) {
		rview->to_set_name = set_name;
		changed = TRUE;
	}
	/* No timeout? Test against the current name.
	 */
	else if( set_name != rview->set_name ) {
		rview->to_set_name = set_name;
		changed = TRUE;
	}

	if( changed ) {
		/* Reset the timeout.
		 */
		FREEFI( gtk_timeout_remove, rview->set_name_tid );
		rview->set_name_tid = gtk_timeout_add( 200, 
			(GtkFunction) rowview_set_name_cb, rview );
	}
}

static void *
rowview_tip_sub( Link *link, BufInfo *buf )
{
	if( link->child->dirty ) {
		symbol_qualified_name( link->child, buf );
		buf_appends( buf, " " );
	}

	return( NULL );
}

static void
rowview_tip( Rowview *rview, BufInfo *buf )
{
	Row *row = ROW( VIEW( rview )->model );

	if( row == row->top_row && row->sym->dirty ) {
		Symbol *sym = row->sym;

		if( sym->ndirtychildren ) {
			buf_appends( buf, ", blocked on " );
			slist_map( sym->topchildren,
				(SListMapFn) rowview_tip_sub, buf );
		}
	}
}

static void
rowview_set_tooltip( Rowview *rview, const char *tip )
{
	if( !rview->last_tooltip || strcmp( rview->last_tooltip, tip ) != 0 ) {
		set_tooltip( rview->but, "%s", tip );
		SETSTR( rview->last_tooltip, tip );
	}
}

static void 
rowview_refresh( View *view )
{
	Rowview *rview = ROWVIEW( view );
	Row *row = ROW( VIEW( rview )->model );
	int pos = MODEL( row )->pos;

#ifdef DEBUG
	printf( "rowview_refresh: " );
	row_name_print( row );
	printf( "\n" );
#endif /*DEBUG*/

	/* Attach widgets to parent in new place.
	 */
        if( rview->rnum != pos ) {
#ifdef DEBUG
		printf( "rowview_refresh: move from row %d to row %d\n", 
			rview->rnum, pos );
#endif /*DEBUG*/

		rview->rnum = pos;

		rowview_attach( rview, rview->spin, 
			0, GTK_FILL, GTK_FILL );
		rowview_attach( rview, rview->but, 
			1, GTK_FILL, GTK_EXPAND | GTK_FILL );
		rowview_attach( rview, rview->led, 
			2, GTK_FILL, GTK_EXPAND | GTK_FILL );
		if( rview->rhsview )
			rowview_attach( rview, GTK_WIDGET( rview->rhsview ), 
				3, 
				GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL );
	}

        set_glabel( rview->label, "%s", row_name( row ) );

	/* Spin visible only if this is a class. 
	 */
	widget_visible( rview->spin, rview->visible && row->isclass );

        /* Set colours.
         */
	if( CALC_DISPLAY_LED ) {
		LedColour colour;

		if( row->selected )
			colour = LED_GREEN;
		else if( row->show == ROW_SHOW_PARENT )
			colour = LED_CYAN;
		else if( row->show == ROW_SHOW_CHILD )
			colour = LED_BLUE;
		else if( row->err )
			colour = LED_RED;
		else if( row->dirty )
			colour = LED_YELLOW;
		else
			colour = LED_OFF;

		led_set_colour( LED( rview->led ), colour ); 
	}
	else {
		gchar *name = "";

		if( row->selected )
			name = "selected_widget";
		else if( row->show == ROW_SHOW_PARENT )
			name = "parent_widget";
		else if( row->show == ROW_SHOW_CHILD )
			name = "child_widget";
		else if( row->err )
			name = "error_widget";
		else if( row->dirty )
			name = "dirty_widget";

		rowview_set_name( rview, name );
	}

	if( row->expr && row->expr->err ) 
		/* Display error message in tooltip.
		 */
		rowview_set_tooltip( rview, row->expr->errstr );
	else if( row->expr ) {
		BufInfo buf;
		char txt[200];

		buf_init_static( &buf, txt, 200 );
		expr_tip( row->expr, &buf );
		rowview_tip( rview, &buf );
		rowview_set_tooltip( rview, buf_all( &buf ) );
	}
	else 
		rowview_set_tooltip( rview, 
			"Left-click to select, "
			"shift-left-click to extend select, "
			"double-left-click to edit, "
			"right-click for menu, "
			"left-drag to move" );

	VIEW_CLASS( parent_class )->refresh( view );
}

/* Single click on button callback.
 */
static void
rowview_single_cb( GtkWidget *wid, Rowview *rview, guint state )
{
	Row *row = ROW( VIEW( rview )->model );

	row_select_modifier( row, state );
}

/* Edit our object.
 */
static gboolean
rowview_edit( Rowview *rview )
{
	Row *row = ROW( VIEW( rview )->model );
	Model *graphic = row->child_rhs->graphic;

	if( !graphic || !MODEL_CLASS( GTK_OBJECT( graphic )->klass )->edit ) {
		ierrors( "object not editable" );
		return( FALSE );
	}

	model_edit( GTK_WIDGET( rview->sview ), graphic );

	return( TRUE );
}

/* Double click on button callback.
 */
static void
rowview_double_cb( GtkWidget *button, Rowview *rview )
{
	if( !rowview_edit( rview ) ) 
		box_alert( button );
}

/* Edit in menu.
 */
static void
rowview_edit_cb( GtkWidget *menu, GtkWidget *button, Rowview *rview )
{
	if( !rowview_edit( rview ) ) 
		box_alert( button );
}

/* Clone the current item.
 */
static void
rowview_clone_cb( GtkWidget *menu, GtkWidget *button, Rowview *rview )
{
	Row *row = ROW( VIEW( rview )->model );
	Workspace *ws = row->top_col->ws;

	/* Only allow clone of top level rows.
	 */
	if( row->top_row != row ) {
		box_info( button, "you can only clone top level rows" );
		return;
	}

        workspace_deselect_all( ws );
        row_select( row );
        if( !workspace_clone_selected( ws ) )
                box_alert( NULL );
        workspace_deselect_all( ws );

        symbol_recalculate_all();
}

/* Ungroup the current item.
 */
static void
rowview_ungroup_cb( GtkWidget *menu, GtkWidget *button, Rowview *rview )
{
	Row *row = ROW( VIEW( rview )->model );

        if( mainw_ungroup( row ) )
                box_alert( NULL );
        symbol_recalculate_all();
}

/* Save the current item.
 */
static void
rowview_save_cb( GtkWidget *menu, GtkWidget *button, Rowview *rview )
{
	iWindow *iwnd = IWINDOW( view_get_toplevel( VIEW( rview ) ) );
	Row *row = ROW( VIEW( rview )->model );
	Model *graphic = row->child_rhs->graphic;

	if( !graphic ) {
		box_error( GTK_WIDGET( iwnd ), "object not saveable" );
		return;
	}

	classmodel_graphic_save( GTK_WIDGET( iwnd ), CLASSMODEL( graphic ) );
}

/* Replace the current item.
 */
static void
rowview_replace_cb( GtkWidget *menu, GtkWidget *button, Rowview *rview )
{
	iWindow *iwnd = IWINDOW( view_get_toplevel( VIEW( rview ) ) );
	Row *row = ROW( VIEW( rview )->model );
	Model *graphic = row->child_rhs->graphic;

	if( !graphic ) {
		box_error( GTK_WIDGET( iwnd ), "object not replaceable" );
		return;
	}

	classmodel_graphic_replace( GTK_WIDGET( iwnd ), CLASSMODEL( graphic ) );
}

/* Recalculate the current item.
 */
static void
rowview_recalc_cb( GtkWidget *menu, GtkWidget *button, Rowview *rview )
{
	Row *row = ROW( VIEW( rview )->model );
	Workspace *ws = row->top_col->ws;

        workspace_deselect_all( ws );
        row_select( row );
        if( !mainw_selected_recalc() )
                box_alert( NULL );
        workspace_deselect_all( ws );
}

/* Reset the current item.
 */
static void
rowview_clear_edited_cb( GtkWidget *menu, GtkWidget *button, Rowview *rview )
{
	Row *row = ROW( VIEW( rview )->model );

	(void) model_map_all( MODEL( row ),
		(model_map_fn) model_clear_edited, NULL );
	symbol_recalculate_all();
}

/* Remove the current item.
 */
static void
rowview_remove_cb( GtkWidget *menu, GtkWidget *button, Rowview *rview )
{
	Row *row = ROW( VIEW( rview )->model );
	Workspace *ws = row->top_col->ws;

	/* Only allow remove of top level rows.
	 */
	if( row->top_row != row ) {
		box_info( button, "you can only remove top level rows" );
		return;
	}

	workspace_deselect_all( ws );
	row_select( row );
	mainw_selected_remove();
}

/* Callback for up/down spin button.
 */
static void
rowview_spin_up_cb( GtkWidget *widget, gpointer client )
{
	Rowview *rview = ROWVIEW( client );
	Row *row = ROW( VIEW( rview )->model );
	Rhs *rhs = row->child_rhs;

	rhs_vislevel_down( rhs );
	workspace_set_modified( row->sym );
}

static void
rowview_spin_down_cb( GtkWidget *widget, gpointer client )
{
	Rowview *rview = ROWVIEW( client );
	Row *row = ROW( VIEW( rview )->model );
	Rhs *rhs = row->child_rhs;

	rhs_vislevel_up( rhs );
	workspace_set_modified( row->sym );
}

/* Scroll to make tally entry visible. If it's in a closed column, pop the
 * column open first.
 */
static void
rowview_scrollto( View *view )
{
	Rowview *rview = ROWVIEW( view );
	Row *row = ROW( VIEW( rview )->model );
	Column *col = row->top_col;
	Columnview *cview = view_get_columnview( VIEW( rview ) );
	Workspaceview *wview = cview->wview;

        int x, y, w, h;

        /* If the column is closed, open it.
         */
        if( !col->open ) {
                col->open  = TRUE;
		model_changed( MODEL( col ) );
        }

        /* Extract position of tally row in RC widget.
         */
        rowview_get_position( rview, &x, &y, &w, &h );
        workspaceview_scroll( wview, x, y, w, h );
}

static void
rowview_drag( Rowview *rview_from, Rowview *rview_to )
{
	Row *row_from = ROW( VIEW( rview_from )->model );
	Row *row_to = ROW( VIEW( rview_to )->model );

	if( row_from->top_col != row_to->top_col ) {
		box_info( NULL, "drag between columns not yet implemented" );
		return;
	}

	model_child_move( MODEL( row_from ), MODEL( row_to )->pos );

	/* Refresh all the rows, to make sure we move all rows to their new
	 * slots.
	 */
	model_map( MODEL( row_from->scol ),
		(model_map_fn) model_changed, NULL, NULL );

        workspace_deselect_all( row_from->ws );
}

static void
rowview_drag_data_get( GtkWidget *but,
	GdkDragContext *context, GtkSelectionData *selection_data,
	guint info, guint time, Rowview *rview )
{
	if( info == ROWVIEW_TARGET_STRING ) {
		/* Send a pointer to us.
 		 */
		gtk_selection_data_set( selection_data,
			selection_data->target,
			8, (const guchar *) &rview, sizeof( Rowview * ) );
	}
}

static void
rowview_drag_data_received( GtkWidget *but,
	GdkDragContext *context, gint x, gint y,
	GtkSelectionData *data, guint info, guint time, Rowview *rview_to )
{
	if( data->length == sizeof( Rowview * ) && data->format == 8 && 
		info == ROWVIEW_TARGET_STRING ) {
		Rowview *rview_from = *((Rowview **) data->data);

		if( IS_ROWVIEW( rview_from ) ) {
			rowview_drag( rview_from, rview_to );
			gtk_drag_finish( context, TRUE, FALSE, time );
			return;
		}
	}

	gtk_drag_finish( context, FALSE, FALSE, time );
}

/* Attach the rowview menu to a widget ... also used by iimageview
 */
guint
rowview_menu_attach( Rowview *rview, GtkWidget *widget )
{
	return( popup_attach( widget, rowview_popup_menu, rview ) );
}

static void
rowview_link( View *view, Model *model, View *parent )
{
	Row *row = ROW( model );
	Rowview *rview = ROWVIEW( view );
	Subcolumnview *sview = SUBCOLUMNVIEW( parent );

	VIEW_CLASS( parent_class )->link( view, model, parent );

	rview->sview = sview;

	/* Only drag n drop top level rows.
	 */
	if( row->top_row == row ) {
		gtk_drag_source_set( rview->but, GDK_BUTTON1_MASK, 
			rowview_target_table, IM_NUMBER( rowview_target_table ), 
			GDK_ACTION_COPY );
		gtk_signal_connect( GTK_OBJECT( rview->but ), "drag_data_get",
			GTK_SIGNAL_FUNC( rowview_drag_data_get ), rview );

		gtk_drag_dest_set( rview->but, GTK_DEST_DEFAULT_ALL,
			rowview_target_table, IM_NUMBER( rowview_target_table ), 
			GDK_ACTION_COPY );
		gtk_signal_connect( GTK_OBJECT( rview->but ), 
			"drag_data_received",
			GTK_SIGNAL_FUNC( rowview_drag_data_received ), rview );
	}

	rview->popup_sid = rowview_menu_attach( rview, rview->but );
}

static void
rowview_child_add( View *parent, View *child )
{
	Rowview *rowview = ROWVIEW( parent );

	assert( IS_RHSVIEW( child ) );
	assert( !rowview->rhsview );

	rowview->rhsview = RHSVIEW( child );

	VIEW_CLASS( parent_class )->child_add( parent, child );
}

static void
rowview_child_remove( View *parent, View *child )
{
	Rowview *rowview = ROWVIEW( parent );

	assert( IS_RHSVIEW( child ) );
	assert( rowview->rhsview == RHSVIEW( child ) );

	rowview->rhsview = NULL;

	VIEW_CLASS( parent_class )->child_remove( parent, child );
}

static void
rowview_class_init( RowviewClass *klass )
{
	GtkWidget *pane;
	GtkObjectClass *object_class = (GtkObjectClass *) klass;
	ViewClass *view_class = (ViewClass *) klass;

	parent_class = gtk_type_class( TYPE_VIEW );

	object_class->destroy = rowview_destroy;

	/* Create signals.
	 */

	/* Init methods.
	 */
	view_class->link = rowview_link;
	view_class->child_add = rowview_child_add;
	view_class->child_remove = rowview_child_remove;
	view_class->refresh = rowview_refresh;
	view_class->scrollto = rowview_scrollto;

        /* Other init.
         */
	pane = rowview_popup_menu = popup_build( "Row menu" );
	popup_add_but( pane, "Edit ...", rowview_edit_cb );
	popup_add_but( pane, "Clone", rowview_clone_cb );
	popup_add_but( pane, "Ungroup", rowview_ungroup_cb );
	popup_add_but( pane, "Save ...", rowview_save_cb );
	popup_add_but( pane, "Replace from file ...", rowview_replace_cb );
	popup_add_but( pane, "Recalculate", rowview_recalc_cb );
	popup_add_but( pane, "Reset", rowview_clear_edited_cb );
	menu_add_sep( pane );
	popup_add_but( pane, "Remove ...", rowview_remove_cb );
}

static void
rowview_flash_help( GtkWidget *widget, Rowview *rview )
{
	Row *row = ROW( VIEW( rview )->model );
	Expr *expr = row->expr;

	char str[MAX_LINELENGTH];
	BufInfo buf;

	/* No symbol? eg. on load error.
	 */
	if( !expr )
		return;

	buf_init_static( &buf, str, MAX_LINELENGTH );

	row_qualified_name( row, &buf );

	if( expr->err ) {
		buf_appends( &buf, ": " );
		buf_appendline( &buf, expr->errstr );
	}
	else if( row->child_rhs->itext ) {
		iText *itext = ITEXT( row->child_rhs->itext );

		buf_appends( &buf, " = " );

		if( !row->top_col->sform )
			buf_appendline( &buf, itext->formula );
		else 
			buf_appendline( &buf, buf_all( &itext->value ) );
	}

	set_glabel( mainw_message, "%s", buf_all( &buf ) );
}

static void *
rowview_show_parent( Link *link, RowShowState show )
{
	if( link->parent->expr && link->parent->expr->row ) 
		row_set_show( link->parent->expr->row, show );

	return( NULL );
}

static void *
rowview_show_child( Link *link, RowShowState show )
{
	if( link->child->expr && link->child->expr->row ) 
		row_set_show( link->child->expr->row, show );

	return( NULL );
}

static void
rowview_show_dependants( GtkWidget *widget, Rowview *rview )
{
	Row *row = ROW( VIEW( rview )->model );
	Symbol *topsym = row->top_row->sym;

#ifdef DEBUG
	printf( "rowview_show_dependants: " );
	row_name_print( ROW( VIEW( rview )->model ) );
	printf( "\n" );
#endif /*DEBUG*/

	slist_map( topsym->topparents,
		(SListMapFn) rowview_show_parent, (void *) ROW_SHOW_PARENT );
	slist_map( topsym->topchildren,
		(SListMapFn) rowview_show_child, (void *) ROW_SHOW_CHILD );
}

static void
rowview_hide_dependants( GtkWidget *widget, Rowview *rview )
{
	Row *row = ROW( VIEW( rview )->model );
	Symbol *topsym = row->top_row->sym;

#ifdef DEBUG
	printf( "rowview_hide_dependants: " );
	row_name_print( ROW( VIEW( rview )->model ) );
	printf( "\n" );
#endif /*DEBUG*/

	slist_map( topsym->topparents,
		(SListMapFn) rowview_show_parent, (void *) ROW_SHOW_NONE );
	slist_map( topsym->topchildren,
		(SListMapFn) rowview_show_child, (void *) ROW_SHOW_NONE );
}

static void
rowview_init( Rowview *rview )
{
        rview->visible = TRUE; 
	rview->rnum = -1;
        rview->popup_sid = 0; 
        rview->set_name_tid = 0; 
        rview->set_name = NULL; 
        rview->to_set_name = NULL; 
        rview->last_tooltip = NULL; 

	/* Make leds.
	 */
	rview->led = led_new();
	led_set_type( LED( rview->led ), 1 );
	gtk_misc_set_alignment( GTK_MISC( rview->led ), 0.5, 0.0 );
	gtk_misc_set_padding( GTK_MISC( rview->led ), 2, 2 );

	if( CALC_DISPLAY_LED ) 
		gtk_widget_show( rview->led );

        /* Make fold/unfold button.
         */
	rview->spin = spin_new();
        gtk_signal_connect( GTK_OBJECT( rview->spin ), "up_click",
                GTK_SIGNAL_FUNC( rowview_spin_up_cb ), rview );
        gtk_signal_connect( GTK_OBJECT( rview->spin ), "down_click",
                GTK_SIGNAL_FUNC( rowview_spin_down_cb ), rview );
        gtk_widget_show( rview->spin );
	set_tooltip( rview->spin, "Click to open or close class" );

        /* Make name button.
         */
        rview->but = gtk_button_new();
        gtk_widget_show( rview->but );
        doubleclick_add( rview->but, FALSE,
                rowview_single_cb, rview, rowview_double_cb, rview );
        rview->label = gtk_label_new( "" );
        gtk_misc_set_alignment( GTK_MISC( rview->label ), 1, 0 );
        gtk_misc_set_padding( GTK_MISC( rview->label ), 2, 0 );
        gtk_container_add( GTK_CONTAINER( rview->but ), rview->label );
        gtk_widget_show( rview->label );
        gtk_signal_connect( GTK_OBJECT( rview->but ), "enter",
                GTK_SIGNAL_FUNC( rowview_flash_help ), rview );
        gtk_signal_connect( GTK_OBJECT( rview->but ), "enter",
                GTK_SIGNAL_FUNC( rowview_show_dependants ), rview );
        gtk_signal_connect( GTK_OBJECT( rview->but ), "leave",
                GTK_SIGNAL_FUNC( rowview_hide_dependants ), rview );
}

GtkType
rowview_get_type( void )
{
	static GtkType rowview_type = 0;

	if( !rowview_type ) {
		static const GtkTypeInfo rview_info = {
			"Rowview",
			sizeof( Rowview ),
			sizeof( RowviewClass ),
			(GtkClassInitFunc) rowview_class_init,
			(GtkObjectInitFunc) rowview_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		rowview_type = gtk_type_unique( TYPE_VIEW, &rview_info );
	}

	return( rowview_type );
}

View *
rowview_new( void )
{
	Rowview *rview = gtk_type_new( TYPE_ROWVIEW );

	return( VIEW( rview ) );
}

/* Find the position and size of a row in the enclosing GtkLayout.
 */
void
rowview_get_position( Rowview *rview, int *x, int *y, int *w, int *h )
{
	Columnview *cview = view_get_columnview( VIEW( rview ) );
	Workspaceview *wview = cview->wview;

	if( CALC_DISPLAY_LED ) {
		*x = rview->led->allocation.x;
		*y = rview->led->allocation.y;

		*w = rview->led->allocation.width;
		*h = rview->led->allocation.height;
	}
	else {
		*x = rview->but->allocation.x;
		*y = rview->but->allocation.y;

		*w = 0;
		*h = 0;
	}

        *w += rview->but->allocation.width;
        *h += rview->but->allocation.height;

        *w += GTK_WIDGET( rview->rhsview )->allocation.width;
        *h += GTK_WIDGET( rview->rhsview )->allocation.height;

        *x += cview->main->allocation.x + wview->vp.left;
        *y += cview->main->allocation.y + wview->vp.top;
}

/* Hide/show a row. We can't use MODEL( row )->display for this, since tables
 * don't like it :-( just _show()/_hide() rather then _create()/_destroy()
 */
void 
rowview_set_visible( Rowview *rview, gboolean visible )
{
	Row *row = ROW( VIEW( rview )->model );

	if( rview->visible != visible ) {
		widget_visible( rview->but, visible );
		widget_visible( rview->spin, visible && row->isclass );
		widget_visible( rview->led, visible && CALC_DISPLAY_LED );
		if( rview->rhsview )
			widget_visible( GTK_WIDGET( rview->rhsview ), visible );
		rview->visible = visible;
	}
}

gboolean 
rowview_get_visible( Rowview *rview )
{
	return( rview->visible );
}
