/* Search paths for files.
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

/* just load .defs/.wses from "."
#define DEBUG
 */

#include "ip.h"

/* List of directories we have visited this session. Added to by fsb stuff.
 */
GSList *path_session = NULL;

/* Default search paths if prefs fail.
 */
GSList *path_start_default = NULL;
GSList *path_search_default = NULL;

/* Compiled reg-exes go here.
 */
static Wild *regular_expression = NULL;

/* Turn a search path (eg. "/pics/lr:/pics/hr") into a list of directory names.
 */
GSList *
path_parse( const char *path )
{
	GSList *op = NULL;
	const char *p;
	const char *e;
	int len;
	char name[PATHLEN + 1];

	for( p = path; *p; p = e ) {
		/* Find the start of the next component, or the NULL
		 * character.
		 */
		if( !(e = strchr( p, IM_PATH_SEP )) )
			e = p + strlen( p );
		len = e - p + 1;

		/* Copy to our buffer, turn to string.
		 */
		im_strncpy( name, p, IM_MIN( len, PATHLEN ) );

		/* Add to path list.
		 */
		op = g_slist_append( op, im_strdupn( name ) );

		/* Skip IM_PATH_SEP.
		 */
		if( *e == IM_PATH_SEP )
			e++;
	}

	return( op );
}

/* Free a path.
 */
void
path_free( GSList *path )
{
	slist_map( path, (SListMapFn) im_free, NULL );
	g_slist_free( path );
}

/* Sub-fn of below. Add length of this string + 1 (for ':').
 */
static int
path_add_component( const char *str, int c )
{
	return( c + strlen( str ) + 1 );
}

/* Sub-fn of below. Copy string to buffer, append ':', return new end.
 */
static char *
path_add_string( const char *str, char *buf )
{
	strcpy( buf, str );
	strcat( buf, IM_PATH_SEP_STR );

	return( buf + strlen( buf ) );
}

/* Turn a list of directory names into a search path.
 */
char *
path_unparse( GSList *path )
{	
	int len = (int) slist_fold( path, 0, 
		(SListFoldFn) path_add_component, NULL );
	char *buf = im_malloc( NULL, len + 1 );

	/* Build new string.
	 */
	slist_fold( path, buf, (SListFoldFn) path_add_string, NULL );

	/* Fix '\0' to remove trailing IM_PATH_SEP.
	 */
	if( len > 0 )
		buf[ len - 1 ] = '\0';

	return( buf );
}

/* Test for string matches pattern. If the match is successful, call a user 
 * function.
 */
static void *
path_match_pattern( const char *dir, const char *name, 
	path_map_fn fn, void *a, void *b, void *c )
{
	/* Match.
	 */
	if( wild_match( regular_expression, name ) ) {
		char buf[PATHLEN + 10];

		/* Make up whole path name.
		 */
		im_snprintf( buf, PATHLEN, 
			"%s" IM_DIR_SEP_STR "%s", dir, name );

		return( fn( buf, a, b, c ) );
	}

	return( NULL );
}

/* Scan a directory, calling a function for every entry. Abort scan if 
 * function returns non-NULL.
 */
static void *
path_scan_dir( const char *dir, path_map_fn fn, void *a, void *b, void *c )
{	
	DIR *dp;
	struct dirent *d;
	void *ru;

	if( !(dp = (DIR *) call_string( (call_string_fn) opendir,
		dir, NULL, NULL, NULL )) ) 
		return( NULL );

	while( (d = readdir( dp )) ) 
		if( (ru = path_match_pattern( dir, d->d_name, fn, a, b, c )) ) {
			if( closedir( dp ) )
				error( "panic: closedir failed!" );
			return( ru );
		}

	if( closedir( dp ) )
		error( "panic: closedir failed!" );

	return( NULL );
}

/* Compile a reg-ex.
 */
static int
path_compile( const char *patt )
{
	FREEF( wild_destroy, regular_expression );

	if( !(regular_expression = wild_new( patt )) )
		return( -1 );

	return( 0 );
}

static void *
path_str_eq( const char *s1, const char *s2 )
{
        if( strcmp( s1, s2 ) == 0 )
                return( (void *) s1 );
        else
                return( NULL );
}

static void *
path_next_item( const char *filename, 
	GSList **previous, path_map_fn fn, void *a )
{
	char *p = strrchr( filename, IM_DIR_SEP );
	void *ru;

	assert( p );
	if( !slist_map( *previous, (SListMapFn) path_str_eq, p + 1 ) ) {
		*previous = g_slist_prepend( *previous, g_strdup( p + 1 ) );
		if( (ru = fn( filename, a, NULL, NULL )) )
			return( ru );
	}

	return( NULL );
}

/* Scan a search path, applying a function to every file name which matches a
 * pattern. If the user function returns NULL, keep looking, otherwise return
 * its result. We return NULL on error, or if the user function returns NULL
 * for all filenames which match. 
 *
 * Remove duplicates: if fred.wombat is in the first and second dirs on the
 * path, only apply to the first occurence.

 	FIXME ... speed up with a hash and a (date based) cache at some point

 */
void *
path_map_exact( GSList *path, const char *patt, path_map_fn fn, void *a )
{	
	GSList *previous;
	void *ru;

	if( path_compile( patt ) )
		return( NULL );

	previous = NULL;
	ru = slist_map4( path, 
		(SListMap4Fn) path_scan_dir, (void *) path_next_item, 
			&previous, (void *) fn, a );
	FREEF( slist_free_all, previous );

	return( ru );
}

/* As above, but walk the session path too.
 */
void *
path_map( GSList *path, const char *patt, path_map_fn fn, void *a )
{	
	GSList *previous;
	void *ru;

	if( path_compile( patt ) )
		return( NULL );

	previous = NULL;
	ru = slist_map4( path, 
		(SListMap4Fn) path_scan_dir, (void *) path_next_item, 
			&previous, (void *) fn, a );
	if( !ru )
		ru = slist_map4( path_session, 
			(SListMap4Fn) path_scan_dir, (void *) path_next_item, 
				&previous, (void *) fn, a );
	FREEF( slist_free_all, previous );

	return( ru );
}

/* As above, but scan a single directory.
 */
void *
path_map_dir( const char *dir, const char *patt, path_map_fn fn, void *a )
{	
	void *ru;

	if( path_compile( patt ) )
		return( NULL );
	if( !(ru = path_scan_dir( dir, fn, a, NULL, NULL )) )
		/* Not found? Maybe - error message anyway.
		 */
		ierrors( "File \"%s\" not found", patt );

	return( ru );
}

/* Search for a file, and return it's path. NULL for not found. Return a new
 * string.
 */
char *
path_find_file( GSList *path, const char *name )
{
	char buf[FILENAME_MAX];
	char *fname;

	/* Try file name exactly.
	 */
	if( existsf( "%s", name ) )
		return( im_strdupn( name ) );

	/* Drop directory and search datapath for filename.
	 */
	expand_variables( name, buf );
	if( !(fname = path_map( path, im__skip_dir( buf ), 
		(path_map_fn) im_strdupn, NULL ) )) {
		ierrors( "file \"%s\" not found on path", im__skip_dir( buf ) );
		return( NULL );
	}

	return( fname );
}

/* Add a directory to the session path, if it's not there already.
 */
void
path_add( const char *dir )
{
        if( !slist_map( path_session, 
		(SListMapFn) path_str_eq, (gpointer) dir ) ) 
                path_session = g_slist_prepend( path_session, 
			im_strdup( NULL, dir ) );
}

void
path_init( void )
{
#ifdef DEBUG
	printf( "path_init: loading start from \".\" only\n" );
	path_start_default = path_parse( "." );
	path_search_default = path_parse( "." );
#else /*DEBUG*/
	path_start_default = path_parse( 
		IP_SAVEDIR IM_DIR_SEP_STR "start" IM_PATH_SEP_STR
		"$VIPSHOME" IM_DIR_SEP_STR "share" IM_DIR_SEP_STR 
		PACKAGE IM_DIR_SEP_STR "start" );
	path_search_default = path_parse( 
		IP_SAVEDIR IM_DIR_SEP_STR "data" IM_PATH_SEP_STR
		"$VIPSHOME" IM_DIR_SEP_STR "share" IM_DIR_SEP_STR 
		PACKAGE IM_DIR_SEP_STR "data" IM_PATH_SEP_STR "." );
#endif /*DEBUG*/
}
