/* an input matrix 
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

/*
#define DEBUG
 */

#include "ip.h"

static ClassmodelClass *parent_class = NULL;

static void
matrix_destroy( GtkObject *object )
{
	Matrix *matrix;

	g_return_if_fail( object != NULL );
	g_return_if_fail( IS_MATRIX( object ) );

	matrix = MATRIX( object );

#ifdef DEBUG
	printf( "matrix_destroy\n" );
#endif /*DEBUG*/

	/* My instance destroy stuff.
	 */
	FREE( matrix->filename );
	FREE( matrix->value );

	GTK_OBJECT_CLASS( parent_class )->destroy( object );
}

/* Rearrange our model for a new width/height.
 */
static gboolean
matrix_resize( Matrix *matrix, int width, int height )
{
	double *value;
	int x, y, i;

	if( width == matrix->width && height == matrix->height )
		return( TRUE );

	if( !(value = IM_ARRAY( NULL, width * height, double )) )
		return( FALSE );

	/* Set what we can with values from the old matrix.
	 */
	for( i = 0, y = 0; y < height; y++ )
		for( x = 0; x < width; x++, i++ ) 
			if( y < matrix->height && x < matrix->width ) 
				value[i] = matrix->value[x + y*matrix->width];
			else 
				value[i] = 0.0;

	/* Install new values.
	 */
	FREE( matrix->value );
	matrix->value = value;
	matrix->width = width;
	matrix->height = height;

	return( TRUE );
}

/* Widgets for matrix edit.
 */
typedef struct _MatrixEdit {
	iDialog *idlg;

	Matrix *matrix;

	GtkWidget *width;
	GtkWidget *height;
	GtkWidget *display;
} MatrixEdit;

/* Done button hit.
 */
/*ARGSUSED*/
static void
matrix_done_cb( iWindow *iwnd, void *client, 
	iWindowNotifyFn nfn, void *sys )
{
	MatrixEdit *eds = (MatrixEdit *) client;

	int width, height;

	/* Parse values.
	 */
	eds->matrix->display = (MatrixDisplayType) get_goption( eds->display );
	if( !get_geditable_pint( eds->width, &width ) ||
		!get_geditable_pint( eds->height, &height ) ||
		!matrix_resize( eds->matrix, width, height ) ) {
		nfn( sys, IWINDOW_ERROR );
		return;
	}

	/* Rebuild object.
	 */
	classmodel_update( CLASSMODEL( eds->matrix ) );
	symbol_recalculate_all();

	nfn( sys, IWINDOW_TRUE );
}

/* Build the insides of matrix edit.
 */
static void
matrix_buildedit( iDialog *idlg, GtkWidget *work, MatrixEdit *eds )
{
	Matrix *matrix = eds->matrix;

        GtkWidget *hb;

        /* Index with MatrixType.
         */
        static const char *display_names[] = {
                "Text",
                "Sliders",
                "Toggle buttons",
                "Text, plus scale and offset"
        };

        hb = gtk_hbox_new( TRUE, 2 );
        eds->width = build_glabeltext( hb, "Width:" );
	idialog_init_entry( idlg, eds->width, 
		"Width of matrix", "%d", matrix->width );
        eds->height = build_glabeltext( hb, "Height:" );
	idialog_init_entry( idlg, eds->height, 
		"Height of matrix", "%d", matrix->height );
        gtk_box_pack_start( GTK_BOX( work ), hb, TRUE, TRUE, 2 );

        eds->display = build_goption( work, "Display as",
                display_names, IM_NUMBER( display_names ), NULL, NULL );
        set_goption( eds->display, matrix->display );

        gtk_widget_show_all( work );
}

/* Pop up a matrix edit box.
 */
static void 
matrix_edit( GtkWidget *parent, Model *model )
{
	Matrix *matrix = MATRIX( model );
	MatrixEdit *eds = IM_NEW( NULL, MatrixEdit );
	GtkWidget *idlg;

	eds->matrix = matrix;

	idlg = idialog_new();
	iwindow_set_title( IWINDOW( idlg ), "Edit matrix" );
	idialog_set_build( IDIALOG( idlg ), 
		(iWindowBuildFn) matrix_buildedit, eds, NULL, NULL );
	idialog_set_callbacks( IDIALOG( idlg ), 
		iwindow_true_cb, NULL, NULL,
		idialog_free_client, eds );
	idialog_add_ok( IDIALOG( idlg ), matrix_done_cb, "Set matrix" );
	idialog_set_parent( IDIALOG( idlg ), parent );
	iwindow_build( IWINDOW( idlg ) );

	gtk_widget_show( GTK_WIDGET( idlg ) );
}

static xmlNode *
matrix_save( Model *model, xmlNode *xnode )
{
	Matrix *matrix = MATRIX( model );

	xmlNode *xthis;
	const int n = matrix->width * matrix->height;

	if( !(xthis = MODEL_CLASS( parent_class )->save( model, xnode )) )
		return( NULL );

	if( CLASSMODEL( model )->edited ) {
		if( !set_dlprop( xthis, "value", matrix->value, n ) ||
			!set_prop( xthis, "width", "%d", matrix->width ) ||
			!set_prop( xthis, "height", "%d", matrix->height ) ||
			!set_prop( xthis, "display", "%d", 
				(int) matrix->display ) ||
			!set_prop( xthis, "scale", "%g", matrix->scale ) ||
			!set_prop( xthis, "offset", "%g", matrix->offset ) ||
			!set_sprop( xthis, "filename", matrix->filename ) )
			return( NULL );
	}

	return( xthis );
}

static gboolean
matrix_load( Model *model, 
	ModelLoadState *state, Model *parent, xmlNode *xnode )
{
	Matrix *matrix = MATRIX( model );
	char buf[NAMELEN];

	if( !IS_RHS( parent ) ) {
		ierrors( "matrix_load: can only add a matrix to a rhs" );
		return( FALSE );
	}

	assert( sizeof( matrix->display ) == sizeof( int ) );

	FREE( matrix->value );
	if( get_dlprop( xnode, "value", &matrix->value ) &&
		get_iprop( xnode, "width", &matrix->width ) &&
		get_iprop( xnode, "height", &matrix->height ) &&
		get_iprop( xnode, "display", (int *) &matrix->display ) &&
		get_dprop( xnode, "scale", &matrix->scale ) &&
		get_dprop( xnode, "offset", &matrix->offset ) &&
		get_sprop( xnode, "filename", buf, NAMELEN ) ) {
		SETSTR( matrix->filename, buf );
		classmodel_set_edited( CLASSMODEL( model ), TRUE );
	}

	return( MODEL_CLASS( parent_class )->load( model, 
		state, parent, xnode ) );
}

/* Update Matrix from heap.
 */
static gboolean
matrix_class_get( Classmodel *classmodel, PElement *root )
{
	Matrix *matrix = MATRIX( classmodel );
	int w, h;
	int display;
	char buf[MAX_STRSIZE];

	if( !class_get_member_matrix_size( root, MEMBER_VALUE, &w, &h ) ||
		!matrix_resize( matrix, w, h ) ||
		!class_get_member_matrix( root, MEMBER_VALUE, 
			matrix->value, matrix->width * matrix->height, 
			&w, &h ) )
		return( FALSE );

	if( class_get_member_int( root, MEMBER_DISPLAY, &display ) ) {
		matrix->display = (MatrixDisplayType) 
			IM_CLIP( 0, display, MATRIX_DISPLAY_LAST - 1 );
	}
	(void) class_get_member_real( root, MEMBER_SCALE, &matrix->scale );
	(void) class_get_member_real( root, MEMBER_OFFSET, &matrix->offset );
	if( class_get_member_string( root, MEMBER_FILENAME, buf, MAX_STRSIZE ) )
		SETSTR( matrix->filename, buf );

	return( TRUE );
}

/* Make a new "fn value" application.
 */
static gboolean
matrix_class_new( Classmodel *classmodel, PElement *fn, PElement *out )
{
	Heap *hi = reduce_context->hi;
	Matrix *matrix = MATRIX( classmodel );

	PElement rhs;

	/* Make application nodes.
	 */
	heap_appl_init( out, fn );
	if( !heap_appl_add( hi, out, &rhs ) || 
		!heap_matrix_new( hi, 
			matrix->width, matrix->height, matrix->value, &rhs ) ||
		!heap_appl_add( hi, out, &rhs ) ||
		!heap_real_new( hi, matrix->scale, &rhs ) ||
		!heap_appl_add( hi, out, &rhs ) ||
		!heap_real_new( hi, matrix->offset, &rhs ) ||
		!heap_appl_add( hi, out, &rhs ) ||
		!heap_string_new( hi, matrix->filename, &rhs ) ||
		!heap_appl_add( hi, out, &rhs ) ||
		!heap_real_new( hi, matrix->display, &rhs ) )
		return( FALSE );

	return( TRUE );
}

static void
matrix_graphic_save_cb( iWindow *iwnd, 
	void *client, iWindowNotifyFn nfn, void *sys )
{
	Filesel *filesel = FILESEL( iwnd );
	const char *filename = filesel_get_filename( filesel );
	Matrix *matrix = MATRIX( client );
	DOUBLEMASK *dmask;

	if( !(dmask = matrix_model_to_dmask( matrix )) ) {
		nfn( sys, IWINDOW_ERROR );
		return;
	}

	if( im_write_dmask_name( dmask, filename ) ) {
		verrors( "Error writing mask file \"%s\"", filename );
		FREEF( im_free_dmask, dmask );

		nfn( sys, IWINDOW_ERROR );
		return;
	}
	FREEF( im_free_dmask, dmask );

	nfn( sys, IWINDOW_TRUE );
}

static void
matrix_graphic_save( GtkWidget *parent, Classmodel *classmodel )
{
	Matrix *matrix = MATRIX( classmodel );
	GtkWidget *filesel = filesel_new();
	BufInfo buf;
	char txt[100];

	buf_init_static( &buf, txt, 100 );
	row_qualified_name( HEAPMODEL( matrix )->row, &buf );
	iwindow_set_title( IWINDOW( filesel ), 
		"Save Matrix %s", buf_all( &buf ) );
	filesel_set_flags( FILESEL( filesel ), FALSE, TRUE );
	filesel_set_type( FILESEL( filesel ), filesel_type_matrix, 0 ); 
	idialog_set_parent( IDIALOG( filesel ), parent );
	filesel_set_done( FILESEL( filesel ), 
		matrix_graphic_save_cb, classmodel );
	iwindow_build( IWINDOW( filesel ) );

	if( matrix->filename )
		filesel_set_filename( FILESEL( filesel ), matrix->filename );

	gtk_widget_show( GTK_WIDGET( filesel ) );
}

static void
matrix_graphic_replace_cb( iWindow *iwnd, 
	void *client, iWindowNotifyFn nfn, void *sys )
{
	Filesel *filesel = FILESEL( iwnd );
	const char *filename = filesel_get_filename( filesel );
	Matrix *matrix = MATRIX( client );
	Row *row = HEAPMODEL( matrix )->row;
	iText *itext = ITEXT( HEAPMODEL( matrix )->rhs->itext );
	DOUBLEMASK *dmask;
	char txt[MAX_STRSIZE];
	BufInfo buf;

	if( !(dmask = im_read_dmask( filename )) ) {
		nfn( sys, IWINDOW_ERROR );
		return;
	}

	buf_init_static( &buf, txt, MAX_STRSIZE );
	matrix_dmask_to_ip( dmask, &buf );
	im_free_dmask( dmask );

	itext_set_formula( itext, buf_all( &buf ) );
	itext_set_edited( itext, TRUE );
	(void) expr_dirty( row->expr, link_serial_new() );

	symbol_recalculate_all();

	nfn( sys, IWINDOW_TRUE );
}

static void
matrix_graphic_replace( GtkWidget *parent, Classmodel *classmodel )
{
	Matrix *matrix = MATRIX( classmodel );
	GtkWidget *filesel = filesel_new();
	BufInfo buf;
	char txt[100];

	buf_init_static( &buf, txt, 100 );
	row_qualified_name( HEAPMODEL( matrix )->row, &buf );
	iwindow_set_title( IWINDOW( filesel ), 
		"Replace Matrix %s", buf_all( &buf ) );
	filesel_set_flags( FILESEL( filesel ), FALSE, FALSE );
	filesel_set_type( FILESEL( filesel ), filesel_type_matrix, 0 ); 
	idialog_set_parent( IDIALOG( filesel ), parent );
	filesel_set_done( FILESEL( filesel ), 
		matrix_graphic_replace_cb, classmodel );
	iwindow_build( IWINDOW( filesel ) );

	if( matrix->filename )
		filesel_set_filename( FILESEL( filesel ), matrix->filename );

	gtk_widget_show( GTK_WIDGET( filesel ) );
}

static void
matrix_class_init( MatrixClass *klass )
{
	GtkObjectClass *object_class = (GtkObjectClass *) klass;
	ModelClass *model_class = (ModelClass *) klass;
	ClassmodelClass *classmodel_class = (ClassmodelClass *) klass;

	parent_class = gtk_type_class( TYPE_CLASSMODEL );

	object_class->destroy = matrix_destroy;

	/* Create signals.
	 */

	/* Init methods.
	 */
	model_class->view_new = matrixview_new;
	model_class->edit = matrix_edit;
	model_class->save = matrix_save;
	model_class->load = matrix_load;

	classmodel_class->class_get = matrix_class_get;
	classmodel_class->class_new = matrix_class_new;
	classmodel_class->graphic_save = matrix_graphic_save;
	classmodel_class->graphic_replace = matrix_graphic_replace;

	/* Static init.
	 */
	model_register_loadable( MODEL_CLASS( klass ) );
}

static void
matrix_init( Matrix *matrix )
{
#ifdef DEBUG
	printf( "matrix_init\n" );
#endif /*DEBUG*/

	matrix->value = NULL;
        matrix->width = 0;
	matrix->height = 0;
	matrix->display = MATRIX_DISPLAY_TEXT;
	matrix->scale = 1.0;
	matrix->offset = 0.0;
	matrix->filename = NULL;

	model_set( MODEL( matrix ), CLASS_MATRIX, NULL );
}

GtkType
matrix_get_type( void )
{
	static GtkType matrix_type = 0;

	if( !matrix_type ) {
		static const GtkTypeInfo info = {
			"Matrix",
			sizeof( Matrix ),
			sizeof( MatrixClass ),
			(GtkClassInitFunc) matrix_class_init,
			(GtkObjectInitFunc) matrix_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		matrix_type = gtk_type_unique( TYPE_CLASSMODEL, &info );
	}

	return( matrix_type );
}

Classmodel *
matrix_new( Rhs *rhs )
{
	Matrix *matrix = gtk_type_new( TYPE_MATRIX );

	model_child_add( MODEL( rhs ), MODEL( matrix ), -1 );

	return( CLASSMODEL( matrix ) );
}

/* Guess a display type from a filename.
 */
static int
matrix_guess_display( const char *fname )
{
	const char *p;
	int i;

	/* Choose display type based on filename suffix ... rec 
	 * displays as 1, mor displays as 2, all others display as 0.
	 */
	static const char *suffs[] = {
		".con",
		".rec",
		".mor"
	};

	if( !fname || !(p = strrchr( fname, '.' )) ) 
		return( 0 );

	for( i = 0; i < IM_NUMBER( suffs ); i++ )
		if( strcasecmp( p, suffs[i] ) == 0 )
			return( i );

	return( 0 );
}

/* Make an ip definition out of a DOUBLEMASK.
 */
void
matrix_dmask_to_ip( DOUBLEMASK *dmask, BufInfo *buf )
{
	int x, y;

	/* Build matrix expression.
	 */
	buf_appends( buf, CLASS_MATRIX " " );

	buf_appends( buf, "[" );
	for( y = 0; y < dmask->ysize; y++ ) {
		buf_appends( buf, "[" );
		for( x = 0; x < dmask->xsize; x++ ) {
			buf_appendf( buf, "%g", 
				dmask->coeff[x + y*dmask->xsize] );
			if( x != dmask->xsize - 1 )
				buf_appends( buf, "," );
		}
		buf_appends( buf, "]" );
		if( y != dmask->ysize - 1 )
			buf_appends( buf, "," );
	}
	buf_appends( buf, "]" );

	buf_appendf( buf, "(%g) (%g) \"%s\" %d", 
		dmask->scale, dmask->offset, dmask->filename,
		matrix_guess_display( dmask->filename ) );
}

/* Make a heap object out of a DOUBLEMASK.
 */
gboolean
matrix_dmask_to_heap( Heap *hi, DOUBLEMASK *dmask, PElement *out )
{
	Symbol *sym = stable_find( symbol_root->expr->compile->locals, 
		CLASS_MATRIX );

	PElement rhs;

	if( !sym || !sym->expr || !PEISCONSTRUCTOR( &sym->expr->root ) )
		return( FALSE );
	heap_appl_init( out, &sym->expr->root );

	if( !heap_appl_add( hi, out, &rhs ) || 
		!heap_matrix_new( hi, 
			dmask->xsize, dmask->ysize, dmask->coeff, &rhs ) ||
		!heap_appl_add( hi, out, &rhs ) ||
		!heap_real_new( hi, dmask->scale, &rhs ) ||
		!heap_appl_add( hi, out, &rhs ) ||
		!heap_real_new( hi, dmask->offset, &rhs ) ||
		!heap_appl_add( hi, out, &rhs ) ||
		!heap_string_new( hi, dmask->filename, &rhs ) ||
		!heap_appl_add( hi, out, &rhs ) ||
		!heap_real_new( hi, 
			matrix_guess_display( dmask->filename ), &rhs ) )
		return( FALSE );

	return( TRUE );
}

/* Cast an IMASK to a DMASK.
 */
DOUBLEMASK *
matrix_imask_to_dmask( INTMASK *imask )
{
	DOUBLEMASK *dmask;
	int i;

	if( !(dmask = im_create_dmask( imask->filename, 
		imask->xsize, imask->ysize )) ) {
		verrors( "unable to make dmask" );
		return( NULL );
	}

	dmask->scale = imask->scale;
	dmask->offset = imask->offset;
	for( i = 0; i < imask->xsize * imask->ysize; i++ )
		dmask->coeff[i] = imask->coeff[i];

	return( dmask );
}

/* Cast a DMASK to an IMASK.
 */
INTMASK *
matrix_dmask_to_imask( DOUBLEMASK *dmask )
{
	INTMASK *imask;
	int i;

	if( !(imask = im_create_imask( dmask->filename, 
		dmask->xsize, dmask->ysize )) ) {
		verrors( "unable to make imask" );
		return( NULL );
	}

	imask->scale = dmask->scale;
	imask->offset = dmask->offset;
	for( i = 0; i < dmask->xsize * dmask->ysize; i++ )
		imask->coeff[i] = dmask->coeff[i];

	return( imask );
}

/* Make a heap object out of an INTMASK.
 */
gboolean
matrix_imask_to_heap( Heap *hi, INTMASK *imask, PElement *out )
{
	DOUBLEMASK *dmask;

	if( !(dmask = matrix_imask_to_dmask( imask )) )
		return( FALSE );
	if( !matrix_dmask_to_heap( hi, dmask, out ) ) {
		im_free_dmask( dmask );
		return( FALSE );
	}
	im_free_dmask( dmask );

	return( TRUE );
}

/* Make a DOUBLEMASK out of an ip value.
 */
DOUBLEMASK *
matrix_ip_to_dmask( PElement *root )
{
	gboolean result;
	char buf[MAX_STRSIZE];
	DOUBLEMASK *dmask;
	double scale, offset;
	char *filename;
	int width, height;

	if( !heap_isclass( root, &result ) ) 
		return( NULL );
	if( !result ) {
		ierrors( "not a class" );
		return( NULL );
	}

	if( !class_get_member_matrix_size( root, 
		MEMBER_VALUE, &width, &height ) )
		return( NULL );

	if( class_get_member_string( root, MEMBER_FILENAME, buf, MAX_STRSIZE ) )
		filename = buf;
	else
		filename = "untitled";

	if( !(dmask = im_create_dmask( filename, width, height )) )
		return( NULL );

	if( !class_get_member_matrix( root, MEMBER_VALUE, 
		dmask->coeff, width * height, &width, &height ) ) {
		FREEF( im_free_dmask, dmask );
		return( FALSE );
	}

	if( !class_get_member_real( root, MEMBER_SCALE, &scale ) )
		scale = 1.0;
	if( !class_get_member_real( root, MEMBER_OFFSET, &offset ) )
		offset = 0.0;
	dmask->scale = scale;
	dmask->offset = offset;

	return( dmask );
}

/* Make an INTMASK out of an ip value.
 */
INTMASK *
matrix_ip_to_imask( PElement *root )
{
	DOUBLEMASK *dmask;
	INTMASK *imask;

	if( !(dmask = matrix_ip_to_dmask( root )) )
		return( NULL );

	if( !(imask = matrix_dmask_to_imask( dmask )) ) {
		FREEF( im_free_dmask, dmask );
		return( NULL );
	}

	return( imask );
}

DOUBLEMASK *
matrix_model_to_dmask( Matrix *matrix )
{
	DOUBLEMASK *dmask;
	int i;

	if( !(dmask = im_create_dmask( matrix->filename, 
		matrix->width, matrix->height )) ) {
		verrors( "unable to make dmask" );
		return( NULL );
	}

	dmask->scale = matrix->scale;
	dmask->offset = matrix->offset;
	for( i = 0; i < matrix->width * matrix->height; i++ )
		dmask->coeff[i] = matrix->value[i];

	return( dmask );
}

gboolean
matrix_dmask_to_model( Matrix *matrix, DOUBLEMASK *dmask )
{
	int i;

	if( !matrix_resize( matrix, dmask->xsize, dmask->ysize ) ) 
		return( FALSE );

	matrix->scale = dmask->scale;
	matrix->offset = dmask->offset;
	for( i = 0; i < matrix->width * matrix->height; i++ )
		matrix->value[i] = dmask->coeff[i];

	matrix_guess_display( dmask->filename );

	return( TRUE );
}

