/* an ip region class object in a workspace
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

/*
#define DEBUG
 */

#include "ip.h"

static IimageClass *parent_class = NULL;

void
iregion_instance_destroy( IregionInstance *instance )
{
	instance->image_class.type = ELEMENT_NOVAL;
	instance->image_class.ele = NULL;
	FREEF( imageinfo_destroy_nonheap, instance->image );
	instance->classmodel = NULL;
	instance->iregiongroup = NULL;
	heap_unregister_element( reduce_context->hi, &instance->image_class );
}

void
iregion_instance_init( IregionInstance *instance, Classmodel *classmodel )
{
	instance->image_class.type = ELEMENT_NOVAL;
	instance->image_class.ele = NULL;
	instance->image = NULL;
	instance->area.left = 0;
	instance->area.top = 0;
	instance->area.width = 0;
	instance->area.height = 0;
	instance->classmodel = classmodel;
	instance->iregiongroup = NULL;

	heap_register_element( reduce_context->hi, &instance->image_class );
}

gboolean
iregion_instance_update( IregionInstance *instance, PElement *root )
{
	PElement image;
	PElement image_class;
	Imageinfo *value;
	int left, top, width, height;

	if( !class_get_member_class( root, MEMBER_IMAGE, "Image", &image ) ||
		!class_get_member_image( &image, MEMBER_VALUE, &value ) ||
		!class_get_member_int( root, MEMBER_LEFT, &left ) ||
		!class_get_member_int( root, MEMBER_TOP, &top ) ||
		!class_get_member_int( root, MEMBER_WIDTH, &width ) ||
		!class_get_member_int( root, MEMBER_HEIGHT, &height ) )
		return( FALSE );

	instance->area.left = left;
	instance->area.top = top;
	instance->area.width = width;
	instance->area.height = height;

	FREEF( imageinfo_destroy_nonheap, instance->image );
	instance->image = value;
	imageinfo_dup_nonheap( value );

	PEPOINTE( &image_class, &instance->image_class );
	PEPUTPE( &image_class, &image );

	return( TRUE );
}

static void
iregion_destroy( GtkObject *object )
{
	Iregion *iregion;

#ifdef DEBUG
	printf( "iregion_destroy\n" );
#endif /*DEBUG*/

	g_return_if_fail( object != NULL );
	g_return_if_fail( IS_IREGION( object ) );

	iregion = IREGION( object );

	/* My instance destroy stuff.
	 */
	iregion_instance_destroy( &iregion->instance );

	GTK_OBJECT_CLASS( parent_class )->destroy( object );
}

/* Widgets for iregion edit.
 */
typedef struct _IregionEdit {
	iDialog *idlg;

	Classmodel *classmodel;

	GtkWidget *left;
	GtkWidget *top;
	GtkWidget *width;
	GtkWidget *height;
} IregionEdit;

/* Done button hit.
 */
static void
iregion_done_cb( iWindow *iwnd, void *client, 
	iWindowNotifyFn nfn, void *sys )
{
	IregionEdit *eds = (IregionEdit *) client;
	IregionInstance *instance = classmodel_get_instance( eds->classmodel );

	int left, top, width, height;

	if( !get_geditable_int( eds->left, &left ) ||
		!get_geditable_int( eds->top, &top ) ||
		!get_geditable_int( eds->width, &width ) ||
		!get_geditable_int( eds->height, &height ) ) {
		nfn( sys, IWINDOW_ERROR );
		return;
	}

	instance->area.left = left;
	instance->area.top = top;
	instance->area.width = width;
	instance->area.height = height;

	classmodel_update( eds->classmodel );
	symbol_recalculate_all();

	nfn( sys, IWINDOW_TRUE );
}

static void
iregion_help_cb( iWindow *iwnd, void *client, 
	iWindowNotifyFn nfn, void *sys )
{
	box_help( GTK_WIDGET( iwnd ), "sec:region" );

	nfn( sys, IWINDOW_TRUE );
}

/* Build the insides of iregion edit.
 */
static void
iregion_buildedit( iDialog *idlg, GtkWidget *vb, IregionEdit *eds )
{
	IregionInstance *instance = classmodel_get_instance( eds->classmodel );

        GtkWidget *hb;

        hb = gtk_hbox_new( FALSE, 2 );
        gtk_container_set_border_width( GTK_CONTAINER( hb ), 2 );
        gtk_box_pack_start( GTK_BOX( vb ), hb, TRUE, TRUE, 2 );

        eds->left = build_glabeltext( hb, "Left:" );
	idialog_init_entry( idlg, eds->left,
		"Left edge of region", "%d", instance->area.left );
        eds->top = build_glabeltext( hb, "Top:" );
	idialog_init_entry( idlg, eds->top,
		"Top edge of region", "%d", instance->area.top );

        hb = gtk_hbox_new( FALSE, 2 );
        gtk_container_set_border_width( GTK_CONTAINER( hb ), 2 );
        gtk_box_pack_start( GTK_BOX( vb ), hb, TRUE, TRUE, 2 );

        eds->width = build_glabeltext( hb, "Width:" );
	idialog_init_entry( idlg, eds->width,
		"Width of region", "%d", instance->area.width );
        eds->height = build_glabeltext( hb, "Height:" );
	idialog_init_entry( idlg, eds->height,
		"Height of region", "%d", instance->area.height );

        gtk_widget_show_all( vb );
}

/* Pop up a iregion edit box. Shared with iarrow.c.
 */
void 
iregion_edit( GtkWidget *parent, Model *model )
{
	Classmodel *classmodel = CLASSMODEL( model );
	IregionEdit *eds = IM_NEW( NULL, IregionEdit );
	GtkWidget *idlg;

	eds->classmodel = classmodel;

	idlg = idialog_new();
	iwindow_set_title( IWINDOW( idlg ), "Edit %s", NN( model->name ) );
	idialog_set_build( IDIALOG( idlg ), 
		(iWindowBuildFn) iregion_buildedit, eds, NULL, NULL );
	idialog_set_callbacks( IDIALOG( idlg ), 
		iwindow_true_cb, iregion_help_cb, NULL, 
		idialog_free_client, eds );
	idialog_add_ok( IDIALOG( idlg ), iregion_done_cb, "Set region" );
	idialog_set_parent( IDIALOG( idlg ), parent );
	iwindow_build( IWINDOW( idlg ) );

	gtk_widget_show( GTK_WIDGET( idlg ) );
}

/* Shared with iarrow.c.
 */
void
iregion_parent_add( Model *child, Model *parent )
{
	MODEL_CLASS( parent_class )->parent_add( child, parent );

	/* Now we're all linked up, make a child model to handle client 
	 * displays on imageviews.
	 */
	(void) iregiongroup_new( CLASSMODEL( child ) );
}

/* Shared with iarrow.c.
 */
xmlNode *
iregion_save( Model *model, xmlNode *xnode )
{
	/* Get our parent class. We can't just use the global parent_class, 
	 * since due to our lame MI scheme, this method may be called for 
	 * iarrow/ipoint etc. as well as iregion ... look up dynamically.
	 */
	gpointer parent_class = PARENT_CLASS_DYNAMIC( model );

	IregionInstance *instance = 
		classmodel_get_instance( CLASSMODEL( model ) );
	Rect *area = &instance->area;

	xmlNode *xthis;

	if( !(xthis = MODEL_CLASS( parent_class )->save( model, xnode )) )
		return( NULL );

	if( CLASSMODEL( model )->edited ) {
		if( !set_prop( xthis, "left", "%d", area->left ) ||
			!set_prop( xthis, "top", "%d", area->top ) ||
			!set_prop( xthis, "width", "%d", area->width ) ||
			!set_prop( xthis, "height", "%d", area->height ) )
			return( NULL );
	}

	return( xthis );
}

/* Shared with iarrow.c.
 */
gboolean
iregion_load( Model *model, 
	ModelLoadState *state, Model *parent, xmlNode *xnode )
{
	gpointer parent_class = PARENT_CLASS_DYNAMIC( model );
	IregionInstance *instance = 
		classmodel_get_instance( CLASSMODEL( model ) );
	Rect *area = &instance->area;

	if( !IS_RHS( parent ) ) {
		ierrors( "iregion_load: can only add an iregion to a rhs" );
		return( FALSE );
	}

	if( get_iprop( xnode, "left", &area->left ) &&
		get_iprop( xnode, "top", &area->top ) &&
		get_iprop( xnode, "width", &area->width ) &&
		get_iprop( xnode, "height", &area->height ) )
		classmodel_set_edited( CLASSMODEL( model ), TRUE );

	return( MODEL_CLASS( parent_class )->load( model, 
		state, parent, xnode ) );
}

/* Need to implement _update_heap(), as not all model fields are directly
 * editable ... some are set only from expr. See also iimage.c. Shared with
 * iarrow.c.
 */
void *
iregion_update_heap( Heapmodel *heapmodel )
{
	gpointer parent_class = PARENT_CLASS_DYNAMIC( heapmodel );
	IregionInstance *instance = 
		classmodel_get_instance( CLASSMODEL( heapmodel ) );
	Expr *expr = heapmodel->row->expr;

	Rect area;
	PElement pe;

	/* Save any model fields that may have been set by _load() and which
	 * might be zapped by _get_instance().
	 */
	area = instance->area;

	/* Look for the base instance, and update from that.
	 */
	if( !class_get_exact( &expr->root, MODEL( heapmodel )->name, &pe ) )
		return( FALSE );
	if( !iregion_instance_update( instance, &pe ) )
		return( heapmodel );

	/* Restore model fields from _load().
	 */
	instance->area = area;

	/* Classmodel _update_heap() will do _instance_new() from the fixed up
	 * model.
	 */
	return( HEAPMODEL_CLASS( parent_class )->update_heap( heapmodel ) );
}

static void *
iregion_caption_update_sub( Iimage *iimage, Iregion *iregion, gboolean *first )
{
	Iimage *our_iimage = IIMAGE( iregion );
	Workspace *ws = HEAPMODEL( iregion )->row->ws;
	Row *row = HEAPMODEL( iimage )->row;

	/* Supress this name in the caption if it's a superclass. If this
	 * thing is on a super, it's on the subclass too ... not helpful to
	 * have it twice.
	 */
	if( !is_super( row->sym ) ) {
		if( *first )
			*first = FALSE;
		else 
			buf_appends( &our_iimage->caption, ", " );

		row_qualified_name_relative( SYMBOL( ws ), 
			row, &our_iimage->caption );
	}

	return( NULL );
}

static void
iregion_caption_update( Iregion *iregion, PElement *root )
{
	const int nimages = g_slist_length( CLASSMODEL( iregion )->iimages );

	gboolean result;
	Iimage *iimage = IIMAGE( iregion );
	gboolean first;
	Compile *compile;

	if( !heap_isclass( root, &result ) || !result )
		return;

	compile = PEGETCLASSCOMPILE( root );
	buf_rewind( &iimage->caption );
	symbol_qualified_name( compile->sym, &iimage->caption );
	buf_appendf( &iimage->caption, " on " );
	if( nimages > 1 )
		buf_appendf( &iimage->caption, "[" );
	first = TRUE;
	slist_map2( CLASSMODEL( iregion )->iimages,
		(SListMap2Fn) iregion_caption_update_sub, iregion, &first );
	if( nimages > 1 )
		buf_appendf( &iimage->caption, "]" );
	buf_appendf( &iimage->caption, " at (%d, %d), size (%d, %d)",
		iregion->instance.area.left, iregion->instance.area.top,
		iregion->instance.area.width, iregion->instance.area.height );
}

static void *
iregion_update_model( Heapmodel *heapmodel )
{
	Iregion *iregion = IREGION( heapmodel );

	if( HEAPMODEL_CLASS( parent_class )->update_model( heapmodel ) )
		return( heapmodel );

	/* Update who-has-displays-on-what stuff.
	 */
	classmodel_iimage_update( CLASSMODEL( iregion ), 
		iregion->instance.image );

	iregion_caption_update( iregion, &heapmodel->row->expr->root );

	model_changed( MODEL( heapmodel ) );

	return( NULL );
}

/* Update Iregion from heap. Shared with iarrow.c.
 */
gboolean
iregion_class_get( Classmodel *classmodel, PElement *root )
{
	gpointer parent_class = PARENT_CLASS_DYNAMIC( classmodel );
	IregionInstance *instance = classmodel_get_instance( classmodel );

#ifdef DEBUG
	printf( "iregion_class_get: " );
	row_name_print( HEAPMODEL( classmodel )->row );
	printf( "\n" );
#endif /*DEBUG*/

	if( !iregion_instance_update( instance, root ) )
		return( FALSE );

	return( CLASSMODEL_CLASS( parent_class )->class_get( 
		classmodel, root ) );
}

/* Make a new "fn value" application. Shared with iarrow.c.
 */
gboolean
iregion_class_new( Classmodel *classmodel, PElement *fn, PElement *out )
{
	Heap *hi = reduce_context->hi;
	IregionInstance *instance = classmodel_get_instance( classmodel );

	PElement rhs;

#ifdef DEBUG
	printf( "iregion_class_new\n" );
#endif /*DEBUG*/

	/* Make application nodes.
	 */
	heap_appl_init( out, fn );
	if( !heap_appl_add( hi, out, &rhs ) ||
		!heap_element_new( hi, &instance->image_class, &rhs ) ||
		!heap_appl_add( hi, out, &rhs ) ||
		!heap_real_new( hi, instance->area.left, &rhs ) ||
		!heap_appl_add( hi, out, &rhs ) ||
		!heap_real_new( hi, instance->area.top, &rhs ) ||
		!heap_appl_add( hi, out, &rhs ) ||
		!heap_real_new( hi, instance->area.width, &rhs ) ||
		!heap_appl_add( hi, out, &rhs ) ||
		!heap_real_new( hi, instance->area.height, &rhs ) )
		return( FALSE );

	return( TRUE );
}

static void *
iregion_get_instance( Classmodel *classmodel )
{
	Iregion *iregion = IREGION( classmodel );

	return( &iregion->instance );
}

static void
iregion_class_init( IregionClass *klass )
{
	GtkObjectClass *object_class = (GtkObjectClass *) klass;
	ModelClass *model_class = (ModelClass *) klass;
	HeapmodelClass *heapmodel_class = (HeapmodelClass *) klass;
	ClassmodelClass *classmodel_class = (ClassmodelClass *) klass;

	parent_class = gtk_type_class( TYPE_IIMAGE );

	/* Create signals.
	 */

	/* Init methods.
	 */
	object_class->destroy = iregion_destroy;

	model_class->view_new = iregionview_new;
	model_class->edit = iregion_edit;
	model_class->parent_add = iregion_parent_add;
	model_class->save = iregion_save;
	model_class->load = iregion_load;

	heapmodel_class->update_heap = iregion_update_heap;
	heapmodel_class->update_model = iregion_update_model;

	classmodel_class->class_get = iregion_class_get;
	classmodel_class->class_new = iregion_class_new;
	classmodel_class->get_instance = iregion_get_instance;

	/* Static init.
	 */
	model_register_loadable( MODEL_CLASS( klass ) );
}

static void
iregion_init( Iregion *iregion )
{
	iregion_instance_init( &iregion->instance, CLASSMODEL( iregion ) );

	model_set( MODEL( iregion ), CLASS_REGION, NULL );
}

GtkType
iregion_get_type( void )
{
	static GtkType iregion_type = 0;

	if( !iregion_type ) {
		static const GtkTypeInfo info = {
			"Iregion",
			sizeof( Iregion ),
			sizeof( IregionClass ),
			(GtkClassInitFunc) iregion_class_init,
			(GtkObjectInitFunc) iregion_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		iregion_type = gtk_type_unique( TYPE_IIMAGE, &info );
	}

	return( iregion_type );
}

Classmodel *
iregion_new( Rhs *rhs )
{
	Iregion *iregion = gtk_type_new( TYPE_IREGION );

	model_child_add( MODEL( rhs ), MODEL( iregion ), -1 );

	return( CLASSMODEL( iregion ) );
}
