/* image management ... a layer over the VIPS IMAGE type
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

/*

jobs:

- reference counting layer ... we im_close() the underlying image when the
  final reference has gone

  imageinfo_new makes images with count == 0, therefore the image will be
  GCed away next time around ... you need to store the pointer somewhere
  and dup it to make it permanent

- cache: several open( "fred.v" )s share a single IMAGE

- lookup table management ... if an operation can work with pixel lookup 
  tables (found by examining a flag in the VIPS function descriptor), then
  instead of operating on the image, the operation runs on the LUT associated
  with that image ... Imageinfo tracks the LUTs representing delayed eval

- dependency tracking ... an imageinfo can require several other imageinfos
  to be open for it to work properly; we follow these dependencies, and
  delay destroying an imageinfo until it's not required by any others

- temp file management ... we can make temp images on disc; we unlink() these
  temps when they're no longer needed

- imageinfo/expr association tracking ... we track when an expr 
  receives an imageinfo as it's value; the info is used to get region views
  to display in the right image ... see expr_value_new()

- paint stuff: also undo/redo buffers, each with a "*_changed" signal

every Imageinfo is associated with a heap: when we GC a heap, we
imageinfo_mark() any references to images inside that heap; after GC we
imageinfo_destroy_heap() any unreferenced imageinfo

extra wrinkle: we allow non-heap pointers to imageinfos (eg. Imagedisplay
classes will hold a pointer to the image they are displaying) ... again,
delay destroy until all non-heap pointers are done as well

 */

#include "ip.h"

/*
#define DEBUG
#define DEBUG_OPEN
#define DEBUG_UNDO
 */

/* Our signals. 
 */
enum {
	CHANGED,	/* Whole thing has changed (eg. imageinfo_file()) */
	AREA_CHANGED,	/* Area of image has changed */
	UNDO_CHANGED,	/* Undo/redo state has changed */
	LAST_SIGNAL
};

static GtkObjectClass *parent_class = NULL;

static guint imageinfo_signals[LAST_SIGNAL] = { 0 };

/* All imageinfos everywhere.
 */
static GSList *imageinfo_all = NULL;

static void *
imageinfo_changed( Imageinfo *imageinfo )
{
	assert( IS_IMAGEINFO( imageinfo ) );

	if( !GTK_OBJECT_DESTROYED( imageinfo ) ) 
		gtk_signal_emit( GTK_OBJECT( imageinfo ), 
			imageinfo_signals[CHANGED] );

	return( NULL );
}

static void *
imageinfo_area_changed( Imageinfo *imageinfo, Rect *dirty )
{
	assert( IS_IMAGEINFO( imageinfo ) );

	if( !GTK_OBJECT_DESTROYED( imageinfo ) ) 
		gtk_signal_emit( GTK_OBJECT( imageinfo ), 
			imageinfo_signals[AREA_CHANGED], dirty );

	return( NULL );
}

static void *
imageinfo_undo_changed( Imageinfo *imageinfo )
{
	assert( IS_IMAGEINFO( imageinfo ) );

	if( !GTK_OBJECT_DESTROYED( imageinfo ) ) 
		gtk_signal_emit( GTK_OBJECT( imageinfo ), 
			imageinfo_signals[UNDO_CHANGED] );

	return( NULL );
}

/* Test imageinfo matches filename.
 */
static void *
imageinfo_test_filename( Imageinfo *imageinfo, const char *name )
{
	if( imageinfo->name && strcmp( imageinfo->name, name ) == 0 )
		return( imageinfo );

	return( NULL );
}

/* Look up an imageinfo by filename. 

	FIXME ... slow!

 */
static Imageinfo *
imageinfo_find( const char *filename )
{
	return( (Imageinfo *) slist_map( imageinfo_all, 
		(SListMapFn) imageinfo_test_filename, (char *) filename ) );
}

/* Prettyprint an imageinfo.
 */
void *
imageinfo_dump( Imageinfo *imageinfo )
{
	printf( "imageinfo->hi = 0x%x\n", (unsigned int) imageinfo->hi );
	if( imageinfo->hi ) 
		dump_heapinfo( imageinfo->hi );

	printf( "imageinfo->marked (refed in heap) = %s\n", 
		bool_to_char( imageinfo->marked ) );
	printf( "imageinfo->count (# non-heap pointers) = %d\n", 
		imageinfo->count );
	printf( "imageinfo->zombie (waiting on last non-heap) = %s\n", 
		bool_to_char( imageinfo->zombie ) );

	printf( "imageinfo->im = 0x%x\n", (unsigned int) imageinfo->im );
	printf( "imageinfo->mapped_im  = 0x%x\n", 
		(unsigned int) imageinfo->mapped_im );
	printf( "imageinfo->identity_lut = 0x%x\n", 
		(unsigned int) imageinfo->identity_lut );
	printf( "imageinfo->underlying = 0x%x\n", 
		(unsigned int) imageinfo->underlying );

	printf( "imageinfo->subii (imageinfo depending on us) = %d\n",
		g_slist_length( imageinfo->subii ) );

	printf( "imageinfo->dfile (unlink on close) = %s\n",
		bool_to_char( imageinfo->dfile ) );
	printf( "imageinfo->name = \"%s\"\n", imageinfo->name );

	printf( "imageinfo->exprs = " );
	slist_map( imageinfo->exprs, 
		(SListMapFn) expr_name_print, NULL );
	printf( "\n" );

	if( imageinfo->underlying ) {
		printf( "underlying:\n" );
		imageinfo_dump( imageinfo->underlying );
	}

	return( NULL );
}

/* Debugging ... check that all imageinfos have been closed, dump any which
 * haven't.
 */
void
imageinfo_check_all_destroyed( void )
{
	slist_map( imageinfo_all, 
		(SListMapFn) imageinfo_dump, NULL );
}

/* imageinfo no longer depends upon in.
 */
static void *
imageinfo_sub_remove( Imageinfo *in, Imageinfo *imageinfo )
{
	assert( g_slist_find( imageinfo->subii, in ) );

	imageinfo->subii = g_slist_remove( imageinfo->subii, in );
	imageinfo_destroy_nonheap( in );

	return( NULL );
}

/* imageinfo depends on in ... add a dependency.
 */
void 
imageinfo_sub_add( Imageinfo *imageinfo, Imageinfo *in )
{
	assert( imageinfo && in );

	imageinfo->subii = g_slist_prepend( imageinfo->subii, in );
	imageinfo_dup_nonheap( in );
}

/* out needs all of in[], add to sub-mark-list.
 */
void 
imageinfo_sub_add_all( Imageinfo *out, int nin, Imageinfo **in )
{
	int i;

	if( out )
		for( i = 0; i < nin; i++ )
			imageinfo_sub_add( out, in[i] );
}

void
imageinfo_expr_add( Imageinfo *imageinfo, Expr *expr )
{
#ifdef DEBUG
	printf( "imageinfo_expr_add: " );
	expr_name_print( expr );
	printf( "has imageinfo \"%s\" as value\n", imageinfo->im->filename );
#endif /*DEBUG*/

	assert( !g_slist_find( imageinfo->exprs, expr ) );
	assert( !expr->imageinfo );

	expr->imageinfo = imageinfo;
	imageinfo->exprs = g_slist_prepend( imageinfo->exprs, expr );
}

void *
imageinfo_expr_remove( Expr *expr, Imageinfo *imageinfo )
{
#ifdef DEBUG
	printf( "imageinfo_expr_remove: " );
	expr_name_print( expr );
	printf( "has lost imageinfo \"%s\" as value\n", 
		imageinfo->im->filename );
#endif /*DEBUG*/

	assert( expr->imageinfo );
	assert( g_slist_find( imageinfo->exprs, expr ) );
	assert( expr->imageinfo == imageinfo );

	expr->imageinfo = NULL;
	imageinfo->exprs = g_slist_remove( imageinfo->exprs, expr );

	return( NULL );
}

GSList *
imageinfo_expr_which( Imageinfo *imageinfo )
{
	return( imageinfo->exprs );
}

/* Find the underlying image in an imageinfo.
 */
IMAGE *
imageinfo_get_underlying( Imageinfo *imageinfo )
{
	if( imageinfo->underlying )
		return( imageinfo_get_underlying( imageinfo->underlying ) );
	else
		return(  imageinfo->im );
}

/* Free up an undo fragment. 
 */
static void
imageinfo_undofragment_free( Undofragment *frag )
{
#ifdef DEBUG_UNDO
	printf( "imageinfo_undofragment_free: 0x%x\n", (unsigned int) frag );
#endif /*DEBUG_UNDO*/

	FREEF( im_close, frag->im );
	FREE( frag );
}

/* Free an undo buffer.
 */
static void
imageinfo_undobuffer_free( Undobuffer *undo )
{
#ifdef DEBUG_UNDO
	printf( "imageinfo_undobuffer_free: 0x%x\n", (unsigned int) undo );
#endif /*DEBUG_UNDO*/

	slist_map( undo->frags, 
		(SListMapFn) imageinfo_undofragment_free, NULL );
	FREEF( g_slist_free, undo->frags );
	FREE( undo );
}

/* Free all undo information attached to an imageinfo.
 */
static void
imageinfo_undo_free( Imageinfo *imageinfo )
{
	slist_map( imageinfo->redo, 
		(SListMapFn) imageinfo_undobuffer_free, NULL );
	FREEF( g_slist_free, imageinfo->redo );
	slist_map( imageinfo->undo, 
		(SListMapFn) imageinfo_undobuffer_free, NULL );
	FREEF( g_slist_free, imageinfo->undo );
	FREEF( imageinfo_undobuffer_free, imageinfo->cundo );
}

static void
imageinfo_destroy( GtkObject *object )
{
	Imageinfo *imageinfo = IMAGEINFO( object );
	IMAGE *im = imageinfo_get_underlying( imageinfo );
	gboolean isfile = im_isfile( im );
	char name[PATHLEN];

#ifdef DEBUG_OPEN
	printf( "imageinfo_destroy: \"%s\" final death\n", 
		NN( imageinfo->name ) );
#endif /*DEBUG_OPEN*/

	assert( imageinfo->count == 0 );

	if( imageinfo->dfile && isfile ) 
		/* We must close before we delete to make sure we
		 * get the desc file too ... save the filename.
		 */
		im_strncpy( name, im->filename, PATHLEN - 5 );

	slist_map( imageinfo->subii, 
		(SListMapFn) imageinfo_sub_remove, imageinfo );
	assert( !imageinfo->subii );
	slist_map( imageinfo->exprs, 
		(SListMapFn) imageinfo_expr_remove, imageinfo );
	assert( !imageinfo->exprs );

	FREEF( im_close, imageinfo->im );
	FREEF( im_close, imageinfo->mapped_im );
	FREEF( im_close, imageinfo->identity_lut );

	imageinfo->hi->itable = 
		g_slist_remove( imageinfo->hi->itable, imageinfo );

	if( imageinfo->dfile && isfile ) {
#ifdef DEBUG_OPEN
		printf( "imageinfo_destroy: unlinking \"%s\"\n", name );
#endif /*DEBUG_OPEN*/

		(void) unlink( name );
		strcpy( name + strlen( name ) - 1, "desc" );
		(void) unlink( name );
		(void) mainw_free_update();
	}

	FREEF( imageinfo_destroy_nonheap, imageinfo->underlying );

	imageinfo_undo_free( imageinfo );

	GTK_OBJECT_CLASS( parent_class )->destroy( object );
}

/* Final death!
 */
static void 
imageinfo_finalize( GtkObject *object )
{
	Imageinfo *imageinfo = IMAGEINFO( object );

#ifdef DEBUG_OPEN
	printf( "imageinfo_finalize: %s\n", imageinfo->name );
#endif /*DEBUG_MAKE*/

	FREE( (char *) imageinfo->name );
	imageinfo_all = g_slist_remove( imageinfo_all, imageinfo );

	GTK_OBJECT_CLASS( parent_class )->finalize( object );
}

/* From heap_gc() ... no heap pointers left, delete if there are no
 * non-heap pointers either.
 */
void
imageinfo_destroy_heap( Imageinfo *imageinfo )
{
	/* Has this already been destroyed as a heap pointer? 
	 */
	if( imageinfo->zombie )
		return;

	/* Are there any non-heap pointers?
	 */
	if( imageinfo->count ) {
#ifdef DEBUG
		printf( "imageinfo_destroy_heap: \"%s\" zombiefying\n", 
			imageinfo->im->filename );
#endif /*DEBUG*/

		/* Setting zombie indicates that the last non-heap pointer
		 * to be removed should trigger destroy.
		 */
		imageinfo->zombie = TRUE;
		return;
	}

#ifdef DEBUG
	printf( "imageinfo_destroy_heap: \"%s\"\n", imageinfo->im->filename );
#endif /*DEBUG*/

	/* Nope ... just destroy.
	 */
        gtk_object_unref( GTK_OBJECT( imageinfo ) );
}

/* destroy() for non-heap pointers.
 */
void
imageinfo_destroy_nonheap( Imageinfo *imageinfo )
{
	assert( imageinfo->count > 0 );

#ifdef DEBUG
	printf( "imageinfo_destroy_nonheap: \"%s\" count = %d\n", 
		imageinfo->im->filename, imageinfo->count );
#endif /*DEBUG*/

	imageinfo->count--;

	/* Delayed destroy?
	 */
	if( !imageinfo->count && imageinfo->zombie ) 
		gtk_object_unref( GTK_OBJECT( imageinfo ) );
}

/* Create a new heap pointer. Heap pointers are GCed for us, we don't need to
 * ref count ... just clear the zombie flag.
 */
void
imageinfo_dup_heap( Imageinfo *imageinfo )
{
	imageinfo->zombie = FALSE;

#ifdef DEBUG
	printf( "imageinfo_dup_heap: \"%s\"\n", imageinfo->im->filename );
#endif /*DEBUG*/
}

/* Create a new non-heap pointer.
 */
void
imageinfo_dup_nonheap( Imageinfo *imageinfo )
{
	assert( imageinfo->count >= 0 );

	imageinfo->count++;

#ifdef DEBUG
	printf( "imageinfo_dup_nonheap: \"%s\" count = %d\n", 
		imageinfo->im->filename, imageinfo->count );
#endif /*DEBUG*/
}

static void
imageinfo_real_changed( Imageinfo *imageinfo )
{
}

static void
imageinfo_real_area_changed( Imageinfo *imageinfo, Rect *dirty )
{
}

static void
imageinfo_real_undo_changed( Imageinfo *imageinfo )
{
}

static void
imageinfo_class_init( ImageinfoClass *klass )
{
	GtkObjectClass *object_class = (GtkObjectClass*) klass;

	parent_class = gtk_type_class( GTK_TYPE_OBJECT );

	object_class->destroy = imageinfo_destroy;
	object_class->finalize = imageinfo_finalize;

	klass->changed = imageinfo_real_changed;
	klass->area_changed = imageinfo_real_area_changed;
	klass->undo_changed = imageinfo_real_undo_changed;

	/* Create signals.
	 */
	imageinfo_signals[CHANGED] = gtk_signal_new( "changed",
		GTK_RUN_FIRST,
		object_class->type,
		GTK_SIGNAL_OFFSET( ImageinfoClass, changed ),
		gtk_marshal_NONE__NONE,
		GTK_TYPE_NONE, 0 );
	imageinfo_signals[AREA_CHANGED] = gtk_signal_new( "area_changed",
		GTK_RUN_FIRST,
		object_class->type,
		GTK_SIGNAL_OFFSET( ImageinfoClass, area_changed ),
		gtk_marshal_NONE__POINTER,
		GTK_TYPE_NONE, 1, GTK_TYPE_POINTER );
	imageinfo_signals[UNDO_CHANGED] = gtk_signal_new( "undo_changed",
		GTK_RUN_FIRST,
		object_class->type,
		GTK_SIGNAL_OFFSET( ImageinfoClass, undo_changed ),
		gtk_marshal_NONE__NONE,
		GTK_TYPE_NONE, 0 );

	gtk_object_class_add_signals( object_class, 
		imageinfo_signals, LAST_SIGNAL );
}

static void
imageinfo_init( Imageinfo *imageinfo )
{
#ifdef DEBUG_OPEN
	printf( "imageinfo_init:\n" );
#endif /*DEBUG_MAKE*/

	imageinfo->hi = NULL;
	imageinfo->marked = FALSE;
	imageinfo->count = 0;
	imageinfo->zombie = FALSE;
	imageinfo->im = NULL;
	imageinfo->mapped_im = NULL;
	imageinfo->identity_lut = NULL;
	imageinfo->underlying = NULL;
	imageinfo->subii = NULL;
	imageinfo->dfile = FALSE;
	imageinfo->name = NULL;
	imageinfo->exprs = NULL;
	imageinfo->ok_to_paint = FALSE;
	imageinfo->undo = NULL;
	imageinfo->redo = NULL;
	imageinfo->cundo = NULL;

	imageinfo_all = g_slist_prepend( imageinfo_all, imageinfo );
}

GtkType
imageinfo_get_type( void )
{
	static GtkType type = 0;

	if( !type ) {
		static const GtkTypeInfo info = {
			"Imageinfo",
			sizeof( Imageinfo ),
			sizeof( ImageinfoClass ),
			(GtkClassInitFunc) imageinfo_class_init,
			(GtkObjectInitFunc) imageinfo_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		type = gtk_type_unique( GTK_TYPE_OBJECT, &info );
	}

	return( type );
}

/* Make a basic imageinfo. No refs, will be destroyed on next GC.
 */
Imageinfo *
imageinfo_new( Heap *hi, IMAGE *im )
{
	Imageinfo *imageinfo = gtk_type_new( TYPE_IMAGEINFO );

#ifdef DEBUG_OPEN
	printf( "imageinfo_new: \"%s\"\n", im->filename );
#endif /*DEBUG_OPEN*/

	if( hi == NULL )
		hi = reduce_context->hi;

	imageinfo->hi = hi;
	imageinfo->im = im;
	hi->itable = g_slist_prepend( hi->itable, imageinfo );

	/* We do our own ref counting because of the GC. Have one gtk_ref,
	 * unref once when our counts hit zero.
	 */
        gtk_object_ref( GTK_OBJECT( imageinfo ) );
        gtk_object_sink( GTK_OBJECT( imageinfo ) );

	return( imageinfo );
}

/* Make a temp image. Deleted on close. No refs: closed on next GC. If you
 * want it to stick around, ref it!
 */
Imageinfo *
imageinfo_new_temp( Heap *hi, const char *mode )
{
	IMAGE *im;
	Imageinfo *imageinfo;
	char tname[FILENAME_MAX];

	if( !temp_name( tname, "v" ) || !(im = im_open( tname, mode )) )
		return( NULL );
	if( !(imageinfo = imageinfo_new( hi, im )) ) {
		im_close( im );
		return( NULL );
	}

	imageinfo->dfile = TRUE;

	return( imageinfo );
}

/* An image is a transformed LUT (eg. after passing a LUT from above through
 * VIPS) ... wrap it around the underlying image.
 */
Imageinfo *
imageinfo_new_modlut( Heap *hi, Imageinfo *imageinfo, IMAGE *im )
{
	Imageinfo *top_imageinfo;

	if( !(top_imageinfo = imageinfo_new( hi, im )) )
		return( NULL );

	/* Link together.
	 */
	top_imageinfo->underlying = imageinfo;
	imageinfo_dup_nonheap( top_imageinfo->underlying );

	return( top_imageinfo );
}

/* Open for read ... returns a non-heap pointer, destroy if it goes in the
 * heap.
 */
static Imageinfo *
imageinfo_open_image_input( const char *filename, Heap *hi )
{
	Imageinfo *imageinfo;
	char tname[FILENAME_MAX];

	if( !temp_name( tname, "v" ) )
		return( NULL );

	if( im_istiff( filename ) && im_istifftiled( filename ) ) {
		/* Tiled TIFF - open and read partially.
		 */
		if( !(imageinfo = imageinfo_new_temp( hi, "p" )) ||
			im_tiff2vips( filename, imageinfo->im ) )
			return( NULL );

#ifdef DEBUG_OPEN
		printf( "imageinfo_open_image_input: "
			"opened tiled TIFF \"%s\"\n", filename );
#endif /*DEBUG_OPEN*/

		imageinfo_dup_nonheap( imageinfo );
	}

	else if( im_istiff( filename ) && !im_istifftiled( filename ) ) {
		/* Plain TIFF ... convert to VIPS in a file.
		 */
		if( !(imageinfo = imageinfo_new_temp( hi, "w" )) )
			return( NULL );
		imageinfo_dup_nonheap( imageinfo );
		if( !mainw_add_eval_callback( imageinfo->im ) ||
			im_tiff2vips( filename, imageinfo->im ) ) {
			imageinfo_destroy_nonheap( imageinfo );
			return( NULL );
		}

#ifdef DEBUG_OPEN
		printf( "imageinfo_open_image_input: "
			"opened plain TIFF \"%s\"\n", filename );
#endif /*DEBUG_OPEN*/
	}

	else if( im_isjpeg( filename ) ) {
		if( !(imageinfo = imageinfo_new_temp( hi, "w" )) )
			return( NULL );
		imageinfo_dup_nonheap( imageinfo );
		if( !mainw_add_eval_callback( imageinfo->im ) ||
			im_jpeg2vips( filename, imageinfo->im ) ) {
			imageinfo_destroy_nonheap( imageinfo );
			return( NULL );
		}

#ifdef DEBUG_OPEN
		printf( "imageinfo_open_image_input: "
			"opened JPEG \"%s\"\n", filename );
#endif /*DEBUG_OPEN*/
	}

	else if( im_ispng( filename ) ) {
		if( !(imageinfo = imageinfo_new_temp( hi, "w" )) )
			return( NULL );
		imageinfo_dup_nonheap( imageinfo );
		if( !mainw_add_eval_callback( imageinfo->im ) ||
			im_png2vips( filename, imageinfo->im ) ) {
			imageinfo_destroy_nonheap( imageinfo );
			return( NULL );
		}

#ifdef DEBUG_OPEN
		printf( "imageinfo_open_image_input: "
			"opened PNG \"%s\"\n", filename );
#endif /*DEBUG_OPEN*/
	}

	else if( im_isppm( filename ) ) {
		if( !(imageinfo = imageinfo_new_temp( hi, "w" )) )
			return( NULL );
		imageinfo_dup_nonheap( imageinfo );
		if( !mainw_add_eval_callback( imageinfo->im ) ||
			im_ppm2vips( filename, imageinfo->im ) ) {
			imageinfo_destroy_nonheap( imageinfo );
			return( NULL );
		}

#ifdef DEBUG_OPEN
		printf( "imageinfo_open_image_input: "
			"opened PPM \"%s\"\n", filename );
#endif /*DEBUG_OPEN*/
	}

	else if( im_ismagick( filename ) ) {
		if( !(imageinfo = imageinfo_new_temp( hi, "w" )) )
			return( NULL );
		imageinfo_dup_nonheap( imageinfo );
		if( !mainw_add_eval_callback( imageinfo->im ) ||
			im_magick2vips( filename, imageinfo->im ) ) {
			imageinfo_destroy_nonheap( imageinfo );
			return( NULL );
		}

#ifdef DEBUG_OPEN
		printf( "imageinfo_open_image_input: "
			"opened via libMagick \"%s\"\n", 
			filename );
#endif /*DEBUG_OPEN*/
	}

	else if( im_isvips( filename ) ) {
		IMAGE *im;

		if( !(im = im_open( filename, "r" )) ) 
			return( NULL );

		/* Junk the old history, we don't want stuff from before this
		 * session confusing the balancer.
		 */
		FREE( im->Hist );
		im_histlin( im, "" );

		if( !(imageinfo = imageinfo_new( hi, im )) ) {
			im_close( im );
			return( NULL );
		}
		imageinfo_dup_nonheap( imageinfo );

#ifdef DEBUG_OPEN
		printf( "imageinfo_open_image_input: opened VIPS \"%s\"\n", 
			filename );
#endif /*DEBUG_OPEN*/
	}

	else {
		im_errormsg( "\"%s\" is not in a supported image file format",
			filename );
		return( NULL );
	}

	/* Get ready for input.
 	 */
	if( im_pincheck( imageinfo->im ) ) 
		return( NULL );

	return( imageinfo );
}

/* Open a filename for input.
 */
Imageinfo *
imageinfo_new_input( Heap *hi, const char *name )
{
	Imageinfo *imageinfo;

	if( (imageinfo = imageinfo_find( name )) ) {
		/* We always make a new non-heap pointer.
		 */
		imageinfo_dup_nonheap( imageinfo );
		return( imageinfo );
	}

        if( !(imageinfo = (Imageinfo *) call_string( 
		(call_string_fn) imageinfo_open_image_input, 
		name, hi, NULL, NULL )) ) {
                verrors( "unable to open image \"%s\" for input", name );
                return( NULL );
        }

	SETSTR( (char *) imageinfo->name, name );

	return( imageinfo );
}

/* Add an identity lut, if this is a LUTtable image.
 */
static IMAGE *
imageinfo_get_identity_lut( Imageinfo *imageinfo )
{
	if( imageinfo->im->Coding == IM_CODING_NONE && 
		imageinfo->im->BandFmt == IM_BANDFMT_UCHAR ) {
		if( !imageinfo->identity_lut ) {
			char tname[FILENAME_MAX];
			IMAGE *im;

			if( !temp_name( tname, "v" ) || 
				!(im = im_open( tname, "p" )) )
				return( NULL );
			imageinfo->identity_lut = im;

			if( im_identity( imageinfo->identity_lut, 
				imageinfo->im->Bands ) ) {
				verrors( "unable to build LUT" );
				return( NULL );
			}
		}

		return( imageinfo->identity_lut );
	}
	else
		return( NULL );
}

static IMAGE *
imageinfo_get_mapped( Imageinfo *imageinfo )
{
	if( !imageinfo->mapped_im ) {
		IMAGE *im = imageinfo_get_underlying( imageinfo );
		IMAGE *mapped_im;
		char name[FILENAME_MAX];

		if( !temp_name( name, "v" ) )
			return( NULL );
		if( !(mapped_im = im_open( name, "p" )) ) {
			verrors( "unable to map LUT" );
			return( NULL );
		}
		if( im_maplut( im, mapped_im, imageinfo->im ) ) {
			verrors( "unable to map LUT" );
			im_close( mapped_im );
			return( NULL );
		}
		imageinfo->mapped_im = mapped_im;
	}

	return( imageinfo->mapped_im );
}

/* Get a lut ... or not!
 */
IMAGE *
imageinfo_get( gboolean use_lut, Imageinfo *imageinfo )
{
	if( !imageinfo ) {
		ierrors( "image has no value" );
		return( NULL );
	}

	if( use_lut && imageinfo->underlying ) 
		return( imageinfo->im );
	if( use_lut && !imageinfo->underlying ) {
		IMAGE *lut;

		if( (lut = imageinfo_get_identity_lut( imageinfo )) )
			return( lut );
		else
			return( imageinfo->im );
	}
	else if( !use_lut && imageinfo->underlying )
		return( imageinfo_get_mapped( imageinfo ) );
	else
		return( imageinfo->im );
}

/* Do a set of II all refer to the same underlying image? Used to spot
 * LUTable optimisations.
 */
gboolean
imageinfo_same_underlying( Imageinfo *imageinfo[], int n )
{
	int i;

	if( n < 2 )
		return( TRUE );
	else {
		IMAGE *first = imageinfo_get_underlying( imageinfo[0] );

		for( i = 1; i < n; i++ )
			if( imageinfo_get_underlying( imageinfo[i] ) != first )
				return( FALSE );

		return( TRUE );
	}
}

/* Write to a filename.
 */
gboolean
imageinfo_write( Imageinfo *imageinfo, const char *filename )
{
	IMAGE *in = imageinfo_get( FALSE, imageinfo );
	IMAGE *out;

	char name[FILENAME_MAX];
	char mode[FILENAME_MAX];

	im__filename_split( filename, name, mode );
	if( imageinfo_find( name ) ) {
		ierrors( "unable to open \"%s\" for output\n"
			"image is already open", name );
		return( FALSE );
	}
        if( !(out = (IMAGE *) call_string( (call_string_fn) im_open,
                filename, "w", NULL, NULL )) ) {
                verrors( "unable to open \"%s\" for output", name );
		return( FALSE );
	}
	if( !mainw_add_eval_callback( out ) ) {
		im_close( out );
		return( FALSE );
	}
	if( im_copy( in, out ) ) {
		verrors( "unable to write image to file" );
		im_close( out );
		return( FALSE );
	}
	if( im_close( out ) ) {
		/* im_close() can fail if we write a TIFF on evalend.
		 */
		verrors( "unable to write image to file" );
		return( FALSE );
	}

	return( TRUE );
}

/* Change an imageinfo to be a file, rather than a memory object.
 */
gboolean
imageinfo_file( Imageinfo *imageinfo )
{
	IMAGE *im;
	char name[FILENAME_MAX];

	/* Check image type.
	 */
	if( !imageinfo_get( FALSE, imageinfo ) )
		return( FALSE );
	if( im_isfile( imageinfo_get( FALSE, imageinfo ) ) )
		return( TRUE );

	/* Save it.
	 */
	if( !temp_name( name, "v" ) )
		return( FALSE );
	if( !imageinfo_write( imageinfo, name ) )
		return( FALSE );

	if( !(im = im_open( name, "r" )) ) {
		verrors( "error reopening temp file!" );
		return( FALSE );
	}

	FREEF( im_close, imageinfo->im );
	FREEF( im_close, imageinfo->mapped_im );
	FREEF( imageinfo_destroy_nonheap, imageinfo->underlying );

	imageinfo->im = im;
	imageinfo->dfile = TRUE;

	imageinfo_changed( imageinfo );

	return( TRUE );
}

static void
imageinfo_check_paintable_cb( iWindow *iwnd, void *client, 
	iWindowNotifyFn nfn, void *sys )
{
	Imageinfo *imageinfo = IMAGEINFO( client );

	if( !imageinfo_file( imageinfo ) ) {
		nfn( sys, IWINDOW_ERROR );
		return;
	}
	if( im_rwcheck( imageinfo->im ) ) {
		verrors( "unable to write to \"%s\"\n"
			"check permissions", imageinfo->im->filename );
		nfn( sys, IWINDOW_ERROR );
		return;
	}

	imageinfo->ok_to_paint = TRUE;

	nfn( sys, IWINDOW_TRUE );
}

/* Check painting is OK. nfn() called on "ok!". Returns FALSE if it's
 * not immediately obvious that we can paint.
 */
gboolean
imageinfo_check_paintable( GtkWidget *parent, Imageinfo *imageinfo,
	iWindowNotifyFn nfn, void *sys )
{
	IMAGE *im = imageinfo_get( FALSE, imageinfo );

	if( im_isfile( im ) && 
		!imageinfo->dfile && !imageinfo->ok_to_paint ) {
		iDialog *idlg;

		idlg = box_yesno( parent,
			imageinfo_check_paintable_cb, 
				iwindow_true_cb, imageinfo,
			nfn, sys,
			"Modify file",
			"this image represents a disc file\n"
			"\n"
			"   %s\n"
			"\n"
			"if you paint on this file, it will be "
				"permanently changed\n"
			"if something goes wrong, you may lose work\n"
			"are you sure you want to modify this file?",
			imageinfo->name );
		idialog_set_object( idlg, GTK_OBJECT( imageinfo ) );

		return( FALSE );
	}
	else if( !im_isfile( im ) && !imageinfo->ok_to_paint ) {
		imageinfo_check_paintable_cb( NULL, imageinfo, nfn, sys );

		return( FALSE );
	}

	nfn( sys, IWINDOW_TRUE );
	return( TRUE );
}

/* Mark an imageinfo as being used ... also mark all sub-images.
 */
void 
imageinfo_mark( Imageinfo *imageinfo )
{
	if( !imageinfo->marked ) {
		imageinfo->marked = TRUE;
		(void) slist_map( imageinfo->subii, 
			(SListMapFn) imageinfo_mark, NULL );
	}
}

/* Try to get an Imageinfo from a symbol.
 */
Imageinfo *
imageinfo_sym_image( Symbol *sym )
{
        PElement *root = &sym->expr->root;

        if( sym->type == SYM_VALUE && PEISIMAGE( root ) )
                return( PEGETII( root ) );
        else
                return( NULL );
}

/* Make an info string about an imageinfo.
 */
void
imageinfo_info( Imageinfo *imageinfo, BufInfo *buf )
{
	IMAGE *im;

	if( !(im = imageinfo_get( FALSE, imageinfo )) ) {
		buf_appends( buf, "No image value" );
		return;
	}

	if( imageinfo->name )
		buf_appendf( buf, "%s, ", im__skip_dir( imageinfo->name ) ); 

	/* Coded? Special warning.
	 */
	if( im->Coding != IM_CODING_NONE ) 
		buf_appendf( buf, "%s, ", im_Coding2char( im->Coding ) );

	/* Main stuff.
	 */
	buf_appendf( buf, "%dx%d %s pixels, %d band%s, %s",
		im->Xsize, im->Ysize, decode_bandfmt( im->BandFmt ),
		im->Bands, im->Bands==1?"":"s", decode_type( im->Type ) );
}

/* Brush definitions.
 */
static PEL imageinfo_brush1[] = {
	0xff
};
static PEL imageinfo_brush2[] = {
	0xff, 0xff,
	0xff, 0xff
};
static PEL imageinfo_brush3[] = {
	0x00, 0xff, 0x00,
	0xff, 0xff, 0xff,
	0x00, 0xff, 0x00
};
static PEL imageinfo_brush4[] = {
	0x00, 0xff, 0xff, 0x00,
	0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0xff, 0xff,
	0x00, 0xff, 0xff, 0x00
};
static PEL imageinfo_brush5[] = {
	0x00, 0xff, 0xff, 0xff, 0x00,
	0xff, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0xff, 0xff, 0xff,
	0x00, 0xff, 0xff, 0xff, 0x00
};
static PEL imageinfo_brush6[] = {
	0x00, 0x00, 0xff, 0xff, 0x00, 0x00,
	0x00, 0xff, 0xff, 0xff, 0xff, 0x00,
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	0x00, 0xff, 0xff, 0xff, 0xff, 0x00,
	0x00, 0x00, 0xff, 0xff, 0x00, 0x00,
};
static PEL imageinfo_brush7[] = {
	0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00,
	0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00,
	0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00,
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00,
	0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00,
	0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00,
};
static PEL imageinfo_brush8[] = {
	0x00, 0xff,
	0xff, 0xff
};
static PEL imageinfo_brush9[] = {
	0x00, 0x00, 0xff,
	0x00, 0xff, 0xff,
	0xff, 0xff, 0x00
};
static PEL imageinfo_brush10[] = {
	0x00, 0x00, 0x00, 0xff,
	0x00, 0x00, 0xff, 0xff,
	0x00, 0xff, 0xff, 0x00,
	0xff, 0xff, 0x00, 0x00
};
static PEL imageinfo_brush11[] = {
	0x00, 0x00, 0x00, 0x00, 0xff,
	0x00, 0x00, 0x00, 0xff, 0xff,
	0x00, 0x00, 0xff, 0xff, 0x00,
	0x00, 0xff, 0xff, 0x00, 0x00,
	0xff, 0xff, 0x00, 0x00, 0x00
};
static PEL imageinfo_brush12[] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0xff,
	0x00, 0x00, 0x00, 0x00, 0xff, 0xff,
	0x00, 0x00, 0x00, 0xff, 0xff, 0x00,
	0x00, 0x00, 0xff, 0xff, 0x00, 0x00,
	0x00, 0xff, 0xff, 0x00, 0x00, 0x00,
	0xff, 0xff, 0x00, 0x00, 0x00, 0x00
};
static PEL imageinfo_brush13[] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

static PEL *imageinfo_brush_masks[] = {
	imageinfo_brush1, imageinfo_brush2, imageinfo_brush3, 
	imageinfo_brush4, imageinfo_brush5, imageinfo_brush6, 
	imageinfo_brush7, imageinfo_brush8, imageinfo_brush9, 
	imageinfo_brush10, imageinfo_brush11, imageinfo_brush12, 
	imageinfo_brush13
};

static Rect imageinfo_brush_shapes[] = {
	{ 0, 0, 1, 1 },			/* PAINTBOX_1ROUND */
	{ -1, -1, 2, 2 },		/* PAINTBOX_2ROUND */
	{ -1, -1, 3, 3 },		/* PAINTBOX_3ROUND */
	{ -2, -2, 4, 4 },		/* PAINTBOX_4ROUND */
	{ -2, -2, 5, 5 },		/* PAINTBOX_5ROUND */
	{ -3, -3, 6, 6 },		/* PAINTBOX_6ROUND */
	{ -5, -5, 10, 10 },		/* PAINTBOX_10ROUND */
	{ -1, -1, 2, 2 },		/* PAINTBOX_2ITALIC */
	{ -1, -1, 3, 3 },		/* PAINTBOX_3ITALIC */
	{ -2, -2, 4, 4 },		/* PAINTBOX_4ITALIC */
	{ -2, -2, 5, 5 },		/* PAINTBOX_5ITALIC */
	{ -3, -3, 6, 6 },		/* PAINTBOX_6ITALIC */
	{ -5, -5, 10, 10 }		/* PAINTBOX_10ITALIC */
};

static Undofragment *
imageinfo_undofragment_new( Undobuffer *undo )
{
	Undofragment *frag;

	if( !(frag = IM_NEW( NULL, Undofragment )) )
		return( NULL );

#ifdef DEBUG_UNDO
	printf( "imageinfo_undofragment_new: 0x%x\n", (unsigned int) frag );
#endif /*DEBUG_UNDO*/

	frag->undo = undo;
	frag->im = NULL;

	return( frag );
}

static Undobuffer *
imageinfo_undobuffer_new( Imageinfo *imageinfo )
{
	Undobuffer *undo;

	if( !(undo = IM_NEW( NULL, Undobuffer )) )
		return( NULL );

#ifdef DEBUG_UNDO
	printf( "imageinfo_undobuffer_new: 0x%x\n", (unsigned int) undo );
#endif /*DEBUG_UNDO*/

	undo->imageinfo = imageinfo;
	undo->frags = NULL;

	/* No pixels in bounding box at the moment.
	 */
	undo->bbox.left = 0;
	undo->bbox.top = 0;
	undo->bbox.width = 0;
	undo->bbox.height = 0;

	return( undo );
}

/* Grab from the image into an IMAGE buffer. Always grab to memory.
 */
static IMAGE *
imageinfo_undo_grab_area( IMAGE *im, Rect *dirty )
{
	IMAGE *save;

	/* Make new image to extract to. 
	 */
	if( !(save = im_open( "undo buffer", "t" )) )
		return( NULL );

	/* Try to extract from im.
	 */
	if( im_extract_area( im, save, 
		dirty->left, dirty->top, dirty->width, dirty->height ) ) {
		verrors( "extract for undo failed with:" );
		im_close( save );
		return( NULL );
	}

	return( save );
}

/* Grab into an undo fragment. Add frag to frag list on undo buffer, expand
 * bounding box.
 */
static Undofragment *
imageinfo_undo_grab( Undobuffer *undo, Rect *dirty )
{
	Imageinfo *imageinfo = undo->imageinfo;
	Undofragment *frag = imageinfo_undofragment_new( undo );
	IMAGE *im = imageinfo_get( FALSE, imageinfo ); 
	Rect bbox;

#ifdef DEBUG_UNDO
	printf( "imageinfo_undo_grab: adding %dx%d pixels to buffer 0x%x\n",
		dirty->width, dirty->height, (unsigned int) undo );
#endif /*DEBUG_UNDO*/

	/* Try to extract from im. Memory allocation happens at this
	 * point, so we must be careful!
	 */
	if( !(frag->im = imageinfo_undo_grab_area( im, dirty )) ) {
		imageinfo_undofragment_free( frag );
		return( NULL );
	}

	/* Note position of this frag.
	 */
	frag->pos = *dirty;

	/* Add frag to frag list on undo buffer.
	 */
	undo->frags = g_slist_prepend( undo->frags, frag );

	/* Find bounding box for saved pixels.
	 */
	im_rect_unionrect( dirty, &undo->bbox, &bbox );
	undo->bbox = bbox;

	/* Return new frag.
	 */
	return( frag );
}

/* Trim the undo buffer if we have more than x items on it.
 */
static void
imageinfo_undo_trim( Imageinfo *imageinfo )
{
	int max = PAINTBOX_MAX_UNDO;
	int len = g_slist_length( imageinfo->undo );

	if( max >= 0 && len > max ) {
		GSList *l;
		int i;

		l = g_slist_reverse( imageinfo->undo );

		for( i = 0; i < len - max; i++ ) {
			Undobuffer *undo = (Undobuffer *) l->data;

			imageinfo_undobuffer_free( undo );
			l = g_slist_remove( l, undo );
		}

		imageinfo->undo = g_slist_reverse( l );
	}

#ifdef DEBUG_UNDO
	printf( "imageinfo_undo_trim: %d items in undo buffer\n", 
		g_slist_length( imageinfo->undo ) );
#endif /*DEBUG_UNDO*/
}

/* Mark the start or end of an undo session. Copy current undo information 
 * to the undo buffers and NULL out the current undo pointer. Junk all redo
 * information: this new undo action makes all that out of date.
 */
void
imageinfo_undo_mark( Imageinfo *imageinfo )
{
	/* Is there an existing undo save area?
	 */
	if( imageinfo->cundo ) {
		/* Left over from the last undo save. Copy to undo save list
		 * and get ready for new undo buffer.
		 */
		imageinfo->undo = 
			g_slist_prepend( imageinfo->undo, imageinfo->cundo );
		imageinfo->cundo = NULL;
	}

	/* Junk all redo information. 
	 */
	slist_map( imageinfo->redo, 
		(SListMapFn) imageinfo_undobuffer_free, NULL );
	FREEF( g_slist_free, imageinfo->redo );

	/* Trim undo buffer.
	 */
	imageinfo_undo_trim( imageinfo );

	/* Update menus.
	 */
	imageinfo_undo_changed( imageinfo );
}

/* Add to the undo buffer. If there is no undo buffer currently under
 * construction, make a new one. If there is an existing undo buffer, try to
 * grow it left/right/up/down so as to just enclose the new bounding box. We
 * assume that our dirty areas are not going to be disconnected. Is this
 * always true? No - if you move smudge or smear quickly, you can get
 * non-overlapping areas. However: if you do lots of little operations in more
 * or less the same place (surely the usual case), then this technique will be
 * far better.
 */
static gboolean
imageinfo_undo_add( Imageinfo *imageinfo, Rect *dirty )
{
	IMAGE *im = imageinfo_get( FALSE, imageinfo ); 
	Undobuffer *undo = imageinfo->cundo;
	Rect over, image, clipped;

	/* Undo disabled? Do nothing.
	 */
	if( PAINTBOX_MAX_UNDO == 0 )
		return( TRUE );

	/* Clip dirty against image size. 
	 */
	image.left = 0;
	image.top = 0;
	image.width = im->Xsize;
	image.height = im->Ysize;
	im_rect_intersectrect( &image, dirty, &clipped );

	/* Is there anything left? If not, can return immediately.
	 */
	if( im_rect_isempty( &clipped ) )
		return( TRUE );

	if( !undo ) {
		/* No current undo buffer ... start a new one for this action.
		 */
		if( !(imageinfo->cundo = undo = 
			imageinfo_undobuffer_new( imageinfo )) )
			return( FALSE );

		return( imageinfo_undo_grab( undo, &clipped ) != NULL );
	}

	/* Existing stuff we are to add to. Try to expand our undo
	 * area to just enclose the new bounding box. We assume that
	 * there is an overlap between the new and old stuff.
	 */

	/* Do we need to expand our saved area to the right?
	 */
	if( IM_RECT_RIGHT( &clipped ) > IM_RECT_RIGHT( &undo->bbox ) ) {
		/* Expand to the right. Calculate the section we need
		 * to add to our bounding box.
		 */
		over.left = IM_RECT_RIGHT( &undo->bbox );
		over.top = undo->bbox.top;
		over.width = IM_RECT_RIGHT( &clipped ) - 
			IM_RECT_RIGHT( &undo->bbox );
		over.height = undo->bbox.height;

		/* Grab new fragment.
		 */
		if( !imageinfo_undo_grab( undo, &over ) )
			return( FALSE );
	}

	/* Do we need to expand our saved area to the left?
	 */
	if( undo->bbox.left > clipped.left ) {
		over.left = clipped.left;
		over.top = undo->bbox.top;
		over.width = undo->bbox.left - clipped.left;
		over.height = undo->bbox.height;

		if( !imageinfo_undo_grab( undo, &over ) )
			return( FALSE );
	}

	/* Do we need to expand our saved area upwards?
	 */
	if( undo->bbox.top > clipped.top ) {
		over.left = undo->bbox.left;
		over.top = clipped.top;
		over.width = undo->bbox.width;
		over.height = undo->bbox.top - clipped.top;

		if( !imageinfo_undo_grab( undo, &over ) )
			return( FALSE );
	}

	/* Do we need to expand our saved area downwards?
	 */
	if( IM_RECT_BOTTOM( &clipped ) > IM_RECT_BOTTOM( &undo->bbox ) ) {
		over.left = undo->bbox.left;
		over.top = IM_RECT_BOTTOM( &undo->bbox );
		over.width = undo->bbox.width;
		over.height = IM_RECT_BOTTOM( &clipped ) - 
			IM_RECT_BOTTOM( &undo->bbox );

		if( !imageinfo_undo_grab( undo, &over ) )
			return( FALSE );
	}

	return( TRUE );
}

/* Paste an undo fragment back into the image.
 */
static void *
imageinfo_undofragment_paste( Undofragment *frag )
{
	Undobuffer *undo = frag->undo;
	Imageinfo *imageinfo = undo->imageinfo;
	IMAGE *im = imageinfo_get( FALSE, imageinfo ); 

	im_insertplace( im, frag->im, frag->pos.left, frag->pos.top );
	imageinfo_area_changed( imageinfo, &frag->pos );

	return( NULL );
}

/* Paste a whole undo buffer back into the image.
 */
static void
imageinfo_undobuffer_paste( Undobuffer *undo )
{
#ifdef DEBUG_UNDO
	printf( "imageinfo_undobuffer_paste: pasting buffer 0x%x\n",
		(unsigned int) undo );
#endif /*DEBUG_UNDO*/

	slist_map( undo->frags, 
		(SListMapFn) imageinfo_undofragment_paste, NULL );
}

/* Undo a paint action.
 */
void
imageinfo_undo( Imageinfo *imageinfo )
{
	Undobuffer *undo;

	/* Find the undo action we are to perform.
	 */
	if( !imageinfo->undo )
		return;
	undo = (Undobuffer *) imageinfo->undo->data;

	/* We are going to undo the first action on the undo list. We must
	 * save the area under the first undo action to the redo list. Do
	 * the save, even if undo is disabled.
	 */
	if( !imageinfo_undo_add( imageinfo, &undo->bbox ) ) {
		box_alert( NULL );
		return;
	}

	/* Add new undo area.
	 */
	imageinfo->redo = g_slist_prepend( imageinfo->redo, imageinfo->cundo );
	imageinfo->cundo = NULL;

	/* Paint undo back.
	 */
	imageinfo_undobuffer_paste( undo );

	/* Junk the undo action we have performed.
	 */
	imageinfo->undo = g_slist_remove( imageinfo->undo, undo );
	imageinfo_undobuffer_free( undo );

	/* Trim undo buffer.
	 */
	imageinfo_undo_trim( imageinfo );

	/* Update menus.
	 */
	imageinfo_undo_changed( imageinfo );
}

/* Redo a paint action, if possible.
 */
void
imageinfo_redo( Imageinfo *imageinfo )
{
	Undobuffer *undo;

	/* Find the redo action we are to perform.
	 */
	if( !imageinfo->redo )
		return;
	undo = (Undobuffer *) imageinfo->redo->data;

	/* We are going to redo the first action on the redo list. We must
	 * save the area under the first redo action to the undo list. Save
	 * even if undo is disabled.
	 */
	if( !imageinfo_undo_add( imageinfo, &undo->bbox ) ) {
		box_alert( NULL );
		return;
	}

	/* Add this new buffer to the undo list.
	 */
	imageinfo->undo = g_slist_prepend( imageinfo->undo, imageinfo->cundo );
	imageinfo->cundo = NULL;

	/* Paint redo back.
	 */
	imageinfo_undobuffer_paste( undo );

	/* We can junk the head of the undo list now.
	 */
	imageinfo->redo = g_slist_remove( imageinfo->redo, undo );
	imageinfo_undobuffer_free( undo );

	/* Trim undo buffer.
	 */
	imageinfo_undo_trim( imageinfo );

	/* Update menus.
	 */
	imageinfo_undo_changed( imageinfo );
}

/* Try to add to the undo buffer. If undo is disabled, junk everything since
 * this new paint action will invalidate all saved undo/redo stuff.
 */
void
imageinfo_undo_clear( Imageinfo *imageinfo )
{
	imageinfo_undo_free( imageinfo );
	imageinfo_undo_changed( imageinfo );
}

/* Draw a line.
 */
gboolean
imageinfo_paint_line( Imageinfo *imageinfo, Imageinfo *ink, 
	int nib, int x1, int y1, int x2, int y2 )
{
	IMAGE *im = imageinfo_get( FALSE, imageinfo ); 
	IMAGE *ink_im = imageinfo_get( FALSE, ink );
	PEL *data = (PEL *) ink_im->data;
	Rect dirty, p1, p2, image, clipped;

	p1 = imageinfo_brush_shapes[nib];
	p1.left += x1;
	p1.top += y1;
	p2 = imageinfo_brush_shapes[nib];
	p2.left += x2;
	p2.top += y2;
	im_rect_unionrect( &p1, &p2, &dirty );

	image.left = 0;
	image.top = 0;
	image.width = im->Xsize;
	image.height = im->Ysize;
	im_rect_intersectrect( &dirty, &image, &clipped );

	if( im_rect_isempty( &clipped ) )
		return( TRUE );

	if( !imageinfo_undo_add( imageinfo, &clipped ) ) 
		return( FALSE );

	if( im_fastlineuser( im, x1, y1, x2, y2, im_plotmask, data, 
		imageinfo_brush_masks[nib], &imageinfo_brush_shapes[nib] ) ) {
		verrors( "line draw failed with:" );
		return( FALSE );
	}

	imageinfo_area_changed( imageinfo, &dirty );

	return( TRUE );
}

/* Smudge a line.
 */
gboolean
imageinfo_paint_smudge( Imageinfo *imageinfo, 
	Rect *oper, int x1, int y1, int x2, int y2 )
{	
	IMAGE *im = imageinfo_get( FALSE, imageinfo ); 
	Rect p1, p2, dirty;

	/* Calculate bounding box for smudge.
	 */
	p1 = *oper;
	p1.left += x1;
	p1.top += y1;
	p2 = *oper;
	p2.left += x2;
	p2.top += y2;
	im_rect_unionrect( &p1, &p2, &dirty );
	if( !imageinfo_undo_add( imageinfo, &dirty ) )
		return( FALSE );

	/* Smudge line connecting old and new points. 
	 */
	if( im_fastlineuser( im, x1, y1, x2, y2, im_smudge, 
		oper, NULL, NULL ) ) {
		verrors( "smudge failed with:" );
		return( FALSE );
	}

	imageinfo_area_changed( imageinfo, &dirty );

	return( TRUE );
}

/* Flood an area.
 */
gboolean
imageinfo_paint_flood( Imageinfo *imageinfo, Imageinfo *ink, 
	int x, int y, gboolean blob )
{
	IMAGE *im = imageinfo_get( FALSE, imageinfo ); 
	IMAGE *ink_im = imageinfo_get( FALSE, ink );
	PEL *data = (PEL *) ink_im->data;
	Rect dirty;
	int result;

	/* Save undo area. We have to save the entire image, as we don't know
	 * how much the flood will change :(
	 */
	dirty.left = 0;
	dirty.top = 0;
	dirty.width = im->Xsize;
	dirty.height = im->Ysize;
	if( !imageinfo_undo_add( imageinfo, &dirty ) ) 
		return( FALSE );

	/* Flood!
	 */
	if( blob )
		result = im_flood_blob( im, x, y, data, &dirty );
	else
		result = im_flood( im, x, y, data, &dirty );
	if( result ) {
		verrors( "flood area failed with:" );
		return( FALSE );
	}

	imageinfo_area_changed( imageinfo, &dirty );

	return( TRUE );
}

gboolean
imageinfo_paint_dropper( Imageinfo *imageinfo, Imageinfo *ink, int x, int y )
{
	IMAGE *im = imageinfo_get( FALSE, imageinfo ); 
	IMAGE *ink_im = imageinfo_get( FALSE, ink );
	PEL *data = (PEL *) ink_im->data;
	Rect dirty;

	if( im_readpoint( im, x, y, data ) ) {
		verrors( "dropper failed with:" );
		return( FALSE );
	}

	dirty.left = 0;
	dirty.top = 0;
	dirty.width = ink_im->Xsize;
	dirty.height = ink_im->Ysize;

	imageinfo_area_changed( ink, &dirty );

	return( TRUE );
}

/* Fill a rect.
 */
gboolean
imageinfo_paint_rect( Imageinfo *imageinfo, Imageinfo *ink, Rect *area )
{	
	IMAGE *im = imageinfo_get( FALSE, imageinfo ); 
	IMAGE *ink_im = imageinfo_get( FALSE, ink );
	PEL *data = (PEL *) ink_im->data;

	if( !imageinfo_undo_add( imageinfo, area ) )
		return( FALSE );

	if( im_paintrect( im, area, data ) ) {
		verrors( "rectangle fill failed with:" );
		return( FALSE );
	}

	imageinfo_area_changed( imageinfo, area );

	return( TRUE );
}

static gboolean
imageinfo_paint_text2( Imageinfo *imageinfo, 
	GdkFont *font, const char *text, Rect *tarea, GdkPixmap *pixmap )
{
	IMAGE *im = imageinfo_get( FALSE, imageinfo );
	PEL *data = (PEL *) im->data;
	GdkGC *fg, *bg;
	GdkImage *image;
	GdkGCValues values;
	int x, y;

	values.foreground.pixel = 1;
	values.background.pixel = 0;
	if( !(fg = gdk_gc_new_with_values( pixmap, 
		&values,
		GDK_GC_FOREGROUND | GDK_GC_BACKGROUND )) ) {
		ierrors( "unable to make GC" );
		return( FALSE );
	}

	values.foreground.pixel = 0;
	values.background.pixel = 1;
	if( !(bg = gdk_gc_new_with_values( pixmap, 
		&values,
		GDK_GC_FOREGROUND | GDK_GC_BACKGROUND )) ) {
		gdk_gc_unref( fg );
		ierrors( "unable to make GC" );
		return( FALSE );
	}

	gdk_draw_rectangle( pixmap, bg, TRUE, 
		0, 0, tarea->width, tarea->height );
	gdk_draw_string( pixmap, font, fg, -tarea->left, -tarea->top, text );

	gdk_gc_unref( fg );
	gdk_gc_unref( bg );

	if( !(image = gdk_image_get( pixmap, 	
		0, 0, tarea->width, tarea->height )) ) {
		ierrors( "unable to get server image" );
		return( FALSE );
	}

	for( y = 0; y < tarea->height; y++ ) 
		for( x = 0; x < tarea->width; x++ )
			if( gdk_image_get_pixel( image, x, y ) )
				data[x + y * tarea->width] = 255;
			else
				data[x + y * tarea->width] = 0;

	gdk_image_destroy( image );

	return( TRUE );
}

/* Make a mask for text.
 */
gboolean
imageinfo_paint_text( Imageinfo *imageinfo, 
	GdkFont *font, const char *text, Rect *tarea )
{
	IMAGE *im = imageinfo_get( FALSE, imageinfo );
	int lbearing, rbearing, width, ascent, descent;
	GdkPixmap *pixmap;

	gdk_text_extents( font, text, strlen( text ), 
		  &lbearing, &rbearing, &width, &ascent, &descent );
	tarea->width = rbearing - lbearing;
	tarea->height = ascent + descent;
	tarea->left = lbearing;
	tarea->top = -ascent;
	im_initdesc( im, tarea->width, tarea->height, 1, 
		IM_BBITS_BYTE, IM_BANDFMT_UCHAR, IM_CODING_NONE, IM_TYPE_B_W,
		1.0, 1.0, 0, 0 );
	if( im_setupout( im ) ) {
		verrors( "unable to build text mask" );
		return( FALSE );
	}

	if( !(pixmap = gdk_pixmap_new( NULL, 
		tarea->width, tarea->height, 1 )) ) {
		ierrors( "unable to build off-screen bitmap" );
		return( FALSE );
	}

	if( !imageinfo_paint_text2( imageinfo, font, text, tarea, pixmap ) ) {
		gdk_pixmap_unref( pixmap );
		return( FALSE );
	}

	gdk_pixmap_unref( pixmap );

	return( TRUE );
}

/* Paint a mask.
 */
gboolean
imageinfo_paint_mask( Imageinfo *imageinfo, 
	Imageinfo *ink, Imageinfo *mask, int x, int y )
{
	IMAGE *im = imageinfo_get( FALSE, imageinfo ); 
	IMAGE *ink_im = imageinfo_get( FALSE, ink );
	IMAGE *mask_im = imageinfo_get( FALSE, mask );
	Rect dirty, image, clipped;

	dirty.left = x;
	dirty.top = y;
	dirty.width = mask_im->Xsize;
	dirty.height = mask_im->Ysize;
	image.left = 0;
	image.top = 0;
	image.width = im->Xsize;
	image.height = im->Ysize;
	im_rect_intersectrect( &dirty, &image, &clipped );

	if( im_rect_isempty( &clipped ) )
		return( TRUE );

	if( !imageinfo_undo_add( imageinfo, &clipped ) ) 
		return( FALSE );

	if( im_plotmask( im, 0, 0, 
		(PEL *) ink_im->data, (PEL *) mask_im->data, &dirty ) ) {
		verrors( "plot mask failed with:" );
		return( FALSE );
	}

	imageinfo_area_changed( imageinfo, &dirty );

	return( TRUE );
}

/* Print a pixel.
 */
void 
imageinfo_to_text( Imageinfo *imageinfo, BufInfo *buf )
{
	IMAGE *im = imageinfo_get( FALSE, imageinfo );
	PEL *p = (PEL *) im->data;
	int i;

#define PRINT_INT( T, I ) buf_appendf( buf, "%d", ((T *)p)[I] );
#define PRINT_FLOAT( T, I ) buf_appendf( buf, "%g", ((T *)p)[I] );

	for( i = 0; i < im->Bands; i++ ) {
		if( i )
			buf_appends( buf, ", " );

		switch( im->BandFmt ) {
		case IM_BANDFMT_UCHAR:
			PRINT_INT( unsigned char, i );
			break;
			
		case IM_BANDFMT_CHAR:
			PRINT_INT( char, i );
			break;
			
		case IM_BANDFMT_USHORT:
			PRINT_INT( unsigned short, i );
			break;
			
		case IM_BANDFMT_SHORT:
			PRINT_INT( short, i );
			break;
			
		case IM_BANDFMT_UINT:
			PRINT_INT( unsigned int, i );
			break;
			
		case IM_BANDFMT_INT:
			PRINT_INT( int, i );
			break;
			
		case IM_BANDFMT_FLOAT:
			PRINT_FLOAT( float, i );
			break;
			
		case IM_BANDFMT_COMPLEX:
			buf_appends( buf, "(" );
			PRINT_FLOAT( float, (i << 1) );
			buf_appends( buf, ", " );
			PRINT_FLOAT( float, (i << 1) + 1 );
			buf_appends( buf, ")" );
			break;
			
		case IM_BANDFMT_DOUBLE:
			PRINT_FLOAT( double, i );
			break;
			
		case IM_BANDFMT_DPCOMPLEX:
			buf_appends( buf, "(" );
			PRINT_FLOAT( double, i << 1 );
			buf_appends( buf, ", " );
			PRINT_FLOAT( double, (i << 1) + 1 );
			buf_appends( buf, ")" );
			break;

		default:
			buf_appends( buf, "???" );
			break;
		}
	}
}

/* Set band i to value.
 */
static void
imageinfo_from_text_band( Imageinfo *imageinfo, int i, double re, double im )
{
	IMAGE *image = imageinfo_get( FALSE, imageinfo );
	PEL *p = (PEL *) image->data;
	double mod = sqrt( re*re + im*im );

	if( i < 0 || i >= image->Bands )
		return;

#define SET_INT( T, I, X ) (((T *)p)[I] = (T) IM_RINT(X)) 
#define SET_FLOAT( T, I, X ) (((T *)p)[I] = (T) (X)) 

	switch( image->BandFmt ) {
	case IM_BANDFMT_UCHAR:
		SET_INT( unsigned char, i, mod );
		break;
		
	case IM_BANDFMT_CHAR:
		SET_INT( char, i, mod );
		break;
		
	case IM_BANDFMT_USHORT:
		SET_INT( unsigned short, i, mod );
		break;
		
	case IM_BANDFMT_SHORT:
		SET_INT( short, i, mod );
		break;
		
	case IM_BANDFMT_UINT:
		SET_INT( unsigned int, i, mod );
		break;
		
	case IM_BANDFMT_INT:
		SET_INT( int, i, mod );
		break;
		
	case IM_BANDFMT_FLOAT:
		SET_FLOAT( float, i, mod );
		break;
		
	case IM_BANDFMT_COMPLEX:
		SET_FLOAT( float, (i << 1), re );
		SET_FLOAT( float, (i << 1) + 1, im );
		break;
		
	case IM_BANDFMT_DOUBLE:
		SET_FLOAT( double, i, mod );
		break;
		
	case IM_BANDFMT_DPCOMPLEX:
		SET_FLOAT( double, i << 1, re );
		SET_FLOAT( double, (i << 1) + 1, im );
		break;

	default:
		break;
	}
}

/* Parse a string to an imageinfo.
 * Strings are from imageinfo_to_text(), ie. of the form:
 *
 *	50, 0, 0
 *	(12,13), (14,15)
 *
 */
gboolean
imageinfo_from_text( Imageinfo *imageinfo, const char *text )
{
	char buf[MAX_LINELENGTH];
	char *p;
	int i;
	Rect dirty;

	im_strncpy( buf, text, MAX_LINELENGTH );

	for( i = 0, p = buf; p += strspn( p, WHITESPACE ), *p; i++ ) {
		double re, im;

		if( p[0] == '(' ) {
			/* Complex constant.
			 */
			re = atof( p + 1 );
			p = break_token( p, "," );
			im = atof( p );
			p = break_token( p, ")" );
		}
		else {
			/* Real constant.
			 */
			re = atof( p );
			im = 0;
		}

		p = break_token( p, "," );

		imageinfo_from_text_band( imageinfo, i, re, im );
	}

	dirty.left = 0;
	dirty.top = 0;
	dirty.width = 1;
	dirty.height = 1;
	imageinfo_area_changed( imageinfo, &dirty );

	return( TRUE );
}

/* Get the image as display RGB in rgb[0-2].
 */
void
imageinfo_to_rgb( Imageinfo *imageinfo, double *rgb )
{
	Conversion *conv = conversion_new( imageinfo );
	Rect area;
	PEL *p;
	int i;

	area.left = 0;
	area.top = 0;
	area.width = 1;
	area.height = 1;

	if( im_prepare_thread( conv->tg, conv->ireg, &area ) ) {
		FREEFO( gtk_object_unref, conv );
		return;
	}
        p = (PEL *) IM_REGION_ADDR( conv->ireg, area.left, area.top );

	if( imageinfo->im->Bands < 3 ) 
		for( i = 0; i < 3; i++ )
			rgb[i] = p[0] / 255.0;
	else 
		for( i = 0; i < 3; i++ )
			rgb[i] = p[i] / 255.0;

	FREEFO( gtk_object_unref, conv );
}

/* Try to overwrite an imageinfo with a display RGB colour.
 */
void
imageinfo_from_rgb( Imageinfo *imageinfo, double *rgb )
{
	IMAGE *im = imageinfo_get( FALSE, imageinfo );
	Imageinfo *in, *out;
	IMAGE *t1, *t2;
	int i;
	Rect dirty;

	/* FIXME ... should let other displays be used here, see
	 * ../scraps/calibrate.[hc]
	 */
	struct im_col_display *display = im_col_displays( 7 );

	/* Make 1 pixel images for conversion in and out.
	 */
	if( !(in = imageinfo_new_temp( reduce_context->hi, "t" )) ||
		!(out = imageinfo_new_temp( reduce_context->hi, "t" )) ) 
		return;
	if( !(t1 = im_open_local( out->im, "imageinfo_from_rgb:1", "t" )) ||
		!(t2 = im_open_local( out->im, "imageinfo_from_rgb:1", "t" )) )
		return;

	/* Fill in with rgb.
	 */
	im_initdesc( in->im, 1, 1, 3, 
		IM_BBITS_BYTE, IM_BANDFMT_UCHAR, IM_CODING_NONE, 
		IM_TYPE_RGB, 1.0, 1.0, 0, 0 );
	if( im_setupout( in->im ) ) 
		return;
	for( i = 0; i < 3; i++ )
		((PEL *) in->im->data)[i] = rgb[i] * 255.0;

	/* To imageinfo->type. Make sure we get a float.
	 */
	switch( imageinfo->im->Type ) {
	case IM_TYPE_XYZ:
		if( im_disp2XYZ( in->im, out->im, display ) )
			return;
		break;

	case IM_TYPE_YXY:
		if( im_disp2XYZ( in->im, t1, display ) ||
			im_XYZ2Yxy( t1, out->im ) )
			return;
		break;

	case IM_TYPE_LAB:
		if( im_disp2Lab( in->im, out->im, display ) )
			return;
		break;

	case IM_TYPE_LCH:
		if( im_disp2Lab( in->im, t1, display ) ||
			im_Lab2LCh( t1, out->im ) )
			return;
		break;

	case IM_TYPE_UCS:
		if( im_disp2Lab( in->im, t1, display ) ||
			im_Lab2LCh( t1, t2 ) ||
			im_LCh2UCS( t2, out->im ) )
			return;
		break;

	case IM_TYPE_sRGB:
		if( im_disp2XYZ( in->im, t1, display ) ||
			im_XYZ2sRGB( t1, t2 ) ||
			im_clip2f( t2, out->im ) )
			return;
		break;

	case IM_TYPE_RGB:
	default:
		if( im_clip2f( in->im, out->im ) )
			return;
		break;
	}

#define SET( TYPE, i ) ((TYPE *) im->data)[i] = ((float *) out->im->data)[i];

	/* Now ... overwrite imageinfo.
	 */
	for( i = 0; i < im->Bands; i++ )
		switch( im->BandFmt ) {
		case IM_BANDFMT_UCHAR:          
			SET( unsigned char, i ); 
			break;

		case IM_BANDFMT_CHAR:           
			SET( signed char, i ); 
			break;

		case IM_BANDFMT_USHORT:         
			SET( unsigned short, i ); 
			break;

		case IM_BANDFMT_SHORT:          
			SET( signed short, i ); 
			break;

		case IM_BANDFMT_UINT:           
			SET( unsigned int, i ); 
			break;

		case IM_BANDFMT_INT:            
			SET( signed int, i );  
			break;

		case IM_BANDFMT_FLOAT:          
			SET( float, i ); 
			break;

		case IM_BANDFMT_DOUBLE:         
			SET( double, i ); 
			break;

		case IM_BANDFMT_COMPLEX:        
			SET( float, i * 2 ); 
			SET( float, i * 2 + 1 ); 
			break;

		case IM_BANDFMT_DPCOMPLEX:      
			SET( double, i * 2 ); 
			SET( double, i * 2 + 1 ); 
			break;

		default:
			assert( FALSE );
		}

	dirty.left = 0;
	dirty.top = 0;
	dirty.width = 1;
	dirty.height = 1;
	imageinfo_area_changed( imageinfo, &dirty );
}

/* Widgets for colour edit.
 */
typedef struct _ColourEdit {
	iDialog *idlg;

	Imageinfo *imageinfo;
	GtkWidget *colour_widget;
} ColourEdit;

/* Done button hit.
 */
static void
imageinfo_colour_done_cb( iWindow *iwnd, void *client, 
	iWindowNotifyFn nfn, void *sys )
{
	ColourEdit *eds = (ColourEdit *) client;
	Imageinfo *imageinfo = eds->imageinfo;
	double rgb[4];

	gtk_color_selection_get_color( 
		GTK_COLOR_SELECTION( eds->colour_widget ), rgb );

	/* This will emit "area_changed" on our imageinfo.
	 */
	imageinfo_from_rgb( imageinfo, rgb );

	nfn( sys, IWINDOW_TRUE );
}

/* Build the insides of colour edit.
 */
static void
imageinfo_colour_buildedit( iDialog *idlg, GtkWidget *work, ColourEdit *eds )
{
	Imageinfo *imageinfo = eds->imageinfo;
	double rgb[4];

	eds->colour_widget = gtk_color_selection_new();
	gtk_color_selection_set_opacity( 
		GTK_COLOR_SELECTION( eds->colour_widget ), FALSE );
	imageinfo_to_rgb( imageinfo, rgb );
	gtk_color_selection_set_color( 
		GTK_COLOR_SELECTION( eds->colour_widget ), rgb );
	gtk_color_selection_get_color( 
		GTK_COLOR_SELECTION( eds->colour_widget ), rgb );
	gtk_color_selection_set_color( 
		GTK_COLOR_SELECTION( eds->colour_widget ), rgb );

        gtk_box_pack_start( GTK_BOX( work ), 
		eds->colour_widget, TRUE, TRUE, 2 );

        gtk_widget_show_all( work );
}

void
imageinfo_colour_edit( GtkWidget *parent, Imageinfo *imageinfo )
{
	ColourEdit *eds = IM_NEW( NULL, ColourEdit );
	GtkWidget *idlg;

	eds->imageinfo = imageinfo;

	idlg = idialog_new();
	iwindow_set_title( IWINDOW( idlg ), "Edit colour" );
	idialog_set_build( IDIALOG( idlg ), 
		(iWindowBuildFn) imageinfo_colour_buildedit, eds, NULL, NULL );
	idialog_set_callbacks( IDIALOG( idlg ), 
		iwindow_true_cb, NULL, NULL, idialog_free_client, eds );
	idialog_add_ok( IDIALOG( idlg ), 
		imageinfo_colour_done_cb, "Set colour" );
	idialog_set_parent( IDIALOG( idlg ), parent );
	iwindow_build( IWINDOW( idlg ) );

	gtk_widget_show( GTK_WIDGET( idlg ) );
}

