/* make and manage base dialogs ... subclass off this for others
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

*/

#include "ip.h"

/*
#define DEBUG
 */

static iWindowClass *parent_class = NULL;

/* Handy destroy callback ... just free client.
 */
void
idialog_free_client( iDialog *idlg, void *client )
{
	FREE( client );
}

/* Notify our parent.
 */
static void
idialog_notify_parent( iDialog *idlg, iWindowResult result )
{
	if( idlg->nfn ) {
		iWindowNotifyFn nfn = idlg->nfn;
		
		idlg->nfn = NULL;
		nfn( idlg->sys, result );
	}
}

static void *
idialog_set_sensitive( GtkWidget *w, gboolean state )
{
	gtk_widget_set_sensitive( w, state );

	return( NULL );
}

/* Set OK sensitivities.
 */
void
idialog_set_ok_button_state( iDialog *idlg, gboolean state )
{
	slist_map( idlg->ok_l, 
		(SListMapFn) idialog_set_sensitive, (void *) state );
}

/* Set all the button sensitivities.
 */
static void
idialog_set_button_state( iDialog *idlg, gboolean state )
{
	idialog_set_ok_button_state( idlg, state );
	if( idlg->but_cancel )
		gtk_widget_set_sensitive( idlg->but_cancel, state );
	if( idlg->but_help )
		gtk_widget_set_sensitive( idlg->but_help, state );
}

/* Sub-fn of below. Come back from a popdown notify.
 */
static void
idialog_popdown_notify( void *sys, iWindowResult result )
{
        iWindowSusp *susp = IWINDOW_SUSP( sys );
	iDialog *idlg = IDIALOG( susp->client );

	if( result == IWINDOW_TRUE ) 
		/* If our caller hasn't been notified yet, post off a FALSE.
		 */
		idialog_notify_parent( idlg, IWINDOW_FALSE );

	/* Pass result on to our suspension (ie. back to iwindow).
	 */
	iwindow_susp_return( susp, result );

	/* Housekeeping.
	 */
	iwindow_notify_return( IWINDOW( idlg ) );
}

/* Our popdown callback ... here from iwindow.
 */
static void 
idialog_popdown_cb( iWindow *iwnd, void *client, 
	iWindowNotifyFn nfn, void *sys ) 
{
	iDialog *idlg = IDIALOG( client );
        iWindowSusp *susp = iwindow_susp_new( NULL, iwnd, idlg, nfn, sys );

#ifdef DEBUG
	printf( "idialog_popdown_cb: %s\n", IWINDOW( idlg )->title );
#endif /*DEBUG*/

	/* Trigger user popdown.
	 */
	iwindow_notify_send( IWINDOW( idlg ), 
		idlg->popdown_cb, idlg->client, idialog_popdown_notify, susp );
}

/* Sub-fn of below. Come back from a done notify.
 */
static void
idialog_done_notify( void *sys, iWindowResult result )
{
	iDialog *idlg = IDIALOG( sys );

#ifdef DEBUG
	printf( "idialog_done_notify: %s\n", IWINDOW( idlg )->title );
#endif /*DEBUG*/

	idialog_set_button_state( idlg, TRUE );

	/* If all ok, popdown and tell our parent.
	 */
	if( result == IWINDOW_TRUE ) {
		/* Unless we're pinned up, that is.
		 */
		if( !(idlg->tog_pin && 
			gtk_toggle_button_get_active(
				GTK_TOGGLE_BUTTON( idlg->tog_pin ) )) ) {
			idialog_notify_parent( idlg, result );
			iwindow_kill( IWINDOW( idlg ) );
		}
	}

	/* Alert on failure.
	 */
	if( result == IWINDOW_ERROR )
		box_alert( GTK_WIDGET( idlg ) );

	/* Clean up.
	 */
	iwindow_notify_return( IWINDOW( idlg ) );
}

/* Make a DONE event happen. Used (for example) by the browse window to force
 * a done in the enclosing FSB on double click on icon. 
 */
void
idialog_done_trigger( iDialog *idlg, int pos )
{
	iWindowFn fn = (iWindowFn) g_slist_nth_data( idlg->ok_cb_l, pos );

#ifdef DEBUG
	printf( "idialog_done_trigger: %s, %d\n", 
		IWINDOW( idlg )->title, pos );
#endif /*DEBUG*/

	/* Trigger user done callback.
	 */
	assert( fn );
	idialog_set_button_state( idlg, FALSE );
	iwindow_notify_send( IWINDOW( idlg ), 
		fn, idlg->client, idialog_done_notify, idlg );
}

/* Sub-fn of below.
 */
static void
idialog_cancel_notify( void *sys, iWindowResult result )
{
	iDialog *idlg = IDIALOG( sys );

#ifdef DEBUG
	printf( "idialog_cancel_notify: %s\n", IWINDOW( idlg )->title );
#endif /*DEBUG*/

	idialog_set_button_state( idlg, TRUE );

	/* Send cancel message back to parent if our client cancel was OK.
	 */
	if( result == IWINDOW_TRUE ) {
		idialog_notify_parent( idlg, IWINDOW_FALSE );
		iwindow_kill( IWINDOW( idlg ) );
	}

	/* Alert on error.
	 */
	if( result == IWINDOW_ERROR )
		box_alert( GTK_WIDGET( idlg ) );

	/* Clean up.
	 */
	iwindow_notify_return( IWINDOW( idlg ) );
}

static void
idialog_cancel_trigger( iDialog *idlg )
{
#ifdef DEBUG
        printf( "idialog_cancel_trigger: %s\n", IWINDOW( idlg )->title );
#endif /*DEBUG*/

        /* Trigger user cancel function.
         */
	idialog_set_button_state( idlg, FALSE );
        iwindow_notify_send( IWINDOW( idlg ), 
		idlg->cancel_cb, idlg->client, idialog_cancel_notify, idlg );
}

/* Sub-fn of below.
 */
static void
idialog_help_notify( void *sys, iWindowResult result )
{
	iDialog *idlg = IDIALOG( sys );

#ifdef DEBUG
	printf( "idialog_help_notify: %s\n", IWINDOW( idlg )->title );
#endif /*DEBUG*/

	idialog_set_button_state( idlg, TRUE );

	/* Alert on error.
	 */
	if( result == IWINDOW_ERROR )
		box_alert( GTK_WIDGET( idlg ) );

	/* Clean up.
	 */
	iwindow_notify_return( IWINDOW( idlg ) );
}

static void
idialog_help_trigger( iDialog *idlg )
{
#ifdef DEBUG
        printf( "idialog_help_trigger: %s\n", IWINDOW( idlg )->title );
#endif /*DEBUG*/

        /* Trigger user help function.
         */
	idialog_set_button_state( idlg, FALSE );
        iwindow_notify_send( IWINDOW( idlg ), 
		idlg->help_cb, idlg->client, idialog_help_notify, idlg );
}

/* Button callbacks from gtk.
 */
static void
idialog_done_cb( GtkWidget *w, iDialog *idlg )
{
	int pos = g_slist_index( idlg->ok_l, w );

	assert( pos != -1 );

	idialog_done_trigger( idlg, pos );
}

static void
idialog_cancel_cb( GtkWidget *w, iDialog *idlg )
{
	idialog_cancel_trigger( idlg );
}

static void
idialog_help_cb( GtkWidget *w, iDialog *idlg )
{
	idialog_help_trigger( idlg );
}

static void
idialog_destroy( GtkObject *object )
{
	iDialog *idlg;

#ifdef DEBUG
	printf( "idialog_destroy\n" );
#endif /*DEBUG*/

	g_return_if_fail( object != NULL );
	g_return_if_fail( IS_IDIALOG( object ) );

	idlg = IDIALOG( object );

#ifdef DEBUG
	printf( "... %s\n", IWINDOW( idlg )->title );
#endif /*DEBUG*/

	/* My instance destroy stuff.
	 */

	if( idlg->destroy_cb ) {
		idlg->destroy_cb( idlg, idlg->client );
		idlg->destroy_cb = NULL;
	}

	FREESID( idlg->parent_sid, idlg->window_parent );
	FREESID( idlg->object_sid, idlg->object );
	slist_map( idlg->ok_txt_l, (SListMapFn) im_free, NULL );
	FREEF( g_slist_free, idlg->ok_cb_l );
	FREEF( g_slist_free, idlg->ok_txt_l );
	FREEF( g_slist_free, idlg->ok_l );

	GTK_OBJECT_CLASS( parent_class )->destroy( object );
}

static void
idialog_realize( GtkWidget *widget )
{
	iDialog *idlg = IDIALOG( widget );

#ifdef DEBUG
	printf( "idialog_realize: %s\n", IWINDOW( idlg )->title );
#endif /*DEBUG*/

	/* Centre over parent. We should be able to just set a hint and let
	 * the WM do this, but it's very unreliable. Do it ourselves.
	 */
        if( idlg->window_parent ) {
		GtkWindow *pwnd = GTK_WINDOW( 
			gtk_widget_get_toplevel( idlg->window_parent ) );
		GtkWidget *wid = GTK_WIDGET( idlg );
		int x, y, w, h;
		int dx, dy;
		const int swidth = gdk_screen_width();
		const int sheight = gdk_screen_height();

		/* get_root_origin allows for WM decorations.
		 */
		gdk_window_get_size( GTK_WIDGET( pwnd )->window, &w, &h );
		gdk_window_get_root_origin( GTK_WIDGET( pwnd )->window, 
			&x, &y );
		dx = x + w / 2 - wid->requisition.width / 2;
		dy = y + h / 2 - wid->requisition.height / 2;

		/* Clip against screen size. Allow a bit for desktop borders
		 * added by the WM.
		 */
		dx = IM_CLIP( 50, dx, swidth - wid->requisition.width - 50 );
		dy = IM_CLIP( 50, dy, sheight - wid->requisition.height - 50 );

		gtk_widget_set_uposition( GTK_WIDGET( idlg ), dx, dy );

#ifdef DEBUG
		printf( "idialog_realize: centering over parent at %dx%d\n",
			dx, dy );
#endif /*DEBUG*/
	}

	GTK_WIDGET_CLASS( parent_class )->realize( widget );

	if( idlg->entry )
		gtk_widget_grab_focus( GTK_WIDGET( idlg->entry ) );
}

/* Our parent has been destroyed, kill us too.
 */
static void
idialog_parent_destroy( GtkWidget *par, iDialog *idlg )
{
#ifdef DEBUG
	printf( "idialog_parent_destroy: %s\n", IWINDOW( idlg )->title );
#endif /*DEBUG*/

	/* Here for dead parent ... if parent is dead, we won't need to remove
	 * the dead-dad signal. 
	 */
	idlg->parent_sid = 0;

	iwindow_kill( IWINDOW( idlg ) );
}

/* The object we represent has been destroyed, kill us too.
 */
static void
idialog_object_destroy( GtkWidget *par, iDialog *idlg )
{
#ifdef DEBUG
	printf( "idialog_object_destroy: %s\n", IWINDOW( idlg )->title );
#endif /*DEBUG*/

	/* Here for dead parent ... if parent is dead, we won't need to remove
	 * the dead-dad signal. 
	 */
	idlg->object_sid = 0;

	iwindow_kill( IWINDOW( idlg ) );
}

static void *
idialog_build_ok( const char *txt, iDialog *idlg )
{
	GtkWidget *but;

	but = build_button( txt, idialog_done_cb, idlg );
	idlg->ok_l = g_slist_prepend( idlg->ok_l, but );
	gtk_box_pack_start( GTK_BOX( idlg->bb ), but, TRUE, TRUE, 0 );
	gtk_widget_show( but );

	return( NULL );
}

static void 
idialog_build( GtkWidget *widget )
{
	iDialog *idlg = IDIALOG( widget );
	iWindow *iwnd = IWINDOW( idlg );
	GtkAccelGroup *accel_group;

#ifdef DEBUG
	printf( "idialog_build: %s\n", iwnd->title );
#endif /*DEBUG*/

	/* Call all builds in superclasses.
	 */
	if( IWINDOW_CLASS( parent_class )->build )
		(*IWINDOW_CLASS( parent_class )->build)( widget );

	/* delete_event and destroy handled by our superclass.
	 */
	iwindow_set_popdown( iwnd, idialog_popdown_cb, idlg );

        gtk_window_set_modal( GTK_WINDOW( idlg ), idlg->modal );

        idlg->work = gtk_vbox_new( FALSE, 2 );
        gtk_container_set_border_width( GTK_CONTAINER( idlg->work ), 5 );
        gtk_box_pack_start( GTK_BOX( iwnd->work ), idlg->work, TRUE, TRUE, 2 );

	if( !idlg->nosep ) {
		GtkWidget *sep;

		sep = gtk_hseparator_new();
		gtk_box_pack_start( GTK_BOX( iwnd->work ), 
			sep, FALSE, FALSE, 2 );
		gtk_widget_show( sep );
	}

	idlg->hb = gtk_hbox_new( FALSE, 0 );
        gtk_box_pack_start( GTK_BOX( iwnd->work ), idlg->hb, FALSE, FALSE, 0 );
        gtk_widget_show( idlg->hb );

        idlg->bb = gtk_hbutton_box_new();
        gtk_button_box_set_layout( GTK_BUTTON_BOX( idlg->bb ), 
		GTK_BUTTONBOX_END );
        gtk_button_box_set_spacing( GTK_BUTTON_BOX( idlg->bb ), 5 );
        gtk_container_set_border_width( GTK_CONTAINER( idlg->bb ), 10 );
        gtk_box_pack_end( GTK_BOX( idlg->hb ), idlg->bb, FALSE, FALSE, 0 );
        gtk_widget_show( idlg->bb );

	if( idlg->pinup ) {
		idlg->tog_pin = gtk_check_button_new_with_label( "Pin up" );
		set_tooltip( idlg->tog_pin, "Check this to pin the dialog up" );
                gtk_box_pack_start( GTK_BOX( idlg->hb ),
                        idlg->tog_pin, FALSE, FALSE, 0 );
		gtk_widget_show( idlg->tog_pin );
	}

        if( idlg->help_cb ) {
		GtkWidget *align;

		align = gtk_alignment_new( 0.0, 0.5, 0.0, 0.0 );
		gtk_container_set_border_width( GTK_CONTAINER( align ), 10 );
                gtk_box_pack_start( GTK_BOX( idlg->hb ),
                        align, FALSE, FALSE, 0 );
		gtk_widget_show( align );

                idlg->but_help = build_button( "Help", idialog_help_cb, idlg );
                gtk_container_add( GTK_CONTAINER( align ), idlg->but_help );
                gtk_widget_show( idlg->but_help );
        }

        if( idlg->cancel_cb ) {
                idlg->but_cancel = build_button( "Cancel", 
			idialog_cancel_cb, idlg );
                gtk_box_pack_start( GTK_BOX( idlg->bb ),
                        idlg->but_cancel, TRUE, TRUE, 0 );
                gtk_widget_show( idlg->but_cancel );
                if( !idlg->nook && !idlg->ok_l )
                        gtk_widget_grab_default( idlg->but_cancel );
	}

        slist_map_rev( idlg->ok_txt_l,
		(SListMapFn) idialog_build_ok, idlg );

	/* Escape triggers cancel, or OK if no cancel
	 */
	accel_group = gtk_accel_group_new();
	gtk_accel_group_attach( accel_group, GTK_OBJECT( idlg ) );
	gtk_widget_add_accelerator( 
		idlg->cancel_cb ? 
			idlg->but_cancel : GTK_WIDGET( idlg->ok_l->data ), 
		"clicked", accel_group, GDK_Escape, 0, 0 );

	/* F1 triggers help.
	 */
	if( idlg->help_cb ) 
		gtk_widget_add_accelerator( 
			idlg->but_help,
			"clicked", accel_group, GDK_F1, 0, 0 );

	if( !idlg->nook && idlg->ok_l )
		gtk_widget_grab_default( GTK_WIDGET( idlg->ok_l->data ) );

	/* Build user dialog contents.
	 */
	if( idlg->build )
		idlg->build( iwnd, idlg->work, 
			idlg->build_a, idlg->build_b, idlg->build_c );

	/* Link to parent.
	 */
        if( idlg->window_parent ) {
                gtk_window_set_transient_for( GTK_WINDOW( idlg ),
                        GTK_WINDOW( gtk_widget_get_toplevel( 
				idlg->window_parent ) ) );
		idlg->parent_sid = gtk_signal_connect( 
			GTK_OBJECT( idlg->window_parent ), 
			"destroy",
			GTK_SIGNAL_FUNC( idialog_parent_destroy ), idlg );
	}
	if( idlg->object )
		idlg->object_sid = gtk_signal_connect( idlg->object,
			"destroy",
			GTK_SIGNAL_FUNC( idialog_object_destroy ), idlg );

        gtk_widget_show( idlg->work );
}

static void
idialog_class_init( iDialogClass *klass )
{
	GtkObjectClass *object_class;
	GtkWidgetClass *widget_class;
	iWindowClass *iwindow_class;

	object_class = (GtkObjectClass *) klass;
	widget_class = (GtkWidgetClass *) klass;
	iwindow_class = (iWindowClass *) klass;

	parent_class = gtk_type_class( TYPE_IWINDOW );

	object_class->destroy = idialog_destroy;
	widget_class->realize = idialog_realize;
	iwindow_class->build = idialog_build;

	/* Create signals.
	 */

	/* Init methods.
	 */
}

static void
idialog_init( iDialog *idlg )
{
#ifdef DEBUG
	printf( "idialog_init: %s\n", IWINDOW( idlg )->title );
#endif /*DEBUG*/

	/* Init our instance fields.
	 */
	idlg->window_parent = NULL;
        idlg->parent_sid = 0;
	idlg->object = NULL;
        idlg->object_sid = 0;

	idlg->work = NULL;

	idlg->ok_l = NULL;
	idlg->ok_txt_l = NULL;
	idlg->but_cancel = NULL;
	idlg->but_help = NULL;
	idlg->tog_pin = NULL;

	idlg->entry = NULL;

	idlg->nook = FALSE;
	idlg->modal = FALSE;
	idlg->pinup = FALSE;
	idlg->nosep = TRUE;

	idlg->ok_cb_l = NULL;
	idlg->cancel_cb = NULL;
	idlg->help_cb = NULL;
	idlg->popdown_cb = NULL;
	idlg->destroy_cb = NULL;
	idlg->client = NULL;

	idlg->arg = NULL;

	idlg->nfn = iwindow_notify_null;
	idlg->sys = NULL;
}

GtkType
idialog_get_type( void )
{
	static GtkType idialog_type = 0;

	if (!idialog_type) {
		static const GtkTypeInfo idlg_info = {
			"iDialog",
			sizeof( iDialog ),
			sizeof( iDialogClass ),
			(GtkClassInitFunc) idialog_class_init,
			(GtkObjectInitFunc) idialog_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		idialog_type = gtk_type_unique( TYPE_IWINDOW, &idlg_info );
	}

	return( idialog_type );
}

GtkWidget *
idialog_new()
{
	iDialog *idlg = gtk_type_new( TYPE_IDIALOG );
	GtkWindow *gwnd = GTK_WINDOW( idlg );

	/* Init gtk base class.
	 */
	gwnd->type = GTK_WINDOW_DIALOG;

	return( GTK_WIDGET( idlg ) );
}

void 
idialog_set_parent( iDialog *idlg, GtkWidget *window_parent )
{
	idlg->window_parent = window_parent;
}

void 
idialog_set_object( iDialog *idlg, GtkObject *object )
{
	idlg->object = object;
}

void 
idialog_set_nook( iDialog *idlg, gboolean nook )
{
	idlg->nook = nook;
}

void 
idialog_set_pinup( iDialog *idlg, gboolean pinup )
{
	idlg->pinup = pinup;
}

void 
idialog_set_modal( iDialog *idlg, gboolean modal )
{
	idlg->modal = modal;
}

void 
idialog_set_nosep( iDialog *idlg, gboolean nosep )
{
	idlg->nosep = nosep;
}

void 
idialog_set_callbacks( iDialog *idlg, 
	iWindowFn cancel_cb, iWindowFn help_cb, iWindowFn popdown_cb, 
	iDialogFreeFn destroy_cb, void *client )
{
	idlg->cancel_cb = cancel_cb;
	idlg->help_cb = help_cb;
	idlg->popdown_cb = popdown_cb;
	idlg->destroy_cb = destroy_cb;
	idlg->client = client;
}

void 
idialog_add_ok( iDialog *idlg, iWindowFn done_cb, const char *fmt, ... )
{
	va_list ap;
	char buf[ 1024 ];

        va_start( ap, fmt );
        (void) im_vsnprintf( buf, 1024, fmt, ap );
        va_end( ap );

	idlg->ok_cb_l = g_slist_append( idlg->ok_cb_l, (void *) done_cb );
	idlg->ok_txt_l = g_slist_append( idlg->ok_txt_l, 
		im_strdup( NULL, buf ) );
}

void 
idialog_set_notify( iDialog *idlg, iWindowNotifyFn nfn, void *sys )
{
	idlg->nfn = nfn;
	idlg->sys = sys;
}

void 
idialog_set_build( iDialog *idlg, 
	iWindowBuildFn build, void *build_a, void *build_b, void *build_c )
{
	idlg->build = build;
	idlg->build_a = build_a;
	idlg->build_b = build_b;
	idlg->build_c = build_c;
}

void 
idialog_set_default_entry( iDialog *idlg, GtkEntry *entry )
{
	if( !idlg->nook && idlg->ok_l ) {
		gtk_signal_connect_object( GTK_OBJECT( entry ), 
			"focus_in_event", 
			GTK_SIGNAL_FUNC( gtk_widget_grab_default ),
			GTK_OBJECT( idlg->ok_l->data ) );
		gtk_signal_connect_object( GTK_OBJECT( entry ), 
			"activate", 
			GTK_SIGNAL_FUNC( gtk_button_clicked ),
			GTK_OBJECT( idlg->ok_l->data ) );
		idlg->entry = entry;
	}
}

/* Set up an entry inside a dialog ... set tooltip, set start
 * value, link to OK button in enclosing dialog.
 */
void
idialog_init_entry( iDialog *idlg, GtkWidget *entry, 
	const char *tip, const char *fmt, ... )
{
	va_list ap;

	va_start( ap, fmt );
        set_gentryv( entry, fmt, ap );
	va_end( ap );
        set_tooltip( entry, "%s", tip );
	idialog_set_default_entry( idlg, GTK_ENTRY( entry ) );
}
