/* gtkutil functions.
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

#include "ip.h"

/*
#define DEBUG
 */

/* All our tooltips.
 */
static GtkTooltips *our_tooltips = NULL;

/* Set two adjustments together.
 */
void
adjustments_set_value( GtkAdjustment *hadj, GtkAdjustment *vadj,
        float hval, float vval )
{
        gboolean hchanged = FALSE;
        gboolean vchanged = FALSE;

        hval = IM_CLIP( hadj->lower, hval, hadj->upper );
        vval = IM_CLIP( vadj->lower, vval, vadj->upper );

        if( hval != hadj->value ) {
                hadj->value = hval;
                hchanged = TRUE;
        }
        if( vval != vadj->value ) {
                vadj->value = vval;
                vchanged = TRUE;
        }

#ifdef DEBUG
        if( hchanged )
                printf( "adjustments_set_value: hadj = %g\n", hval );
        if( vchanged )
                printf( "adjustments_set_value: vadj = %g\n", vval );
#endif /*DEBUG*/

        if( hchanged )
                gtk_adjustment_value_changed( hadj );
        if( vchanged )
                gtk_adjustment_value_changed( vadj );
}

/* Like gtk_object_destroy, but return NULL for list maps.
 */
void *
object_destroy( void *obj )
{
	gtk_object_destroy( GTK_OBJECT( obj ) );

	return( NULL );
}

/* Like g_free, but return NULL for list maps.
 */
void *
null_g_free( void *obj )
{
	g_free( obj );

	return( NULL );
}

/* Set visible/not.
 */
void
widget_visible( GtkWidget *widget, gboolean visible )
{
	if( widget && visible )
		gtk_widget_show( widget );
	else if( widget && !visible )
		gtk_widget_hide( widget );
}

/* Turn an xy coordinate to a row/column position in a table.
 */
gboolean
table_find_pos( GtkWidget *widget, int x, int y, int *row, int *column )
{
        GtkTable *table = GTK_TABLE( widget );
        int right = widget->allocation.x;
        int bottom = widget->allocation.y;
        int i;

        if( x < right || y < bottom )
                return( FALSE );

        *column = -1;
        for( i = 0; i < table->ncols; i++ ) {
                right += table->cols[i].allocation;

                if( x < right ) {
                        *column = i;
                        break;
                }
        }

        *row = -1;
        for( i = 0; i < table->nrows; i++ ) {
                bottom += table->rows[i].allocation;

                if( y < bottom ) {
                        *row = i;
                        break;
                }
        }

#ifdef DEBUG
	g_message( "table_find_pos: row=%d, column=%d", *row, *column );
#endif /*DEBUG*/

        return( *row >= 0 && *column >= 0 );
}

static void
allocation2rect( GtkAllocation *from, Rect *to )
{
        to->left = from->x;
        to->top = from->y;
        to->width = from->width;
        to->height = from->height;
}

/* Turn an xy pos to a widget in a table.
 */
GtkWidget *
table_find_child( GtkWidget *widget, int x, int y )
{
        GtkTable *table = GTK_TABLE( widget );
        GList *p;

#ifdef DEBUG
	g_message( "table_find_child: x=%d, y=%d", x, y );
#endif /*DEBUG*/

        for( p = table->children; p; p = p->next ) {
                GtkTableChild *table_child = p->data;
                Rect child;

		/* Skip hidden widgets.
		 */
		if( !GTK_WIDGET_VISIBLE( table_child->widget ) )
			continue;

                allocation2rect( &table_child->widget->allocation, &child );

#ifdef DEBUG
		g_message( "table_find_child: "
			"child at x=%d, y=%d, w=%d, h=%d", 
			child.left, child.top, child.width, child.height );
#endif /*DEBUG*/

                if( im_rect_includespoint( &child, x, y ) )
                        return( table_child->widget );
        }

        return( NULL );
}

/* Make a button widget.
 */
GtkWidget *
build_button( const char *name, GtkSignalFunc cb, gpointer user )
{
	GtkWidget *but;

	but = gtk_button_new_with_label( name );
	GTK_WIDGET_SET_FLAGS( but, GTK_CAN_DEFAULT );
	gtk_signal_connect( GTK_OBJECT( but ), 
		"clicked", GTK_SIGNAL_FUNC( cb ), user );

	return( but );
}

/* Calculate the bounding box for a string rendered with a widget's default
 * font. Set geo to a rect with 0,0 positioned on the left-hand baseline.
 */
void
get_geo( GtkWidget *widget, const char *text, Rect *geo )
{
	int lbearing, rbearing, width, ascent, descent;
	GdkFont *font = gtk_widget_get_style( widget )->font;

	gdk_text_extents( font, text, strlen( text ), 
		  &lbearing, &rbearing, &width, &ascent, &descent );

	geo->width = rbearing + lbearing;
	geo->height = ascent + descent;
	geo->left = lbearing;
	geo->top = -ascent;
}

/* Set a widget to a fixed size ... width in characters.
 */
void
set_fixed( GtkWidget *widget, int nchars )
{
	Rect geo;

	get_geo( widget, "W", &geo );
	gtk_widget_set_usize( widget, geo.width * nchars, geo.height );
}

/* Build a GtkEntry, with a widget width specified in characters.
 */
GtkWidget *
build_entry( int nchars )
{
	GtkWidget *entry;
	Rect geo;

	entry = gtk_entry_new();
	get_geo( entry, "W", &geo );
	gtk_widget_set_usize( entry, geo.width * nchars, -2 );

	return( entry );
}

/* Build a new menu.
 */
GtkWidget *
menu_build( const char *name )
{
	GtkWidget *menu;

	menu = gtk_menu_new();
	gtk_menu_set_title( GTK_MENU( menu ), name );

	return( menu );
}

/* Add a menu item.
 */
GtkWidget *
menu_add_but( GtkWidget *menu, const char *name, GtkSignalFunc cb, void *user )
{
	GtkWidget *but;

	but = gtk_menu_item_new_with_label( name );
	gtk_menu_append( GTK_MENU( menu ), but );
	gtk_widget_show( but );
	gtk_signal_connect( GTK_OBJECT( but ), "activate", 
		GTK_SIGNAL_FUNC( cb ), user );

	return( but );
}

/* Add a toggle item.
 */
GtkWidget *
menu_add_tog( GtkWidget *menu, const char *name, GtkSignalFunc cb, void *user )
{
	GtkWidget *tog;

	tog = gtk_check_menu_item_new_with_label( name );
	gtk_menu_append( GTK_MENU( menu ), tog );
	gtk_widget_show( tog );
	gtk_signal_connect( GTK_OBJECT( tog ), "toggled", 
		GTK_SIGNAL_FUNC( cb ), user );

	return( tog );
}

/* Add a separator.
 */
GtkWidget *
menu_add_sep( GtkWidget *menu )
{
	GtkWidget *sep;

	sep = gtk_menu_item_new();
	gtk_widget_set_sensitive( GTK_WIDGET( sep ), FALSE );
	gtk_menu_append( GTK_MENU( menu ), sep );
	gtk_widget_show( sep );

	return( sep );
}

/* Four quarks: each menu item has a quark linking back to the main pane, 
 * plus a quark for the user signal. The main pane has a quark linking to the
 * widget the menu was popped from, and that has the userdata for this context.
 * One more quark holds the popup in a host.
 */
static GQuark quark_main = 0;
static GQuark quark_host = 0;
static GQuark quark_data = 0;
static GQuark quark_popup = 0;

/* Build a new popup menu.
 */
GtkWidget *
popup_build( const char *name )
{
	/* Build our quarks.
	 */
	if( !quark_main ) {
		quark_main = g_quark_from_static_string( "ip_mainpane" );
		quark_host = g_quark_from_static_string( "ip_hostitem" );
		quark_data = g_quark_from_static_string( "ip_userdata" );
		quark_popup = g_quark_from_static_string( "ip_popup" );
	}

	return( menu_build( name ) );
}

/* Activate function for a popup menu item.
 */
static void
popup_activate_cb( GtkWidget *item, GtkSignalFunc cb )
{
	GtkWidget *qmain = gtk_object_get_data_by_id( 
		GTK_OBJECT( item ), quark_main );
	GtkWidget *qhost = gtk_object_get_data_by_id( 
		GTK_OBJECT( qmain ), quark_host );
	gpointer qdata = gtk_object_get_data_by_id( 
		GTK_OBJECT( qhost ), quark_data );

	(*cb)( item, qhost, qdata );
}

/* Add a menu item to a popup.
 */
GtkWidget *
popup_add_but( GtkWidget *popup, const char *name, GtkSignalFunc cb )
{
	GtkWidget *but = menu_add_but( popup, name, 
		popup_activate_cb, (void *) cb );

	gtk_object_set_data_by_id( GTK_OBJECT( but ), quark_main, popup );

	return( but );
}

/* Add a toggle item to a popup.
 */
GtkWidget *
popup_add_tog( GtkWidget *popup, const char *name, GtkSignalFunc cb )
{
	GtkWidget *tog = menu_add_tog( popup, name, 
		popup_activate_cb, (void *) cb );

	gtk_object_set_data_by_id( GTK_OBJECT( tog ), quark_main, popup );

	return( tog );
}

/* Show the popup.
 */
void
popup_show( GtkWidget *host, GdkEvent *ev )
{
	GtkWidget *popup = gtk_object_get_data_by_id( 
		GTK_OBJECT( host ), quark_popup );

	gtk_object_set_data_by_id( GTK_OBJECT( popup ), quark_host, host );
	gtk_menu_popup( GTK_MENU( popup ), NULL, NULL,
		(GtkMenuPositionFunc) NULL, NULL, 3, ev->button.time );
}

/* Event handler for popupshow.
 */
static gboolean
popup_handle_event( GtkWidget *host, GdkEvent *ev, gpointer dummy )
{
        if( ev->type == GDK_BUTTON_PRESS && ev->button.button == 3 ) {
		gtk_signal_emit_stop_by_name( GTK_OBJECT( host ), "event" );
		popup_show( host, ev );

		return( TRUE );
	}

	return( FALSE );
}

/* Link a host to a popup.
 */
void
popup_link( GtkWidget *host, GtkWidget *popup, gpointer data )
{
	gtk_object_set_data_by_id( GTK_OBJECT( host ), quark_popup, popup );
	gtk_object_set_data_by_id( GTK_OBJECT( host ), quark_data, data );
}

/* Add a callback to show a popup.
 */
guint
popup_attach( GtkWidget *host, GtkWidget *popup, gpointer data )
{
	guint sid;

	popup_link( host, popup, data );
        sid = gtk_signal_connect( GTK_OBJECT( host ), "event",
                GTK_SIGNAL_FUNC( popup_handle_event ), NULL );

	return( sid );
}

void
popup_detach( GtkWidget *host, guint sid )
{
	gtk_signal_disconnect( GTK_OBJECT( host ), sid );
}

static void
set_tooltip_events( GtkWidget *wid )
{
	GdkEventMask mask;

	assert( wid->window );

	mask = gdk_window_get_events( wid->window );
	mask |= GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK;
	gdk_window_set_events( wid->window, mask );
}

/* Set the tooltip on a widget.
 */
void
set_tooltip( GtkWidget *wid, const char *fmt, ... )
{
	va_list ap;
	char *txt;

	if( !wid )
		return;

	if( !fmt )
		fmt = "";

	va_start( ap, fmt );
	txt = g_strdup_vprintf( fmt, ap );
	va_end( ap );

	if( !our_tooltips ) {
		our_tooltips = gtk_tooltips_new();
		gtk_tooltips_force_window( our_tooltips );
		set_name( our_tooltips->tip_window, "tooltip_widget" );
	}

	gtk_tooltips_set_tip( our_tooltips, wid, txt, NULL );

	if( !GTK_WIDGET_REALIZED( wid ) )
		gtk_signal_connect( GTK_OBJECT( wid ), "realize",
			GTK_SIGNAL_FUNC( set_tooltip_events ), NULL );
	else
		set_tooltip_events( wid );

	g_free( txt );
}

/* Set a GtkEditable.
 */
void
set_gentryv( GtkWidget *edit, const char *fmt, va_list ap )
{
	char buf[1000];
	gint position;
	int i;
	int len;

	if( !edit )
		return;

	if( !fmt )
		fmt = "";

	(void) im_vsnprintf( buf, 1000, fmt, ap );

	/* Filter out /n and /t ... they confuse gtkentry terribly
	 */
	len = strlen( buf );
	for( i = 0; i < len; i++ )
		if( buf[i] == '\n' || buf[i] == '\t' )
			buf[i] = ' ';

	gtk_editable_delete_text( GTK_EDITABLE( edit ), 0, -1 );
	position = 0;
	gtk_editable_insert_text( GTK_EDITABLE( edit ), 
		buf, strlen( buf ), &position );
}

/* Set a GtkEditable.
 */
void
set_gentry( GtkWidget *edit, const char *fmt, ... )
{
	va_list ap;

	va_start( ap, fmt );
	set_gentryv( edit, fmt, ap );
	va_end( ap );
}

/* Set a GtkLabel.
 */
/*VARARGS1*/
void
set_glabel( GtkWidget *label, const char *fmt, ... )
{
	va_list ap;
	char buf[1000];

	if( !label )
		return;

	if( !fmt )
		fmt = "";

	va_start( ap, fmt );
	(void) im_vsnprintf( buf, 1000, fmt, ap );
	va_end( ap );

	gtk_label_set_text( GTK_LABEL( label ), buf );
}

gboolean
get_geditable_name( GtkWidget *text, char *out, int sz )
{
	char *name;
	char *tname;

	name = gtk_editable_get_chars( GTK_EDITABLE( text ), 0, -1 );
	tname = trim_nonalpha( name );
	if( !tname ) {
		FREEF( g_free, name );
		ierrors( "enter an identifier" );
		return( FALSE );
	}
	im_strncpy( out, tname, sz );
	FREEF( g_free, name );

	return( TRUE );
}

gboolean
get_geditable_string( GtkWidget *text, char *out, int sz )
{
	char *str;

	str = gtk_editable_get_chars( GTK_EDITABLE( text ), 0, -1 );
	im_strncpy( out, str, sz );
	FREEF( g_free, str );

	return( TRUE );
}

gboolean
get_geditable_filename( GtkWidget *text, char *out, int sz )
{
	char *filename;
	char *tfilename;

	filename = gtk_editable_get_chars( GTK_EDITABLE( text ), 0, -1 );
	tfilename = trim_white( filename );
	if( !tfilename ) {
		FREEF( g_free, filename );
		ierrors( "enter a filename" );
		return( FALSE );
	}
	im_strncpy( out, tfilename, sz );
	FREEF( g_free, filename );

	return( TRUE );
}

/* Get a geditable as a double.
 */
gboolean
get_geditable_double( GtkWidget *text, double *out )
{
	char *txt = gtk_editable_get_chars( GTK_EDITABLE( text ), 0, -1 );
	double t;

	if( sscanf( txt, "%lg", &t ) != 1 ) {
		ierrors( "\"%s\" is not a float", txt );
		FREEF( g_free, txt );

		return( FALSE );
	}
	FREEF( g_free, txt );
	*out = t;

	return( TRUE );
}

/* Get as int.
 */
gboolean
get_geditable_int( GtkWidget *text, int *n )
{
	int i;
	char *txt = gtk_editable_get_chars( GTK_EDITABLE( text ), 0, -1 );

	/* Parse values.
	 */
	if( sscanf( txt, "%i", &i ) != 1 ) {
		ierrors( "\"%s\" is not an int", txt );
		FREEF( g_free, txt );

		return( FALSE );
	}
	FREEF( g_free, txt );
	*n = i;

	return( TRUE );
}

/* Get as unsigned int.
 */
gboolean
get_geditable_uint( GtkWidget *text, int *n )
{
	int i;

	if( !get_geditable_int( text, &i ) || i < 0 ) {
		ierrors( "not an unsigned int!" );

		return( FALSE );
	}
	*n = i;

	return( TRUE );
}

/* Get as positive int.
 */
gboolean
get_geditable_pint( GtkWidget *text, int *n )
{
	int i;

	if( !get_geditable_int( text, &i ) || i <= 0 ) {
		ierrors( "not a positive int!" );
		return( FALSE );
	}
	*n = i;

	return( TRUE );
}

/* Set a name for a widget and all it's children ... so they pick up a style.
 */
void 
set_name( GtkWidget *widget, const char *name )
{
	gtk_widget_set_name( widget, name );
	if( GTK_IS_CONTAINER( widget ) )
		gtk_container_foreach( GTK_CONTAINER( widget ), 
                           (GtkCallback) set_name, (char *) name );
}

/* Pack into labeled frame.
 */
GtkWidget *
build_glabelframe( GtkWidget *box, const char *label )
{	
	GtkWidget *frame;

        frame = gtk_frame_new( label );
        gtk_container_set_border_width( GTK_CONTAINER( frame ), 4 );
        gtk_container_add( GTK_CONTAINER( frame ), box );

	return( frame );
}

/* Indent widget, label above.
 */
GtkWidget *
build_glabelframe2( GtkWidget *widget, const char *name )
{
	GtkWidget *lab;
	GtkWidget *vb;
	GtkWidget *hb;
	GtkWidget *inv;

        hb = gtk_hbox_new( FALSE, 2 );
        inv = gtk_label_new( "" );
        gtk_box_pack_start( GTK_BOX( hb ), inv, FALSE, FALSE, 15 );
        gtk_box_pack_start( GTK_BOX( hb ), widget, TRUE, TRUE, 0 );

        vb = gtk_vbox_new( FALSE, 2 );
	lab = gtk_label_new( name );
	gtk_misc_set_alignment( GTK_MISC( lab ), 0.0, 0.5 );
        gtk_box_pack_start( GTK_BOX( vb ), lab, FALSE, FALSE, 0 );
        gtk_box_pack_start( GTK_BOX( vb ), hb, TRUE, TRUE, 0 );

	return( vb );
}

/* Make a text field + label. Return text widget. Position divide at 50%.
 */
GtkWidget *
build_glabeltext( GtkWidget *box, const char *label )
{	
	GtkWidget *hb;
	GtkWidget *lab;
	GtkWidget *txt;

        hb = gtk_hbox_new( TRUE, 2 );
        lab = gtk_label_new( label );
        gtk_box_pack_start( GTK_BOX( hb ), lab, TRUE, TRUE, 2 );
	gtk_misc_set_alignment( GTK_MISC( lab ), 1.0, 0.5 );
        txt = build_entry( 5 );
        gtk_box_pack_start( GTK_BOX( hb ), txt, TRUE, TRUE, 2 );

        gtk_box_pack_start( GTK_BOX( box ), hb, TRUE, TRUE, 2 );

	return( txt );
}

/* Make a text field + label. Return text widget. Make label small.
 */
GtkWidget *
build_glabeltext2( GtkWidget *box, const char *label, int nchars )
{
	GtkWidget *hb;
	GtkWidget *lab;
	GtkWidget *txt;

        hb = gtk_hbox_new( FALSE, 2 );
        lab = gtk_label_new( label );
        gtk_box_pack_start( GTK_BOX( hb ), lab, FALSE, FALSE, 2 );
        txt = build_entry( nchars );
        gtk_box_pack_start( GTK_BOX( hb ), txt, TRUE, TRUE, 2 );

        gtk_box_pack_start( GTK_BOX( box ), hb, FALSE, FALSE, 2 );

	return( txt );
}

/* Make a text field + label. Indent the text on a new line.
 */
GtkWidget *
build_glabeltext3( GtkWidget *box, const char *label )
{	
	GtkWidget *txt;
	GtkWidget *vb;

        txt = gtk_entry_new();
        vb = build_glabelframe2( txt, label );
        gtk_box_pack_start( GTK_BOX( box ), vb, FALSE, FALSE, 0 );

	return( txt );
}

/* Make a labeled toggle.
 */
GtkWidget *
build_gtoggle( GtkWidget *box, const char *caption )
{
        GtkWidget *hb;
        GtkWidget *inv;
	GtkWidget *toggle;

	/* Indent left a bit.
	 */
        inv = gtk_label_new( "" );
        hb = gtk_hbox_new( FALSE, 0 );
        gtk_box_pack_start( GTK_BOX( hb ), inv, FALSE, FALSE, 2 );
        toggle = gtk_check_button_new_with_label( caption );
        gtk_box_pack_start( GTK_BOX( hb ), toggle, FALSE, FALSE, 0 );

        gtk_box_pack_start( GTK_BOX( box ), hb, FALSE, FALSE, 0 );

	return( toggle );
}

/* Make a label plus option menu.
 */
GtkWidget *
build_goption( GtkWidget *box, 
	const char *name, const char *item_names[], int nitem,
	GtkSignalFunc fn, void *value )
{
	GtkWidget *hb;
	GtkWidget *label;
	GtkWidget *om;
	GtkWidget *pane;
	int i;

        hb = gtk_hbox_new( FALSE, 2 );
        label = gtk_label_new( name );
        gtk_box_pack_start( GTK_BOX( hb ), label, FALSE, TRUE, 2 );
	om = gtk_option_menu_new();
        gtk_box_pack_start( GTK_BOX( hb ), om, FALSE, TRUE, 2 );
        set_tooltip( om, "Left-click to change value" );

	pane = gtk_menu_new();
	for( i = 0; i < nitem; i++ ) {
		GtkWidget *item;

		item = gtk_menu_item_new_with_label( item_names[i] );
		gtk_menu_append( GTK_MENU( pane ), item );
		if( fn ) 
			gtk_signal_connect( GTK_OBJECT( item ), "activate", 
				fn, value );
	}
        gtk_widget_show_all( pane );
	gtk_option_menu_set_menu( GTK_OPTION_MENU( om ), pane );
        gtk_box_pack_start( GTK_BOX( box ), hb, FALSE, TRUE, 2 );

	return( om );
}

/* Which of an optionMenu is chosen? Should call gtk_option_menu_get_history(),
 * when its added to the api (1.4?).
 */
int
get_goption( GtkWidget *widget )
{
        GtkOptionMenu *om = GTK_OPTION_MENU( widget );

        g_return_val_if_fail( om != NULL, -1 );
        g_return_val_if_fail( om->menu != NULL, -1 );
        g_return_val_if_fail( om->menu_item != NULL, -1 );

        return( g_list_index( 
                GTK_MENU_SHELL( om->menu )->children, om->menu_item ) );
}

/* Which of a radio menu is chosen? 
 */
int
get_gradio( GtkWidget *widget )
{
	GtkMenu *menu = GTK_MENU( widget );
	GtkWidget *item = gtk_menu_get_active( menu );

	return( g_list_index( GTK_MENU_SHELL( menu )->children, item ) );
}

/* Set one element of an optionMenu.
 */
void
set_goption( GtkWidget *widget, int i )
{
	gtk_option_menu_set_history( GTK_OPTION_MENU( widget ), i );
}

/* Process all (and only) graphics_expose events. Used to catch up after a
 * scroll.
 */
void
process_events_graphics_expose( GtkWidget *widget )
{
	GdkEvent *event;

	if( !GTK_WIDGET_REALIZED( widget ))
		return;

	while( (event = gdk_event_get_graphics_expose( widget->window )) ) {
		gtk_widget_event( widget, event );
		if( event->expose.count == 0 ) {
			gdk_event_free( event );
			break;
		}
		gdk_event_free( event );
	}
}

/* Register a widget as a filename drag receiver.
 */
typedef struct {
	GtkWidget *widget;
	FiledropFunc fn;
	void *client;
} FiledropInfo;

static void
filedrop_drag_data_received( GtkWidget *widget, 
	GdkDragContext *context,
	gint x, gint y, GtkSelectionData *data, guint info, guint time,
	FiledropInfo *fdi )
{
	gchar *sPath = NULL;
	gchar *pFrom, *pTo; 
	gboolean result;

#ifdef DEBUG
	printf( "filedrop_drag_data_received: seen \"%s\"\n", data->data );
#endif /*DEBUG*/

        pFrom = strstr( (char *) data->data, "file:" );

        while( pFrom ) {
#if !GLIB_CHECK_VERSION (2,0,0)
		pFrom += 5; /* remove 'file:' */
#else
		GError *error = NULL;
#endif

		pTo = pFrom;
		while( *pTo != 0 && *pTo != 0xd && *pTo != 0xa ) 
			pTo += 1;

		sPath = g_strndup( pFrom, pTo - pFrom );

#if !GLIB_CHECK_VERSION (2,0,0)
#ifdef DEBUG
		printf( "filedrop_drag_data_received: "
			"invoking callback with \"%s\"\n", sPath );
#endif /*DEBUG*/

		result = fdi->fn( fdi->client, sPath );
#else
		/* format changed with Gtk+1.3, use conversion */
		pFrom = g_filename_from_uri( sPath, NULL, &error );

#ifdef DEBUG
		printf( "filedrop_drag_data_received: "
			"invoking callback with \"%s\"\n", pFrom );
#endif /*DEBUG*/

		result = fdi->fn( fdi->client, pFrom );
		g_free( pFrom );
#endif

		g_free( sPath );

		if( !result )
			box_alert( fdi->widget );

		pFrom = strstr( pTo, "file:" );
        } 

        gtk_drag_finish( context, TRUE, FALSE, time );
}

/* HB: file dnd stuff lent by The Gimp via Dia, not fully understood 
 * but working ...
 */
enum {
	TARGET_URI_LIST,
	TARGET_TEXT_PLAIN
};

static void
filedrop_destroy( GtkWidget *widget, FiledropInfo *fdi )
{
	im_free( fdi );
}

void
filedrop_register( GtkWidget *widget, FiledropFunc fn, void *client )
{
	static GtkTargetEntry target_table[] = {
		{ "text/uri-list", 0, TARGET_URI_LIST },
		{ "text/plain", 0, TARGET_TEXT_PLAIN }
	};

	FiledropInfo *fdi = IM_NEW( NULL, FiledropInfo );

	fdi->widget = widget;
	fdi->fn = fn;
	fdi->client = client;
	gtk_signal_connect( GTK_OBJECT( widget ), "destroy",
                GTK_SIGNAL_FUNC( filedrop_destroy ), fdi );

	gtk_drag_dest_set( GTK_WIDGET( widget ),
		GTK_DEST_DEFAULT_ALL,
		target_table, IM_NUMBER( target_table ),
		GDK_ACTION_COPY );
	gtk_signal_connect( GTK_OBJECT( widget ), "drag_data_received",
		GTK_SIGNAL_FUNC( filedrop_drag_data_received ), fdi );
}
