/* Stuff to parse and compile text.
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

/* regular (and very slow) sanity checks on symbols ... needs DEBUG in
 * symbol.c as well
#define DEBUG_SANITY
 */

/* 
#define DEBUG
 */

#include "ip.h"

static GtkObject *parent_class = NULL;

Compile *
compile_get_parent( Compile *compile )
{
	if( !compile->sym->tab )
		return( NULL );

	return( compile->sym->tab->compile );
}

void *
compile_name_print( Compile *compile )
{
	printf( "compile(0x%x) ", (unsigned int) compile );
	symbol_name_print( compile->sym );
	printf( " " );

	return( NULL );
}

static Compile *
compile_map_all_sub( Symbol *sym, map_compile_fn fn, void *a )
{
	if( !sym->expr || !sym->expr->compile )
		return( NULL );
	else 
		return( compile_map_all( sym->expr->compile, fn, a ) );
}

/* Apply a function to a compile ... and any local compiles.
 */
Compile *
compile_map_all( Compile *compile, map_compile_fn fn, void *a )
{
	Compile *res;

	/* Apply to this compile.
	 */
	if( (res = fn( compile, a )) )
		return( res );

	/* And over any locals.
	 */
	if( (res = (Compile *) stable_map( compile->locals, 
		(symbol_map_fn) compile_map_all_sub, (void *) fn, a, NULL )) )
		return( res );

	return( NULL );
}

/* Make a dependency. Text in compile refers to sym.
 */
void
compile_link_make( Compile *compile, Symbol *child )
{
	/* Already a dependency? Don't make a second link.
	 */
	if( !g_slist_find( compile->children, child ) ) {
		/* New link, each direction.
		 */
		compile->children = g_slist_prepend( compile->children, child );
		child->parents = g_slist_prepend( child->parents, compile );

		/* If the child is a forward reference, we may have to patch 
		 * this later. Save the pointer-to-child pointer on child.
		 */
		if( child->type == SYM_ZOMBIE )
			(void) symbol_patch_add( 
				&compile->children->data, child );
	}

#ifdef DEBUG_SANITY
	/* Sanity check.
	 */
	symbol_sanity( child );
	symbol_leaf_set_sanity();
#endif /*DEBUG_SANITY*/
}

/* Break a dependency. Text in compile referred to child.
 */
void *
compile_link_break( Compile *compile, Symbol *child )
{
	/* Sanity check.
	 */
#ifdef DEBUG_SANITY
	symbol_sanity( child );
	symbol_leaf_set_sanity();
#endif /*DEBUG_SANITY*/

	/* Must be there.
	 */
	assert( g_slist_find( compile->children, child ) &&
		g_slist_find( child->parents, compile ) );

	compile->children = g_slist_remove( compile->children, child );
	child->parents = g_slist_remove( child->parents, compile );

	/* Sanity check.
	 */
#ifdef DEBUG_SANITY
	symbol_sanity( child );
	symbol_leaf_set_sanity();
#endif /*DEBUG_SANITY*/

	return( NULL );
}

void *
compile_expr_link_break( Compile *compile, Expr *expr )
{
	assert( expr->compile == compile );
	assert( g_slist_find( compile->exprs, expr ) );

	expr->compile = NULL;
	compile->exprs = g_slist_remove( compile->exprs, expr );

	gtk_object_unref( GTK_OBJECT( compile ) );

	return( NULL );
}

void *
compile_expr_link_break_rev( Expr *expr, Compile *compile )
{
	return( compile_expr_link_break( compile, expr ) );
}

void 
compile_expr_link_make( Compile *compile, Expr *expr )
{
	assert( !expr->compile );
	assert( !g_slist_find( compile->exprs, expr ) );
	assert( compile->sym == expr->sym );

	expr->compile = compile;
	compile->exprs = g_slist_prepend( compile->exprs, expr );

        gtk_object_ref( GTK_OBJECT( compile ) );
        gtk_object_sink( GTK_OBJECT( compile ) );
}

static void
compile_destroy( GtkObject *object )
{
	Compile *compile;

	g_return_if_fail( object != NULL );
	g_return_if_fail( IS_COMPILE( object ) );

	compile = COMPILE( object );

#ifdef DEBUG
	printf( "compile_destroy: " );
	compile_name_print( compile );
	printf( "\n" );
#endif /*DEBUG*/

	/* My instance destroy stuff.
	 */

	/* Junk parse tree.
	 */
	slist_map( compile->treefrag, (SListMapFn) tree_node_destroy, NULL );
	FREEF( g_slist_free, compile->treefrag );
	compile->tree = NULL;

	/* Junk all locals.
	 */
	FREEF( stable_destroy, compile->locals );
	FREEF( g_slist_free, compile->param );
	compile->nparam = 0;
	FREEF( g_slist_free, compile->secret );
	compile->nsecret = 0;
	compile->this = NULL;
	compile->super = NULL;
	(void) slist_map( compile->children, 
		(SListMapFn) symbol_link_break, compile );
	FREEF( g_slist_free, compile->children );

	/* Junk heap.
	 */
	if( compile->heap ) {
		compile->base.type = ELEMENT_NOVAL;
		heap_unregister_element( compile->heap, &compile->base ); 
		FREEF( heap_destroy, compile->heap );
	}

	/* Junk text.
	 */
	FREE( compile->text );
	FREE( compile->prhstext );
	FREE( compile->rhstext );

	compile->sym = NULL;

	/* Unlink from any exprs.
	 */
	(void) slist_map( compile->exprs, 
		(SListMapFn) compile_expr_link_break_rev, compile );
	assert( !compile->exprs );

	GTK_OBJECT_CLASS( parent_class )->destroy( object );
}

static void
compile_class_init( CompileClass *klass )
{
	GtkObjectClass *object_class;

	object_class = (GtkObjectClass *) klass;

	parent_class = gtk_type_class( GTK_TYPE_OBJECT );

	object_class->destroy = compile_destroy;

	/* Create signals.
	 */

	/* Init default methods.
	 */
}

static void
compile_init( Compile *compile )
{
	/* Init our instance fields.
	 */
	compile->sym = NULL;

	compile->exprs = NULL;

	compile->is_klass = FALSE;
	compile->has_super = FALSE;

	compile->text = NULL;
	compile->prhstext = NULL;
	compile->rhstext = NULL;

	compile->tree = NULL;
	compile->treefrag = NULL;

	compile->locals = NULL;
	compile->nparam = 0;
	compile->param = NULL;
	compile->nsecret = 0;
	compile->secret = NULL;
	compile->this = NULL;
	compile->super = NULL;
	compile->children = NULL;

	compile->base.type = ELEMENT_NOVAL;
	compile->heap = NULL;
}

GtkType
compile_get_type( void )
{
	static GtkType compile_type = 0;

	if( !compile_type ) {
		static const GtkTypeInfo info = {
			"Compile",
			sizeof( Compile ),
			sizeof( CompileClass ),
			(GtkClassInitFunc) compile_class_init,
			(GtkObjectInitFunc) compile_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		compile_type = gtk_type_unique( GTK_TYPE_OBJECT, &info );
	}

	return( compile_type );
}

/* Make a compile linked to an expr.
 */
static Compile *
compile_new( Expr *expr )
{
	Compile *compile = gtk_type_new( TYPE_COMPILE );

	compile->sym = expr->sym;

	/* Junk any old compile.
	 */
	if( expr->compile )
		compile_expr_link_break( expr->compile, expr );

	compile_expr_link_make( compile, expr );

#ifdef DEBUG
	printf( "compile_new: " );
	compile_name_print( compile );
	printf( "\n" );
#endif /*DEBUG*/

	return( compile );
}

/* Make a compile for holding a root symbol.
 */
Compile *
compile_new_root( Expr *expr )
{
	Compile *compile = compile_new( expr );

	if( !(compile->locals = stable_new( compile, TABLE_SIZE )) ) {
		FREEO( compile );
		return( NULL );
	}

	return( compile );
}

/* Max cells function for symbols. Enough to compile something big.
 */
static int
compile_heap_max_fn( Heap *heap )
{
	return( 10000 );
}

/* Make a exprinfo suitable for a top-level symbol.
 */
Compile *
compile_new_toplevel( Expr *expr )
{
	Compile *compile = compile_new( expr );

	if( !(compile->locals = stable_new( compile, SMALL_TABLE_SIZE )) ||
		!(compile->heap = heap_new( compile, 
			compile_heap_max_fn, 100, 1000 )) ) {
		FREEO( compile );
		return( NULL );
	}

	heap_register_element( compile->heap, &compile->base ); 

	return( compile );
}

/* Make a exprinfo suitable for a local.
 */
Compile *
compile_new_local( Expr *expr )
{
	Compile *compile = compile_new( expr );

	if( !(compile->locals = stable_new( compile, SMALL_TABLE_SIZE )) ||
		!(compile->heap = heap_new( compile, 
			compile_heap_max_fn, 100, 100 )) ) {
		FREEO( compile );
		return( NULL );
	}

	heap_register_element( compile->heap, &compile->base ); 

	return( compile );
}

/* Code generation.
 */

/* Generate a (.sym x) pair. Set x to be NULL and point rhs at it .. caller
 * fills in later.
 */
static gboolean
compile_dotsym( Compile *compile, Symbol *sym, PElement *rhs, PElement *out )
{
	Heap *hi = compile->heap;
	HeapNode *hn1;
	PElement lhs;

	if( NEWNODE( hi, hn1 ) )
		return( FALSE );
	hn1->type = TAG_APPL;
	PPUT( hn1, ELEMENT_ELIST, NULL, ELEMENT_ELIST, NULL ); 
	PEPUTP( out, ELEMENT_NODE, hn1 );

	PEPOINTLEFT( hn1, &lhs );
	if( !heap_dot_sym_new( hi, sym, compile, &lhs ) )
		return( FALSE );

	PEPOINTRIGHT( hn1, rhs );

	return( TRUE );
}

/* Compile a reference to sym from expr.
 */
static gboolean
compile_reference( Compile *compile, Symbol *sym, PElement *out )
{
	Heap *hi = compile->heap;
	Compile *parent = compile_get_parent( compile );

#ifdef DEBUG
	printf( "generate_reference: ref to " );
	symbol_name_print( sym );
	printf( " inside " );
	compile_name_print( compile );
	printf( "\n" );
#endif /*DEBUG*/

	if( g_slist_find( compile->param, sym ) || 
		g_slist_find( compile->secret, sym ) ) {
		/* sym is a simple parameter, easy!
		 */
		PEPUTP( out, ELEMENT_SYMBOL, sym );
	}
	else if( is_class( parent ) && 
		(symbol_get_parent( sym ) == parent->sym ||
		g_slist_find( parent->secret, sym )) ) {
		Symbol *ths = parent->this;

		/* sym is a member of the same class as expr, or sym is a
		 * secret to our constructor (in which case it'll be in this
		 * as well) ... generate (.sym this)
		 *
		 * Optimisation: don't generate (.this this)
		 */
		if( sym == ths ) {
			PEPUTP( out, ELEMENT_SYMBOL, ths );
		}
		else {
			PElement rhs;

			if( !compile_dotsym( compile, sym, &rhs, out ) )
				return( FALSE );
			PEPUTP( &rhs, ELEMENT_SYMBOL, ths );
		}
	}
	else if( is_member_enclosing( compile, sym ) ) {
		Symbol *sths = symbol_get_parent( sym )->expr->compile->this;
		PElement rhs;

		/* Sym is a member of an enclosing class ...
		 * generate (.sym ref-to-this-for-that-class)
		 */
		if( !compile_dotsym( compile, sym, &rhs, out ) ||
			!compile_reference( compile, sths, &rhs ) )
			return( FALSE );
	}
	else {
		/* some other reference ... generate (sym secret1 .. secretn)
		 * recurse for secrets, since we may have to fetch them from 
		 * "this"
		 */
		PElement e = *out;
		PElement f;
		GSList *l;

		PEPUTP( &e, ELEMENT_SYMBOL, sym );

		/* Build secret args to this sym.
		 */
		if( sym->expr && sym->expr->compile )
			for( l = sym->expr->compile->secret; l; l = l->next ) {
				Symbol *arg = SYMBOL( l->data );
				HeapNode *hn1;

				if( NEWNODE( hi, hn1 ) )
					return( FALSE );
				hn1->type = TAG_APPL;
				PEPUTLEFT( hn1, &e );
				PPUTRIGHT( hn1, ELEMENT_ELIST, NULL ); 
				PEPUTP( &e, ELEMENT_NODE, hn1 );

				PEPOINTRIGHT( hn1, &f );
				if( !compile_reference( compile, arg, &f ) )
					return( FALSE );
			}
	}

	return( TRUE );
}

/* Fwd.
 */
static gboolean
compile_graph( Compile *compile, ParseNode *pn, PElement *out );

static gboolean
compile_dot( Compile *compile, ParseNode *arg, const char *str, PElement *out )
{
	Heap *hi = compile->heap;
	HeapNode *hn1;
	PElement e1, e2;

	if( NEWNODE( hi, hn1 ) )
		return( FALSE );
	hn1->type = TAG_APPL;
	PPUT( hn1, ELEMENT_ELIST, NULL, ELEMENT_ELIST, NULL );
	PEPUTP( out, ELEMENT_NODE, hn1 );

	PEPOINTLEFT( hn1, &e1 );
	if( !heap_dot_tag_new( hi, str, compile, &e1 ) )
		return( FALSE );

	PEPOINTRIGHT( hn1, &e2 );
	if( !compile_graph( compile, arg, &e2 ) )
		return( FALSE );

	return( TRUE );
}

/* Build a graph with vars still in it. Write result to *out.
 */
static gboolean
compile_graph( Compile *compile, ParseNode *pn, PElement *out )
{
	Heap *hi = compile->heap;
	HeapNode *hn1, *hn2, *hn3;
	PElement e1, e2, e3;
	GSList *l;

	switch( pn->type ) {
	case NODE_APPLY:
		/* Build apply node.
		 */
		if( NEWNODE( hi, hn1 ) )
			return( FALSE );
		hn1->type = TAG_APPL;
		PPUT( hn1, ELEMENT_ELIST, NULL, ELEMENT_ELIST, NULL );
		PEPUTP( out, ELEMENT_NODE, hn1 );

		/* Make sides.
		 */
		PEPOINTLEFT( hn1, &e1 );
		PEPOINTRIGHT( hn1, &e2 );
		if( !compile_graph( compile, pn->arg1, &e1 ) ||
			!compile_graph( compile, pn->arg2, &e2 ) )
			return( FALSE );

		break;

	case NODE_UOP:
		/* Build apply node.
		 */
		if( NEWNODE( hi, hn1 ) )
			return( FALSE );
		hn1->type = TAG_APPL;
		PPUT( hn1, ELEMENT_UNOP, pn->uop, ELEMENT_ELIST, NULL );
		PEPUTP( out, ELEMENT_NODE, hn1 );

		/* Build arg.
		 */
		PEPOINTRIGHT( hn1, &e2 );
		if( !compile_graph( compile, pn->arg1, &e2 ) )
			return( FALSE );

		break;

	case NODE_BINOP:
		if( pn->biop == BI_DOT ) {
			assert( pn->arg2->type == NODE_TAG );

			if( !compile_dot( compile, 
				pn->arg1, pn->arg2->tag, out ) )
				return( FALSE );
		}
		else {
			/* Build apply nodes.
			 */
			if( NEWNODE( hi, hn1 ) )
				return( FALSE );
			hn1->type = TAG_APPL;
			PPUT( hn1, ELEMENT_ELIST, NULL, ELEMENT_ELIST, NULL );
			PEPUTP( out, ELEMENT_NODE, hn1 );
			PEPOINTLEFT( hn1, &e1 );

			if( NEWNODE( hi, hn2 ) )
				return( FALSE );
			hn2->type = TAG_APPL;
			PPUT( hn2, ELEMENT_BINOP, pn->biop, 
				ELEMENT_ELIST, NULL );
			PEPUTP( &e1, ELEMENT_NODE, hn2 );

			/* Build args.
			 */
			PEPOINTRIGHT( hn1, &e2 );
			PEPOINTRIGHT( hn2, &e3 );
			if( !compile_graph( compile, pn->arg1, &e3 ) ||
				!compile_graph( compile, pn->arg2, &e2 ) )
				return( FALSE );
		}

		break;

	case NODE_COMPOSE:
		if( NEWNODE( hi, hn1 ) )
			return( FALSE );
		hn1->type = TAG_APPL;
		PPUT( hn1, ELEMENT_ELIST, NULL, ELEMENT_ELIST, NULL );
		PEPUTP( out, ELEMENT_NODE, hn1 );
		PEPOINTLEFT( hn1, &e1 );

		if( NEWNODE( hi, hn2 ) )
			return( FALSE );
		hn2->type = TAG_APPL;
		PPUT( hn2, ELEMENT_COMB, COMB_SR, 
			ELEMENT_ELIST, NULL );
		PEPUTP( &e1, ELEMENT_NODE, hn2 );

		/* Build args.
		 */
		PEPOINTRIGHT( hn1, &e2 );
		PEPOINTRIGHT( hn2, &e3 );
		if( !compile_graph( compile, pn->arg1, &e3 ) ||
			!compile_graph( compile, pn->arg2, &e2 ) )
			return( FALSE );

		break;

	case NODE_LEAF:
		/* A reference to a symbol. 
		 */
		if( !compile_reference( compile, pn->leaf, out ) )
			return( FALSE );

		break;

	case NODE_CLASS:
		/* Output constructor.
		 */
		PEPUTP( out, ELEMENT_CONSTRUCTOR, pn->klass );
		break;

	case NODE_TAG:
		/* RHS of projection. 
		 */
		PEPUTP( out, ELEMENT_TAG, pn->tag );
		break;

	case NODE_GENERATOR:
		/* Build apply nodes.
		 */
		if( NEWNODE( hi, hn1 ) )
			return( FALSE );
		hn1->type = TAG_APPL;
		PPUT( hn1, ELEMENT_ELIST, NULL, ELEMENT_ELIST, NULL );
		PEPUTP( out, ELEMENT_NODE, hn1 );
		PEPOINTLEFT( hn1, &e1 );

		if( NEWNODE( hi, hn2 ) )
			return( FALSE );
		hn2->type = TAG_APPL;
		PPUT( hn2, ELEMENT_ELIST, NULL, ELEMENT_ELIST, NULL );
		PEPUTP( &e1, ELEMENT_NODE, hn2 );
		PEPOINTLEFT( hn2, &e2 );

		if( NEWNODE( hi, hn3 ) )
			return( FALSE );
		hn3->type = TAG_APPL;
		PPUT( hn3, ELEMENT_COMB, COMB_GEN, ELEMENT_ELIST, NULL );
		PEPUTP( &e2, ELEMENT_NODE, hn3 );

		/* Build args.
		 */
		PEPOINTRIGHT( hn1, &e3 );
		PEPOINTRIGHT( hn2, &e2 );
		PEPOINTRIGHT( hn3, &e1 );
		if( !compile_graph( compile, pn->arg1, &e1 ) )
			return( FALSE );
		if( pn->arg2 )
			if( !compile_graph( compile, pn->arg2, &e2 ) )
				return( FALSE );
		if( pn->arg3 )
			if( !compile_graph( compile, pn->arg3, &e3 ) )
				return( FALSE );

		break;

	case NODE_LISTCONST:
	case NODE_SUPER:
		/* List of expressions.
		 */

		/* Make first RHS ... the end of the list. 
		 */
		e1 = *out;
		PEPUTP( &e1, ELEMENT_ELIST, NULL );

		/* Build @':' for each element.
		 */
		for( l = pn->elist; l; l = l->next ) {
			ParseNode *arg = (ParseNode *) l->data;

			/* Build apply nodes.
			 */
			if( NEWNODE( hi, hn1 ) )
				return( FALSE );
			hn1->type = TAG_APPL;
			PPUT( hn1, ELEMENT_ELIST, NULL, ELEMENT_ELIST, NULL );
			PEPUTP( &e1, ELEMENT_NODE, hn1 );
			PEPOINTLEFT( hn1, &e2 );

			if( NEWNODE( hi, hn2 ) )
				return( FALSE );
			hn2->type = TAG_APPL;
			PPUT( hn2, ELEMENT_BINOP, BI_CONS, 
				ELEMENT_ELIST, NULL );
			PEPUTP( &e2, ELEMENT_NODE, hn2 );

			/* Build arg.
			 */
			PEPOINTRIGHT( hn2, &e3 );
			if( !compile_graph( compile, arg, &e3 ) )
				return( FALSE );

			/* APPL is now our LHS.
			 */
			PEPOINTRIGHT( hn1, &e1 );
		}

		break;

	case NODE_CONST:
		/* Constant.
		 */
		switch( pn->con.type ) {
		case PARSE_CONST_STR:
			if( !heap_string_new( hi, pn->con.val.str, out ) )
				return( FALSE );
			break;

		case PARSE_CONST_CHAR:
			PEPUTP( out, ELEMENT_CHAR, pn->con.val.ch );
			break;

		case PARSE_CONST_BOOL:
			PEPUTP( out, ELEMENT_BOOL, pn->con.val.bool );
			break;

		case PARSE_CONST_ELIST:
			PEPUTP( out, ELEMENT_ELIST, NULL );
			break;

		case PARSE_CONST_NUM:
			if( !heap_real_new( hi, pn->con.val.num, out ) )
				return( FALSE );
			break;

		default:
			assert( FALSE );
		}

		break;

	case NODE_NONE:
	default:
		assert( FALSE );
	}

	return( TRUE );
}

/* Parameter abstraction.
 */

/* Abstract a symbol from the body of a piece of graph. Set *used if we found 
 * the symbol in this piece of graph ... ie. if our caller should add an
 * Sx-combinator for us.  Update *root with the new piece of graph.
 */
static int
compile_abstract_body( Compile *compile, 
	PElement *root, Symbol *sym, gboolean *used )
{
	Heap *hi = compile->heap;
	HeapNode *hn;
	HeapNode *hn1;
	PElement e1, e2;
	gboolean b1, b2;
	CombinatorType comb;

	switch( PEGETTYPE( root ) ) {
	case ELEMENT_NODE:
		hn = PEGETVAL( root );
		switch( hn->type ) {
		case TAG_APPL:		
		case TAG_CONS:
			b1 = FALSE; b2 = FALSE;
			PEPOINTLEFT( hn, &e1 );
			PEPOINTRIGHT( hn, &e2 );
			if( compile_abstract_body( compile, &e1, sym, &b1 ) ||
				compile_abstract_body( compile, 
					&e2, sym, &b2 ) )
				return( -1 );

			if( PEISCOMB( &e2 ) && 
				PEGETCOMB( &e2 ) == COMB_I && !b1 && b2 &&
				hn->type == TAG_APPL ) {
				PEPUTPE( root, &e1 );
				*used = TRUE;
			}
			else if( b1 || b2 ) {
				if( b1 && !b2 ) 
					comb = COMB_SL;
				else if( !b1 && b2 ) 
					comb = COMB_SR;
				else 
					comb = COMB_S;

				/* Generate Sx combinator.
				 */
				if( NEWNODE( hi, hn1 ) )
					return( -1 );
				hn1->type = TAG_APPL;
				PPUTLEFT( hn1, ELEMENT_COMB, comb );
				PEPUTRIGHT( hn1, &e1 );
				PEPUTP( &e1, ELEMENT_NODE, hn1 );

				/* We've used the var too!
				 */
				*used = TRUE;
			}

			break;

		case TAG_DOUBLE:
		case TAG_COMPLEX:
		case TAG_CLASS:
		case TAG_GEN:
		case TAG_DOT:
			break;

		case TAG_FREE:	
		default:
			assert( FALSE );
		}

		break;

	case ELEMENT_SYMBOL:
		if( SYMBOL( PEGETVAL( root ) ) == sym ) {
			/* Found an instance! Make an I combinator.
			 */
			*used = TRUE;
			PEPUTP( root, ELEMENT_COMB, COMB_I );
		}
		break;

	case ELEMENT_CONSTRUCTOR:
		/* set used .. to stop K being generated for this
		 * class parameter.
		 */
		*used = TRUE;
		break;

	case ELEMENT_CHAR:
	case ELEMENT_BOOL:
	case ELEMENT_BINOP:
	case ELEMENT_UNOP:
	case ELEMENT_COMB:
	case ELEMENT_ELIST:
	case ELEMENT_IMAGE:
	case ELEMENT_SYMREF:
	case ELEMENT_COMPILEREF:
	case ELEMENT_NOVAL:
	case ELEMENT_TAG:
		/* Leave alone.
		 */
		break;

	default:
		assert( FALSE );
	}

	return( 0 );
}

/* Abstract a symbol from a graph. As above, but make a K if the symbol is
 * entirely unused. 
 */
static void *
compile_abstract_symbol( Symbol *sym, Compile *compile, PElement *root )
{
	Heap *hi = compile->heap;
	gboolean b;

#ifdef DEBUG
	printf( "abstracting " );
	symbol_name_print( sym );
	printf( "\n" );
#endif /*DEBUG*/

	b = FALSE;
	if( compile_abstract_body( compile, root, sym, &b ) )
		return( sym );

	if( !b ) {
		HeapNode *hn1;

		/* Parameter not used! Need a K.
		 */
		if( NEWNODE( hi, hn1 ) )
			return( sym );
		hn1->type = TAG_APPL;
		PPUTLEFT( hn1, ELEMENT_COMB, COMB_K );
		PEPUTRIGHT( hn1, root );

		/* Update root.
		 */
		PEPUTP( root, ELEMENT_NODE, hn1 );
	}

	return( NULL );
}

/* Common sub-expression elimination.
 */

/* Found two equal sub-expressions. We can change hn1 to just be a reference
 * to hn2.
 */
static int
compile_transform_reference( Compile *compile, HeapNode *hn1, HeapNode *hn2 )
{
#ifdef DEBUG
	Heap *hi = compile->heap;
	BufInfo buf;
	char txt[ 100 ];
#endif /*DEBUG*/

#ifdef DEBUG
	buf_init_static( &buf, txt, 80 );
	graph_node( hi, &buf, hn1, TRUE );
	printf( "Found common subexpression: %s\n", buf_all( &buf ) );
#endif /*DEBUG*/

	/* Zap nodes to indicate sharing. 
	 */
	hn1->type = TAG_REFERENCE;
	PPUTLEFT( hn1, ELEMENT_NODE, hn2 );
	PPUTRIGHT( hn1, ELEMENT_NODE, NULL );

	return( 0 );
}

/* Test two sub-trees for equality.
 */
static gboolean
compile_tree_equal( HeapNode *hn1, HeapNode *hn2 )
{
	/* Test for pointer equality.
	 */
	if( hn1 == hn2 )
		return( TRUE );

	/* Test type tags for equality.
	 */
	if( hn1->type != hn2->type )
		return( FALSE );

	/* If double, test immediately.
	 */
	if( hn1->type == TAG_DOUBLE ) {
		if( hn1->body.num == hn2->body.num )
			return( TRUE );
		else
			return( FALSE );
	}

	/* If complex, test immediately.
	 */
	if( hn1->type == TAG_COMPLEX ) {
		if( GETLEFT( hn1 )->body.num == GETLEFT( hn2 )->body.num &&
			GETRIGHT( hn1 )->body.num == GETRIGHT( hn2 )->body.num )
			return( TRUE );
		else
			return( FALSE );
	}

	/* If compound type, something is wrong! Only built by reduce.
	 */
	assert( hn1->type != TAG_CLASS );

	/* In two parts, test tags.
	 */
	if( GETLT( hn1 ) != GETLT( hn2 ) )
		return( FALSE );
	if( GETRT( hn1 ) != GETRT( hn2 ) )
		return( FALSE );

	/* Test non-subtree parts.
	 */
	if( GETLT( hn1 ) != ELEMENT_NODE ) 
		if( GETLEFT( hn1 ) != GETLEFT( hn2 ) )
			return( FALSE );
	if( GETRT( hn1 ) != ELEMENT_NODE ) 
		if( GETRIGHT( hn1 ) != GETRIGHT( hn2 ) )
			return( FALSE );

	/* If sub-trees, test them.
	 */
	if( GETLT( hn1 ) == ELEMENT_NODE ) 
		if( !compile_tree_equal( GETLEFT( hn1 ), GETLEFT( hn2 ) ) )
			return( FALSE );
	if( GETRT( hn1 ) == ELEMENT_NODE ) 
		if( !compile_tree_equal( GETRIGHT( hn1 ), GETRIGHT( hn2 ) ) )
			return( FALSE );

	return( TRUE );
}

/* Searching for a match for hn2 ... is it hn1?
 */
static HeapNode *
compile_find_match( HeapNode *hn1, HeapNode *hn2 )
{
	/* If pointers equal, can drop search. 
	 */
	if( hn1 == hn2 )
		return( hn2 );

	/* If not APPL or CONS, no need to test hn1.
	 */
	if( hn1->type != TAG_APPL && hn1->type != TAG_CONS ) 
		return( NULL );

	/* Found a candidate in hn1, test for equality.
	 */
	if( compile_tree_equal( hn1, hn2 ) )
		/* Found a match!
		 */
		return( hn1 );

	return( NULL );
}

/* Search whole of graph for this function for match for hn.
 */
static void *
compile_find_common( HeapNode *hn, Compile *compile, HeapNode *root )
{
	HeapNode *hn1;

	/* Is hn a candidate for elimination?
	 */
	if( hn->type == TAG_APPL || hn->type == TAG_CONS ) 
		/* Search whole of graph for match for hn. Ignore matches with
		 * self.
		 */
		if( (hn1 = heap_map( root,
			(heap_map_fn) compile_find_match, hn, NULL )) && 
			hn1 != hn ) 
			/* We can remove hn!
			 */
			if( compile_transform_reference( compile, hn, hn1 ) )
				return( hn );

	return( NULL );
}

/* Use this to generate an id for each SHARE node.
 */
static int compile_share_number = 0;

/* If this is a REF node, make sure dest is a SHARE node.
 */
static void *
compile_transform_share( HeapNode *hn, Compile *compile )
{
	Heap *hi = compile->heap;

#ifdef DEBUG
	BufInfo buf;
	char txt[ 100 ];
#endif /*DEBUG*/

	if( hn->type == TAG_REFERENCE ) {
		HeapNode *hn1 = GETLEFT( hn );

		if( hn1->type != TAG_SHARED ) {
			HeapNode *hn2;

#ifdef DEBUG
			buf_init_static( &buf, txt, 80 );
			graph_node( hi, &buf, hn1, TRUE );
			printf( "Found shared code: %s\n", buf_all( &buf ) );
#endif /*DEBUG*/

			if( NEWNODE( hi, hn2 ) )
				return( hn );
			*hn2 = *hn1;
			hn1->type = TAG_SHARED;
			PPUT( hn1, 
				ELEMENT_NODE, hn2, 
				ELEMENT_CHAR, compile_share_number );

			compile_share_number++;
			if( compile_share_number == MAX_RELOC ) {
				ierrors( "Too many shared nodes in graph\n"
					"Disable optimisation, or raise "
					"MAX_RELOC" );
				return( hn );
			}
		}
	}

	return( NULL );
}

/* Do common-subexpression elimination. 
 */
static gboolean
compile_remove_subexpr( Compile *compile, PElement *root )
{
	HeapNode *rootn = PEGETVAL( root );

	if( PEGETTYPE( root ) != ELEMENT_NODE )
		/* Nowt to do.
		 */
		return( TRUE );

	/* Walk the tree for this function, searching for common expressions
	 * and generating REFERENCE nodes.
	 */
	if( heap_map( rootn, 
		(heap_map_fn) compile_find_common, compile, rootn ) )
		return( FALSE );

	/* Now search for destinations of reference nodes, and mark all shared
	 * sections. Each shared section is given a number ... saves a lookup
	 * during copy.
	 */
	compile_share_number = 0;
	if( heap_map( rootn, 
		(heap_map_fn) compile_transform_share, compile, NULL ) )
		return( FALSE );

	return( TRUE );
}

/* Top-level compiler driver.
 */

static void *compile_symbol( Symbol *sym, gboolean optimise );

/* Compile a symbol into a heap. 
 */
void *
compile_heap( Compile *compile, gboolean optimise )
{
	PElement root;
#ifdef DEBUG
	BufInfo buf;
	char txt[1024];
#endif /*DEBUG*/

	PEPOINTE( &root, &compile->base );

	/* Compile all our sub-definitions first.
	 */
	if( stable_map( compile->locals, 
		(symbol_map_fn) compile_symbol, 
		(void *) optimise, NULL, NULL ) )
		return( compile->sym );

	/* Is there an existing function root? GC it away.
	 */
	if( PEGETTYPE( &root ) != ELEMENT_NOVAL ) {
		PEPUTP( &root, ELEMENT_NOVAL, NULL );
		if( !heap_gc( compile->heap ) )
			return( compile->sym );

		return( NULL );
	}

#ifdef DEBUG
	printf( "compile_expr: about to compile " );
	symbol_name_print( compile->sym );
	printf( "\n" );
	if( compile->tree )
		dump_tree( compile->tree );
#endif /*DEBUG*/

	/* Compile function body. Tree can be NULL for classes.
	 */
	if( compile->tree ) {
		if( !compile_graph( compile, compile->tree, &root ) )
			return( compile->sym );
	}
	else {
		PEPUTP( &root, ELEMENT_NOVAL, NULL );
	}

#ifdef DEBUG
	buf_init_static( &buf, txt, 1024 );
	graph_pelement( compile->heap, &buf, &root, TRUE );
	printf( "before var abstraction, compiled \"%s\" to: %s\n", 
		MODEL( compile->sym )->name, buf_all( &buf ) );
#endif /*DEBUG*/

	/* Abstract real parameters.
	 */
#ifdef DEBUG
	printf( "abstracting real params ...\n" );
#endif /*DEBUG*/
	if( slist_map2_rev( compile->param,
		(SListMap2Fn) compile_abstract_symbol, compile, &root ) )
			return( compile->sym );

	/* Abstract secret parameters. 
	 */
#ifdef DEBUG
	printf( "abstracting secret params ...\n" );
#endif /*DEBUG*/
	if( slist_map2_rev( compile->secret,
		(SListMap2Fn) compile_abstract_symbol, compile, &root ) )
			return( compile->sym );

	/* Remove common sub-expressions.
	 */
	if( optimise && !compile_remove_subexpr( compile, &root ) )
		return( compile->sym );

#ifdef DEBUG
	buf_init_static( &buf, txt, 1024 );
	graph_pelement( compile->heap, &buf, &root, TRUE );
	printf( "compiled \"%s\" to: %s\n", 
		MODEL( compile->sym )->name, buf_all( &buf ) );
#endif /*DEBUG*/

	return( NULL );
}

/* Compile a symbol.
 */
static void *
compile_symbol( Symbol *sym, gboolean optimise )
{
	if( sym->expr && sym->expr->compile )
		return( compile_heap( sym->expr->compile, optimise ) );

	return( NULL );
}

/* Parse support.
 */

/* Do end-of-parse checks. Called after every 'A=...' style definition (not 
 * just top-level syms). Used to do lots of checks, not much left now.
 */
void
compile_check( Compile *compile )
{
	Symbol *sym = compile->sym;
	Symbol *parent = symbol_get_parent( sym );

	/* Check "check" member. 
	 */ 
	if( is_member( sym ) &&
		strcmp( MODEL( sym )->name, MEMBER_CHECK ) == 0 ) {
		if( compile->nparam != 0 ) 
			yyerror( "member \"%s\" of class "
				"%s should have no arguments",
				MEMBER_CHECK, symbol_name( parent ) );
	}
}

/* Mark error on all exprs using this compile.
 */
void
compile_error_set( Compile *compile )
{
	(void) slist_map( compile->exprs, (SListMapFn) expr_error_set, NULL );
}
