/* a column button in a workspace
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

/* 
#define DEBUG
 */

#include "ip.h"

static FilemodelClass *parent_class = NULL;

/* Offset for this column load/save.
 */
static int column_left_offset = 0;
static int column_top_offset = 0;

/* Map down a column.
 */
void *
column_map( Column *col, row_map_fn fn, void *a, void *b )
{
	Subcolumn *scol = col->scol;

	return( subcolumn_map( scol, fn, a, b ) );
}

void *
column_map_symbol_sub( Row *row, symbol_map_fn fn, void *a )
{
	return( fn( row->sym, a, NULL, NULL ) );
}

/* Map down a column, applying to the symbol of the row.
 */
void *
column_map_symbol( Column *col, symbol_map_fn fn, void *a )
{
	return( column_map( col, 
		(row_map_fn) column_map_symbol_sub, (void *) fn, a ) );
}

static void
column_destroy( GtkObject *object )
{
	Column *col;

#ifdef DEBUG
	printf( "column_destroy\n" );
#endif /*DEBUG*/

	g_return_if_fail( object != NULL );
	g_return_if_fail( IS_COLUMN( object ) );

	col = COLUMN( object );

	GTK_OBJECT_CLASS( parent_class )->destroy( object );
}

/* Select all things in a column.
 */
void *
column_select_symbols( Column *col )
{
	return( column_map( col, (row_map_fn) row_select_extend, NULL, NULL ) );
}

static Subcolumn *
column_get_subcolumn( Column *col )
{
	assert( g_slist_length( MODEL( col )->children ) == 1 );

	return( SUBCOLUMN( MODEL( col )->children->data ) );
}

static void
column_child_add( Model *parent, Model *child, int pos )
{
	Column *col = COLUMN( parent );

	MODEL_CLASS( parent_class )->child_add( parent, child, pos );

	/* Update our context.
	 */
	col->scol = column_get_subcolumn( col );
}

static Workspace *
column_get_workspace( Column *col )
{
	return( WORKSPACE( MODEL( col )->parent ) );
}

static void
column_parent_add( Model *child, Model *parent )
{
	Column *col = COLUMN( child );

	assert( IS_WORKSPACE( parent ) );

	MODEL_CLASS( parent_class )->parent_add( child, parent );

	/* Update our context.
	 */
	col->ws = column_get_workspace( col );
}

static xmlNode *
column_save( Model *model, xmlNode *xnode )
{
	Column *col = COLUMN( model );
	int x = IM_MAX( 0, col->x - column_left_offset );
	int y = IM_MAX( 0, col->y - column_top_offset );

	xmlNode *xthis;

	if( !(xthis = MODEL_CLASS( parent_class )->save( model, xnode )) )
		return( NULL );

	if( !set_prop( xthis, "x", "%d", x ) ||
		!set_prop( xthis, "y", "%d", y ) ||
		!set_prop( xthis, "open", bool_to_char( col->open ) ) ||
		!set_prop( xthis, "selected", bool_to_char( col->selected ) ) ||
		!set_prop( xthis, "sform", bool_to_char( col->sform ) ) ||
		!set_prop( xthis, "next", "%d", col->next ) ) 
		return( NULL );

	return( xthis );
}

static void *
row_is_selected( Row *row )
{
	if( row->selected )
		return( row );

	return( NULL );
}

static gboolean
column_save_test( Model *model )
{
	Column *col = COLUMN( model );
	Workspace *ws = col->ws;

	if( ws->save_type == WORKSPACE_SAVE_SELECTED ) 
		/* Only save columns containing selected rows.
		 */
		return( column_map( col, 
			(row_map_fn) row_is_selected, NULL, NULL ) != NULL );

	return( TRUE );
}

static gboolean
column_load( Model *model, 
	ModelLoadState *state, Model *parent, xmlNode *xnode )
{
	Column *col = COLUMN( model );

	int x = col->x;
	int y = col->y;

	if( !IS_WORKSPACE( parent ) ) {
		ierrors( "column_load: can only add a column to a "
			"workspace" );
		return( FALSE );
	}

	if( !get_iprop( xnode, "x", &x ) ||
		!get_iprop( xnode, "y", &y ) ||
		!get_bprop( xnode, "open", &col->open ) ||
		!get_bprop( xnode, "selected", &col->selected ) ||
		!get_bprop( xnode, "sform", &col->sform ) ||
		!get_iprop( xnode, "next", &col->next ) )
		return( FALSE );

	col->x = x + column_left_offset;
	col->y = y + column_top_offset;

	return( MODEL_CLASS( parent_class )->load( model, 
		state, parent, xnode ) );
}

static void
column_class_init( ColumnClass *klass )
{
	GtkObjectClass *object_class = (GtkObjectClass*) klass;
	ModelClass *model_class = (ModelClass *) klass;
	FilemodelClass *filemodel_class = (FilemodelClass *) klass;

	parent_class = gtk_type_class( TYPE_FILEMODEL );

	object_class->destroy = column_destroy;

	/* Create signals.
	 */

	/* Init methods.
	 */
	model_class->view_new = columnview_new;
	model_class->child_add = column_child_add;
	model_class->parent_add = column_parent_add;
	model_class->save = column_save;
	model_class->save_test = column_save_test;
	model_class->load = column_load;

	filemodel_class->filetype = filesel_type_workspace;

	/* Static init.
	 */
	model_register_loadable( MODEL_CLASS( klass ) );
}

static void
column_init( Column *col )
{
#ifdef DEBUG
	printf( "column_init\n" );
#endif /*DEBUG*/

	col->scol = NULL;
	col->ws = NULL;

        col->x = 0;
        col->y = 0;
        col->open = TRUE;
        col->selected = FALSE;
        col->sform = FALSE;

        col->next = 1;
        col->last_select = NULL;
}

GtkType
column_get_type( void )
{
	static GtkType column_type = 0;

	if( !column_type ) {
		static const GtkTypeInfo col_info = {
			"Column",
			sizeof( Column ),
			sizeof( ColumnClass ),
			(GtkClassInitFunc) column_class_init,
			(GtkObjectInitFunc) column_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		column_type = gtk_type_unique( TYPE_FILEMODEL, &col_info );
	}

	return( column_type );
}

Column *
column_new( Workspace *ws, const char *name )
{
	Column *col;

	if( workspace_column_find( ws, name ) ) {
		ierrors( "Can't create column \"%s\".\n"
			"A column with that name already exists.", name );
		return( NULL );
	}

	col = gtk_type_new( TYPE_COLUMN );

	model_set( MODEL( col ), name, "untitled" );
	model_child_add( MODEL( ws ), MODEL( col ), -1 );

        subcolumn_new( NULL, col );

	return( col );
}

/* Find the bottom of the column.
 */
Row *
column_get_bottom( Column *col )
{
	Subcolumn *scol = col->scol;
	GSList *children = MODEL( scol )->children;

	if( children ) {
		Row *row = ROW( g_slist_last( children )->data );

		return( row );
	}
	else {
		ierrors( "No objects in current column" );
		return( NULL );
	}
}

/* Add the last n names from a column to a buffer. Error if there are too few 
 * there.
 */
gboolean
column_add_n_names( Column *col, BufInfo *buf, int nparam )
{
	Subcolumn *scol = col->scol;
	GSList *children = MODEL( scol )->children;
	int len = g_slist_length( children );
	GSList *i;

	assert( nparam >= 0 );

	if( nparam > 0 && nparam > len ) {
		ierrors( "too few items in column for operation" );
		return( FALSE );
	}

	for( i = g_slist_nth( children, len - nparam ); i; i = i->next ) {
		Row *row = ROW( i->data );

		buf_appends( buf, " " );
		buf_appends( buf, MODEL( row->sym )->name );
	}

	return( TRUE );
}

/* Is a column empty?
 */
gboolean
column_isempty( Column *col )
{
	Subcolumn *scol = col->scol;
	GSList *children = MODEL( scol )->children;

	return( children == NULL );
}

/* Set the load/save offsets.
 */
void
column_set_offset( int x_off, int y_off )
{
#ifdef DEBUG
	printf( "column_set_offset: load offset %d x %d\n", x_off, y_off );
#endif /*DEBUG*/

	column_left_offset = x_off;
	column_top_offset = y_off;
}

char *
column_name_new( Column *col )
{
	char buf[256];

	im_snprintf( buf, 256, "%s%d", MODEL( col )->name, col->next++ );

	assert( !stable_find( SYMBOL( col->ws )->expr->compile->locals, buf ) );

	return( im_strdup( NULL, buf ) );
}
