# -*- coding: utf-8 -*-
#
#  bln.py - a easyballoon compatible Saori module for ninix
#  Copyright (C) 2002-2011 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#

# TODO:
# - font.face

import os
import random
import math
import time
import sys
import logging

import gtk
import glib
import pango
import cairo

import ninix.pix
import ninix.script
from ninix.dll import SAORI


class Saori(SAORI):

    def __init__(self):
        SAORI.__init__(self)
        self.blns = {}
        self.__sakura = None

    def need_ghost_backdoor(self, sakura):
        self.__sakura = sakura

    def check_import(self):
        return 1 if self.__sakura else 0

    def setup(self):
        self.blns = self.read_bln_txt(self.dir)
        if self.blns:
            self.__sakura.attach_observer(self)
            return 1
        else:
            return 0

    def read_bln_txt(self, dir):
        blns = {}
        try:
            with open(os.path.join(dir, 'bln.txt'), 'r') as f:
                data = {}
                name = ''
                for line in f:
                    line = line.strip()
                    if not line:
                        continue
                    if line.startswith('//'):
                        continue
                    if '[' in line:
                        if name:
                            blns[name] = (data, {})
                        data = {}
                        start = line.find('[')
                        end = line.find(']')
                        if end < 0:
                            end = len(line)
                        name = line[start + 1:end]
                    else:
                        if ',' in line:
                            key, value = [x.strip() for x in line.split(',', 1)]
                            data[key] = value
            if name:
                blns[name] = (data, {})
            return blns
        except:
            return None

    def finalize(self):
        for name in self.blns:
            data, bln = self.blns[name]
            for bln_id in bln.keys():
                if bln[bln_id]:
                    bln[bln_id].destroy()
                    del bln[bln_id]
        self.blns = {}
        self.__sakura.detach_observer(self)
        return 1

    def observer_update(self, event, args): ## FIXME
        if event == 'set scale':
            for name in self.blns:
                data, bln = self.blns[name]
                for bln_id in bln.keys():
                    if bln[bln_id]:
                        bln[bln_id].reset_scale()

    def execute(self, argument):
        if not argument:
            return self.RESPONSE[400]
        name = argument[0]
        text = argument[1] if len(argument) >= 2 else ''
        if len(argument) >= 3:
            try:
                offset_x = int(argument[2])
            except:
                offset_x = 0
        else:
            offset_x = 0
        if len(argument) >= 4:
            try:
                offset_y = int(argument[3])
            except:
                offset_y = 0
        else:
            offset_y = 0
        bln_id = argument[4] if len(argument) >= 5 else ''
        if len(argument) >= 6 and argument[5] in ['1', '2']:
            update = int(argument[5])
        else:
            update = 0
        if name in self.blns:
            data, bln = self.blns[name]
            if bln_id in bln and update == 0:
                bln[bln_id].destroy()
                del bln[bln_id]
            if text:
                if update == 0 or bln_id not in bln:
                    bln[bln_id] = Balloon(self.__sakura, self.dir,
                                          data, text,
                                          offset_x, offset_y, name, bln_id)
                else:
                    bln[bln_id].update_script(text, update)
            self.blns[name] = (data, bln)
        elif name == 'clear':
            for name in self.blns:
                data, bln = self.blns[name]
                if bln_id in bln and \
                   bln[bln_id].get_state() != 'orusuban':
                    bln[bln_id].destroy()
                    del bln[bln_id]
                    self.blns[name] = (data, {})
        return None


class Balloon(object):

    def __init__(self, sakura, dir, data,
                 text, offset_x, offset_y, name, bln_id):
        self.dir = dir
        self.__sakura = sakura
        self.name = name
        self.id = bln_id
        self.data = data # XXX
        self.window = ninix.pix.TransparentWindow()
        self.window.set_focus_on_map(False)
        self.window.set_title(name)
        self.window.set_skip_taskbar_hint(True)
        self.window.set_decorated(False)
        self.window.set_resizable(False)
        self.window.connect('delete_event', self.delete)
        self.position = data.get('position', 'sakura')
        left, top, scrn_w, scrn_h = ninix.pix.get_workarea()
        # -1: left, 1: right
        if self.position == 'sakura':
            s0_x, s0_y, s0_w, s0_h = self.get_sakura_status('SurfaceSakura')
            self.direction = -1 if s0_x + s0_w / 2 > left + scrn_w / 2 else 1
        elif self.position == 'kero':
            s1_x, s1_y, s1_w, s1_h = self.get_sakura_status('SurfaceKero')
            self.direction = -1 if s1_x + s1_w / 2 > left + scrn_w / 2 else 1
        else:
            self.direction = 1 # XXX
        if self.position in ['sakura', 'kero']:
            if self.direction == -1:
                skin = data.get('skin.left', data.get('skin'))
            else:
                skin = data.get('skin.right', data.get('skin'))
        else:
            skin = data.get('skin')
        if skin is None:
            self.destroy()
            return
        path = os.path.join(self.dir, skin.replace('\\', '/'))
        try:
            balloon_pixbuf = ninix.pix.create_pixbuf_from_file(path)
        except:
            self.destroy()
            return
        self.balloon_pixbuf = balloon_pixbuf
        w = balloon_pixbuf.get_width()
        h = balloon_pixbuf.get_height()
        self.x = self.y = 0
        if 'offset.x' in data:
            self.x += self.direction * int(data['offset.x'])
        if 'offset.y' in data:
            self.y += int(data['offset.y'])
        if 'offset.random' in data:
            self.x += int(data['offset.random']) * random.randrange(-1, 2)
            self.y += int(data['offset.random']) * random.randrange(-1, 2)
        self.x += self.direction * int(offset_x)
        self.y += int(offset_y)
        self.action_x = 0
        self.action_y = 0
        self.vx = 0
        self.vy = 0
        self.left = int(data.get('disparea.left', 0))
        self.right = int(data.get('disparea.right', w))
        self.top = int(data.get('disparea.top', 0))
        self.bottom = int(data.get('disparea.bottom', h))
        self.script = None
        self.darea = gtk.DrawingArea()
        self.darea.set_events(gtk.gdk.EXPOSURE_MASK|
                              gtk.gdk.BUTTON_PRESS_MASK|
                              gtk.gdk.BUTTON_RELEASE_MASK|
                              gtk.gdk.POINTER_MOTION_MASK|
                              gtk.gdk.LEAVE_NOTIFY_MASK)
        self.darea.connect('expose_event', self.redraw)
        self.darea.connect('button_press_event', self.button_press)
        self.darea.connect('button_release_event', self.button_release)
        self.darea.connect('motion_notify_event', self.motion_notify)
        self.darea.connect('leave_notify_event', self.leave_notify)
        self.darea.show()
        self.window.add(self.darea)
        self.darea.realize()
        self.window.realize()
        self.window.window.set_back_pixmap(None, False)
        self.set_skin()
        self.set_position()
        self.layout = None
        if text != 'noscript' and \
           (self.right - self.left) and (self.bottom - self.top):
            self.script = text
            if 'font.color' in data:
                fontcolor_r = int(data['font.color'][:2], 16) / 255.0
                fontcolor_g = int(data['font.color'][2:4], 16) /255.0
                fontcolor_b = int(data['font.color'][4:6], 16) / 255.0
                self.fontcolor = (fontcolor_r, fontcolor_g, fontcolor_b)
            default_font_size = 12 # for Windows environment
            self.font_size = int(data.get('font.size', default_font_size))
            self.layout = pango.Layout(self.darea.get_pango_context())
            self.font_desc = pango.FontDescription()
            self.font_desc.set_family('Sans') # FIXME
            if data.get('font.bold') == 'on':
                self.font_desc.set_weight(pango.WEIGHT_BOLD)
            self.layout.set_wrap(pango.WRAP_CHAR)
            self.set_layout()
        self.slide_vx = int(data.get('slide.vx', 0))
        self.slide_vy = int(data.get('slide.vy', 0))
        self.slide_autostop = int(data.get('slide.autostop', 0))
        if data.get('action.method') in ['sinwave', 'vibrate']:
            action = data['action.method']
            ref0 = int(data.get('action.reference0', 0))
            ref1 = int(data.get('action.reference1', 0))
            ref2 = int(data.get('action.reference2', 0))
            ref3 = int(data.get('action.reference3', 0))
            if ref2:
                self.action = {'method': action,
                               'ref0': ref0,
                               'ref1': ref1,
                               'ref2': ref2,
                               'ref3': ref3}
            else:
                self.action = None
        else:
            self.action = None
        self.move_notify_time = None
        self.life_time = None
        self.state = ''
        life = data.get('life', 'auto')
        if life == 'auto':
            self.life_time = 16000
        elif life in ['infinitely', '0']:
            pass
        elif life == 'orusuban':
            self.state = 'orusuban'
        else:
            try:
                self.life_time = int(life)
            except:
                pass
        self.start_time = time.time()
        self.startdelay = int(data.get('startdelay', 0))
        self.nooverlap = int('nooverlap' in data)
        self.talking = self.get_sakura_is_talking()
        self.move_notify_time = time.time()
        self.timeout_id = None
        self.visible = 0
        self.x_root = None
        self.y_root = None
        self.processed_script = None
        self.processed_text = ''
        self.text = ''
        self.script_wait = None
        self.quick_session = 0
        self.script_parser = ninix.script.Parser(error='loose')
        try:
            self.processed_script = self.script_parser.parse(self.script)
        except ninix.script.ParserError as e:
            self.processed_script = None
            logging.error('-' * 50)
            logging.error(e)
            logging.error(self.script)
        self.timeout_id = glib.timeout_add(10, self.do_idle_tasks)

    def set_position(self):
        if self.window is None:
            return
        new_x = self.base_x + int((self.x + self.action_x + self.vx) * self.scale / 100)
        new_y = self.base_y + int((self.y + self.action_y + self.vy) * self.scale / 100)
        self.window.resize_move(new_x, new_y)

    def set_skin(self):
        if self.window is None:
            return
        self.scale = self.get_sakura_status('SurfaceScale') ## FIXME
        balloon_pixbuf = self.balloon_pixbuf
        w = balloon_pixbuf.get_width()
        h = balloon_pixbuf.get_height()
        w = max(8, int(w * self.scale / 100))
        h = max(8, int(h * self.scale / 100))
        balloon_pixbuf = balloon_pixbuf.scale_simple(
            w, h, gtk.gdk.INTERP_BILINEAR)
        mask_pixmap = gtk.gdk.Pixmap(None, w, h, 1)
        balloon_pixbuf.render_threshold_alpha(
            mask_pixmap, 0, 0, 0, 0, w, h, 1)
        self.window.mask = mask_pixmap
        self.current_balloon_pixbuf = balloon_pixbuf
        self.darea.set_size_request(w, h)
        self.base_x, self.base_y = self.get_coordinate(w, h)

    def set_layout(self):
        if self.window is None:
            return
        if self.layout is None:
            return
        font_size = self.font_size * 3 / 4 # convert from Windows to GTK+
        font_size = font_size * self.scale / 100 * pango.SCALE
        self.font_desc.set_size(font_size)
        self.layout.set_font_description(self.font_desc)
        self.layout.set_width(
            int((self.right - self.left) * 1024 * self.scale / 100))

    def reset_scale(self):
        if self.window is None:
            return
        self.set_skin()
        self.set_position()
        self.set_layout()
        self.darea.queue_draw()

    @property
    def clickerase(self):
        return self.data.get('clickerase', 'on') == 'on'

    @property
    def dragmove_horizontal(self):
        return self.data.get('dragmove.horizontal') == 'on'

    @property
    def dragmove_vertical(self):
        return self.data.get('dragmove.vertical') == 'on'

    def get_coordinate(self, w, h):
        left, top, scrn_w, scrn_h = ninix.pix.get_workarea()
        s0_x, s0_y, s0_w, s0_h = self.get_sakura_status('SurfaceSakura')
        s1_x, s1_y, s1_w, s1_h = self.get_sakura_status('SurfaceKero')
        b0_x, b0_y, b0_w, b0_h = self.get_sakura_status('BalloonSakura')
        b1_x, b1_y, b1_w, b1_h = self.get_sakura_status('BalloonKero')
        x = left
        y = top
        if self.position == 'lefttop':
            pass
        elif self.position == 'leftbottom':
            y = top + scrn_h - h
        elif self.position == 'righttop':
            x = left + scrn_w - w
        elif self.position == 'rightbottom':
            x = left + scrn_w - w
            y = top + scrn_h - h
        elif self.position == 'center':
            x = left + (scrn_w - w) / 2
            y = top + (scrn_h - h) / 2
        elif self.position == 'leftcenter':
            y = top + (scrn_h - h) / 2
        elif self.position == 'rightcenter':
            x = left + scrn_w - w
            y = top + (scrn_h - h) / 2
        elif self.position == 'centertop':
            x = left + (scrn_w - w) / 2
        elif self.position == 'centerbottom':
            x = left + (scrn_w - w) / 2
            y = top + scrn_h - h
        elif self.position == 'sakura':
            if self.direction: # right
                x = s0_x + s0_w
            else:
                x = s0_x - w
            y = s0_y
        elif self.position == 'kero':
            if self.direction: # right
                x = s1_x + s1_w
            else:
                x = s1_x - w
            y = s1_y
        elif self.position == 'sakurab':
            x = b0_x
            y = b0_y
        elif self.position == 'kerob':
            x = b1_x
            y = b1_y
        return x, y

    def update_script(self, text, mode):
        if not text:
            return
        if mode == 2 and self.script is not None:
            self.script = ''.join((self.script, text))
        else:
            self.script = text
        self.processed_script = None
        self.processed_text = ''
        self.text = ''
        self.script_wait = None
        self.quick_session = 0
        try:
            self.processed_script = self.script_parser.parse(self.script)
        except ninix.script.ParserError as e:
            self.processed_script = None
            logging.error('-' * 50)
            logging.error(e)
            logging.error(self.script)

    def get_sakura_is_talking(self):
        talking = 0
        try:
            talking = int(self.__sakura.is_talking())
        except:
            pass
        return talking

    def get_sakura_status(self, key):
        if key == 'SurfaceScale':
            result = self.__sakura.get_surface_scale()
        elif key == 'SurfaceSakura_Shown':
            result = int(self.__sakura.surface_is_shown(0))
        elif key == 'SurfaceSakura':
            try:
                s0_x, s0_y = self.__sakura.get_surface_position(0)
                s0_w, s0_h = self.__sakura.get_surface_size(0)
            except:
                s0_x, s0_y = 0, 0
                s0_w, s0_h = 0, 0
            result = s0_x, s0_y, s0_w, s0_h
        elif key == 'SurfaceKero_Shown':
            result = int(self.__sakura.surface_is_shown(1))
        elif key == 'SurfaceKero':
            try:
                s1_x, s1_y = self.__sakura.get_surface_position(1)
                s1_w, s1_h = self.__sakura.get_surface_size(1)
            except:
                s1_x, s1_y = 0, 0
                s1_w, s1_h = 0, 0
            result = s1_x, s1_y, s1_w, s1_h
        elif key == 'BalloonSakura_Shown':
            result = int(self.__sakura.balloon_is_shown(0))
        elif key == 'BalloonSakura':
            try:
                b0_x, b0_y = self.__sakura.get_balloon_position(0)
                b0_w, b0_h = self.__sakura.get_balloon_size(0)
            except:
                b0_x, b0_y = 0, 0
                b0_w, b0_h = 0, 0
            result = b0_x, b0_y, b0_w, b0_h
        elif key == 'BalloonKero_Shown':
            result = int(self.__sakura.balloon_is_shown(1))
        elif key == 'BalloonKero':
            try:
                b1_x, b1_y = self.__sakura.get_balloon_position(1)
                b1_w, b1_h = self.__sakura.get_balloon_size(1)
            except:
                b1_x, b1_y = 0, 0
                b1_w, b1_h = 0, 0
            result = b1_x, b1_y, b1_w, b1_h
        else:
            result = None
        return result

    def do_idle_tasks(self):
        if not self.window:
            return None
        s0_shown = self.get_sakura_status('SurfaceSakura_Shown')
        s1_shown = self.get_sakura_status('SurfaceKero_Shown')
        b0_shown = self.get_sakura_status('BalloonSakura_Shown')
        b1_shown = self.get_sakura_status('BalloonKero_Shown')
        sakura_talking = self.get_sakura_is_talking()
        if self.state == 'orusuban':
            if self.visible:
                if s0_shown or s1_shown:
                    self.destroy()
                    return None
            else:
                if not s0_shown and not s1_shown:
                    self.start_time = time.time()
                    self.visible = 1
                    self.window.show()
                    self.life_time = 300000
        else:
            if self.visible:
                if (self.position == 'sakura' and not s0_shown) or \
                        (self.position == 'kero' and not s1_shown) or \
                        (self.position == 'sakurab' and not b0_shown) or \
                        (self.position == 'kerob' and not b1_shown) or \
                        (self.nooverlap and not self.talking and sakura_talking):
                    self.destroy()
                    return None
            else:
                if time.time() - self.start_time >= self.startdelay * 0.001:
                    self.start_time = time.time()
                    self.visible = 1
                    self.window.show()
        if self.visible:
            if self.life_time:
                if time.time() - self.start_time >= self.life_time * 0.001 and \
                   not (self.processed_script or self.processed_text):
                    self.destroy()
                    return None
            if self.action:
                if  self.action['method'] == 'sinwave':
                    offset = self.action['ref1'] \
                             * math.sin(2.0 * math.pi
                                        * float(int((time.time() - \
                                                     self.start_time) * 1000) \
                                                % self.action['ref2'])
                                        / self.action['ref2'])
                    if self.action['ref0']:
                        self.action_y = int(offset)
                    else:
                        self.action_x = int(offset)
                elif self.action['method'] == 'vibrate':
                    offset = (int((time.time() - self.start_time) * 1000) / \
                              self.action['ref2']) % 2
                    self.action_x = int(offset * self.action['ref0'])
                    self.action_y = int(offset * self.action['ref1'])
            if (self.slide_vx != 0 or self.slide_vy != 0) and \
                   self.slide_autostop > 0 and \
                   self.slide_autostop * 0.001 + 0.05 <= time.time() - self.start_time:
                self.vx = self.direction * int((self.slide_autostop / 50.0 + 1) * self.slide_vx)
                self.slide_vx = 0
                self.vy = int((self.slide_autostop / 50.0 + 1) * self.slide_vy)
                self.slide_vy = 0
            if self.slide_vx != 0:
                self.vx = self.direction * int(((time.time() - self.start_time) * self.slide_vx) / 50 * 1000.)
            if self.slide_vy != 0:
                self.vy = int(((time.time() - self.start_time) * self.slide_vy) / 50 * 1000.)
            self.set_position()
            if self.processed_script or self.processed_text:
                self.interpret_script()
        self.talking = 0 if self.talking and not sakura_talking else sakura_talking
        return True

    def redraw(self, darea, event):
        cr = darea.window.cairo_create()
        cr.save()
        cr.set_operator(cairo.OPERATOR_CLEAR)
        cr.paint()
        cr.restore()
        cr.set_source_pixbuf(self.current_balloon_pixbuf, 0, 0)
#        cr.paint_with_alpha(self.__alpha_channel)
        cr.paint()
        if self.layout:
            cr.set_source_rgb(*self.fontcolor)
            cr.move_to(int(self.left * self.scale / 100),
                       int(self.top * self.scale / 100))
            cr.show_layout(self.layout)
        del cr

    def get_state(self):
        return self.state

    def interpret_script(self):
        if self.script_wait is not None:
            if time.time() < self.script_wait:
                return
            self.script_wait = None
        if self.processed_text:
            if self.quick_session or self.state == 'orusuban':
                self.text = ''.join((self.text, self.processed_text))
                self.draw_text(self.text)
                self.processed_text = ''
            else:
                self.text = ''.join((self.text, self.processed_text[0]))
                self.draw_text(self.text)
                self.processed_text = self.processed_text[1:]
                self.script_wait = time.time() + 0.014
            return
        node = self.processed_script.pop(0)
        if node[0] == ninix.script.SCRIPT_TAG:
            name, args = node[1], node[2:]
            if name == r'\n':
                self.text = ''.join((self.text, '\n'))
                self.draw_text(self.text)
            elif name == r'\w':
                if args:
                    try:
                        amount = int(args[0]) * 0.05 - 0.01
                    except ValueError:
                        amount = 0
                else:
                    amount = 1 * 0.05 - 0.01
                if amount > 0:
                    self.script_wait = time.time() + amount
            elif name == r'\b':
                if args:
                    try:
                        amount = int(args[0])
                    except ValueError:
                        amount = 0
                else:
                    amount = 1
                if amount > 0:
                    self.text = self.text[:-amount]
            elif name == r'\c':
                self.text = ''
            elif name == r'\_q':
                self.quick_session = not self.quick_session
            elif name == r'\l':
                self.life_time = None
                self.update_script('', 2)
        elif node[0] == ninix.script.SCRIPT_TEXT:
            text = ''
            for chunk in node[1]:
                text = ''.join((text, chunk[1]))
            self.processed_text = text

    def draw_text(self, text):
        self.layout.set_text(text)
        self.darea.queue_draw()

    def button_press(self, widget, event):
        self.x_root = event.x_root
        self.y_root = event.y_root
        if event.type == gtk.gdk._2BUTTON_PRESS:
            x = int(event.x * 100 / self.scale)
            y = int(event.y * 100 / self.scale)
            self.__sakura.notify_event(
                'OnEBMouseDoubleClick', self.name, x, y, self.id)
        return True

    def button_release(self, widget, event):
        self.x_root = None
        self.y_root = None
        x = int(event.x * 100 / self.scale)
        y = int(event.y * 100 / self.scale)
        if event.type == gtk.gdk.BUTTON_RELEASE:
            if event.button == 1:
                self.__sakura.notify_event(
                    'OnEBMouseClick', self.name, x, y, self.id, 0)
            elif event.button == 3:
                self.__sakura.notify_event(
                    'OnEBMouseClick', self.name, x, y, self.id, 1)
        if self.clickerase:
            self.destroy()
        return True

    def motion_notify(self, widget, event):
        scale = self.scale
        if self.x_root is not None and \
           self.y_root is not None:
            x_delta = int(event.x_root - self.x_root)
            y_delta = int(event.y_root - self.y_root)
            if event.state & gtk.gdk.BUTTON1_MASK:
                if self.dragmove_horizontal:
                    self.x += x_delta * 100 / scale
                if self.dragmove_vertical:
                    self.y += y_delta * 100 / scale
                self.set_position()
                self.x_root = event.x_root
                self.y_root = event.y_root
        if self.move_notify_time is None or \
           time.time() - self.move_notify_time > 500 * 0.001:
            x = int(event.x * 100 / scale)
            y = int(event.y * 100 / scale)
            self.__sakura.notify_event(
                'OnEBMouseMove', self.name, x, y, self.id)
            self.move_notify_time = time.time()
        return True

    def leave_notify(self, widget, event):
        self.move_notify_time = None

    def delete(self, window, event):
        return True

    def destroy(self):
        self.visible = 0
        if self.window:
            self.window.destroy()
            self.window = None
        if self.timeout_id:
            glib.source_remove(self.timeout_id)
            self.timeout_id = None
