# -*- coding: utf-8 -*-
#
#  Copyright (C) 2001, 2002 by Tamito KAJIYAMA
#  Copyright (C) 2002, 2003 by MATSUMURA Namihiko <nie@counterghost.net>
#  Copyright (C) 2002-2011 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#  Copyright (C) 2003 by Shun-ichi TAHARA <jado@flowernet.gr.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#

import codecs
import os
import random
import re
import select
import socket
import sys
import time
import webbrowser
import StringIO

if 'DISPLAY' in os.environ:
    import gtk ## FIXME
    import glib

try:
    import gconf ## FIXME
except:
    gconf = None

# For some reasons gst parses sys.argv on import. This is a workaround.
# (See https://bugzilla.gnome.org/show_bug.cgi?id=549879 .)
argv = sys.argv
sys.argv = []
import pygst
pygst.require("0.10")
import gst
sys.argv = argv

import ninix.surface
import ninix.balloon
import ninix.dll
import ninix.makoto
import ninix.pix
import ninix.script
import ninix.version
import ninix.update
from ninix.home import get_ninix_home


class Sakura:

    BALLOON_LIFE   = 10  # sec (0: never closed automatically)
    SELECT_TIMEOUT = 15  # sec
    PAUSE_TIMEOUT  = 30  # sec
    SILENT_TIME    = 15  # sec
    # script modes
    BROWSE_MODE = 1
    SELECT_MODE = 2
    PAUSE_MODE  = 3
    # script origins
    FROM_SSTP_CLIENT = 1
    FROM_GHOST       = 2
    # HTML entity definitions
    try:
        from htmlentitydefs import name2codepoint
    except:
        name2codepoint = None

    def __init__(self, data, default_path, callback, debug=0):
        self.callback = callback
        self.debug = debug
        self.sstp_handle = None
        self.sstp_entry_db = None
        self.sstp_request_handler = None
        if self.debug & 16384: ## FIXME
            self.script_parser = ninix.script.Parser(error='strict')
        else:
            self.script_parser = ninix.script.Parser(error='loose')
        self.char = 2 # 'sakura' and 'kero'
        self.script_queue = []
        self.script_mode = self.BROWSE_MODE
        self.script_post_proc = []
        self.event_queue = []
        self.__current_script = ''
        self.__balloon_life = 0
        self.__surface_life = 0
        self.__boot = [0, 0]
        self.surface_mouse_motion = None ## FIXME
        self.time_critical_session = 0
        self.lock_repaint = 0
        self.passivemode = 0
        self.__running = 0
        self.anchor = None
        self.clock = (0, 0)
        self.synchronized_session = []
        ##
        self.old_otherghostname = None ## FIXME
        # create vanish dialog
        vanish_dialog_callback = {
            'notify_vanish_confirmation': self.notify_vanish_confirmation,
            'notify_event': self.notify_event,
            }
        self.__vanish_dialog = VanishDialog(vanish_dialog_callback)
        self.cantalk = 1
        self.__sender = 'ninix-aya'
        self.__charset = 'Shift_JIS'
        saori_lib = ninix.dll.Library(
            'saori', default_path, sakura=self)
        self.__dll = ninix.dll.Library(
            'shiori', default_path, saori_lib=saori_lib)
        self.__temp_mode = 0
        self.__observer = []
        balloon_callback = {
            'is_paused': self.is_paused,
            'notify_balloon_click': self.notify_balloon_click,
            'notify_event': self.notify_event,
            'notify_link_selection': self.notify_link_selection,
            'notify_user_teach': self.notify_user_teach,
            'position_balloons': self.position_balloons,
            'reset_idle_time': self.reset_idle_time,
            }
        for key in ['get_preference']:
            balloon_callback[key] = self.callback[key]
        self.balloon = ninix.balloon.Balloon(balloon_callback, debug)
        surface_callback = {
            'about': self.about,
            'busy': self.busy,
            'close_sakura': self.close,
            'enqueue_event': self.enqueue_event,
            'getstring': self.getstring,
            'get_balloon_size': self.get_balloon_size,
            'get_current_balloon_directory': self.get_current_balloon_directory,
            'get_shell_list': self.get_shell_list,
            'get_keroname': self.get_keroname,
            'get_prefix': self.get_prefix,
            'get_selfname': self.get_selfname,
            'is_running': self.is_running,
            'network_update': self.network_update,
            'notify_deiconified': self.notify_deiconified,
            'notify_event': self.notify_event,
            'notify_iconified': self.notify_iconified,
            'notify_observer': self.notify_observer,
            'notify_site_selection': self.notify_site_selection,
            'notify_surface_click': self.notify_surface_click,
            'notify_surface_mouse_motion': self.notify_surface_mouse_motion,
            'reset_idle_time': self.reset_idle_time,
            'select_shell': self.select_shell,
            'select_balloon': self.select_balloon,
            'select_kinoko': self.select_kinoko,
            'select_nekodorif': self.select_nekodorif,
            'select_sakura': self.select_sakura,
            'set_balloon_direction': self.set_balloon_direction,
            'set_balloon_position': self.set_balloon_position,
            'vanish_sakura': self.vanish,
            'get_current_item': self.get_current_item,
            }
        for key in ['close_all', 'get_preference', 'edit_preferences',
                    'open_ghost_manager', 'show_usage', 'start_sakura',
                    'get_ghost_list', 'get_balloon_list',
                    'get_nekodorif_list', 'get_kinoko_list',
                    'get_plugin_list', 'select_plugin']:
            surface_callback[key] = self.callback[key]
        self.surface = ninix.surface.Surface(surface_callback, debug)
        self.keep_silence(False)
        update_callback = {
            'enqueue_event': self.enqueue_event,
            'check_event_queue': self.check_event_queue,
            }
        self.updateman = ninix.update.NetworkUpdate(update_callback)
        self.audio_player = gst.element_factory_make("playbin", "player")
        fakesink = gst.element_factory_make("fakesink", "fakesink")
        self.audio_player.set_property("video-sink", fakesink)
        self.new(*data)

    def get_current_item(self):
        return self.current

    def select_sakura(self, event, item):
        if self.busy():
            gtk.gdk.beep()
            return
        self.callback['change_sakura'](self, item, 'manual')

    def select_nekodorif(self, event, item):
        self.callback['select_nekodorif'](item, self)

    def select_kinoko(self, event, item):
        self.callback['select_kinoko'](item, self)

    def set_observer(self, observer):
        if observer not in self.__observer:
            self.__observer.append(observer)

    def notify_observer(self, event, args=()):
        for observer in self.__observer:
            observer.notify_observer(event, args)

    def delete_observer(self, observer):
        for i in range(len(self.__observer)):
            if self.__observer[i] == observer:
                del self.__observer[i]
                break

    def new(self, desc, shiori_dir, use_makoto, surface_set, prefix,
            shiori_dll, shiori_name): ## FIXME
        self.shiori = None
        self.desc = desc
        self.shiori_dir = shiori_dir
        self.use_makoto = use_makoto
        self.surface_set = surface_set
        self.prefix = prefix ## FIXME: ninix_home should not be included.(append in get_prefix())
        self.shiori_dll = shiori_dll
        self.shiori_name = shiori_name
        name = (shiori_dll, shiori_name)
        self.shiori = self.__dll.request(name)
        char = 2
        while self.desc.get('char%d.seriko.defaultsurface' % char) is not None:
            char += 1
        if char > 2:
            self.char = char
        # XXX
        if self.desc.get('name') == 'BTH小っちゃいってことは便利だねっ':
            self.set_SSP_mode(1)
        else:
            self.set_SSP_mode(0)

    def set_SSP_mode(self, flag): # XXX
        self.__sender = 'SSP' if flag else 'ninix-aya'

    def save_history(self):
        path = os.path.join(self.get_prefix(), 'HISTORY')
        try:
            f = open(path, 'w')
        except IOError, (code, message):
            sys.stderr.write('cannot write %s\n' % path)
        else:
            f.write('time, %s\n' % self.ghost_time)
            f.write('vanished_count, %s\n' % self.vanished_count)
            f.close()

    def save_settings(self):
        path = os.path.join(self.get_prefix(), 'SETTINGS')
        try:
            f = open(path, 'w')
        except IOError, (code, message):
            sys.stderr.write('cannot write %s\n' % path)
        else:
            if self.balloon_directory is not None:
                f.write('balloon_directory, %s\n' % self.balloon_directory)
            f.close()

    def load_history(self):
        path = os.path.join(self.get_prefix(), 'HISTORY')
        if os.path.exists(path):
            ghost_time = 0
            ghost_vanished_count = 0
            try:
                f = open(path, 'r')
            except IOError, (code, message):
                sys.stderr.write('cannot read %s\n' % path)
            else:
                for line in f:
                    if ',' not in line:
                        continue
                    key, value = [x.strip() for x in line.split(',', 1)]
                    if key == 'time':
                        try:
                            ghost_time = int(value)
                        except:
                            pass
                    elif key == 'vanished_count':
                        try:
                            ghost_vanished_count = int(value)
                        except:
                            pass
                f.close()
            self.ghost_time = ghost_time
            self.vanished_count = ghost_vanished_count
        else:
            self.ghost_time = 0
            self.vanished_count = 0

    def load_settings(self):
        path = os.path.join(self.get_prefix(), 'SETTINGS')
        if os.path.exists(path):
            balloon_directory = None
            try:
                f = open(path, 'r')
            except IOError, (code, message):
                sys.stderr.write('cannot read %s\n' % path)
            else:
                for line in f:
                    if ',' not in line:
                        continue
                    key, value = [x.strip() for x in line.split(',', 1)]
                    if key == 'balloon_directory':
                        balloon_directory = value
                f.close()
            self.balloon_directory = balloon_directory
        else:
            self.balloon_directory = None

    def load_shiori(self):
        if self.shiori and self.shiori.load(self.shiori_dir):
            if getattr(self.shiori, 'show_description', None):
                self.shiori.show_description()
        else:
            sys.stderr.write('%s cannot load SHIORI(%s)\n' % \
                             (self.get_selfname(), self.shiori_name))
        self.__charset = 'Shift_JIS' # default
        charset = self.get_event_response('charset')
        if charset:
            try:
                codecs.lookup(charset)
            except:
                sys.stderr.write('Unsupported charset %s' % repr(charset))
            else:
                self.__charset = charset
        else:
            pass
        self.get_event_response('basewareversion',
                                ninix.version.VERSION,
                                'ninix-aya',
                                ninix.version.NUMBER,
                                event_type='NOTIFY')

    def finalize(self):
        if not self.__temp_mode:
            self.shiori.unload()
        self.stop()

    def enter_temp_mode(self):
        if not self.__temp_mode:
            self.__temp_mode = 2

    def leave_temp_mode(self):
        self.__temp_mode = 0

    def set_surface(self, desc, alias, surface, name, dir, tooltips):
        self.surface.new(desc, alias, surface, name, dir, tooltips)
        for side in range(2, self.char):
            default = self.desc.get('char%d.seriko.defaultsurface' % side)
            self.surface.add_window(side, default)
        icon = self.desc.get('icon', None)
        if icon is not None:
            icon_path = os.path.join(self.shiori_dir, icon)
            if not os.path.exists(icon_path):
                icon_path = None
        else:
            icon_path = None
        self.surface.set_icon(icon_path)

    def set_balloon(self, desc, balloon):
        self.balloon.new(desc, balloon)
        for side in range(2, self.char):
            self.balloon.add_window(side)
        for side in range(self.char):
            balloon_win = self.balloon.get_window(side)
            surface_win = self.surface.get_window(side)
            balloon_win.set_transient_for(surface_win)

    def enqueue_script(self, script, sender, handle,
                       host, show_sstp_marker, use_translator,
                       db=None, request_handler=None):
        if not self.script_queue and \
           not self.time_critical_session and not self.passivemode:
            if self.sstp_request_handler:
                self.sstp_request_handler.send_sstp_break()
                self.sstp_request_handler = None
            self.reset_script(1)
        self.script_queue.append((script, sender, handle, host,
                                  show_sstp_marker, use_translator,
                                  db, request_handler))

    reset_event = ['OnGhostChanging', 'OnShellChanging', 'OnVanishSelected']

    def check_event_queue(self):
        return bool(self.event_queue)

    def enqueue_event(self, event, *arglist, **argdict): ## FIXME
        for key in argdict:
            assert key in ['proc'] # trap typo, etc.
        if event in self.reset_event:
            self.reset_script(1)
        self.event_queue.append((event, arglist, argdict))

    EVENT_SCRIPTS = {
        'OnUpdateBegin': \
        ''.join((r'\t\h\s[0]',
                 unicode(_('Network Update has begun.'), 'utf-8'),
                 r'\e')),
        'OnUpdateComplete': \
        ''.join((r'\t\h\s[5]',
                 unicode(_('Network Update completed successfully.'), 'utf-8'),
                 r'\e')),
        'OnUpdateFailure': \
        ''.join((r'\t\h\s[4]',
                 unicode(_('Network Update failed.'), 'utf-8'),
                 r'\e')),
        }

    def handle_event(self): ## FIXME
        while self.event_queue:
            event, arglist, argdict = self.event_queue.pop(0)
            proc = argdict.get('proc', None)
            argdict = {'default': self.EVENT_SCRIPTS.get(event)}
            if self.notify_event(event, *arglist, **argdict):
                if proc is not None:
                    self.script_post_proc.append(proc)
                return 1
            elif proc is not None:
                proc()
                return 1
        return 0

    def is_running(self):
        return self.__running

    def is_paused(self):
        return self.script_mode == self.PAUSE_MODE

    def is_talking(self):
        return int(self.processed_script or self.processed_text)

    def busy(self, check_updateman=True):
        return bool(self.time_critical_session or \
                        self.balloon.user_interaction or \
                        self.event_queue or \
                        self.passivemode or \
                        (check_updateman and self.updateman.is_active()))

    def get_silent_time(self):
        return self.silent_time

    def keep_silence(self, quiet):
        if quiet:
            self.silent_time = time.time()
        else:
            self.silent_time = 0
            self.reset_idle_time()

    def get_idle_time(self):
        now = time.time()
        idle = now - self.idle_start
        return idle

    def reset_idle_time(self):
        self.idle_start = time.time()

    def notify_preference_changed(self): ## FIXME
        self.balloon.reset_fonts()
        flag = self.callback['get_preference']('use_pna')
        self.set_use_pna(flag)
        scale = self.callback['get_preference']('surface_scale')
        self.set_surface_scale(scale)
        flag = self.callback['get_preference']('balloon_scalling')
        self.set_balloon_scalling(flag)
        alpha = self.callback['get_preference']('surface_alpha')
        self.set_surface_alpha(alpha)
        alpha = self.callback['get_preference']('balloon_alpha')
        self.set_balloon_alpha(alpha)
        quality = self.callback['get_preference']('animation_quality')
        self.set_animation_quality(quality)
        flag = self.callback['get_preference']('seriko_inactive')
        self.set_seriko_inactive(flag)

    def set_use_pna(self, flag): ## FIXME
        self.surface.set_use_pna(flag)
        self.balloon.set_use_pna(flag)

    def set_surface_alpha(self, alpha): ## FIXME
        self.surface.set_alpha_channel(alpha)

    def set_balloon_alpha(self, alpha): ## FIXME
        self.balloon.set_alpha_channel(alpha)

    def set_animation_quality(self, quality): ## FIXME
        self.surface.set_animation_quality(quality)

    def set_seriko_inactive(self, flag): ## FIXME
        self.surface.set_seriko_inactive(flag)

    def set_balloon_scalling(self, flag): ## FIXME
        self.balloon.set_scalling(flag)

    def set_debug(self, debug): ## FIXME
        self.debug = debug
        self.surface.set_debug(debug)

    def get_surface_position(self, side):
        result = self.surface.get_position(side)
        return result if result is not None else (0, 0)

    def set_balloon_position(self, side, x, y):
        self.balloon.set_position(side, x, y)

    def set_balloon_direction(self, side, direction):
        self.balloon.set_direction(side, direction)

    def get_balloon_size(self, side):
        result = self.balloon.get_balloon_size(side)
        return result if result is not None else (0, 0)

    def get_balloon_position(self, side):
        result = self.balloon.get_position(side)
        return result if result is not None else (0, 0)

    def balloon_is_shown(self, side):
        return int(self.balloon and self.balloon.is_shown(side))

    def surface_is_shown(self, side):
        return int(self.surface and self.surface.is_shown(side))

    def is_URL(self, s):
        return s.startswith('http://') or \
               s.startswith('ftp://') or \
               s.startswith('file:/')

    def is_anchor(self, link_id):
        return int(len(link_id) == 2 and link_id[0] == 'anchor')

    def vanish(self):
        if self.busy():
            gtk.gdk.beep() ## FIXME
            return
        self.notify_event('OnVanishSelecting')
        self.__vanish_dialog.show()

    def vanish_by_myself(self):
        self.vanished_count += 1
        self.ghost_time = 0
        self.callback['stop_sakura'](self.callback['vanish_sakura'])

    def get_name(self):
        return self.desc.get('name', unicode(_('Sakura&Unyuu'), 'utf-8'))

    def get_username(self):
        return self.getstring('username') or \
               self.surface.get_username() or \
               self.desc.get('user.defaultname', unicode(_('User'), 'utf-8'))

    def get_selfname(self):
        return self.surface.get_selfname() or \
               self.desc.get('sakura.name', unicode(_('Sakura'), 'utf-8'))

    def get_selfname2(self):
        return self.surface.get_selfname2() or \
               self.desc.get('sakura.name2', unicode(_('Sakura'), 'utf-8'))

    def get_keroname(self):
        return self.surface.get_keroname() or \
               self.desc.get('kero.name', unicode(_('Unyuu'), 'utf-8'))

    def get_friendname(self):
        return self.surface.get_friendname() or \
               self.desc.get('sakura.friend.name',
                             unicode(_('Tomoyo'), 'utf-8'))

    def getaistringrandom(self): # obsolete
        result = self.get_event_response('OnAITalk')
        return self.translate(result)

    def getdms(self):
        result = self.get_event_response('dms')
        return self.translate(result)

    def getword(self, word_type):
        result = self.get_event_response('%s' % word_type)
        return self.translate(result)

    def getstring(self, name): ## FIXME
        return self.get_event_response(name)

    def translate(self, s):
        if s is not None:
            if self.use_makoto:
                s = ninix.makoto.execute(s)
            else:
                r = self.get_event_response('OnTranslate', s, translate=0)
                if r:
                    s = r
        return s

    def get_value(self, response): # FIXME: check return code
        header = StringIO.StringIO(response)
        result = {}
        to = None
        for line in header:
            line = line.strip()
            if not line:
                continue
            if ':' not in line:
                continue
            key, result[key] = [x.strip() for x in line.split(':', 1)]
        for key in result.keys():
            result[key] = unicode(result[key], self.__charset, 'ignore')
        if 'Reference0' in result:
            to = result['Reference0']
        if 'Value' in result:
            return result['Value'], to
        else:
            return None, to

    def get_event_response_with_communication(self, event, *arglist, **argdict):
        if self.__temp_mode == 1:
            return ''
        for key in argdict:
            assert key in ['event_type', 'translate'] # trap typo, etc.
        ref = arglist
        event_type = argdict.get('event_type', 'GET')
        translate = argdict.get('translate', 1)
        header = ''.join(('%s SHIORI/3.0\r\n' % event_type,
                          'ID: %s\r\n' % event,
                          'Sender: %s\r\n' % self.__sender,
                          'SecurityLevel: local\r\n'))
        for i in range(len(ref)):
            if ref[i] is not None:
                header = ''.join((header,
                                  'Reference', str(i), ': ',
                                  str(ref[i]), '\r\n'))
        header = ''.join((header, '\r\n'))
        header = header.encode(self.__charset, 'ignore')
        response = self.shiori.request(header)
        if event_type != 'NOTIFY' and self.cantalk:
            result, to = self.get_value(response)
            if translate:
                result = self.translate(result)
        else:
            result, to = '', None
        if self.__temp_mode == 2 and event != 'charset':
            result, to = '', None
        if result is None:
            result = ''
        if to and result:
            def proc():
                self.callback['send_message'](
                    to, self.get_selfname(), result)
            communication = proc
        else:
            communication = None
        return result, communication

    def get_event_response(self, event, *arglist, **argdict):
        result, communication = self.get_event_response_with_communication(event, *arglist, **argdict)
        return result

    ###   CALLBACK   ###
    def notify_start(self, init, vanished, ghost_changed, name, path):
        if self.__temp_mode:
            default = None
        else:
            default = ninix.version.VERSION_INFO
        if init:
            if self.ghost_time == 0:
                if not self.notify_event('OnFirstBoot', self.vanished_count,
                                         None, None, None, None, None, None,
                                         self.surface.name):
                    self.notify_event('OnBoot', self.surface.name, default=default)
            else:
                self.notify_event('OnBoot', self.surface.name, default=default)
            left, top, scrn_w, scrn_h = ninix.pix.get_workarea()
            self.notify_event('OnDisplayChange',
                              gtk.gdk.visual_get_best_depth(),
                              scrn_w, scrn_h, event_type='NOTIFY')
        elif vanished:
            if self.ghost_time == 0:
                if self.notify_event('OnFirstBoot', self.vanished_count,
                                     None, None, None, None, None, None,
                                     self.surface.name):
                    return
            elif self.notify_event('OnVanished', name):
                return
            self.notify_event('OnBoot', self.surface.name, default=default)
        elif ghost_changed:
            if self.ghost_time == 0:
                if self.notify_event('OnFirstBoot', self.vanished_count,
                                     None, None, None, None, None, None,
                                     self.surface.name):
                    return
            elif self.notify_event('OnGhostChanged', name):
                return
            self.notify_event('OnBoot', self.surface.name, default=default)
        else:
            pass ## FIXME

    def notify_vanish_confirmation(self):
        def proc(self=self):
            self.vanished_count += 1
            self.ghost_time = 0
            glib.idle_add(self.callback['vanish_sakura'], self)
        self.enqueue_event('OnVanishSelected', proc=proc)
        self.vanished = 1 ## FIXME

    def notify_iconified(self):
        self.cantalk = 0
        self.callback['select_current_sakura']()
        if not self.passivemode:
            self.reset_script(1)
            self.stand_by(1)
            self.notify_event('OnWindowStateMinimize')

    def notify_deiconified(self):
        if not self.cantalk:
            self.cantalk = 1
            self.callback['select_current_sakura']()
            if not self.passivemode:
                self.notify_event('OnWindowStateRestore')

    def notify_link_selection(self, link_id, text, number):
        if self.script_origin == self.FROM_SSTP_CLIENT and \
           self.sstp_request_handler is not None:
            self.sstp_request_handler.send_answer(text)
            self.sstp_request_handler = None
        if self.is_anchor(link_id):
            self.notify_event('OnAnchorSelect', link_id[1])
        elif self.is_URL(link_id):
            webbrowser.open(link_id)
            self.reset_script(1)
            self.stand_by(0)
        elif self.sstp_entry_db:
            # leave the previous sstp message as it is
            self.start_script(self.sstp_entry_db.get(link_id, r'\e'))
            self.sstp_entry_db = None
        elif not self.notify_event('OnChoiceSelect', link_id, text, number):
            self.reset_script(1)
            self.stand_by(0)

    def notify_site_selection(self, event, args):
        title, url = args
        if self.is_URL(url):
            webbrowser.open(url)
        self.enqueue_event('OnRecommandedSiteChoice', title, url)

    def notify_surface_click(self, button, click, side, x, y):
        if button == 1 and click == 1:
            self.raise_all()
        if self.vanished:
            if side == 0 and button == 1:
                if self.sstp_request_handler:
                    self.sstp_request_handler.send_sstp_break()
                    self.sstp_request_handler = None
                self.reset_script(1)
                self.notify_event('OnVanishButtonHold', default=r'\e')
                self.vanished = 0
            return
        if self.updateman.is_active():
            if button == 1 and click == 2:
                self.updateman.interrupt()
            return
        if self.time_critical_session:
            return
        elif button in [1, 3] and click == 1:
            if self.passivemode and \
               self.processed_script is not None:
                return
            part = self.surface.get_touched_region(side, x, y)
            self.notify_event('OnMouseClick', x, y, 0, side, part, button)
        elif self.passivemode:
            return
        elif button == 1 and click == 2:
            if self.sstp_request_handler:
                self.sstp_request_handler.send_sstp_break()
                self.sstp_request_handler = None
            part = self.surface.get_touched_region(side, x, y)
            self.notify_event('OnMouseDoubleClick', x, y, '', side, part)

    def notify_balloon_click(self, button, click, side):
        if self.script_mode == self.PAUSE_MODE:
            self.script_mode = self.BROWSE_MODE
            self.balloon.clear_text_all()
            self.balloon.hide_all()
            self.script_side = 0
        elif button == 1 and click == 1:
            self.raise_all()
        if self.vanished:
            return
        if self.updateman.is_active():
            if button == 1 and click == 2:
                self.updateman.interrupt()
            return
        if self.time_critical_session:
            self.time_critical_session = 0
            return
        elif self.passivemode:
            return
        elif button == 1 and click == 2:
            if self.sstp_request_handler:
                self.sstp_request_handler.send_sstp_break()
                self.sstp_request_handler = None
        elif button == 3 and click == 1:
            if self.sstp_request_handler:
                self.sstp_request_handler.send_sstp_break()
                self.sstp_request_handler = None
            self.reset_script(1)
            self.stand_by(0)

    def notify_surface_mouse_motion(self, side, x, y, part):
        if self.surface_mouse_motion is not None:
            return
        if part:
            self.surface_mouse_motion = (side, x, y, part)
        else:
            self.surface_mouse_motion = None

    def notify_user_teach(self, word):
        if word is not None:
            script = self.translate(self.get_event_response('OnTeach', word))
            if script:
                self.start_script(script)
                self.balloon.hide_sstp_message()


    month_names = ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun',
                   'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec']
    boot_event = ['OnBoot', 'OnFirstBoot', 'OnGhostChanged', 'OnShellChanged',
                  'OnUpdateComplete']
    reset_event = ['OnVanishSelecting', 'OnVanishCancel'] ## FIXME

    def notify_event(self, event, *arglist, **argdict):
        #if self.time_critical_session:
        #    return 0
        if event in self.reset_event:
            self.reset_script(1)
        for key in argdict:
            assert key in ['event_type', 'default'] # trap typo, etc.
        event_type = argdict.get('event_type', 'GET')
        default = argdict.get('default', None)
        argdict = {'event_type': event_type} ## FIXME
        script, communication = self.get_event_response_with_communication(event, *arglist, **argdict) or (default, None)
        if self.debug & 1 and script or \
           self.debug & 2 and not script and event != 'OnSecondChange':
            t = time.localtime(time.time())
            m = self.month_names[t[1] - 1]
            print '\n[%02d/%s/%d:%02d:%02d:%02d %+05d]' % (
                t[2], m, t[0], t[3], t[4], t[5], - time.timezone / 36)
            print 'Event:', event
            for n in range(len(arglist)):
                value = arglist[n]
                if value is not None:
                    print 'Reference%d:' % n, '%s' % \
                          str(value).encode('utf-8', 'ignore')
        if not script: # an empty script is ignored
            if event in self.boot_event:
                self.surface_bootup()
            if event == 'OnMouseClick' and arglist[5] == 3:
                self.surface.open_popup_menu(arglist[5], arglist[3])
            return 0
        if self.debug & 1:
            print '=> "%s"' % script.encode('utf-8', 'ignore')
        if self.passivemode and \
           (event == 'OnSecondChange' or event == 'OnMinuteChange'):
            return 0
        self.start_script(script)
        self.balloon.hide_sstp_message()
        if event in self.boot_event:
            self.script_post_proc.append(self.surface_bootup)
        if communication is not None:
            self.script_post_proc.append(communication)
        return 1

    def get_prefix(self):
        return self.prefix

    def get_shell_list(self):
        shell_list = []
        set_type, i, j = self.current
        for num in range(len(self.surface_set)):
            name = self.surface_set[num][0]
            value = (set_type, i, num)
            working = 1 if num == j else 0 
            shell_list.append((name, value, working))
        return shell_list

    def get_current_balloon_directory(self):
        return self.balloon.get_balloon_directory()

    def select_shell(self, widget, item):
        set_type, i, j = item
        assert self.current[0] == set_type
        assert self.current[1] == i
        assert self.current[2] != j
        assert set_type == 'g'
        assert self.surface_set and len(self.surface_set) > j
        name, surface_dir, surface_desc, surface_alias, surface, surface_tooltips = \
            self.surface_set[j]
        if not name: ## FIXME
            surface_name = unicode(_('Master'), 'utf-8')
        else:
            surface_name = name
        def proc(self=self, item=item):
            print 'ghost', item
            self.current = item
            self.set_surface(surface_desc, surface_alias, surface, surface_name,
                             surface_dir, surface_tooltips)
            self.surface.reset_alignment()
            self.position_all()
            self.notify_event('OnShellChanged',
                              surface_name, surface_name, surface_dir)
        self.enqueue_event('OnShellChanging', surface_name, surface_dir,
                           proc=proc)

    def select_balloon(self, widget, item):
        if not widget.active:
            return
        if item == self.get_current_balloon_directory(): # no change
            # need reloadning?
            if self.balloon_directory != item:
                self.balloon_directory = item # save user's choice
            return
        desc, balloon = self.callback['get_balloon_description'](item)
        subdir = balloon['balloon_dir'][0]
        path = os.path.join(get_ninix_home(), 'balloon', subdir)
        self.balloon.hide_all()
        self.set_balloon(desc, balloon)
        if item == subdir:
            self.balloon_directory = subdir # save user's choice
        self.balloon.set_balloon_default()
        self.position_balloons()
        name = desc.get('name', '')
        print 'balloon', name, path
        self.notify_event('OnBalloonChange', name, path)

    def surface_bootup(self):
        for side in [0, 1]:
            if not self.__boot[side]:
                self.set_surface_default(side)
                self.surface.show(side)

    def get_uptime(self):
        uptime = int(time.time() - self.start_time) / 3600
        if uptime < 0:
            self.start_time = time.time()
            return 0
        return uptime

    def hide_all(self):
        self.surface.hide_all()
        self.balloon.hide_all()

    def position_balloons(self):
        self.surface.reset_balloon_position()

    def align_top(self, side):
        self.surface.set_alignment(side, 1)
        self.position_balloons()

    def align_bottom(self, side):
        self.surface.set_alignment(side, 0)
        self.position_balloons()

    def align_current(self):
        self.surface.set_alignment_current()
        self.position_balloons()

    def identify_window(self, win):
        return bool(self.surface.identify_window(win) or \
                        self.balloon.identify_window(win))

    def set_surface_default(self, side=None):
        self.surface.set_surface_default(side)

    def get_surface_scale(self):
        return self.surface.get_scale()

    def set_surface_scale(self, scale):
        self.surface.set_scale(scale)
        self.balloon.set_scale(scale)

    def get_surface_size(self, side):
        result = self.surface.get_surface_size(side)
        return result if result is not None else (0, 0)

    def set_surface_position(self, side, x, y):
        self.surface.set_position(side, x, y)

    def set_surface_id(self, side, id):
        self.surface.set_surface(side, id)

    def get_surface_id(self, side):
        return self.surface.get_surface(side)

    def surface_is_shown(self, side):
        return bool(self.surface and self.surface.is_shown(side))

    def get_kinoko_position(self, baseposition):
        side = 0
        x, y = self.get_surface_position(side)
        w, h = self.get_surface_size(side)
        if baseposition == 1:
            rect = self.surface.get_collision_area(side, 'face')
            if rect is not None:
                x1, y1, x2, y2 = rect
                return x + (x2 - x1) / 2, y + (y2 - y1) / 2
            else:
                return x + w / 2, y + h / 4
        elif baseposition == 2:
            rect = self.surface.get_collision_area(side, 'bust')
            if rect is not None:
                x1, y1, x2, y2 = rect
                return x + (x2 - x1) / 2, y + (y2 - y1) / 2
            else:
                return x + w / 2, y + h / 2
        elif baseposition == 3:
            centerx, centery = self.surface.get_center(side)
            if centerx is None:
                centerx = w / 2
            if centery is None:
                centery = h / 2
            return x + centerx, y + centery
        else: # baseposition == 0 or baseposition not in [1, 2, 3]: # AKF
            centerx, centery = self.surface.get_kinoko_center(side)
            if centerx is None or centery is None:
                rect = self.surface.get_collision_area(side, 'head')
                if rect is not None:
                    x1, y1, x2, y2 = rect
                    return x + (x2 - x1) / 2, y + (y2 - y1) / 2
                else:
                    return x + w / 2, y + h / 8
            return x + centerx, y + centery

    def raise_surface(self, side):
        self.surface.raise_(side)

    def lower_surface(self, side):
        self.surface.lower(side)

    def position_all(self):
        self.surface.reset_position()
        self.position_balloons()

    def raise_all(self):
        self.surface.raise_all()
        self.balloon.raise_all()

    def lower_all(self):
        self.surface.lower_all()
        self.balloon.lower_all()

    ###   STARTER   ###
    def stand_by(self, reset_surface):
        self.balloon.hide_all()
        self.balloon.hide_sstp_message()
        if reset_surface:
            self.set_surface_default()
            self.notify_event('OnSurfaceChange',
                              self.get_surface_id(0),
                              self.get_surface_id(1))
            self.balloon.set_balloon_default()
        elif self.get_surface_id(0) != '0' or \
             self.get_surface_id(1) != '10':
            self.__surface_life = random.randint(20, 30)
            ##print 'surface_life =', self.__surface_life

    def start(self, item, init, temp, vanished, ghost_changed, prev_name):
        if self.is_running():
            if temp:
                self.enter_temp_mode()
            else:
                if self.__temp_mode == 1:
                    self.__temp_mode = 2
                    self.load_shiori()
                    self.notify_start(init, vanished, ghost_changed,
                                      prev_name, '')
            return
        self.ghost_time = 0
        self.vanished_count = 0
        self.__running = 1
        self.__temp_mode = temp
        self.current = item
        set_type, i, j = item
        assert set_type == 'g'
        print 'ghost', item
        assert self.surface_set and len(self.surface_set) > j
        name, surface_dir, surface_desc, surface_alias, surface, surface_tooltips = \
            self.surface_set[j]
        surface_set = self.surface_set
        if not name: ## FIXME
            surface_name = unicode(_('Master'), 'utf-8')
        else:
            surface_name = name
        if ghost_changed:
            name = prev_name
        self.set_surface(surface_desc, surface_alias, surface, surface_name,
                         surface_dir, surface_tooltips)
        self.load_settings()
        if self.callback['get_preference']('ignore_default'): ## FIXME: change prefs key
            default_balloon = self.callback['get_preference']('default_balloon')
            balloon = self.callback['get_balloon_description'](default_balloon)
        elif self.balloon_directory is not None:
            balloon = self.callback['get_balloon_description'](self.balloon_directory)
        else:
            balloon_path = self.desc.get('deault.balloon.path', '')
            balloon_name = self.desc.get('balloon', '')
            balloon = None
            if balloon_path:
                balloon = self.callback['find_balloon_by_subdir'](balloon_path)
            if balloon is None and balloon_name:
                balloon = self.callback['find_balloon_by_name'](balloon_name)
            if balloon is None:
                default_balloon = self.callback['get_preference']('default_balloon')
                balloon = self.callback['get_balloon_description'](default_balloon)
        desc, balloon = balloon
        self.set_balloon(desc, balloon)
        if not temp:
            self.load_shiori()
        self.restart()
        self.start_time = time.time()
        self.notify_start(init, vanished, ghost_changed, name, surface_dir)
        glib.timeout_add(10, self.do_idle_tasks) # 10ms

    def restart(self):
        self.load_history()
        self.vanished = 0
        self.__boot = [0, 0]
        self.old_otherghostname = None ## FIXME
        self.reset_script(1)
        self.surface.reset_alignment()
        self.stand_by(1)
        self.position_all()
        self.reset_idle_time()
        self.__running = 1

    def stop(self):
        if not self.__running:
            return
        self.notify_observer('finalize')
        self.__running = 0
        self.save_settings()
        self.save_history()
        self.callback['rebuild_ghostdb'](self, None)
        self.hide_all()
        self.surface.finalize()
        self.balloon.finalize()
        self.audio_player.set_state(gst.STATE_NULL)

    def process_script(self):
        now = time.time()
        idle = self.get_idle_time()
        minute, second = time.localtime(now)[4:6]
        if self.clock[0] != second: ## FIXME
            if not self.__temp_mode:
                self.ghost_time += 1
            self.callback['rebuild_ghostdb'](
                self,
                self.get_selfname(),
                self.get_surface_id(0),
                self.get_surface_id(1))
            otherghostname = self.callback['get_otherghostname'](
                self.get_selfname())
            if otherghostname != self.old_otherghostname:
                args = []
                args.extend(otherghostname)
                args.insert(0, 'otherghostname')
                args = tuple(args)
                keyword = {'event_type': 'NOTIFY'}
                self.notify_event(*args, **keyword)
            self.old_otherghostname = otherghostname
        if not self.__running:
            pass
        elif self.script_mode == self.PAUSE_MODE:
            ##if idle > self.PAUSE_TIMEOUT:
            ##    self.script_mode = self.BROWSE_MODE
            pass
        elif self.processed_script or self.processed_text:
            self.interpret_script()
        elif self.script_post_proc:
            for proc in self.script_post_proc:
                proc()
            self.script_post_proc = []
        elif self.script_mode == self.SELECT_MODE:
            if self.passivemode:
                pass
            elif idle > self.SELECT_TIMEOUT:
                self.script_mode = self.BROWSE_MODE
                if self.sstp_request_handler:
                    self.sstp_request_handler.send_timeout()
                    self.sstp_request_handler = None
                if not self.notify_event('OnChoiceTimeout'):
                    self.stand_by(0)
        elif self.sstp_handle is not None:
            self.close_sstp_handle()
        elif self.balloon.user_interaction:
            pass
        elif idle > self.__balloon_life > 0 and not self.passivemode:
            self.__balloon_life = 0
            self.stand_by(0)
            self.notify_event('OnBalloonClose', self.__current_script)
            if self.callback['get_preference']('sink_after_talk'):
                self.surface.lower_all()
        elif self.event_queue and self.handle_event():
            pass
        elif self.script_queue and not self.passivemode:
            if self.get_silent_time() > 0:
                self.keep_silence(True) # extend silent time
            script, sender, self.sstp_handle, \
                    host, show_sstp_marker, use_translator, \
                    self.sstp_entry_db, self.sstp_request_handler = \
                    self.script_queue.pop(0)
            if self.cantalk:
                if show_sstp_marker:
                    self.balloon.show_sstp_message(sender, host)
                else:
                    self.balloon.hide_sstp_message()
                # XXX: how about the use_translator flag?
                self.start_script(script, self.FROM_SSTP_CLIENT)
        elif self.get_silent_time() > 0:
            if now - self.get_silent_time() > self.SILENT_TIME:
                self.keep_silence(False)
        elif self.clock[0] != second and \
             self.notify_event('OnSecondChange', self.get_uptime(),
                               self.surface.get_mikire(),
                               self.surface.get_kasanari(),
                               not self.passivemode and self.cantalk):
            pass
        elif self.clock[1] != minute and \
             self.notify_event('OnMinuteChange', self.get_uptime(),
                               self.surface.get_mikire(),
                               self.surface.get_kasanari(),
                               not self.passivemode and self.cantalk):
            pass
        elif self.surface_mouse_motion is not None:
            side, x, y, part = self.surface_mouse_motion
            self.notify_event('OnMouseMove', x, y, '', side, part)
            self.surface_mouse_motion = None
        elif idle > self.__surface_life > 0 and not self.passivemode:
            self.__surface_life = 0
            self.notify_event('OnSurfaceRestore',
                              self.get_surface_id(0),
                              self.get_surface_id(1))
        self.clock = (second, minute)

    reload_event = None

    def do_idle_tasks(self):
        if not self.__running:
            return False
        if self.__temp_mode:
            self.process_script()
            if not self.busy() and \
               not self.script_queue and \
               not (self.processed_script or \
                    self.processed_text):
                if self.__temp_mode == 1:
                    time.sleep(1.4)
                    self.finalize()
                    self.callback['close_ghost'](self)
                    self.callback['reset_sstp_flag']()
                    return False
                else:
                    self.callback['reset_sstp_flag']()
                    self.leave_temp_mode()
                    return True
            else:
                return True
        if self.reload_event and not self.busy() and \
           not (self.processed_script or self.processed_text):
            self.hide_all()
            sys.stdout.write('reloading....\n')
            self.shiori.unload()
            self.callback['stop_sakura'](self, self.callback['reload_current_sakura'], self)
            self.load_settings()
            self.restart() ## FIXME
            sys.stdout.write('done.\n')
            self.enqueue_event(*self.reload_event)
            self.reload_event = None
        # continue network update (enqueue events)
        if self.updateman.is_active():
            self.updateman.run()
            while 1:
                event = self.updateman.get_event()
                if not event:
                    break
                if event[0] == 'OnUpdateComplete' and event[1] == 'changed':
                    self.reload_event = event
                else:
                    self.enqueue_event(*event)
        self.process_script()
        return True

    def quit(self):
        self.callback['stop_sakura'](self)

    ###   SCRIPT PLAYER   ###
    def start_script(self, script, origin=None):
        if not script:
            return
        self.script_origin = origin or self.FROM_GHOST
        self.reset_script(1)
        self.__current_script = script
        if not script.rstrip().endswith(r'\e'):
            script = ''.join((script, r'\e'))
        self.processed_script = []
        while 1:
            try:
                self.processed_script.extend(self.script_parser.parse(script))
            except ninix.script.ParserError, e:
                sys.stderr.write(''.join(('-' * 50, '\n')))
                sys.stderr.write(('%s\n' % e).encode('utf-8', 'ignore'))
                done, script = e
                self.processed_script.extend(done)
            else:
                break
        self.script_mode = self.BROWSE_MODE
        self.script_wait = None
        self.script_side = 0
        self.time_critical_session = 0
        self.quick_session = 0
        self.set_synchronized_session(reset=1)
        self.balloon.hide_all()
        self.balloon.clear_text_all()
        self.balloon.set_balloon_default()
        self.current_time = time.localtime(time.time())
        self.reset_idle_time()
        if self.callback['get_preference']('raise_before_talk'):
            self.raise_all()

    def __yen_e(self, args):
        surface_id = self.get_surface_id(self.script_side)
        self.surface.invoke_yen_e(self.script_side, surface_id)
        self.reset_script()
        self.__balloon_life = self.BALLOON_LIFE
    
    def __yen_0(self, args):
        ##self.balloon.show(0)
        self.script_side = 0

    def __yen_1(self, args):
        ##self.balloon.show(1)
        self.script_side = 1

    def __yen_p(self, args):
        try:
            chr_id = int(args[0])
        except:
            return
        if chr_id >= 0:
            self.script_side = chr_id

    def __yen_4(self, args):
        if self.script_side == 0:
            sw, sh = self.get_surface_size(1)
            sx, sy = self.get_surface_position(1)
        elif self.script_side == 1:
            sw, sh = self.get_surface_size(0)
            sx, sy = self.get_surface_position(0)
        else:
            return
        w, h = self.get_surface_size(self.script_side)
        x, y = self.get_surface_position(self.script_side)
        left, top, scrn_w, scrn_h = ninix.pix.get_workarea()
        if sx + sw / 2 > left + scrn_w / 2:
            new_x = min(x - scrn_w // 20, sx - scrn_w // 20)
        else:
            new_x = max(x + scrn_w // 20, sx + scrn_w // 20)
        step = -10 if x > new_x else 10
        for current_x in range(x, new_x, step):
            self.set_surface_position(self.script_side, current_x, y)
        self.set_surface_position(self.script_side, new_x, y)

    def __yen_5(self, args):
        if self.script_side == 0:
            sw, sh = self.get_surface_size(1)
            sx, sy = self.get_surface_position(1)
        elif self.script_side == 1:
            sw, sh = self.get_surface_size(0)
            sx, sy = self.get_surface_position(0)
        else:
            return
        w, h = self.get_surface_size(self.script_side)
        x, y = self.get_surface_position(self.script_side)
        left, top, scrn_w, scrn_h = ninix.pix.get_workarea()
        if x < sx + sw / 2 < x + w or sx < x + w / 2 < sx + sw:
            return
        if sx + sw / 2 > x + w / 2:
            new_x = sx - w / 2 + 1
        else:
            new_x = sx + sw - w / 2 - 1
        new_x = max(new_x, left)
        new_x = min(new_x, left + scrn_w - w)
        step = -10 if x > new_x else 10
        for current_x in range(x, new_x, step):
            self.set_surface_position(self.script_side, current_x, y)
        self.set_surface_position(self.script_side, new_x, y)

    def __yen_s(self, args):
        surface_id = args[0]
        if surface_id == '-1':
            self.surface.hide(self.script_side)
        else:
            self.set_surface_id(self.script_side, surface_id)
            self.surface.show(self.script_side)
            ## FIXME: self.script_side > 1
            self.notify_event('OnSurfaceChange',
                              self.get_surface_id(0),
                              self.get_surface_id(1))
            self.position_balloons()
        if self.script_side in [0, 1] and not self.__boot[self.script_side]:
            self.__boot[self.script_side] = 1

    def __yen_b(self, args):
        if args[0] == '-1':
            self.balloon.hide(self.script_side)
        else:
            try:
                balloon_id = int(args[0]) / 2
            except ValueError:
                balloon_id = 0
            else:
                self.balloon.set_balloon(self.script_side, balloon_id)

    def __yen__b(self, args):
        try:
            filename, x, y = self.expand_meta(args[0]).split(',')
        except:
            filename, param = self.expand_meta(args[0]).split(',')
            assert param == 'inline'
            x, y = 0, 0 ## FIXME
        filename = filename.lower()
        path = os.path.join(self.get_prefix(), 'ghost/master',
                            filename.replace('\\', '/'))
        if os.path.isfile(path):
            self.balloon.append_image(self.script_side, path, x, y)
        else:
            path = ''.join((path, '.png'))
            if os.path.isfile(path):
                self.balloon.append_image(self.script_side, path, x, y)

    def __yen_n(self, args):
        if args and self.expand_meta(args[0]) == 'half':
            self.balloon.append_text(self.script_side, u'\n[half]')
        else:
            self.balloon.append_text(self.script_side, u'\n')

    def __yen_c(self, args):
        self.balloon.clear_text(self.script_side)

    def __set_weight(self, value, unit):
        try:
            amount = int(value) * unit - 0.01
        except ValueError:
            amount = 0
        if amount > 0:
            self.script_wait = time.time() + amount

    def __yen_w(self, args):
        script_speed = self.callback['get_preference']('script_speed')
        if not self.quick_session and script_speed >= 0:
            self.__set_weight(args[0], 0.05) # 50ms

    def __yen__w(self, args):
        script_speed = self.callback['get_preference']('script_speed')
        if not self.quick_session and script_speed >= 0:
            self.__set_weight(args[0], 0.001) # 1ms

    def __yen_t(self, args):
        self.time_critical_session = not self.time_critical_session

    def __yen__q(self, args):
        self.quick_session = not self.quick_session

    def __yen__s(self, args):
        self.set_synchronized_session([int(arg) for arg in args])

    def __yen__e(self, args):
        self.balloon.hide(self.script_side)
        self.balloon.clear_text(self.script_side)

    def __yen_q(self, args):
        newline_required = 0
        if len(args) == 3: # traditional syntax
            num, link_id, text = args
            newline_required = 1
        else: # new syntax
            text, link_id = args
        text = self.expand_meta(text)
        self.balloon.append_link(self.script_side, link_id, text,
                                 newline_required)
        self.script_mode = self.SELECT_MODE

    def __yen_URL(self, args):
        text = self.expand_meta(args[0])
        if len(args) == 1:
            link = text
        else:
            link = '#cancel'
        self.balloon.append_link(self.script_side, link, text)
        for i in range(1, len(args), 2):
            link = self.expand_meta(args[i])
            text = self.expand_meta(args[i + 1])
            self.balloon.append_link(self.script_side, link, text)
        self.script_mode = self.SELECT_MODE

    def __yen__a(self, args):
        if self.anchor:
            anchor_id = self.anchor[0]
            text = self.anchor[1]
            self.balloon.append_link_out(self.script_side, anchor_id, text)
            self.anchor = None
        else:
            anchor_id = args[0]
            self.anchor = [('anchor', anchor_id), '']
            self.balloon.append_link_in(self.script_side, self.anchor[0])

    def __yen_x(self, args):
        if self.script_mode == self.BROWSE_MODE:
            self.script_mode = self.PAUSE_MODE

    def __yen_a(self, args):
        self.start_script(self.getaistringrandom())

    def __yen_i(self, args):
        try:
            actor_id = int(args[0])
        except ValueError:
            pass
        else:
            self.surface.invoke(self.script_side, actor_id)

    def __yen_j(self, args):
        jump_id = args[0]
        if self.is_URL(jump_id):
            webbrowser.open(jump_id)
        elif self.sstp_entry_db:
            self.start_script(self.sstp_entry_db.get(jump_id, r'\e'))

    def __yen_minus(self, args):
        self.quit()

    def __yen_plus(self, args):
        self.callback['select_ghost'](self, 1)

    def __yen__plus(self, args):
        self.callback['select_ghost'](self, 0)

    def __yen_m(self, args):
        self.write_sstp_handle(self.expand_meta(args[0]))

    def __yen_and(self, args):
        if self.name2codepoint is not None:
            try:
                text = unichr(self.name2codepoint.get(args[0]))
            except:
                text = None
        else:
            text = None
        if text is None:
            text = '?'
        self.balloon.append_text(self.script_side, text)

    def __yen__m(self, args):
        try:
            num = int(args[0], 16)
        except ValueError:
            num = 0
        if 0x20 <= num <= 0x7e:
            text = chr(num)
        else:
            text = '?'
        self.balloon.append_text(self.script_side, text)

    def __yen__u(self, args):
        if re.match('0x[a-fA-F0-9]{4}', args[0]):
            text = eval(''.join(('u"\\u', args[0][2:], '"')))
            self.balloon.append_text(self.script_side, text)
        else:
            self.balloon.append_text(self.script_side, u'?')

    def __yen__v(self, args):
        filename = self.expand_meta(args[0])
        filename = filename.lower()
        path = os.path.join(self.get_prefix(), 'ghost/master', filename)
        if os.path.isfile(path):
            self.audio_player.set_state(gst.STATE_NULL)
            self.audio_player.set_property("uri", "file://" + path)
            self.audio_player.set_state(gst.STATE_PLAYING)

    def __yen_exclamation(self, args): ## FIXME
        if not args:
            return
        argc = len(args)
        args = [self.expand_meta(s) for s in args]
        if args[0] == 'raise' and argc >= 2:
            self.notify_event(*args[1:10])
        elif args[0:2] == ['open', 'browser'] and argc > 2:
            webbrowser.open(args[2])
        elif args[0:2] == ['open', 'communicatebox']:
            if not self.passivemode:
                self.balloon.open_communicatebox()
        elif args[0:2] == ['open', 'teachbox']:
            if not self.passivemode:
                self.balloon.open_teachbox()
        elif args[0:2] == ['open', 'inputbox'] and argc > 2:
            if not self.passivemode:
                if argc > 4:
                    self.balloon.open_inputbox(args[2], args[3], args[4])
                elif argc == 4:
                    self.balloon.open_inputbox(args[2], args[3])
                else:
                    self.balloon.open_inputbox(args[2])
        elif args[0:2] == ['open', 'configurationdialog']:
            self.callback['edit_preferences']()
        elif args[0:2] == ['change', 'shell'] and argc > 2:
            set_type, i, j = self.current
            for num in range(len(self.surface_set)):
                shell_name = self.surface_set[num][0]
                if shell_name == args[2]:
                    item = (set_type, i, num)
                    self.select_shell(None, item) ## FIXME
                    break
        elif args[0:2] == ['change', 'ghost'] and argc > 2:
            if args[2] == 'random':
                self.callback['select_ghost'](self, 0, 0)
            else:
                self.callback['select_ghost_by_name'](self, args[2], 0)
        elif args[0:1] == ['updatebymyself']:
            if not self.busy(check_updateman=False):
                self.__update()
        elif args[0:1] == ['vanishbymyself']:
            self.vanished = 1 ## FIXME
            self.vanish_by_myself()
        elif args[1:2] == ['repaint']:
            if args[0:1] == ['lock']:
                self.lock_repaint = 1
            elif args[0:1] == ['unlock']:
                self.lock_repaint = 0
        elif args[1:2] == ['passivemode']:
            if args[0:1] == ['enter']:
                self.passivemode = 1
            elif args[0:1] == ['leave']:
                self.passivemode = 0
        elif args[0:2] == ['set', 'alignmentondesktop']:
            if args[2] == 'bottom':
                if self.synchronized_session:
                    for chr_id in self.synchronized_session:
                        self.align_bottom(chr_id)
                else:
                    self.align_bottom(self.script_side)
            elif args[2] == 'top':
                if self.synchronized_session:
                    for chr_id in self.synchronized_session:
                        self.align_top(chr_id)
                else:
                    self.align_top(self.script_side)
        elif args[0:2] == ['set', 'alignmenttodesktop']:
            if args[2] == 'free':
                if self.synchronized_session:
                    for chr_id in self.synchronized_session:
                        self.surface.set_alignment(chr_id, 2)
                else:
                    self.surface.set_alignment(self.script_side, 2)
        elif args[0:2] == ['set', 'windowstate']:
            if args[2] == 'minimize':
                self.surface.window_iconify(True)
            ##elif args[2] == '!minimize':
            ##    self.surface.window_iconify(False)
            elif args[2] == 'stayontop':
                self.surface.window_stayontop(True)
            elif args[2] == '!stayontop':
                self.surface.window_stayontop(False)
        elif args[0:2] == ['set', 'wallpaper']:
            path = os.path.join(self.get_prefix(), args[2])
            opt = None
            if len(args) > 3:
                # possible picture_options value:
                # "none", "wallpaper", "centered", "scaled", "stretched"
                options = {
                    'center': 'centered',
                    'tile': 'wallpaper',
                    'stretch': 'stretched'
                    }
                if args[3] not in options:
                    opt = None
                else:
                    opt = options[args[3]]
            if opt is None:
                opt = 'centered' # default
            if os.path.exists(path) and gconf is not None:
                client = gconf.client_get_default() 
                gconf_background_dir = '/desktop/gnome/background/'
                client.set_string(''.join((gconf_background_dir,
                                           'picture_filename')),
                                   path) 
                client.set_string(''.join((gconf_background_dir,
                                           'picture_options')),
                                   opt) 
        elif args[0] == '*':
            self.balloon.append_sstp_marker(self.script_side)
        else:
            pass ## FIXME

    def __yen___c(self, args):
        self.balloon.open_communicatebox()

    def __yen___t(self, args): 
        self.balloon.open_teachbox()

    def __yen_v(self, args):
        self.raise_surface(self.script_side)

    def __yen_f(self, args):
        if len(args) != 2: ## FIXME
            return
        tag = None
        if args[0] == 'sup':
            tag = '<sup>' if args[1] == 'true' else '</sup>'
        elif args[0] == 'sub':
            tag = '<sub>' if args[1] == 'true' else '</sub>'
        elif args[0] == 'strike':
            tag = '<s>' if args[1] in ['true', '1', 1] else '</s>'
        elif args[0] == 'underline':
            tag = '<u>' if args[1] in ['true', '1', 1] else '</u>'
        else:
            pass ## FIXME
        if tag is not None:
            self.balloon.append_meta(self.script_side, tag)

    __script_tag = {
        r'\e': __yen_e,
        r'\0': __yen_0,
        r'\h': __yen_0,
        r'\1': __yen_1,
        r'\u': __yen_1,
        r'\p': __yen_p,
        r'\4': __yen_4,
        r'\5': __yen_5,
        r'\s': __yen_s,
        r'\b': __yen_b,
        r'\_b': __yen__b,
        r'\n': __yen_n,
        r'\c': __yen_c,
        r'\w': __yen_w,
        r'\_w': __yen__w,
        r'\t': __yen_t,
        r'\_q': __yen__q,
        r'\_s': __yen__s,
        r'\_e': __yen__e,
        r'\q': __yen_q,
        r'\URL': __yen_URL,
        r'\_a': __yen__a,
        r'\x': __yen_x,
        r'\a': __yen_a, # Obsolete: only for old SHIORI
        r'\i': __yen_i,
        r'\j': __yen_j,
        r'\-': __yen_minus,
        r'\+': __yen_plus,
        r'\_+': __yen__plus,
        r'\m': __yen_m,
        r'\&': __yen_and,
        r'\_m': __yen__m,
        r'\_u': __yen__u,
        r'\_v': __yen__v,
        r'\!': __yen_exclamation,
        r'\__c': __yen___c,
        r'\__t': __yen___t, 
        r'\v': __yen_v,
        r'\f': __yen_f,
        }

    def interpret_script(self):
        if self.script_wait is not None:
            if time.time() < self.script_wait:
                return
            self.script_wait = None
        if self.processed_text:
            self.balloon.show(self.script_side)
            self.balloon.append_text(self.script_side, self.processed_text[0])
            self.processed_text = self.processed_text[1:]
            surface_id = self.get_surface_id(self.script_side)
            count = self.balloon.get_text_count(self.script_side)
            if self.surface.invoke_talk(self.script_side, surface_id, count):
                self.balloon.reset_text_count(self.script_side)
            script_speed = self.callback['get_preference']('script_speed')
            if script_speed > 0:
                self.script_wait = time.time() + script_speed * 0.02
            return
        node = self.processed_script.pop(0)
        if node[0] == ninix.script.SCRIPT_TAG:
            name, args = node[1], node[2:]
            if name in self.__script_tag:
                self.__script_tag[name](self, args)
            else:
                pass ## FIMXE
        elif node[0] == ninix.script.SCRIPT_TEXT:
            text = self.expand_meta(node[1])
            if self.anchor:
                self.anchor[1] = ''.join((self.anchor[1], text))
            script_speed = self.callback['get_preference']('script_speed')
            if not self.quick_session and script_speed >= 0:
                self.processed_text = text
            else:
                self.balloon.append_text(self.script_side, text)

    def reset_script(self, reset_all=0):
        if reset_all:
            self.script_mode = self.BROWSE_MODE
            self.script_post_proc = []
            self.__current_script = ''
        self.processed_script = None
        self.processed_text = ''
        self.time_critical_session = 0
        self.quick_session = 0
        self.set_synchronized_session(reset=1)
        self.reset_idle_time()

    def set_synchronized_session(self, list=[], reset=0):
        if reset:
            self.synchronized_session = []
        elif not list:
            if self.synchronized_session:
                self.synchronized_session = []
            else:
                self.synchronized_session = [0, 1]
        else:
            self.synchronized_session = list
        self.balloon.synchronize(self.synchronized_session)

    def expand_meta(self, text_node):
        buf = []
        for chunk in text_node:
            if chunk[0] == ninix.script.TEXT_STRING:
                buf.append(chunk[1])
            elif chunk[1] == '%month':
                buf.append(str(self.current_time[1]))
            elif chunk[1] == '%day':
                buf.append(str(self.current_time[2]))
            elif chunk[1] == '%hour':
                buf.append(str(self.current_time[3]))
            elif chunk[1] == '%minute':
                buf.append(str(self.current_time[4]))
            elif chunk[1] == '%second':
                buf.append(str(self.current_time[5]))
            elif chunk[1] in ['%username', '%c']:
                buf.append(self.get_username())
            elif chunk[1] == '%selfname':
                buf.append(self.get_selfname())
            elif chunk[1] == '%selfname2':
                buf.append(self.get_selfname2())
            elif chunk[1] == '%keroname':
                buf.append(self.get_keroname())
            elif chunk[1] == '%friendname':
                buf.append(self.get_friendname())
            elif chunk[1] == '%screenwidth':
                left, top, scrn_w, scrn_h = ninix.pix.get_workarea()
                buf.append(str(scrn_w))
            elif chunk[1] == '%screenheight':
                left, top, scrn_w, scrn_h = ninix.pix.get_workarea()
                buf.append(str(scrn_h))
            elif chunk[1] == '%et':
                buf.append(
                    unicode('%d万年' % self.current_time[7], 'utf-8'))
            elif chunk[1] == '%exh':
                buf.append(str(self.get_uptime()))
            elif chunk[1] in ['%ms', '%mz', '%ml', '%mc', '%mh', \
                              '%mt', '%me', '%mp', '%m?']:
                buf.append(
                    self.getword(''.join(('\\', chunk[1][1:]))))
            elif chunk[1] == '%dms':
                buf.append(self.getdms())
            else: # %c, %songname
                buf.append(chunk[1])
        return ''.join(buf)

    ###   SEND SSTP/1.3   ###
    def _send_sstp_handle(self, data):
        r, w, e = select.select([], [self.sstp_handle], [], 0)
        if not w:
            return
        try:
            self.sstp_handle.send(''.join((data, '\n')))
        except socket.error:
            pass

    def write_sstp_handle(self, data):
        if self.sstp_handle is None:
            return
        self._send_sstp_handle(''.join(('+', data)))
        ##print 'write_sstp_handle(%s)' % repr(data)

    def close_sstp_handle(self):
        if self.sstp_handle is None:
            return
        self._send_sstp_handle('-')
        ##print 'close_sstp_handle()'
        try:
            self.sstp_handle.close()
        except socket.error:
            pass
        self.sstp_handle = None

    def close(self):
        if self.busy():
            gtk.gdk.beep() ## FIXME
            return
        self.reset_script(1)
        self.enqueue_event('OnClose')

    def about(self):
        if self.busy():
            gtk.gdk.beep() ## FIXME
            return
        self.start_script(ninix.version.VERSION_INFO)

    def __update(self):
        if self.updateman.is_active():
            return
        homeurl = self.getstring('homeurl')
        if not homeurl:
            self.start_script(
                ''.join((r'\t\h\s[0]',
                         unicode(_("I'm afraid I don't have Network Update yet."), 'utf-8'),
                         r'\e')))
            self.balloon.hide_sstp_message()
            return
        ghostdir = self.get_prefix()
        print 'homeurl =', homeurl
        print 'ghostdir =', ghostdir
        self.updateman.start(homeurl, ghostdir)

    def network_update(self):
        if self.busy():
            gtk.gdk.beep() ## FIXME
            return
        self.__update()


class VanishDialog:

    def __init__(self, callback):
        self.callback = callback
        self.window = gtk.Dialog()
        self.window.connect('delete_event', self.cancel)
        self.window.set_title('Vanish')
        self.window.set_modal(True)
        self.window.set_position(gtk.WIN_POS_CENTER)
        self.label = gtk.Label(unicode(_('Vanish'), 'utf-8'))
        self.window.vbox.pack_start(self.label, padding=10)
        self.label.show()
        box = gtk.HButtonBox()
        box.set_layout(gtk.BUTTONBOX_END)
        self.window.action_area.pack_start(box)
        box.show()
        button = gtk.Button(unicode(_('Yes'), 'utf-8'))
        button.connect('clicked', self.ok)
        box.add(button)
        button.show()
        button = gtk.Button(unicode(_('No'), 'utf-8'))
        button.connect('clicked', self.cancel)
        box.add(button)
        button.show()

    def set_message(self, message):
        self.label.set_text(message)

    def show(self):
        self.window.show()

    def ok(self, widget, event=None):
        self.window.hide()
        self.callback['notify_vanish_confirmation']()
        return True

    def cancel(self, widget, event=None):
        self.window.hide()
        self.callback['notify_event']('OnVanishCancel')
        return True


def test():
    pass

if __name__ == '__main__':
    test()
