# -*- coding: utf-8 -*-
#
#  Copyright (C) 2001, 2002 by Tamito KAJIYAMA
#  Copyright (C) 2002, 2003 by MATSUMURA Namihiko <nie@counterghost.net>
#  Copyright (C) 2004-2009 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#  Copyright (C) 2003-2005 by Shun-ichi TAHARA <jado@flowernet.gr.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#

import mimetools
import os

import gtk
import gobject

import ninix.home


range_scale = [(' 100%',  100),
               ('  90%',   90),
               ('  80%',   80),
               ('  70%',   70),
               ('  60%',   60),
               ('  50%',   50),
               ('  40%',   40),
               (' 200%',  200),
               (' 300%',  300),
               ('1000%', 1000)]

range_script_speed = [(_('None'), -1), # ordinal (-1: no wait)
                      (''.join(('1 (', _('Fast'), ')')), 0),
                      ('2', 1),
                      ('3', 2),
                      ('4', 3),
                      ('5', 4),
                      ('6', 6),
                      (''.join(('7 (', _('Slow'), ')')), 8)]

# default settings
DEFAULT_BROWSER = "firefox %s"
DEFAULT_BALLOON_FONTS = 'Sans'

def get_default_surface_scale():
    return range_scale[0][1]

def get_default_script_speed():
    i = len(range_script_speed) / 2
    return range_script_speed[i][1]


class Preferences(dict):

    def __init__(self, filename):
        dict.__init__(self)
        self.filename = filename

    def load(self):
        try:
            f = open(self.filename)
        except IOError:
            return
        prefs = mimetools.Message(f)
        for key, value in prefs.items():
            self[key] = value
        f.close()

    def save(self):
        try:
            os.makedirs(os.path.dirname(self.filename))
        except OSError:
            pass
        f = open(self.filename, 'w')
        keys = self.keys()
        keys.sort()
        for key in keys:
            f.write('%s: %s\n' % (key, self[key]))
        f.close()

    def _get(self, name, default, conv):
        value = self.get(name)
        if value:
            try:
                return conv(value)
            except ValueError:
                pass
        return default

    def getint(self, name, default=None):
        return self._get(name, default, int)

    def getfloat(self, name, default=None):
        return self._get(name, default, float)


def new_prefs():
    filename = ninix.home.get_preferences()
    return Preferences(filename)

def load_prefs():
    prefs = new_prefs()
    prefs.load()
    return prefs


class PreferenceDialog:

    PREFS_HELPER_PATTERN    = 'helper_%d_pattern'
    PREFS_HELPER_COMMAND    = 'helper_%d_command'

    PREFS_TYPE = {'sakura_name': None,
                  'sakura_surface': int,
                  'default_balloon': None,
                  'ignore_default': int,
                  'script_speed': int,
                  'surface_scale': int,
                  'balloon_scalling': int,
                  'browser': None,
                  'helper_%d_pattern': None,
                  'helper_%d_command': None,
                  'balloon_fonts': None,
                  'allowembryo': int,
                  'check_collision': int,
                  'use_pna': int,
                  'sink_after_talk': int,
                  'raise_before_talk': int,
                  'surface_alpha': float,
                  'balloon_alpha': float,
                  'animation_quality': float,
                  'seriko_inactive': int,
                  }

    def __init__(self, app):
        self.app = app
        self.window = gtk.Dialog()
        self.window.set_title('Preferences')
        self.window.connect('delete_event', self.cancel)
        self.notebook = gtk.Notebook()
        self.notebook.set_tab_pos(gtk.POS_TOP)
        self.window.vbox.pack_start(self.notebook)
        self.notebook.show()
        for name, constructor in [
            (_('Font'),            self.make_page_fonts),
            (_('Browser'),         self.make_page_browser),
            (_('Helper'),          self.make_page_helper),
            (_('Surface&Balloon'), self.make_page_surface_n_balloon),
            (_('Misc'),            self.make_page_misc),
            (_('Debug'),           self.make_page_debug),
            ]:
            self.notebook.append_page(constructor(),
                                      gtk.Label(unicode(name, 'utf-8')))
        box = gtk.HButtonBox()
        box.set_layout(gtk.BUTTONBOX_END)
        self.window.action_area.pack_start(box)
        box.show()
        button = gtk.Button('OK')
        button.connect('clicked', self.ok)
        box.add(button)
        button.show()
        button = gtk.Button('Apply')
        button.connect('clicked', self.apply)
        box.add(button)
        button.show()
        button = gtk.Button('Cancel')
        button.connect('clicked', self.cancel)
        box.add(button)
        button.show()
        self.rule_editor = RuleEditor(self.window)

    def load(self):
        self.__prefs = load_prefs()
        self.reset()
        self.__saved_prefs = self.__prefs.copy()
        self.notify() ## FIXME
        
    def reset(self): ### FIXME ###
        self.set_balloon_fonts(self.get('balloon_fonts', DEFAULT_BALLOON_FONTS))
        self.set_browser(self.get('browser', DEFAULT_BROWSER))
        helper_list = []
        while 1:
            pattern = self.__prefs.get(self.PREFS_HELPER_PATTERN % len(helper_list)) ## FIXME
            command = self.__prefs.get(self.PREFS_HELPER_COMMAND % len(helper_list)) ## FIXME
            if not pattern or not command:
                break
            helper_list.append((pattern, command))
        self.set_helpers(helper_list)
        name = self.get('default_balloon')
        if self.app.find_balloon_by_name(name) is not None:
            self.set_default_balloon(name)
        self.set_ignore_default(self.get('ignore_default', 0))
        self.set_surface_scale(self.get('surface_scale',
                                        get_default_surface_scale()))
        self.set_script_speed(self.get('script_speed',
                                       get_default_script_speed()))
        self.set_balloon_scalling(self.get('balloon_scalling'))
        self.set_allowembryo(self.get('allowembryo'))
        self.set_check_collision(self.get('check_collision', 0))
        self.set_use_pna(self.get('use_pna'))
        self.set_sink_after_talk(self.get('sink_after_talk'))
        self.set_surface_alpha(self.get('surface_alpha'))
        self.set_balloon_alpha(self.get('balloon_alpha'))
        self.set_raise_before_talk(self.get('raise_before_talk'))
        self.set_animation_quality(self.get('animation_quality'))
        self.set_seriko_inactive(self.get('seriko_inactive'))

    def get(self, name, default=None):
        assert name in self.PREFS_TYPE
        if self.PREFS_TYPE[name] == int:
            return self.__prefs.getint(name, default)
        elif self.PREFS_TYPE[name] == float:
            return self.__prefs.getfloat(name, default)
        else:
            return self.__prefs.get(name, default)

    def save(self): ## FIXME: check key is valid
        name, surface = self.app.get_current_sakura_name() ## FIXME
        self.__prefs['sakura_name'] = self.__saved_prefs['sakura_name'] = name
        self.__prefs['sakura_surface'] = self.__saved_prefs['sakura_surface'] = surface
        self.__prefs.save()

    def get_helpers_regex(self):
        helpers_regex = []
        for pattern, command in self.get_helpers():
            try:
                regex = re.compile(pattern)
            except re.error:
                continue
            helpers_regex.append((regex, command))
        return helpers_regex

    def edit_preferences(self):
        self.show()

    def update(self): ## FIXME
        self.__prefs['allowembryo'] = str(1 if self.allowembryo_button.get_active() else 0)
        self.__prefs['balloon_fonts'] = self.fontsel.get_font_name()
        self.__prefs['browser'] = self.browser.get_text()
        lstore = self.helpers.get_model()
        listiter = lstore.get_iter_first()
        i = 0 ## FIXME
        while listiter:
            self.__prefs[self.PREFS_HELPER_PATTERN % i] = lstore.get_value(listiter, 0)
            self.__prefs[self.PREFS_HELPER_COMMAND % i] = lstore.get_value(listiter, 1)
            i += 1
            listiter = lstore.iter_next(listiter)
        name = None
        selected = self.balloon_treeview.get_selection().get_selected()
        if selected:
            model, listiter = selected
            name = model.get_value(listiter, 0)
        if self.app.find_balloon_by_name(name) is not None:
            self.__prefs['default_balloon'] = name
        self.__prefs['ignore_default'] = 1 if self.ignore_button.get_active() else 0
        label, value = range_scale[self.surface_scale_combo.get_active()]
        self.__prefs['surface_scale'] = str(int(value))
        label, value = range_script_speed[self.script_speed_combo.get_active()]
        self.__prefs['script_speed'] = str(int(value))
        self.__prefs['balloon_scalling'] = str(1 if self.balloon_scalling_button.get_active() else 0)
        self.__prefs['check_collision'] = str(1 if self.check_collision_button.get_active() else 0)
        self.__prefs['use_pna'] = str(1 if self.use_pna_button.get_active() else 0)
        self.__prefs['sink_after_talk'] = str(1 if self.sink_after_talk_button.get_active() else 0)
        self.__prefs['raise_before_talk'] = str(1 if self.raise_before_talk_button.get_active() else 0)
        self.__prefs['surface_alpha'] = str(float(self.surface_alpha_adjustment.get_value()))
        self.__prefs['balloon_alpha'] = str(float(self.balloon_alpha_adjustment.get_value()))
        self.__prefs['animation_quality'] = str(float(self.animation_quality_adjustment.get_value()))
        self.__prefs['seriko_inactive'] = str(1 if self.seriko_inactive_button.get_active() else 0)

    def ok(self, widget):
        self.hide()
        self.update()
        self.__saved_prefs.update(self.__prefs)
        self.notify() ## FIXME

    def apply(self, widget):
        self.update()
        self.notify() ## FIXME

    def cancel(self, widget, event=None):
        self.hide()
        self.__prefs.update(self.__saved_prefs)
        self.reset()
        self.notify() ## FIXME
        return True

    def show(self):
        self.window.show()

    def hide(self):
        self.window.hide()

    def make_page_fonts(self):
        page = gtk.VBox(spacing=5)
        page.set_border_width(5)
        # font
        frame = gtk.Frame(unicode(_('Font(s) for balloons'), 'utf-8'))
        frame.set_size_request(480, -1)
        page.pack_start(frame)
        frame.show()
        self.fontsel = gtk.FontSelection()
        self.fontsel.show()
        frame.add(self.fontsel)
        page.show()
        return page

    def make_page_browser(self):
        page = gtk.VBox(spacing=5)
        page.set_border_width(5)
        # browser
        frame = gtk.Frame(unicode(_('Browser'), 'utf-8'))
        frame.set_size_request(480, -1)
        page.pack_start(frame)
        frame.show()
        box = gtk.VBox(spacing=2)
        box.set_border_width(5)
        frame.add(box)
        box.show()
        self.browser = gtk.Entry()
        box.pack_start(self.browser, False)
        self.browser.show()
        # help messages
        frame = gtk.Frame()
        frame.set_size_request(480, -1)
        page.pack_start(frame, False)
        frame.show()
        box = gtk.VBox(spacing=2)
        box.set_border_width(5)
        frame.add(box)
        box.show()
        label = gtk.Label(unicode(_('- %s in this command line will be replaced with the URL'), 'utf-8'))
        label.set_alignment(0, -1)
        box.pack_start(label, False)
        label.show()
        label = gtk.Label(unicode(_('- trailing & is not required.(automagically added)'), 'utf-8'))
        label.set_alignment(0, -1)
        box.pack_start(label, False)
        label.show()
        page.show()
        return page

    def make_page_helper(self):
        page = gtk.VBox(spacing=5)
        page.set_border_width(5)
        frame = gtk.Frame(unicode(_('Application'), 'utf-8'))
        frame.set_size_request(480, -1)
        page.pack_start(frame)
        frame.show()
        box = gtk.VBox(spacing=5)
        box.set_border_width(5)
        frame.add(box)
        box.show()
        swin = gtk.ScrolledWindow()
        swin.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
        box.pack_start(swin)
        swin.show()
        lstore = gtk.ListStore(gobject.TYPE_STRING, gobject.TYPE_STRING)
        self.helpers = gtk.TreeView(lstore)
        column = gtk.TreeViewColumn('Pattern', gtk.CellRendererText(), text=0)
        self.helpers.append_column(column)
        column = gtk.TreeViewColumn('Command', gtk.CellRendererText(), text=1)
        self.helpers.append_column(column)
        swin.add(self.helpers)
        self.helpers.show()
        bbox = gtk.HButtonBox()
        bbox.set_spacing(10)
        bbox.set_layout(gtk.BUTTONBOX_EDGE)
        bbox.set_border_width(5)
        box.pack_start(bbox, False)
        bbox.show()
        button = gtk.Button('New')
        button.connect('clicked', self.rule_new)
        bbox.pack_start(button)
        button.show()
        button = gtk.Button('Edit')
        button.connect('clicked', self.rule_edit)
        bbox.pack_start(button)
        button.show()
        button = gtk.Button('Delete')
        button.connect('clicked', self.rule_delete)
        bbox.pack_start(button)
        button.show()
        button = gtk.Button('Up')
        button.connect('clicked', self.rule_up)
        bbox.pack_start(button)
        button.show()
        button = gtk.Button('Down')
        button.connect('clicked', self.rule_down)
        bbox.pack_start(button)
        button.show()
        frame = gtk.Frame()
        frame.set_size_request(480, -1)
        page.pack_start(frame, False)
        frame.show()
        box = gtk.VBox(spacing=2)
        box.set_border_width(5)
        frame.add(box)
        box.show()
        label = gtk.Label(unicode(_('- %s in this command line will be replaced with the filename'), 'utf-8'))
        label.set_alignment(0, -1)
        box.pack_start(label, False)
        label.show()
        label = gtk.Label(unicode(_('- trailing & is not required.(automagically added)'), 'utf-8'))
        label.set_alignment(0, -1)
        box.pack_start(label, False)
        label.show()
        page.show()
        return page

    def make_page_surface_n_balloon(self):
        page = gtk.VBox(spacing=5)
        page.set_border_width(5)
        page.show()
        frame = gtk.Frame(unicode(_('Surface Scaling'), 'utf-8'))
        frame.set_size_request(480, -1)
        page.pack_start(frame, False)
        frame.show()
        box = gtk.VBox(spacing=5)
        box.set_border_width(5)
        frame.add(box)
        box.show()
        hbox = gtk.HBox(spacing=5)
        box.pack_start(hbox, False)
        hbox.show()
        label = gtk.Label(unicode(_('Default Setting'), 'utf-8'))
        hbox.pack_start(label, False)
        label.show()
        self.surface_scale_combo = gtk.combo_box_new_text()
        for label, value in range_scale:
            self.surface_scale_combo.append_text(label)
        hbox.pack_start(self.surface_scale_combo, False)
        self.surface_scale_combo.show()
        button = gtk.CheckButton(unicode(_('Scale Balloon'), 'utf-8'))
        self.balloon_scalling_button = button
        box.pack_start(button, False)
        button.show()
        frame = gtk.Frame(unicode(_('Default Balloon'), 'utf-8'))
        frame.set_size_request(480, -1)
        page.pack_start(frame)
        frame.show()
        box = gtk.VBox(spacing=5)
        box.set_border_width(5)
        frame.add(box)
        box.show()
        scrolled = gtk.ScrolledWindow()
        scrolled.set_policy(gtk.POLICY_NEVER, gtk.POLICY_ALWAYS)
        scrolled.set_shadow_type(gtk.SHADOW_ETCHED_IN)
        box.pack_start(scrolled, True)
        scrolled.show()
        model = gtk.ListStore(gobject.TYPE_STRING)
        for desc, balloon in self.app.balloons:
            name = desc.get('name', '')
            listiter = model.append()
            model.set_value(listiter, 0, name)
        treeview = gtk.TreeView(model)
        column = gtk.TreeViewColumn(
            _('Balloon Name'), gtk.CellRendererText(), text=0)
        treeview.append_column(column)
        treeview.get_selection().set_mode(gtk.SELECTION_SINGLE)
        self.balloon_treeview = treeview
        scrolled.add(treeview)
        treeview.show()
        button = gtk.CheckButton(unicode(_('Always Use This Balloon'), 'utf-8'))
        self.ignore_button = button
        box.pack_start(button, False)
        button.show()
        frame = gtk.Frame(unicode(_('Translucency'), 'utf-8'))
        frame.set_size_request(480, -1)
        page.pack_start(frame, False)
        frame.show()
        box = gtk.VBox(spacing=5)
        box.set_border_width(5)
        frame.add(box)
        box.show()
        button = gtk.CheckButton(unicode(_('Use PNA file'), 'utf-8'))
        self.use_pna_button = button
        box.pack_start(button, False)
        button.show()
        hbox = gtk.HBox(spacing=5)
        box.add(hbox)
        hbox.show()
        label = gtk.Label(unicode(_("Surfaces' alpha channel"), 'utf-8'))
        hbox.pack_start(label, False)
        label.show()
        self.surface_alpha_adjustment = gtk.Adjustment(1.0, 0.1, 1.0, 0.1, 0.5)
        button = gtk.SpinButton(self.surface_alpha_adjustment, 0.2, 1)
        hbox.pack_start(button, False)
        button.show()
        hbox = gtk.HBox(spacing=5)
        box.add(hbox)
        hbox.show()
        label = gtk.Label(unicode(_("Balloons' alpha channel"), 'utf-8'))
        hbox.pack_start(label, False)
        label.show()
        self.balloon_alpha_adjustment = gtk.Adjustment(1.0, 0.1, 1.0, 0.1, 0.5)
        button = gtk.SpinButton(self.balloon_alpha_adjustment, 0.2, 1)
        hbox.pack_start(button, False)
        button.show()
        frame = gtk.Frame(unicode(_('Animation'), 'utf-8'))
        frame.set_size_request(480, -1)
        page.pack_start(frame, False)
        frame.show()
        box = gtk.VBox(spacing=5)
        box.set_border_width(5)
        frame.add(box)
        box.show()
        hbox = gtk.HBox(spacing=5)
        box.add(hbox)
        hbox.show()
        label = gtk.Label(unicode(_("Quality"), 'utf-8'))
        hbox.pack_start(label, False)
        label.show()        
        self.animation_quality_adjustment = gtk.Adjustment(1.0, 0.1, 1.0, 0.1, 0.1)
        button = gtk.SpinButton(self.animation_quality_adjustment, 0.2, 1)
        hbox.pack_start(button, False)
        button.show()
        hbox.show()
        button = gtk.CheckButton(unicode(_('SERIKO INACTIVE'), 'utf-8'))
        self.seriko_inactive_button = button
        box.pack_start(button, False)
        button.show()
        return page        

    def make_page_misc(self):
        page = gtk.VBox(spacing=5)
        page.set_border_width(5)
        page.show()
        frame = gtk.Frame(unicode(_('SSTP Setting'), 'utf-8'))
        frame.set_size_request(480, -1)
        page.pack_start(frame, False)
        frame.show()
        button = gtk.CheckButton(unicode(_('Allowembryo'), 'utf-8'))
        self.allowembryo_button = button
        frame.add(button)
        button.show()
        frame = gtk.Frame(unicode(_('Script Wait'), 'utf-8'))
        frame.set_size_request(480, -1)
        page.pack_start(frame, False)
        frame.show()
        hbox = gtk.HBox(spacing=5)
        frame.add(hbox)
        hbox.show()
        label = gtk.Label(unicode(_('Default Setting'), 'utf-8'))
        hbox.pack_start(label, False)
        label.show()
        self.script_speed_combo = gtk.combo_box_new_text()
        for label, value in range_script_speed:
            self.script_speed_combo.append_text(label)
        hbox.pack_start(self.script_speed_combo, False)
        self.script_speed_combo.show()
        frame = gtk.Frame(unicode(_('Raise & Lower'), 'utf-8'))
        frame.set_size_request(480, -1)
        page.pack_start(frame, False)
        frame.show()
        box = gtk.VBox(spacing=5)
        box.set_border_width(5)
        frame.add(box)
        box.show()
        button = gtk.CheckButton(unicode(_('Sink after Talk'), 'utf-8'))
        self.sink_after_talk_button = button
        box.pack_start(button, False)
        button.show()
        button = gtk.CheckButton(unicode(_('Raise before Talk'), 'utf-8'))
        self.raise_before_talk_button = button
        box.pack_start(button, False)
        button.show()
        return page

    def make_page_debug(self):
        page = gtk.VBox(spacing=5)
        page.set_border_width(5)
        page.show()
        frame = gtk.Frame(unicode(_('Surface Debugging'), 'utf-8'))
        frame.set_size_request(480, -1)
        page.pack_start(frame, False)
        frame.show()
        button = gtk.CheckButton(unicode(_('Display Collision Area'), 'utf-8'))
        self.check_collision_button = button
        frame.add(button)
        button.show()
        return page

    def rule_new(self, widget):
        self.rule_editor.set_pattern('')
        self.rule_editor.set_command('')
        if self.rule_editor.run('New rule...'):
            pattern = self.rule_editor.get_pattern()
            command = self.rule_editor.get_command()
            lstore = self.helpers.get_model()
            listiter = lstore.append()
            lstore.set(listiter, 0, pattern, 1, command)

    def rule_edit(self, widget):
        selection = self.helpers.get_selection()
        lstore, listiter = selection.get_selected()
        if not listiter:
            return
        pattern = lstore.get_value(listiter, 0)
        command = lstore.get_value(listiter, 1)
        self.rule_editor.set_pattern(pattern)
        self.rule_editor.set_command(command)
        if self.rule_editor.run('Edit rule...'):
            pattern = self.rule_editor.get_pattern()
            command = self.rule_editor.get_command()
            lstore.set(listiter, 0, pattern, 1, command)

    def rule_delete(self, widget):
        selection = self.helpers.get_selection()
        lstore, listiter = selection.get_selected()
        if not listiter:
            return
        lstore.remove(listiter)

    def rule_up(self, widget):
        selection = self.helpers.get_selection()
        lstore, listiter = selection.get_selected()
        if not listiter:
            return
        path = lstore.get_path(listiter)
        if path[0] == 0:
            return
        listiter_ = lstore.get_iter(path[0] - 1)
        if listiter_:
            lstore.swap(listiter_, listiter)

    def rule_down(self, widget):
        selection = self.helpers.get_selection()
        lstore, listiter = selection.get_selected()
        if not listiter:
            return
        listiter_ = lstore.iter_next(listiter)
        if listiter_:
            lstore.swap(listiter_, listiter)

    def notify(self): ## FIXME
        for sakura in self.app.get_working_ghost():
            sakura.reset_balloon_fonts() ## FIXME
            flag = self.get('balloon_scalling')
            sakura.set_balloon_scalling(flag) ## FIXME
            scale = self.get('surface_scale')
            sakura.set_surface_scale(scale) ## FIXME
            flag = self.get('use_pna')
            sakura.set_use_pna(flag) ## FIXME
            alpha = self.get('surface_alpha')
            sakura.set_surface_alpha(alpha) ## FIXME
            alpha = self.get('balloon_alpha')
            sakura.set_balloon_alpha(alpha) ## FIXME
            quality = self.get('animation_quality')
            sakura.set_animation_quality(quality) ## FIXME
            flag = self.get('seriko_inactive')
            sakura.set_seriko_inactive(flag) ## FIXME

    def set_balloon_fonts(self, name):
        self.fontsel.set_font_name(name)

    def set_browser(self, command):
        self.browser.set_text(command)

    def set_helpers(self, helper_list):
        lstore = self.helpers.get_model()
        lstore.clear()
        for pattern, command in helper_list:
            listiter = lstore.append()
            lstore.set(listiter, 0, pattern, 1, command)

    def get_helpers(self):
        helper_list = []
        while 1:
            pattern = self.__prefs.get(self.PREFS_HELPER_PATTERN % len(helper_list))
            command = self.__prefs.get(self.PREFS_HELPER_COMMAND % len(helper_list))
            if not pattern or not command:
                break
            helper_list.append((pattern, command))
        return helper_list

    def set_balloon_scalling(self, flag):
        self.balloon_scalling_button.set_active(True if flag else False)

    def set_script_speed(self, speed):
        index = 0
        for label, value in range_script_speed:
            if speed == value:
                self.script_speed_combo.set_active(index)
                break
            index += 1

    def set_surface_scale(self, scale):
        index = 0
        for label, value in range_scale:
            if scale == value:
                self.surface_scale_combo.set_active(index)
                break
            index += 1

    def set_default_balloon(self, name):
        model = self.balloon_treeview.get_model()
        listiter = model.get_iter_first()
        while listiter is not None:
            value = model.get_value(listiter, 0)
            if value == name or name is None:
                self.balloon_treeview.get_selection().select_iter(listiter)
                break
            listiter = model.iter_next(listiter) 
        else:
            listiter = model.get_iter_first()
            assert listiter is not None
            self.balloon_treeview.get_selection().select_iter(listiter)

    def set_ignore_default(self, flag):
        self.ignore_button.set_active(True if flag else False)

    def set_check_collision(self, flag):
        self.check_collision_button.set_active(True if flag else False)

    def set_allowembryo(self, flag):
        self.allowembryo_button.set_active(True if flag else False)

    def set_use_pna(self, flag):
        self.use_pna_button.set_active(True if flag else False)

    def set_sink_after_talk(self, flag):
        self.sink_after_talk_button.set_active(True if flag else False)

    def set_raise_before_talk(self, flag):
        self.raise_before_talk_button.set_active(True if flag else False)

    def set_surface_alpha(self, alpha):
        self.surface_alpha_adjustment.set_value(1.0 if alpha is None else alpha)

    def set_balloon_alpha(self, alpha):
        self.balloon_alpha_adjustment.set_value(1.0 if alpha is None else alpha)

    def set_animation_quality(self, quality):
        self.animation_quality_adjustment.set_value(1.0 if quality is None else quality)

    def set_seriko_inactive(self, flag):
        self.seriko_inactive_button.set_active(True if flag else False)


class RuleEditor:

    def __init__(self, master=None):
        self.dialog = gtk.Dialog()
        self.dialog.connect('delete_event', self.cancel)
        self.dialog.set_modal(True)
        self.dialog.set_position(gtk.WIN_POS_CENTER)
        if master is not None:
            self.dialog.set_transient_for(master)
        # entries
        table = gtk.Table(2, 2)
        table.set_row_spacings(5)
        table.set_col_spacings(5)
        table.set_border_width(10)
        self.dialog.vbox.pack_start(table)
        label = gtk.Label('Pattern')
        table.attach(label, 0, 1, 0, 1, xoptions=gtk.FILL)
        self.pattern_entry = gtk.Entry()
        self.pattern_entry.set_size_request(300, -1)
        self.pattern_entry.connect('changed', self.changed)
        table.attach(self.pattern_entry, 1, 2, 0, 1)
        label = gtk.Label('Command')
        table.attach(label, 0, 1, 1, 2, xoptions=gtk.FILL)
        self.command_entry = gtk.Entry()
        self.command_entry.set_size_request(300, -1)
        self.command_entry.connect('changed', self.changed)
        table.attach(self.command_entry, 1, 2, 1, 2)
        self.dialog.vbox.show_all()
        # buttons
        self.ok_button = gtk.Button('OK')
        self.ok_button.connect('clicked', self.ok)
        self.dialog.action_area.pack_start(self.ok_button)
        button = gtk.Button('Cancel')
        button.connect('clicked', self.cancel)
        self.dialog.action_area.pack_start(button)
        self.dialog.action_area.show_all()

    def set_pattern(self, text):
        self.pattern_entry.set_text(text)

    def get_pattern(self):
        return self.pattern_entry.get_text()

    def set_command(self, text):
        self.command_entry.set_text(text)

    def get_command(self):
        return self.command_entry.get_text()

    def changed(self, widget, event=None):
        if self.pattern_entry.get_text() and \
           self.command_entry.get_text():
            self.ok_button.set_sensitive(True)
        else:
            self.ok_button.set_sensitive(False)

    def run(self, title):
        self.dialog.set_title(title)
        self.dialog.show()
        gtk.main()
        return self.done

    def hide(self):
        self.dialog.hide()
        gtk.main_quit()

    def ok(self, widget, event=None):
        self.done = 1
        self.hide()
        return True

    def cancel(self, widget, event=None):
        self.done = 0
        self.hide()
        return True
