# -*- coding: utf-8 -*-
#
#  Copyright (C) 2004-2009 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#

# TODO:
# - 「きのこ」へのステータス送信.
# - 「きのこ」の情報の参照.
# - SERIKO/1.2ベースのアニメーション
# - (スキン側の)katochan.txt
# - balloon.txt
# - surface[0/1/2]a.txt(@ゴースト)
# - 自爆イベント
# - headrect.txt : 頭の当たり判定領域データ
#   当たり領域のleft／top／right／bottomを半角カンマでセパレートして記述.
#   1行目がsurface0、2行目がsurface1の領域データ.
#   このファイルがない場合、領域は自動計算される.
# - speak.txt
# - katochan が無い場合の処理.(本体の方のpopup menuなども含めて)
# - 設定ダイアログ : [会話/反応]タブ -> [SEND SSTP/1.1] or [SHIORI]
# - 見切れ連続20s、もしくは画面内で静止20sでアニメーション記述ミスと見なし自動的に落ちる
# - 発言中にバルーンをダブルクリックで即閉じ
# - @ゴースト名は#nameと#forには使えない. もし書いても無視されすべて有効になる
# - 連続落し不可指定
#   チェックしておくと落下物を2個以上同時に落とせなくなる
# - スキンチェンジ時も起動時のトークを行う
# - ファイルセット設定機能
#   インストールされたスキン／落下物のうち使用するものだけを選択できる
# - ターゲットのアイコン化への対応
# - アイコン化されているときは自動落下しない
# - アイコン化されているときのDirectSSTP SEND/DROPリクエストはエラー(Invisible)
# - 落下物の透明化ON/OFF
# - 落下物が猫どりふ自身にも落ちてくる
#   不在時に1/2、ランダム/全員落し時に1/10の確率で自爆
# - 一定時間間隔で勝手に物を落とす
# - ターゲット指定落し、ランダム落し、全員落し
# - 出現即ヒットの場合への対応

# - 複数ゴーストでの当たり判定.
# - 透明ウィンドウ

import os
import random

if 'DISPLAY' in os.environ:
    import gtk
    import gobject
    import ninix.pix

import ninix.home

class Menu:

    def __init__(self, nekoninni, accelgroup):
        self.__nekoninni = nekoninni
        ui_info = '''
        <ui>
          <popup name='popup'>
            <menuitem action='Settings'/>
            <menu action='Katochan'>
            </menu>
            <separator/>
            <menuitem action='Exit'/>
          </popup>
        </ui>
        '''
        self.__menu_list = {
            'settings': [('Settings', None, _('Settings...(_O)'), None,
                          '', self.__nekoninni.edit_preferences),
                         '/ui/popup/Settings'],
            'katochan': [('Katochan', None,_('Katochan(_K)'), None),
                         '/ui/popup/Katochan'],
            'exit':     [('Exit', None,_('Exit(_Q)'), None,
                          '', self.__nekoninni.close),
                         '/ui/popup/Exit'],
            }
        self.__katochan_list = None
        actions = gtk.ActionGroup('Actions')
        entry = []
        for value in self.__menu_list.itervalues():
            entry.append(value[0])
        actions.add_actions(tuple(entry))
        ui_manager = gtk.UIManager()
        ui_manager.insert_action_group(actions, 0)
        ui_manager.add_ui_from_string(ui_info)
        self.__popup_menu = ui_manager.get_widget('/ui/popup')
        for key in self.__menu_list:
            path = self.__menu_list[key][-1]
            self.__menu_list[key][1] = ui_manager.get_widget(path)

    def popup(self, button):
        katochan_list = self.__nekoninni.get_katochan_list()
        self.__set_katochan_menu(katochan_list)
        self.__popup_menu.popup(
            None, None, None, button, gtk.get_current_event_time())

    def __set_katochan_menu(self, list): ## FIXME
        key = 'katochan'
        if list:
            menu = gtk.Menu()
            for katochan in list:
                item = gtk.MenuItem(katochan['name'])
                item.connect(
                    'activate', self.__nekoninni.select_katochan, (katochan))
                menu.add(item)
                item.show()
            self.__menu_list[key][1].set_submenu(menu)
            menu.show()
            self.__menu_list[key][1].show()
        else:
            self.__menu_list[key][1].hide()

class Nekoninni:

    def __init__(self):
        self.mode = 1 # 0: SEND SSTP1.1, 1: SHIORI/2.2
        self.__running = 0
        self.skin = None
        self.katochan = None

    def notify_observer(self, event, args): ## FIXME
        if event in ['set position', 'set surface']:
            if self.skin is not None:
                self.skin.set_position()
            if self.katochan is not None and self.katochan.loaded:
                self.katochan.set_position()
        elif event == 'set scale': ## FIXME
            scale = self.target.prefs.get('surface_scale')
            if self.skin is not None:
                self.skin.set_scale(scale)
            if self.katochan is not None:
                self.katochan.set_scale(scale)
        elif event == 'finalize':
            self.finalize()
        else:
            #print 'OBSERVER(nekodorif): ignore -', event
            pass

    def load(self, dir, katochan, target):
        if not katochan:
            return 0
        self.dir = dir
        self.target = target
        self.target.set_observer(self)
        self.accelgroup = gtk.AccelGroup()
        scale = self.target.prefs.get('surface_scale')
        self.skin = Skin(self.dir, self.accelgroup, self, scale)
        if self.skin is None:
            return 0
        self.katochan_list = katochan
        self.katochan = None
        self.launch_katochan(self.katochan_list[0])
        self.__running = 1
        gobject.timeout_add(50, self.do_idle_tasks) # 50ms
        return 1

    def do_idle_tasks(self):
        if not self.__running:
            return False
        self.skin.update()
        if self.katochan is not None:
            self.katochan.update()
        #self.process_script()
        return True

    def send_event(self, event):
        if event not in ['Emerge', # 可視領域内に出現
                         'Hit',    # ヒット
                         'Drop',   # 再落下開始
                         'Vanish', # ヒットした落下物が可視領域内から消滅
                         'Dodge'   # よけられてヒットしなかった落下物が可視領域内から消滅
                         ]:
            return
        args = (self.katochan.get_name(),
                self.katochan.get_ghost_name(),
                self.katochan.get_category(),
                self.katochan.get_kinoko_flag(),
                self.katochan.get_target())
        ##print 'KATOCHAN:', args
        ##print 'TARGET:', args[4]
        self.target.notify_event('OnNekodorifObject%s' % event, *args)

    def has_katochan(self):
        if self.katochan is not None:
            return 1
        else:
            return 0

    def select_katochan(self, widget, args):
        self.launch_katochan(args)

    def drop_katochan(self):
        self.katochan.drop()

    def delete_katochan(self):
        self.katochan.destroy()
        self.katochan = None
        self.skin.reset()

    def launch_katochan(self, katochan):
        if self.katochan:
            self.katochan.destroy()
        self.katochan = Katochan(self, self.target)
        self.katochan.load(katochan)

    def edit_preferences(self, action):
        pass

    def finalize(self):
        self.__running = 0
        self.target.delete_observer(self)
        if self.katochan is not None:
            self.katochan.destroy()
        if self.skin is not None:
            self.skin.destroy()
        ##if self.balloon is not None:
        ##    self.balloon.destroy()

    def close(self, action):
        self.finalize()

    def get_katochan_list(self): ## FIXME
        return self.katochan_list

class Skin:

    def __init__(self, dir, accelgroup, nekoninni, scale):
        self.dir = dir
        self.accelgroup = accelgroup
        self.nekoninni = nekoninni
        self.__dragged = 0
        self.x_root = None
        self.y_root = None
        self.__scale = scale
        self.__menu = Menu(self.nekoninni, self.accelgroup)
        path = os.path.join(self.dir, 'omni.txt')
        if os.path.isfile(path) and os.path.getsize(path) == 0:
            self.omni = 1
        else:
            self.omni = 0
        self.window = gtk.Window()
        name, top_dir = ninix.home.read_profile_txt(dir) # XXX
        self.window.set_title(name)
        self.window.set_decorated(False)
        self.window.set_resizable(False)
        self.window.connect('delete_event', self.delete)
        self.window.connect('button_press_event', self.button_press)
        self.window.connect('button_release_event', self.button_release)
        self.window.connect('motion_notify_event', self.motion_notify)
        self.window.connect('leave_notify_event', self.leave_notify)
        self.window.set_events(gtk.gdk.BUTTON_PRESS_MASK|
                               gtk.gdk.BUTTON_RELEASE_MASK|
                               gtk.gdk.POINTER_MOTION_MASK|
                               gtk.gdk.POINTER_MOTION_HINT_MASK|
                               gtk.gdk.LEAVE_NOTIFY_MASK)
        self.window.realize()
        self.window.add_accel_group(self.accelgroup)
        self.darea = gtk.DrawingArea()
        self.darea.set_events(gtk.gdk.EXPOSURE_MASK)
        self.darea.connect('expose_event', self.redraw)
        self.id = [0, None]
        self.darea.show()
        self.window.add(self.darea)
        self.darea.realize()
        self.set_surface()
        self.set_position(reset=1)
        self.window.show()

    def set_scale(self, scale):
        self.__scale = scale
        self.set_surface()
        self.set_position()

    def redraw(self, widget, event):
        pass

    def delete(self, widget, event):
        self.nekoninni.finalize()

    def destroy(self): ## FIXME
        self.window.destroy()

    def button_press(self,  widget, event): ## FIXME
        self.x_root = event.x_root
        self.y_root = event.y_root
        if event.button == 1:
            if event.type == gtk.gdk.BUTTON_PRESS:
                ##self.window.begin_move_drag(
                ##    event.button, int(event.x_root), int(event.y_root),
                ##    gtk.get_current_event_time())
                pass
            elif event.type == gtk.gdk._2BUTTON_PRESS: # double click
                if self.nekoninni.has_katochan():
                    self.start() ## FIXME
                    self.nekoninni.drop_katochan()
        elif event.button == 3:                
            if event.type == gtk.gdk.BUTTON_PRESS:
                self.__menu.popup(event.button)
        return True

    def set_surface(self):
        if self.id[1] is not None:
            path = os.path.join(self.dir,
                                'surface%d%d.png' % (self.id[0], self.id[1]))
            if not os.path.exists(path):
                self.id[1] = None
                self.set_surface()
                return
        else:
            path = os.path.join(self.dir, 'surface%d.png' % self.id[0])
        try:
            pixbuf = ninix.pix.create_pixbuf_from_file(path)
            scale = self.__scale
            if scale != 100:
                w = max(1, pixbuf.get_width() * scale / 100)
                h = max(1, pixbuf.get_height() * scale / 100)
                pixbuf = pixbuf.scale_simple(w, h, gtk.gdk.INTERP_BILINEAR)
            image, mask = pixbuf.render_pixmap_and_mask(255)
            del pixbuf
        except:
            self.nekoninni.finalize()
            return
        self.w, self.h = image.get_size()
        self.darea.set_size_request(self.w, self.h)
        self.darea.window.set_back_pixmap(image, False)        
        self.window.shape_combine_mask(mask, 0, 0)

    def set_position(self, reset=0):
        left, top, scrn_w, scrn_h = ninix.pix.get_workarea()
        if reset:
            self.x = left
            self.y = top + scrn_h - self.h
        else:
            self.x = min(left + scrn_w - self.w, self.x)
            self.x = max(left, self.x)
            self.y = min(top + scrn_h - self.h, self.y)
            self.y = max(top, self.y)
        self.window.move(self.x, self.y)

    def move(self, x_delta, y_delta): ## FIXME
        self.x = self.x + x_delta
        if self.omni:
            self.y = self.y + y_delta
        self.set_position()

    def update(self):
        if self.id[1] is not None:
            self.id[1] += 1
        else:
            if not random.choice(range(100)): ## XXX
                self.id[1] = 0
        self.set_surface()

    def start(self): ## FIXME
        self.id[0] = 1
        self.set_surface()

    def reset(self): ## FIXME
        self.id[0] = 0
        self.set_surface()

    def button_release(self,  widget, event): ## FIXME
        if self.__dragged:
            self.__dragged = 0
            self.x_root = None
            self.y_root = None
        return True

    def motion_notify(self,  widget, event): ## FIXME
        if event.is_hint:
            x, y, state = self.darea.window.get_pointer()
        else:
            x, y, state = event.x, event.y, event.state
        if state & gtk.gdk.BUTTON1_MASK:
            if self.x_root is not None and \
               self.y_root is not None:
                x_delta = int(event.x_root - self.x_root)
                y_delta = int(event.y_root - self.y_root)
                self.__dragged = 1
                self.move(x_delta, y_delta)
                self.x_root = event.x_root
                self.y_root = event.y_root
        return True

    def leave_notify(self,  widget, event): ## FIXME
        pass

class Balloon:

    def __init__(self):
        pass

    def destroy(self): ## FIXME
        pass

class Katochan:

    CATEGORY_LIST = ['pain',      # 痛い
                     'stab',      # 刺さる
                     'surprise',  # びっくり
                     'hate',      # 嫌い、気持ち悪い
                     'huge',      # 巨大
                     'love',      # 好き、うれしい
                     'elegant',   # 風流、優雅
                     'pretty',    # かわいい
                     'food',      # 食品
                     'reference', # 見る／読むもの
                     'other'      # 上記カテゴリに当てはまらないもの
                     ]

    def __init__(self, nekoninni, target):
        self.side = 0
        self.target = target
        self.nekoninni = nekoninni
        self.settings = {}
        self.settings['state'] = 'before'
        self.settings['fall.type'] = 'gravity'
        self.settings['fall.speed'] = 1
        self.settings['slide.type'] = 'none'
        self.settings['slide.magnitude'] = 0
        self.settings['slide.sinwave.degspeed'] = 30
        self.settings['wave'] = None
        self.settings['wave.loop'] = 0
        self.__scale = 100
        self.loaded = False

    def get_name(self):
        return self.data['name']

    def get_category(self):
        return self.data['category']

    def get_kinoko_flag(self): ## FIXME
        return 0 # 0/1 = きのこに当たっていない(ない場合を含む)／当たった

    def get_target(self): ## FIXME
        if self.side == 0 :
            return self.target.get_selfname()
        else:
            return self.target.get_keroname()

    def get_ghost_name(self):
        if 'for' in self.data: # 落下物が主に対象としているゴーストの名前
            return self.data['for']
        else:
            return ''

    def destroy(self):
        self.window.destroy()

    def delete(self, widget, event):
        self.destroy()

    def redraw(self, widget, event):
        pass

    def set_movement(self, timing):
        key = ''.join((timing, 'fall.type'))
        if key in self.data and \
           self.data[key] in ['gravity', 'evenspeed', 'none']:
            self.settings['fall.type'] = self.data[key]
        else:
            self.settings['fall.type'] = 'gravity'
        self.settings['fall.speed'] = self.data.get(
            ''.join((timing, 'fall.speed')), 1)
        if self.settings['fall.speed'] < 1:
            self.settings['fall.speed'] = 1
        if self.settings['fall.speed'] > 100:
            self.settings['fall.speed'] = 100
        key = ''.join((timing, 'slide.type'))
        if key in self.data and \
           self.data[key] in ['none', 'sinwave', 'leaf']:
            self.settings['slide.type'] = self.data[key]
        else:
            self.settings['slide.type'] = 'none'
        self.settings['slide.magnitude'] = self.data.get(
            ''.join((timing, 'slide.magnitude')), 0)
        self.settings['slide.sinwave.degspeed'] = self.data.get(
            ''.join((timing, 'slide.sinwave.degspeed')), 30)
        self.settings['wave'] = self.data.get(''.join((timing, 'wave')), None)
        if self.data.get(''.join((timing, 'wave.loop'))) == 'on':
            self.settings['wave.loop'] = 1
        else:
            self.settings['wave.loop'] = 0

    def set_scale(self, scale):
        self.__scale = scale
        self.set_surface()
        self.set_position()

    def set_position(self):
        if self.settings['state'] != 'before':
            return
        target_x, target_y = self.target.get_surface_position(self.side)
        target_w, target_h = self.target.get_surface_size(self.side)
        scale = self.__scale
        left, top, scrn_w, scrn_h = ninix.pix.get_workarea()
        self.x = target_x + target_w / 2 - self.w / 2 + \
                 self.offset_x * scale / 100
        self.y = top + self.offset_y * scale / 100
        self.window.move(self.x, self.y)

    def set_surface(self):
        path = os.path.join(self.data['dir'], 'surface%d.png' % self.id)
        try:
            pixbuf = ninix.pix.create_pixbuf_from_file(path)
            scale = self.__scale
            if scale != 100:
                w = pixbuf.get_width() * scale / 100
                h = pixbuf.get_height() * scale / 100
                pixbuf = pixbuf.scale_simple(w, h, gtk.gdk.INTERP_BILINEAR)
            image, mask = pixbuf.render_pixmap_and_mask(255)
            del pixbuf
        except:
            self.nekoninni.finalize()
            return
        self.w, self.h = image.get_size()
        self.darea.set_size_request(self.w, self.h)
        self.darea.window.set_back_pixmap(image, False)
        self.window.shape_combine_mask(mask, 0, 0)
        self.darea.queue_draw_area(0, 0, self.w, self.h)

    def load(self, data):
        self.data = data
        self.__scale = self.target.prefs.get('surface_scale')
        self.set_state('before')
        ##print 'DATA:', self.data
        if 'category' in self.data:
            category = self.data['category'].split(',')
            if category:
                if category[0] not in self.CATEGORY_LIST:
                    print 'WARNING: unknown major category - ', category[0]
                    ##self.data['category'] = self.CATEGORY_LIST[-1]
            else:
                self.data['category'] = self.CATEGORY_LIST[-1]
        else:
            self.data['category'] = self.CATEGORY_LIST[-1]
        if 'target' in self.data:
            if self.data['target'] == 'sakura':
                self.side = 0
            elif self.data['target'] == 'kero':
                self.side = 1
            else:
                self.side = 0
        else:
            self.side = 0
        if self.nekoninni.mode == 1:
            self.nekoninni.send_event('Emerge')
        else:
            if 'before.script' in self.data:
                pass ## FIXME
            else:
                pass ## FIXME
        self.set_movement('before')
        if 'before.appear.direction' in self.data:
            pass ## FIXME
        else:
            pass ## FIXME
        offset_x =  self.data.get('before.appear.ofset.x', 0)
        if offset_x < -32768:
            offset_x = -32768
        if offset_x > 32767:
            offset_x = 32767
        offset_y =  self.data.get('before.appear.ofset.y', 0)
        if offset_y < -32768:
            offset_y = -32768
        if offset_y > 32767:
            offset_y = 32767
        self.offset_x = offset_x
        self.offset_y = offset_y
        self.window = gtk.Window()
        self.window.set_title(self.data['name'])
        self.window.set_decorated(False)
        self.window.set_resizable(False)
        self.window.set_skip_taskbar_hint(True) # XXX
        self.window.connect('delete_event', self.delete)
        self.darea = gtk.DrawingArea()
        self.darea.set_events(gtk.gdk.EXPOSURE_MASK)
        self.darea.connect('expose_event', self.redraw)
        self.darea.show()
        self.window.add(self.darea)
        self.darea.realize()
        self.window.show()
        self.id = 0
        self.set_surface()
        self.set_position()
        self.loaded = True

    def drop(self): ## FIXME
        self.set_state('fall')

    def set_state(self, state):
        self.settings['state'] = state
        self.time = 0
        self.hit = 0
        self.hit_stop = 0

    def update_surface(self): ## FIXME
        pass

    def update_position(self): ## FIXME
        ##print 'MOVE:', self.settings
        if self.settings['slide.type'] == 'leaf':
            pass
        else:
            if self.settings['fall.type'] == 'gravity':
                self.y += int(self.settings['fall.speed'] * \
                              (self.time / 20.0)**2)
            elif  self.settings['fall.type'] == 'evenspeed':
                self.y += self.settings['fall.speed']
            else:
                pass
            if self.settings['slide.type'] == 'sinwave':
                pass ## FIXME
            else:
                pass
        ##print 'POSITION:', self.x, self.y
        self.window.move(self.x, self.y)

    def check_collision(self): ## FIXME: check self position
        for side in [0, 1]:
            target_x, target_y = self.target.get_surface_position(side)
            target_w, target_h = self.target.get_surface_size(side)
            center_x = self.x + self.w / 2
            center_y = self.y + self.h / 2
            if target_x < center_x < target_x + target_w and \
               target_y < center_y < target_y + target_h:
                self.side = side
                return 1
        else:
            return 0

    def check_mikire(self):
        left, top, scrn_w, scrn_h = ninix.pix.get_workarea()
        if self.x + self.w - self.w / 3 > left + scrn_w or \
                self.x + self.w / 3 < left or \
                self.y + self.h - self.h / 3 > top + scrn_h or \
                self.y + self.h / 3 < top:
            return 1
        else:
            return 0

    def update(self): ## FIXME
        if self.settings['state'] == 'fall':
            self.update_surface()
            self.update_position()
            if self.check_collision():
                self.set_state('hit')
                self.hit = 1
                if self.nekoninni.mode == 1:
                    self.id = 1
                    self.set_surface()
                    self.nekoninni.send_event('Hit')
                else:
                    pass ## FIXME
            if self.check_mikire():
                self.set_state('dodge')
        elif self.settings['state'] == 'hit':
            if self.hit_stop >= self.data.get('hit.waittime', 0):
                self.set_state('after')
                self.set_movement('after')
                if self.nekoninni.mode == 1:
                    self.id = 2
                    self.set_surface()
                    self.nekoninni.send_event('Drop')
                else:
                    pass ## FIXME
            else:
                self.hit_stop += 1
                self.update_surface()
        elif self.settings['state'] == 'after':
            self.update_surface()
            self.update_position()
            if self.check_mikire():
                self.set_state('end')
        elif self.settings['state'] == 'end':
            if self.nekoninni.mode == 1:
                self.nekoninni.send_event('Vanish')
            else:
                pass ## FIXME
            self.nekoninni.delete_katochan()
            return False
        elif self.settings['state'] == 'dodge':
            if self.nekoninni.mode == 1:
                self.nekoninni.send_event('Dodge')
            else:
                pass ## FIXME
            self.nekoninni.delete_katochan()
            return False
        else:
            pass ## check collision and mikire
        self.time += 1
        return True
