#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
#  ninix-install.py - a command-line installer for ninix
#  Copyright (C) 2001, 2002 by Tamito KAJIYAMA
#  Copyright (C) 2002, 2003 by MATSUMURA Namihiko <nie@counterghost.net>
#  Copyright (C) 2002-2009 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#  Copyright (C) 2003 by Shun-ichi TAHARA <jado@flowernet.gr.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#

# env.var: NINIX_HOME, NINIX_USER, NINIX_ARCHIVE, TMPDIR
# command: cp, mv, rm, mkdir, lha, zip, find, chmod


import getopt
import os
import re
import socket
import sys
import tempfile
import urllib

import gettext
gettext.install('ninix')

if 'DISPLAY' in os.environ:
    del os.environ['DISPLAY']

import ninix.alias
import ninix.config
import ninix.home
import ninix.version

PROGRAM_NAME = os.path.basename(sys.argv[0])

USAGE = '''\
Usage: %s [options] [file ...]
Options:
  -g, --ghost            assume that all files are ghosts
  -s, --shell            assume that all files are shells
  -b, --balloon          assume that all files are balloons
  -P, --plugin           assume that all files are plug-ins
  -K, --kinoko           assume that all files are kinoko skins
  -N, --nekoninni        assume that all files are nekodorif skins
  -D, --katochan         assume that all files are nekodorif katochans
  -S, --supplement NAME  the name of the supplement target
  -i, --interactive      prompt before overwrite
  -d, --download         download files into the archive directory
  -H, --homedir DIR      ninix home directory (default: ~/.ninix)
  -U, --userdir DIR      user data directory (default: ninix home directory)
  -A, --arcdir DIR       archive directory (default: ~/.ninix/archive)
  -T, --tempdir DIR      temporary directory (default: Python default)
  -r, --reload           send "reload" request to local SSTP server
  -p, --port NUM         port number for SSTP connection (default: 9801)
  -L, --lower            lower file names with upper case letters
  -h, --help             show this message
'''

def usage():
    raise SystemExit, USAGE % PROGRAM_NAME


class InstallError(Exception):
    pass


def fatal(error):
    raise InstallError, error

# mode masks
INTERACTIVE = 1
DOWNLOAD    = 2

def main():
    global mode, arcdir
    try:
        options, files = getopt.getopt(
            sys.argv[1:], 'gsbS:PKNDRridxH:U:A:T:rp:q:Lh',
            ['ghost', 'shell', 'balloon', 'supplement=', 'plugin',
             'kinoko', 'nekoninni', 'katochan',
             'homedir=', 'userdir=', 'arcdir=', 'tempdir=',
             'interactive', 'download', 'reload', 'port=',
             'lower', 'help'])
    except getopt.error, e:
        sys.stderr.write('Error: %s\n' % str(e))
        usage()
    homedir = os.environ.get('NINIX_HOME')
    userdir = os.environ.get('NINIX_USER')
    arcdir = os.environ.get('NINIX_ARCHIVE', '~/.ninix/archive')
    tempdir = None
    filetype = None
    target = None
    mode = 0
    to_lower = 0
    to_reload = 0
    host = ''
    port = 9801
    for opt, val in options:
        if opt in ['-h', '--help']:
            usage()
        elif opt in ['-g', '--ghost']:
            filetype = 'ghost'
        elif opt in ['-s', '--shell']:
            filetype = 'shell'
        elif opt in ['-b', '--balloon']:
            filetype = 'balloon'
        elif opt in ['-P', '--plugin']:
            filetype = 'plugin'
        elif opt in ['-K', '--kinoko']:
            filetype = 'kinoko'
        elif opt in ['-N', '--nekoninni']:
            filetype = 'nekoninni'
        elif opt in ['-D', '--katochan']:
            filetype = 'katochan'
        elif opt in ['-S', '--supplement']:
            filetype = 'supplement'
            target = val
        elif opt in ['-H', '--homedir']:
            homedir = val
        elif opt in ['-U', '--userdir']:
            userdir = val
        elif opt in ['-A', '--arcdir']:
            arcdir = val
        elif opt in ['-T', '--tempdir']:
            tempdir = val
        elif opt in ['-i', '--interactive']:
            mode = mode | INTERACTIVE
        elif opt in ['-d', '--download']:
            mode = mode | DOWNLOAD
        elif opt in ['-r', '--reload']:
            to_reload = 1
        elif opt in ['-p', '--port']:
            port = int(val)
        elif opt in ['-L', '--lower']:
            to_lower = 1
    if homedir:
        ninix.home.NINIX_HOME = homedir
    if userdir:
        ninix.home.NINIX_USER = userdir
    if tempdir:
        if os.path.exists(tempdir):
            tempfile.tempdir = os.path.expanduser(tempdir)
        else:
            fatal('%s: no such directory' % tempdir)
    arcdir = os.path.expanduser(arcdir)
    if to_lower:
        n = 0
        homedir = ninix.home.get_ninix_home()
        for data in ['ghost', 'balloon']:
            try:
                buf = os.listdir(os.path.join(homedir, data))
            except OSError:
                continue
            for subdir in buf:
                path = os.path.join(homedir, data, subdir)
                if os.path.isdir(path):
                    n += lower_files(path)
        if n > 0:
            print n, 'files renamed'
    buf = []
    for filename in files:
        if filename.startswith('@'):
            try:
                lines = open(filename[1:]).readlines()
            except IOError, e:
                sys.stderr.write("cannot read '%s' (skipped)\n" % filename[1:])
            else:
                for line in [line.strip() for line in lines]:
                    if line and not line.startswith('#'):
                        buf.append(line)
        else:
            buf.append(filename)
    for filename in buf:
        try:
            install(filename, filetype, target, ninix.home.get_ninix_home())
        except InstallError, error:
            sys.stderr.write('%s: %s (skipped)\n' % (filename, error))
    if to_reload:
        reload_(host, port)

def escape(s):
    for c in ['"', '`']:
        s = s.replace(c, ''.join(('\\', c)))
    return ''.join(('"', s, '"'))

def install(filename, filetype, target, homedir):
    # check archive format
    if filename.lower().endswith('.nar') or \
       filename.lower().endswith('.zip'):
        archiver = 'unzip -o %(f)s -d %(d)s'
    elif filename.lower().endswith('.lzh'):
        archiver = 'lha xfw=%(d)s %(f)s'
    else:
        fatal('unknown archive format')
    # extract files from the archive
    tmpdir = tempfile.mkdtemp('ninix')
    if not run('mkdir -p %s' % escape(tmpdir)):
        fatal('cannot make temporary directory')
    url = None
    if filename.startswith('http:') or filename.startswith('ftp:'):
        url = filename
        filename = download(filename, tmpdir)
        if filename is None:
            os.system('rm -rf %s' % escape(tmpdir))
            fatal('cannot download the archive file')
    if not run(archiver % {'f': escape(filename), 'd': escape(tmpdir)}):
        os.system('rm -rf %s' % escape(tmpdir))
        fatal('cannot extract files from the archive')
    if url and not mode & DOWNLOAD:
        run('rm %s' % escape(filename))
    os.system('find %s -type d -exec chmod u+rwx {} \\;' % escape(tmpdir))
    os.system('find %s -type f -exec chmod u+rw  {} \\;' % escape(tmpdir))
    rename_files(tmpdir)
    # check the file type (ghost, shell, balloon or supplement)
    error = None
    inst = ninix.home.read_install_txt(tmpdir)
    if inst:
        guess = inst.get('type')
        if guess == 'ghost':
            if os.path.exists(os.path.join(tmpdir, 'ghost', 'master')):
                guess = 'ghost.redo'
            else:
                guess = 'ghost.inverse'
        elif guess == 'ghost with balloon':
            guess = 'ghost.inverse'
        elif guess in ['shell', 'shell with balloon']:
            if 'accept' in inst:
                guess = 'supplement'
            else:
                guess = 'shell.inverse'
        elif guess == 'skin':
            guess = 'nekoninni'
        elif guess == 'katochan':
            guess = 'katochan'
    else:
        guess = None
        for f, t in [('ghost/master/', 'ghost.redo'),
                     ('shell/',        'shell.redo'),
                     ('kawari.ini',    'ghost.inverse'),
                     ('ai.txt',        'ghost.inverse'),
                     ('ai.dtx',        'ghost.inverse'),
                     ('surface/',      'shell.inverse'),
                     ('balloons0.png', 'balloon'),
                     ('balloons0.jpg', 'balloon'),
                     ('plugin.txt',    'plugin'),
                     ('kinoko.ini',    'kinoko')]:
            if os.path.exists(os.path.join(tmpdir, f)):
                guess = t
                break
    if filetype is None and guess is None:
        error = 'cannot guess the file type; use option -g, -s, -b, -P or -S'
    elif filetype is None:
        filetype = guess
    elif filetype == 'ghost' and guess in ['ghost.redo', 'ghost.inverse']:
        filetype = guess
    elif filetype == 'shell' and guess in ['ghost.redo', 'shell.redo',
                                           'supplement']:
        filetype = 'shell.redo'
    elif filetype == 'shell' and guess == 'ghost.inverse':
        filetype = 'shell.inverse'
    elif guess is not None and filetype != guess:
        error = ''.join(('wrong file type option; correct file type is ',
                         guess))
    try:
        if error is not None:
            fatal(error)
        elif filetype == 'ghost.redo':
            install_redo_ghost(filename, tmpdir, homedir)
        elif filetype == 'shell.redo':
            install_redo_shell(filename, tmpdir, homedir)
        elif filetype == 'supplement':
            install_redo_supplement(filename, tmpdir, homedir, target)
        elif filetype == 'ghost.inverse':
            install_inverse_ghost(filename, tmpdir, homedir)
        elif filetype == 'shell.inverse':
            install_inverse_shell(filename, tmpdir, homedir)
        elif filetype == 'balloon':
            install_balloon(filename, tmpdir, homedir)
        elif filetype == 'plugin':
            install_plugin(filename, tmpdir, homedir)
        elif filetype == 'kinoko':
            install_kinoko(filename, tmpdir, homedir)
        elif filetype == 'nekoninni':
            install_nekoninni(filename, tmpdir, homedir)
        elif filetype == 'katochan':
            install_katochan(filename, tmpdir, homedir)
    finally:
        run('rm -rf %s' % escape(tmpdir))


class URLopener(urllib.FancyURLopener):
    version = 'ninix/%s' % ninix.version.VERSION


urllib._urlopener = URLopener()

def download(url, basedir):
    print 'downloading', url
    try:
        ifile = urllib.urlopen(url)
    except IOError:
        return None
    headers = ifile.info()
    if 'content-length' in headers:
        print '(size = %s bytes)' % headers.get('content-length')
    if mode & DOWNLOAD:
        if not os.path.exists(arcdir):
            run('mkdir -p %s' % escape(arcdir))
        basedir = arcdir
    filename = os.path.join(basedir, os.path.basename(url))
    try:
        ofile = open(filename, 'w')
    except IOError:
        return None
    while 1:
        data = ifile.read(4096)
        if not data:
            break
        ofile.write(data)
    ifile.close()
    ofile.close()
    # check the format of the downloaded file
    if filename.lower().endswith('.nar') or \
       filename.lower().endswith('.zip'):
        archiver = 'unzip -t'
    elif filename.lower().endswith('.lzh'):
        archiver = 'lha t'
    else:
        fatal('unknown archive format')
    if os.system('%s %s >/dev/null 2>&1' % (archiver, escape(filename))):
        return None
    return filename

def rename_files(basedir):
    for filename in os.listdir(basedir):
        filename2 = filename.lower()
        path = os.path.join(basedir, filename2)
        if filename != filename2:
            os.rename(os.path.join(basedir, filename), path)
        if os.path.isdir(path):
            rename_files(path)

def list_all_directories(top, basedir):
    dirlist = []
    for path in os.listdir(os.path.join(top, basedir)):
        if os.path.isdir(os.path.join(top, basedir, path)):
            dirlist.extend(list_all_directories(top, os.path.join(basedir, path)))
            dirlist.append(os.path.join(basedir, path))
    return dirlist

def remove_files_and_dirs(mask, target_dir):
    path = os.path.abspath(target_dir)
    if not os.path.isdir(path):
        return
    os.path.walk(path, remove_files, mask)
    dirlist = list_all_directories(path, '')
    dirlist.sort()
    dirlist.reverse()
    for name in dirlist:
        current_path = os.path.join(path, name)
        if os.path.isdir(current_path):
            head, tail = os.path.split(current_path)
            if tail not in mask and not os.listdir(current_path):
                run('rm -rf %s' % escape(current_path))

def remove_files(mask, top_dir, filelist):
    for name in filelist:
        path = os.path.join(top_dir, name)
        if os.path.isdir(path) or name in mask:
            pass
        else:
            run('rm -f %s' % escape(path))

def lower_files(top_dir):
    n = 0
    for filename in os.listdir(top_dir):
        filename2 = filename.lower()
        path = os.path.join(top_dir, filename2)
        if filename != filename2:
            os.rename(os.path.join(top_dir, filename), path)
            print 'renamed', os.path.join(top_dir, filename)
            n += 1
        if os.path.isdir(path):
            n += lower_files(path)
    return n

def confirm_overwrite(path):
    if not mode & INTERACTIVE:
        return 1
    print ''.join((PROGRAM_NAME, ': overwrite "%s"? (yes/no)' % path))
    try:
        answer = raw_input()
    except EOFError:
        answer = None
    except KeyboardInterrupt:
        raise SystemExit
    if not answer:
        return 0
    return answer.lower().startswith('y')

def confirm_removal(path):
    if not mode & INTERACTIVE:
        return 1
    print ''.join((PROGRAM_NAME, ': remove "%s"? (yes/no)' % path))
    try:
        answer = raw_input()
    except EOFError:
        answer = None
    except KeyboardInterrupt:
        raise SystemExit
    if not answer:
        return 0
    return answer.lower().startswith('y')

def uninstall_plugin(homedir, name):
    try:
        dirlist = os.listdir(os.path.join(homedir, 'plugin'))
    except OSError:
        return
    for subdir in dirlist:
        path = os.path.join(homedir, 'plugin', subdir)
        plugin = ninix.home.read_plugin_txt(path)
        if plugin is None:
            continue
        plugin_name, plugin_dir, startup, menu_items = plugin
        if plugin_name == name:
            plugin_dir = os.path.join(homedir, 'plugin', subdir)
            if confirm_removal(plugin_dir):
                run('rm -rf %s' % escape(plugin_dir))

def install_redo_ghost(archive, tmpdir, homedir):
    global mode
    # find install.txt
    inst = ninix.home.read_install_txt(tmpdir)
    if inst is None:
        fatal('install.txt not found')
    target_dir = inst.get('directory')
    if target_dir is None:
        fatal('"directory" not found in install.txt')
    target_dir = target_dir.encode('utf-8')
    prefix = os.path.join(homedir, 'ghost', target_dir)
    ghost_src = os.path.join(tmpdir, 'ghost', 'master')
    shell_src = os.path.join(tmpdir, 'shell')
    ghost_dst = os.path.join(prefix, 'ghost', 'master')
    shell_dst = os.path.join(prefix, 'shell')
    filelist = []
    filelist.append((os.path.join(tmpdir, 'install.txt'),
                     os.path.join(prefix, 'install.txt')))
    # find ghost/master/descript.txt
    desc = ninix.home.read_descript_txt(ghost_src)
    if desc:
        filelist.append((os.path.join(ghost_src, 'descript.txt'),
                         os.path.join(ghost_dst, 'descript.txt')))
    # find ghost/master/alias.txt
    path = os.path.join(ghost_src, 'alias.txt')
    if os.path.exists(path):
        filelist.append((path, os.path.join(ghost_dst, 'alias.txt')))
        aliases = ninix.alias.open(path)
    else:
        aliases = {}
    # find pseudo AI
    pseudo_ai = find_pseudo_ai(ghost_src, ghost_dst)
    if pseudo_ai is None:
        fatal('pseudo AI not found - cannot be used as a ghost')
    filelist.extend(pseudo_ai)
    # XXX: ad hoc work around for Mayura v3.10
    path = os.path.join(ghost_src, 'makotob.dll')
    if os.path.exists(path):
        os.system('mv %s %s' % (escape(path),
                                escape(os.path.join(ghost_src, 'makoto.dll'))))
    # find makoto.dll
    for filename in aliases.get('makoto', ['makoto.dll']):
        path = os.path.join(ghost_src, filename)
        try:
            data = open(path).read()
        except IOError:
            continue
        if data.find('Makoto Basic with Select and Repeat') > 0:
            os.system('rm -f %s' % escape(path))
            os.system('touch %s' % escape(path))
            filelist.append((path, os.path.join(ghost_dst, filename)))
    # find shell
    shell = find_redo_shell(shell_src, shell_dst)
    if shell:
        filelist.extend(shell)
    # find balloon
    balloon_dir = inst and inst.get('balloon.directory')
    if balloon_dir:
        balloon_dir = balloon_dir.encode('utf-8').lower()
        filelist.extend(find_balloon(os.path.join(tmpdir,    balloon_dir),
                                     os.path.join(ghost_dst, balloon_dir)))
    if os.path.exists(prefix):
        inst_dst = ninix.home.read_install_txt(prefix)
        if not inst_dst or inst_dst.get('name') != inst.get('name'):
            mode |= INTERACTIVE
        if inst.getint('refresh', 0):
            # uninstall older versions of the ghost
            if confirm_removal(prefix):
                mask = inst.get('refreshundeletemask', '').lower().split(':')
                mask.append('HISTORY')
                remove_files_and_dirs(mask, prefix)
            else:
                return
        else:
            if not confirm_overwrite(prefix):
                return
    # install files
    print 'installing', archive, '(ghost)'
    install_files(filelist)

def install_redo_shell(archive, tmpdir, homedir):
    global mode
    # find install.txt
    inst = ninix.home.read_install_txt(tmpdir)
    if inst is None:
        fatal('install.txt not found')
    target_dir = inst.get('directory')
    if target_dir is None:
        fatal('"directory" not found in install.txt')
    target_dir = target_dir.encode('utf-8')
    prefix = os.path.join(homedir, 'ghost', target_dir)
    ghost_src = os.path.join(tmpdir, 'ghost', 'master')
    shell_src = os.path.join(tmpdir, 'shell')
    ghost_dst = os.path.join(prefix, 'ghost', 'master')
    shell_dst = os.path.join(prefix, 'shell')
    filelist = []
    filelist.append((os.path.join(tmpdir, 'install.txt'),
                     os.path.join(prefix, 'install.txt')))
    # find ghost/master/descript.txt
    desc = ninix.home.read_descript_txt(ghost_src)
    if desc:
        filelist.append((os.path.join(ghost_src, 'descript.txt'),
                         os.path.join(ghost_dst, 'descript.txt')))
    # find shell
    shell = find_redo_shell(shell_src, shell_dst)
    if shell:
        filelist.extend(shell)
    # find balloon
    balloon_dir = inst and inst.get('balloon.directory')
    if balloon_dir:
        balloon_dir = balloon_dir.encode('utf-8').lower()
        filelist.extend(find_balloon(os.path.join(tmpdir,    balloon_dir),
                                     os.path.join(ghost_dst, balloon_dir)))
    if os.path.exists(prefix):
        inst_dst = ninix.home.read_install_txt(prefix)
        if not inst_dst or inst_dst.get('name') != inst.get('name'):
            mode |= INTERACTIVE
        if inst.getint('refresh', 0):
            # uninstall older versions of the ghost
            if confirm_removal(prefix):
                mask = inst.get('refreshundeletemask', '').lower().split(':')
                mask.append('HISTORY')
                remove_files_and_dirs(mask, prefix)
            else:
                return
        else:
            if not confirm_overwrite(prefix):
                return
    # install files
    print 'installing', archive, '(shell)'
    install_files(filelist)

def install_redo_supplement(archive, tmpdir, homedir, target):
    inst = ninix.home.read_install_txt(tmpdir)
    if inst and 'accept' in inst:
        print 'searching supplement target ...',
        candidates = []
        try:
            dirlist = os.listdir(os.path.join(homedir, 'ghost'))
        except OSError:
            dirlist = []
        for dirname in dirlist:
            path = os.path.join(homedir, 'ghost', dirname)
            if os.path.exists(os.path.join(path, 'shell', 'surface.txt')):
                continue
            desc = ninix.home.read_descript_txt(
                os.path.join(path, 'ghost', 'master'))
            if desc and desc.get('sakura.name') == inst.get('accept'):
                candidates.append(dirname)
        if not candidates:
            print 'not found'
        elif target and target not in candidates:
            print "'%s' not found" % target
            return
        elif len(candidates) == 1 or target in candidates:
            if target:
                path = os.path.join(homedir, 'ghost', target)
            else:
                path = os.path.join(homedir, 'ghost', candidates[0])
            if 'directory' in inst:
                if inst.get('type') == 'shell':
                    path = os.path.join(path, 'shell', inst['directory'])
                else:
                    if 'type' not in inst:
                        print 'supplement type not specified'
                    else:
                        print 'unsupported supplement type:', inst['type']
                    return
            print 'found'
            if not os.path.exists(path):
                run('mkdir -p %s' % escape(path))
            run('rm -f %s' % escape(os.path.join(tmpdir, 'install.txt')))
            run('cp -r %s/* %s' % (escape(tmpdir), escape(path)))
            return
        else:
            print 'multiple candidates found'
            for candidate in candidates:
                print '   ', candidate
            fatal('try -S option with a target ghost name')
    # install supplement as a stand alone shell
    install_redo_shell(archive, tmpdir, homedir)

def find_redo_shell(srcdir, dstdir):
    if not os.path.exists(srcdir):
        return None
    filelist = []
    for dirname in os.listdir(srcdir):
        shell_src = os.path.join(srcdir, dirname)
        shell_dst = os.path.join(dstdir, dirname)
        if not os.path.isdir(shell_src):
            continue
        path = os.path.join(shell_src, 'surfaces.txt')
        if os.path.exists(path):
            filelist.append((path, os.path.join(shell_dst, 'surfaces.txt')))
            for key, config in ninix.home.read_surfaces_txt(shell_src):
                if key.startswith('surface'):
                    for e in find_surface_elements(shell_src, shell_dst,
                                                   config):
                        if e not in filelist:
                            filelist.append(e)
        desc = ninix.home.read_descript_txt(shell_src)
        if desc:
            filelist.append((os.path.join(shell_src, 'descript.txt'),
                             os.path.join(shell_dst, 'descript.txt')))
            for name, default in [('readme', 'readme.txt'),
                                  ('menu.sidebar.bitmap.filename',
                                   'menu_sidebar.png'),
                                  ('menu.background.bitmap.filename',
                                   'menu_background.png'),
                                  ('menu.foreground.bitmap.filename',
                                   'menu_foreground.png')]:
                name = desc.get(name, default).replace('\\', '/')
                name = name.encode('utf-8')
                path = os.path.join(shell_src, name)
                if os.path.exists(path):
                    filelist.append((path, os.path.join(shell_dst, name)))
        for name in ['alias.txt', 'thumbnail.png']:
            path = os.path.join(shell_src, name)
            if os.path.exists(path):
                filelist.append((path, os.path.join(shell_dst, name)))
        filelist.extend(find_surface_set(shell_src, shell_dst))
    return filelist

def find_surface_elements(shell_src, shell_dst, config):
    filelist = []
    for key, method, filename, x, y in \
            ninix.home.list_surface_elements(config):
        filename = filename.lower()
        path = os.path.join(shell_src, filename)
        basename, suffix = os.path.splitext(filename)
        if not os.path.exists(path):
            path = os.path.join(shell_src, ''.join((basename, '.dgp')))
            if os.path.exists(path):
                filename = ''.join((basename, '.dgp'))
            else:
                path = os.path.join(shell_src, ''.join((basename, '.ddp')))
                if os.path.exists(path):
                    filename = ''.join((basename, '.ddp'))
                else:
                    print '%s file not found: %s' % (key, filename)
                    continue
        if suffix not in ['.png', '.dgp', '.ddp']:
            print 'unsupported image format for %s: %s' % (key, filename)
            continue
        filelist.append((path, os.path.join(shell_dst, filename)))
        filename = ''.join((basename, '.pna'))
        path = os.path.join(shell_src, filename)
        if os.path.exists(path):
            filelist.append((path, os.path.join(shell_dst, filename)))
    return filelist

def install_inverse_ghost(archive, tmpdir, homedir):
    global mode
    # find install.txt
    inst = ninix.home.read_install_txt(tmpdir)
    if inst is None:
        fatal('install.txt not found')
    target_dir = inst.get('directory')
    if target_dir is None:
        fatal('"directory" not found in install.txt')
    target_dir = target_dir.encode('utf-8')
    prefix = os.path.join(homedir, 'ghost', target_dir)
    ghost_dst = os.path.join(prefix, 'ghost', 'master')
    shell_dst = os.path.join(prefix, 'shell')
    filelist = []
    filelist.append((os.path.join(tmpdir, 'install.txt'),
                     os.path.join(prefix, 'install.txt')))
    # find descript.txt
    desc = ninix.home.read_descript_txt(tmpdir)
    if desc:
        filelist.append((os.path.join(tmpdir,    'descript.txt'),
                         os.path.join(ghost_dst, 'descript.txt')))
    # find pseudo AI dictionaries
    pseudo_ai = find_pseudo_ai(tmpdir, ghost_dst)
    if pseudo_ai is None:
        fatal('pseudo AI not found - cannot be used as a ghost')
    filelist.extend(pseudo_ai)
    # find makoto.dll
    path = os.path.join(tmpdir, 'makoto.dll')
    try:
        data = open(path).read()
    except IOError:
        pass
    else:
        if data.find('Makoto Basic with Select and Repeat') > 0:
            os.system('rm -f %s' % escape(path))
            os.system('touch %s' % escape(path))
            filelist.append((path, os.path.join(ghost_dst, 'makoto.dll')))
    # find shell
    shell = find_inverse_shell(tmpdir, shell_dst)
    if shell:
        filelist.extend(shell)
    # find balloon
    balloon_dir = inst and inst.get('balloon.directory')
    if balloon_dir:
        balloon_dir = balloon_dir.encode('utf-8').lower()
        filelist.extend(find_balloon(os.path.join(tmpdir,    balloon_dir),
                                     os.path.join(ghost_dst, balloon_dir)))
    if os.path.exists(prefix):
        inst_dst = ninix.home.read_install_txt(prefix)
        if not inst_dst or inst_dst.get('name') != inst.get('name'):
            mode |= INTERACTIVE
        if inst.getint('refresh', 0):
            # uninstall older versions of the ghost
            if confirm_removal(prefix):
                mask = inst.get('refreshundeletemask', '').lower().split(':')
                mask.append('HISTORY')
                remove_files_and_dirs(mask, prefix)
            else:
                return
        else:
            if not confirm_overwrite(prefix):
                return
    # install files
    print 'installing', archive, '(ghost)'
    install_files(filelist)

def install_inverse_shell(archive, tmpdir, homedir):
    global mode
    # find install.txt
    inst = ninix.home.read_install_txt(tmpdir)
    if inst is None:
        fatal('install.txt not found')
    target_dir = inst.get('directory')
    if target_dir is None:
        fatal('"directory" not found in install.txt')
    target_dir = target_dir.encode('utf-8')
    prefix = os.path.join(homedir, 'ghost', target_dir)
    ghost_dst = os.path.join(prefix, 'ghost', 'master')
    shell_dst = os.path.join(prefix, 'shell')
    filelist = []
    filelist.append((os.path.join(tmpdir, 'install.txt'),
                     os.path.join(prefix, 'install.txt')))
    # find descript.txt
    desc = ninix.home.read_descript_txt(tmpdir)
    if desc:
        filelist.append((os.path.join(tmpdir,    'descript.txt'),
                         os.path.join(ghost_dst, 'descript.txt')))
    # find shell
    shell = find_inverse_shell(tmpdir, shell_dst)
    if shell:
        filelist.extend(shell)
    # find balloon
    balloon_dir = inst and inst.get('balloon.directory')
    if balloon_dir:
        balloon_dir = balloon_dir.encode('utf-8').lower()
        filelist.extend(find_balloon(os.path.join(tmpdir,    balloon_dir),
                                     os.path.join(ghost_dst, balloon_dir)))
    if os.path.exists(prefix):
        inst_dst = ninix.home.read_install_txt(prefix)
        if not inst_dst or inst_dst.get('name') != inst.get('name'):
            mode |= INTERACTIVE
        if inst.getint('refresh', 0):
            # uninstall older versions of the shell
            if confirm_removal(prefix):
                mask = inst.get('refreshundeletemask', '').lower().split(':')
                mask.append('HISTORY')
                remove_files_and_dirs(mask, prefix)
            else:
                return
        else:
            if not confirm_overwrite(prefix):
                return
    # install files
    print 'installing', archive, '(shell)'
    install_files(filelist)

def find_inverse_shell(srcdir, dstdir):
    filelist = []
    # find descript.txt
    path = os.path.join(srcdir, 'descript.txt')    
    if os.path.exists(path):
        descript_txt = path
    else:
        descript_txt = None
    # find alias.txt
    path = os.path.join(srcdir, 'alias.txt')
    if os.path.exists(path):
        alias_txt = path
    else:
        alias_txt = None
    # find surface sets
    dirlist = []
    path = os.path.join(srcdir, 'surface.txt')
    if os.path.exists(path):
        filelist.append((path, os.path.join(dstdir, 'surface.txt')))
        config = ninix.config.open(path)
        for name, subdir in config.items():
            dirlist.append((name, subdir.lower()))
    else:
        dirlist.append((None, 'surface'))
    for name, subdir in dirlist:
        shell_src = os.path.join(srcdir, subdir)
        shell_dst = os.path.join(dstdir, subdir)
        if not os.path.isdir(shell_src):
            continue
        if descript_txt:
            # make a copy
            filelist.append((descript_txt,
                             os.path.join(shell_dst, 'descript.txt')))
        if alias_txt:
            # make a copy
            filelist.append((alias_txt,
                             os.path.join(shell_dst, 'alias.txt')))
        filelist.extend(find_surface_set(shell_src, shell_dst))
    return filelist

def install_balloon(archive, srcdir, homedir):
    global mode
    filelist = []
    # find install.txt
    inst = ninix.home.read_install_txt(srcdir)
    if inst is None:
        fatal('install.txt not found')
    #dstdir = os.path.join(homedir, 'balloon', os.path.basename(archive)[:-4])
    target_dir = inst.get('directory')
    if target_dir is None:
        fatal('"directory" not found in install.txt')
    target_dir = target_dir.encode('utf-8')
    dstdir = os.path.join(homedir, 'balloon', target_dir)
    # find balloon
    balloon = find_balloon(srcdir, dstdir)
    if not balloon:
        fatal('balloon not found')
    filelist.extend(balloon)
    filelist.append((os.path.join(srcdir, 'install.txt'),
                     os.path.join(dstdir, 'install.txt')))
    if os.path.exists(dstdir):
        inst_dst = ninix.home.read_install_txt(dstdir)
        if not inst_dst or inst_dst.get('name') != inst.get('name'):
            mode |= INTERACTIVE
        if inst.getint('refresh', 0):
            # uninstall older versions of the balloon
            if confirm_removal(dstdir):
                mask = inst.get('refreshundeletemask', '').lower().split(':')
                remove_files_and_dirs(mask, dstdir)
            else:
                return
        else:
            if not confirm_overwrite(dstdir):
                return
    # install files
    print 'installing', archive, '(balloon)'
    install_files(filelist)

def install_plugin(archive, srcdir, homedir):
    filelist = []
    dstdir = os.path.join(homedir, 'plugin', os.path.basename(archive)[:-4])
    # find plugin.txt
    plugin = ninix.home.read_plugin_txt(srcdir)
    if plugin is None:
        fatal('failed to read plugin.txt')
    plugin_name, plugin_dir, startup, menu_items = plugin
    # find files
    for filename in os.listdir(srcdir):
        path = os.path.join(srcdir, filename)
        if os.path.isfile(path):
            filelist.append((path, os.path.join(dstdir, filename)))
    # uninstall older versions of the plugin
    uninstall_plugin(homedir, plugin_name)
    # install files
    print 'installing', archive, '(plugin)'
    install_files(filelist)

def uninstall_kinoko(homedir, name):
    try:
        dirlist = os.listdir(os.path.join(homedir, 'kinoko'))
    except OSError:
        return
    for subdir in dirlist:
        path = os.path.join(homedir, 'kinoko', subdir)
        kinoko = ninix.home.read_kinoko_ini(path)
        if kinoko is None:
            continue
        kinoko_name = kinoko['title']
        if kinoko_name == name:
            kinoko_dir = os.path.join(homedir, 'kinoko', subdir)
            if confirm_removal(kinoko_dir):
                run('rm -rf %s' % escape(kinoko_dir))

def install_kinoko(archive, srcdir, homedir):
    filelist = []
    # find kinoko.ini
    kinoko = ninix.home.read_kinoko_ini(srcdir)
    if kinoko is None:
        fatal('failed to read kinoko.ini')
    kinoko_name = kinoko['title']
    if kinoko['extractpath'] is not None:
        dstdir = os.path.join(
            homedir, 'kinoko', kinoko['extractpath'].encode('utf-8', 'ignore'))
    else:
        dstdir = os.path.join(
            homedir, 'kinoko', os.path.basename(archive)[:-4])
    # find files
    for filename in os.listdir(srcdir):
        path = os.path.join(srcdir, filename)
        if os.path.isfile(path):
            filelist.append((path, os.path.join(dstdir, filename)))
    # uninstall older versions of the kinoko
    uninstall_kinoko(homedir, kinoko_name)
    # install files
    print 'installing', archive, '(kinoko)'
    install_files(filelist)

def uninstall_nekoninni(homedir, dir):
    nekoninni_dir = os.path.join(homedir, 'nekodorif', 'skin', dir)
    if not os.path.exists(nekoninni_dir):
        return
    if confirm_removal(nekoninni_dir):
        run('rm -rf %s' % escape(nekoninni_dir))

def install_nekoninni(archive, srcdir, homedir):
    global mode
    filelist = []
    # find install.txt
    inst = ninix.home.read_install_txt(srcdir)
    if inst is None:
        fatal('install.txt not found')
    target_dir = inst.get('directory')
    if target_dir is None:
        fatal('"directory" not found in install.txt')
    target_dir = target_dir.encode('utf-8')
    dstdir = os.path.join(homedir, 'nekodorif', 'skin', target_dir)
    # find files
    for filename in os.listdir(srcdir):
        path = os.path.join(srcdir, filename)
        if os.path.isfile(path):
            filelist.append((path, os.path.join(dstdir, filename)))
    # uninstall older versions of the skin
    uninstall_nekoninni(homedir, target_dir)
    # install files
    print 'installing', archive, '(nekodorif skin)'
    install_files(filelist)

def uninstall_katochan(homedir, target_dir):
    katochan_dir = os.path.join(homedir, 'nekodorif', 'katochan', target_dir)
    if not os.path.exists(katochan_dir):
        return
    if confirm_removal(katochan_dir):
        run('rm -rf %s' % escape(katochan_dir))

def install_katochan(archive, srcdir, homedir):
    global mode
    filelist = []
    # find install.txt
    inst = ninix.home.read_install_txt(srcdir)
    if inst is None:
        fatal('install.txt not found')
    target_dir = inst.get('directory')
    if target_dir is None:
        fatal('"directory" not found in install.txt')
    target_dir = target_dir.encode('utf-8')
    dstdir = os.path.join(homedir, 'nekodorif', 'katochan', target_dir)
    # find files
    for filename in os.listdir(srcdir):
        path = os.path.join(srcdir, filename)
        if os.path.isfile(path):
            filelist.append((path, os.path.join(dstdir, filename)))
    # uninstall older versions of the skin
    uninstall_katochan(homedir, target_dir)
    # install files
    print 'installing', archive, '(nekodorif katochan)'
    install_files(filelist)

def list_all_files(top, target_dir):
    filelist = []
    for path in os.listdir(os.path.join(top, target_dir)):
        if os.path.isdir(os.path.join(top, target_dir, path)):
            filelist.extend(list_all_files(top, os.path.join(target_dir, path)))
        else:
            filelist.append(os.path.join(target_dir, path))
    return filelist

def find_pseudo_ai(srcdir, dstdir): # redo, inverse
    # FIXME: This is kluge.
    filelist = []
    for path in list_all_files(srcdir, ''):
        filelist.append((os.path.join(srcdir, path),
                         os.path.join(dstdir, path)))
    return filelist

re_surface_img = re.compile(r'surface[0-9]+\.(png|dgp|pna|ddp)')
re_surface_txt = re.compile(r'surface[0-9]+[sa]\.txt')

def find_surface_set(srcdir, dstdir):
    filename_alias = {}
    path = os.path.join(srcdir, 'alias.txt')
    if os.path.exists(path):
        dic = ninix.alias.open(path)
        for basename, alias in dic.iteritems():
            if basename.startswith('surface'):
                filename_alias[alias] = basename
    filelist = []
    for filename in os.listdir(srcdir):
        if re_surface_img.match(filename):
            filelist.append((os.path.join(srcdir, filename),
                             os.path.join(dstdir, filename)))
            continue
        elif re_surface_txt.match(filename):
            filelist.append((os.path.join(srcdir, filename),
                             os.path.join(dstdir, filename)))
            continue
        basename, suffix = os.path.splitext(filename)
        if not (basename in filename_alias and \
                suffix in ['.png', '.dgp', '.pna', '.ddp']):
            continue
        filelist.append((os.path.join(srcdir, filename),
                         os.path.join(dstdir, filename)))
        for suffix in ['s.txt', 'a.txt']:
            filename = ''.join((basename, suffix))
            if os.path.exists(os.path.join(srcdir, filename)):
                filelist.append((os.path.join(srcdir, filename),
                                 os.path.join(dstdir, filename)))
    return filelist

re_balloon_img = re.compile(r'(balloon[skc][0-9]+|arrow[01]|sstp|online[0-9]+)\.(png|pna)')
re_balloon_txt = re.compile(r'balloon[sk][0-9]+s\.txt')

def find_balloon(srcdir, dstdir): # redo, inverse
    filelist = []
    if not os.path.exists(srcdir):
        return filelist
    path = os.path.join(srcdir, 'descript.txt')
    if os.path.exists(path):
        filelist.append((path, os.path.join(dstdir, 'descript.txt')))
    for filename in os.listdir(srcdir):
        if re_balloon_img.match(filename):
            basename, suffix = os.path.splitext(filename)
            filelist.append((
                os.path.join(srcdir, filename),
                os.path.join(dstdir, ''.join((basename, suffix)))))
        elif re_balloon_txt.match(filename):
            filelist.append((
                os.path.join(srcdir, filename),
                os.path.join(dstdir, filename)))
    return filelist

def install_files(filelist):
    for from_path, to_path in filelist:
        dirname, filename = os.path.split(to_path)
        if not os.path.exists(dirname):
            run('mkdir -p %s' % escape(dirname))
        run('cp %s %s' % (escape(from_path), escape(to_path)))

def reload_(host, port):
    s = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
    try:
        s.connect((host, port))
    except socket.error:
        return
    s.send('EXECUTE SSTP/1.5\r\n'
           'Command: Reload\r\n'
           'Sender: ninix-aya/%s\r\n'
           '\r\n' % ninix.version.VERSION)
    s.recv(1024)
    s.close()

def run(command):
    print command
    if os.system(command):
        print '***FAILED***'
        return 0
    return 1

def readable(path):
    try:
        open(path).read(64)
    except IOError:
        return 0
    return 1

if __name__ == '__main__':
    main()
