/*
 * Copyright (C) 2004,2006 Olivier Rossiny <mrfreeze@tuxfamily.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <stdarg.h>
#include "netswitch.h"
#include <unistd.h>
#include <fcntl.h>
#include <stdlib.h>
#include <string.h>
#include <pty.h>
#include <pthread.h>
#include <fam.h>
#include <glib.h>
#include <glibtop/netlist.h>

int pprintf(process *p, char *format, ...);
int pgetc(process *p);
int isready(process *p);
void my_fork(process *prog, char *args[]);
typedef struct transproc_t{
  int pipe;
  process *proc;
} transproc;

#define UNLOCK(p)		pthread_mutex_unlock((p)->lock);
#define INIT_LOCK(p)		pthread_mutex_init((p)->lock, NULL);
#define WAIT_FOR_LOCK(p)	pthread_mutex_lock((p)->lock);

void my_fork(process *prog, char *args[])
{
  int p[2];
  pipe(p);
  prog->pid = forkpty(&prog->write_fd, NULL, NULL, NULL);
  prog->stop_func = 0;
  if (prog->pid <0){
    fprintf(stderr, "Can't launch the command %s\n", args[0]);
    exit (255);
  } else if (prog->pid == 0) { /* Child Part */
    unsetenv("LC_ALL");
    unsetenv("LC_MESSAGES");
    unsetenv("LANG");
    unsetenv("LANGUAGE");
    dup2(p[1], 1);
    dup2(p[1], 2);
    close(p[0]);
    execv(args[0], args);
    exit(255);
  } else { /* Master part */
    close(p[1]);
    prog->lock = malloc(sizeof(pthread_mutex_t));
    INIT_LOCK(prog);
    prog->read_fd = p[0];
    prog->read_stream = fdopen(prog->read_fd, "r");
    prog->write_stream = fdopen(prog->write_fd, "w");
    setvbuf(prog->read_stream, NULL, _IONBF, 0);
    fcntl(prog->read_fd, F_SETFL, 0);
  }
}

int open_su_term(process *p, netswitch_pwd *getpwd)
{
  char *cmds[6];
  int id = -1;
#ifndef SUDO_MODE
  cmds[++id] = "/bin/su";
#else
  cmds[++id] = "/usr/bin/sudo";
  cmds[++id] = "-S";
  cmds[++id] = "/bin/sh";
#endif
  cmds[++id] = "-c";
  //if (getuid())
    cmds[++id] = "echo Ok;exec /bin/sh";
  //else
  //  cmds[++id] = NULL;
  cmds[++id] = NULL;

  my_fork(p, cmds);
  char *ligne = malloc(20);
  fflush(p->read_stream);
  if (getuid() && isready(p)) {
    fgets(ligne, 10, p->read_stream);
    if (!memcmp(ligne, "Password:", 9)) {
      char *pwd=NULL;
      (*getpwd)(&pwd,
#ifndef SUDO_MODE
      SU
#else
      SUDO
#endif
      );
      if (pwd)
        pprintf(p, "%s\n", pwd);
      else
        pprintf(p, "\n");
      fgets(ligne, 10, p->read_stream);
      memset(pwd, 0, strlen(pwd));
      free(pwd);
    }
    
    /* On essaye de lire le "Ok" */
    if ( (!strcmp(ligne, "Ok\n")) || 
         (fgets(ligne, 7, p->read_stream) && !strcmp(ligne, "Ok\n")) )
      UNLOCK(p)
    else {
      free(ligne);
      return 0;
    }
  }

  if (!getuid()) UNLOCK(p)
  
  free(ligne);
  return 1;
}

int pprintf(process *p, char *format, ...)
{
  va_list list;
  va_start(list, format);
  int res = vfprintf(p->write_stream, format, list);
  va_end(list);
  fflush(p->write_stream);
  return res;
}

int isready(process *p)
{
  fd_set fd;
  struct timeval tv;
  FD_ZERO(&fd);
  FD_SET(p->read_fd, &fd);
  tv.tv_sec = 2;
  tv.tv_usec = 0;
  return select(1 + p->read_fd, &fd, NULL, NULL, &tv);
}

int pgetc(process *p)
{
  if (isready(p))
    return fgetc(p->read_stream);
  else
    return EOF;
}

int process_system(process *p, char *cmd)
{
  char *tmp = malloc(16);
  WAIT_FOR_LOCK(p)
  pprintf(p, "%s 1>/dev/null 2>&1; echo *#£@=$?\n", cmd);
  do {
    if (!fgets(tmp, 15, p->read_stream)) {
      UNLOCK(p)
      return 127;
    }
  } while (tmp && memcmp(tmp, "*#£@=", 5));
  UNLOCK(p)
  int r = atoi(tmp + 6);
  free(tmp);
  return r;
}

void *transfer(void *truc)
{
  char *res, *tmp = malloc(1001);
  transproc *tp = (transproc*) truc;
  FILE *f = fdopen(tp->pipe, "w");
  do {
    if (!isready(tp->proc))
      break;
    res = fgets(tmp, 1000, tp->proc->read_stream);
    //printf(tmp);
    if (!res || !memcmp(tmp, "*#£@=", 5))
      break;
    fputs(tmp, f);
  } while (res);
  free(tmp);
  fclose(f);
  UNLOCK(tp->proc)
  pthread_exit(NULL);
}

FILE *process_read(process *p, char *cmd)
{
  int mypipe[2];
  WAIT_FOR_LOCK(p)
  pprintf(p, "%s ; echo *#£@=$?\n", cmd);
  transproc *p1 = malloc(sizeof(transproc));
  pipe(mypipe);
  p1->pipe = mypipe[1];
  p1->proc = p;
  pthread_t pt;
  pthread_create(&pt, NULL, transfer, p1);
  return fdopen(mypipe[0], "r");
}

void *check_devices_thread(void *data)
{
  int number, last_number = -1;
  glibtop_netlist buf, *last;
  char **devices, **p;
  while (1) {
    number=0;
    devices = glibtop_get_netlist(&buf);
    p = devices;
    while (*p) {
      ++number;
      ++p;
    }
    if (last_number > 0 && last_number != number) (*((callback*)data))();
    last_number = number;
    sleep(1);
  }
  pthread_exit(NULL);
}

void *check_event_thread(void *data)
{
  FAMConnection fc;
  FAMRequest *req = malloc(3 *sizeof(FAMRequest));
  FAMEvent fe;
  int rc;

  if ( FAMOpen(&fc) == 0)
  {
    // request for the monitoring
    FAMMonitorDirectory(&fc, PROG_DIR "/profiles", req+1 , NULL);
    //FAMMonitorDirectory(&fc, "/sys/class/net", req +2, NULL); 
    
    rc = FAMPending(&fc);
    if (rc == -1)
      perror("FAM");

    while (1)
    {
      if (FAMNextEvent(&fc, &fe) <= 0) {
        perror("Event");
        break;
      }
      if (fe.code == FAMDeleted ||
          fe.code == FAMChanged ||
          fe.code == FAMCreated) {
	    (*((callback*)data))();
      }
    }
  }
  FAMClose(&fc);
  free(req);
  pthread_exit(NULL);
}

void check_event(callback *profiles)
{
  pthread_t pt1, pt2;
  pthread_create(&pt1, NULL, check_event_thread, (void*)profiles);
  pthread_create(&pt2, NULL, check_devices_thread, (void*)profiles);
}
