/*
 * Copyright (C) 2004,2005 Olivier Rossiny <mrfreeze@tuxfamily.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <netswitch.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <net/if.h>
#include <glib.h>
#include <glibtop/netload.h>

char basecmd[] = PROG_DIR "/netswitch";
char *cmd = NULL;
char *rootcmd = NULL;
process *proc = NULL;
int services;

#define TMP_DIR		"/var/tmp/"
#define LOCK_FILE       TMP_DIR ".netswitch.pid"

FILE *launch_read(char *param, char root);

void netswitch_delete_lock()
{
  unlink(LOCK_FILE);
}

int netswitch_init(netswitch_pwd *p )
{
  FILE* lock=NULL;
  if (lock=fopen(LOCK_FILE, "r")) {
    int pid=0;
    fscanf(lock, "%d", &pid);
    fclose(lock);
    char *com = (char*) malloc(400);
    sprintf(com, "ps -ef|awk '{print $2}'|grep %d", pid);
    if ( (lock = popen(com, "r")) && getc(lock) == EOF) {
      pclose(lock);
      unlink(LOCK_FILE);
    } else {
      if (lock) pclose(lock);
      kill(pid,12);
      return -1;
    }
  } else {
    if (lock = fopen(LOCK_FILE, "w+")) {
      fprintf(lock, "%d", getpid());
      fclose(lock);
    } else return -1;
  }
  atexit(netswitch_delete_lock);

  rootcmd = malloc(2 + strlen(basecmd));
  strcpy(rootcmd, basecmd);
  cmd = rootcmd;
  if (proc) free(proc);
  proc = malloc(sizeof(process));
  strcat(cmd, " ");
  /* Looking for the services availables */
  FILE *check = launch_read("check", 0);
  char *buf = (char*)malloc(100);
  services = 0;
  if (fgets(buf, 100, check)) {
    if (strstr(buf, "SMTP"))
      services = services | SERVICES_SMTP;
    if (strstr(buf, "SVC"))
      services = services | SERVICES_SERVICES;
    if (strstr(buf, "DHCP"))
      services = services | SERVICES_DHCP;
    if (strstr(buf, "WIFI"))
      services = services | SERVICES_WIFI;
    if (strstr(buf, "PPP"))
      services = services | SERVICES_PPP;
    if (strstr(buf, "WPA"))
      services = services | SERVICES_WPA;
  }
  free(buf);
  pclose(check);
  return open_su_term(proc, p);
}

FILE *launch_read(char *param, char root)
{
  if (!cmd) exit(234);
  char *tmp;
  tmp = malloc(2 + strlen(rootcmd) + strlen(param));
  strcpy(tmp, cmd);
  strcat(tmp, param);
  FILE *f;
  if (root)
    f = process_read(proc, tmp);
  else
    f = popen(tmp, "r");
  free(tmp);
  return f;
}

int launch(char *param, char root)
{
  if (!cmd) exit(234);
  char *tmp = malloc(30 + strlen(cmd) + strlen(param));
  strcpy(tmp, cmd);
  strcat(tmp, param);
  int i;
  if (root)
    i = process_system(proc, tmp);
  else {
    FILE *f = popen(tmp, "w");
    i = pclose(f);
  }
  free(tmp);
  return i;
}

int __lire_ligne(FILE *f, char *buf, int max)
{
  int c, i = EOF;
  while ((i < max) && ( (c=fgetc(f) ) != EOF) && (c != '\n') ) {
    buf[++i] = c;
  }
  if (c == '\n')
    buf[++i] = c;
  buf[++i] = '\0';
  return i;
}

netswitch *netswitch_new()
{
  if ( !cmd) return NULL;
  netswitch *net = malloc(sizeof(netswitch));
  memset(net, 0, sizeof(netswitch));
  net->next = net;
  return net;
}

void netswitch_free_profiles(netswitch *n)
{
  profiles_list *q, *p = n->profiles;
  if (p) do {
    if (p->name)
      free(p->name);
    q=p->next;
    free(p);
    p=q;
  } while  (p != n->profiles);
}

void netswitch_free(netswitch *n)
{
  netswitch *l,*m = n;
  do {
    if (m->iface) free(m->iface);
    m->iface = NULL;
    netswitch_free_profiles(m);
    l = m->next;
    free(m);
    m = l;
  } while (m !=n);
  n = NULL;
}

void netswitch_profiles_list(netswitch *n)
{
  FILE *f = launch_read("profiles", 0);
  char *ligne = malloc(50);
  int taille = 0;
  netswitch *m = NULL;
  profiles_list *last_profile = NULL;
  while (fgets(ligne, 50, f)) {
    taille = strlen(ligne);

    if (memcmp("[Wifi:", ligne, 6) == 0) {
      if (m) {
        m->next = malloc(sizeof(netswitch));
        m = m->next;
      } else {
        m = n;
      }
      m->type=WIRELESS;
      m->next=n;
      taille = strchr(ligne, ']') - strchr(ligne, ':');
      m->iface = malloc(1 + taille);
      memcpy(m->iface, ligne + 6, taille);
      m->iface[taille - 1] = '\0';
      // Création d'un pointeur de profil "NULL"
      m->profiles = malloc(sizeof (profiles_list));
      last_profile = m->profiles;
      last_profile->name = NULL;
      last_profile->next = last_profile;
      continue;
    } else if (memcmp("[Eth:", ligne, 5) == 0) {
      if (m) {
        m->next = malloc(sizeof(netswitch));
        m = m->next;
      } else {
        m = n;
      }
      m->type = ETHERNET;
      m->next=n;
      taille = strchr(ligne, ']') - strchr(ligne, ':');
      m->iface = malloc(1 + taille);
      memcpy(m->iface, ligne + 5, taille);
      m->iface[taille - 1] = '\0';
      // Création d'un pointeur de profil "NULL"
      m->profiles = malloc(sizeof (profiles_list));
      last_profile = m->profiles;
      last_profile->name = NULL;
      last_profile->next = last_profile;
      continue;
    }

    if (last_profile && taille > 1) {
      last_profile->next = malloc(sizeof (profiles_list));
      last_profile = last_profile->next;
      last_profile->name = malloc(1 + taille);
      memcpy(last_profile->name, ligne, taille);
      last_profile->name[taille]='\0';
      if (last_profile->name[taille-1] == '\n')
        last_profile->name[taille-1]='\0';
      last_profile->next = m->profiles;
    }
  }
  pclose(f);
}

void netswitch_active_profiles(netswitch *n)
{
  char *ligne = malloc(50);
  int d;
  FILE *f = launch_read("list", 0);
  if (!f) {
    n = NULL;
    return;
  }
  netswitch *m;
  profiles_list *p, *pp;
  while (fgets(ligne, 50, f)) {
    if (ligne[strlen(ligne) -1] == '\n')
      ligne[strlen(ligne) -1] = '\0';
    m = n;
    while ( (d = memcmp(m->iface, ligne, strlen(m->iface))) && m->next != n)
      m = m->next;
    if (d) continue;

    p = m->profiles;
    if (p != p->next) {// Il existe au moins un profil non nul
      if (!p->name) // Le premier est nul
        p = p->next;
      pp = p;
      d = strlen(ligne + strlen(m->iface) + 1);
      if (d) {
        while (strcmp(p->name, ligne + strlen(m->iface) + 1)) {
          p = p->next;
          if (!p->name) p = p->next;
	  if (p == m->profiles) {
	    while (p->name) p = p->next;
            break;
	  }
        }
      } else
        while (p->name)
          p = p->next;
      m->profiles = p;
    }
  }
  free(ligne);
  pclose(f);
}

void netswitch_active_all(netswitch *n, char *profile)
{
  char *p = malloc(14 + strlen(profile));
  printf(p, "profile on '%s'", profile);
  launch(p, 1);
  free(p);
  netswitch_active_profiles(n);
}

void netswitch_active(netswitch *n, char *profile)
{
  char *p = malloc(15 + strlen(n->iface) + strlen(profile));
  sprintf(p, "profile on %s '%s'", n->iface, profile);
  launch(p, 1);
  free(p);
  netswitch_active_profiles(n);
}

void netswitch_inactive_profile(netswitch *n, char *profil)
{
  char *p = malloc(17 + strlen(profil));
  sprintf(p, "profile off '%s'", profil);
  launch(p, 1);
  free(p);
  netswitch_active_profiles(n);
}

void netswitch_inactive(netswitch *n)
{
  char *p = malloc(20 + strlen(n->iface));
  sprintf(p, "profile off %s", n->iface);
  launch(p, 1);
  free(p);
  netswitch_active_profiles(n);
}

void netswitch_inactive_all_iface(netswitch *n)
{
  launch("profile off", 1);
  netswitch_active_profiles(n);
}

void netswitch_delete(netswitch *n, char *profile)
{
  char *p = malloc(16 + strlen(n->iface) + strlen(profile));
  sprintf(p, "profile del %s '%s'", n->iface, profile);
  launch(p, 1);
  if (n->profiles->name)
    strcpy(p, n->profiles->name);
  else {
    free(p);
    p = NULL;
  }
  profiles_list *m = n->profiles, *q = NULL;
  do {
    if (m->next->name && !strcmp(m->next->name, profile)) {
      q = m->next;
      m->next = q->next;
      free(q->name);
      free(q);
      if (p) {
        if (strcmp(p, profile))
          break;
	else {
	  while (m->name) m = m->next;
          n->profiles = m;
          break;
        }
      } else break;
    }
    m = m->next;
    /* On a pas trouvé de profil correspondant et on a fait un tour */
    if (q == NULL && m == n->profiles)
      break;
  } while (1);
  if (p) free(p);
}

int netswitch_install(netswitch *n, profile *p, char *name)
{
  char *fic = malloc(10 + strlen(name));
  sprintf(fic, TMP_DIR "%s", name);
  FILE *f = fopen(fic, "w+");
  if (!f) {
    printf("Err:%s\n", fic);
    free(fic);
    return 0;
  }
  chmod(fic, 0666);
  int i;
  char **t = p->nameservers;
  if (p->hwaddr) fprintf(f, "HARDWARE=%s\n", p->hwaddr);
  if (p->ipaddr) {
    fprintf(f, "IPADDR=%s\n", p->ipaddr);
    if (p->netmask) fprintf(f, "NETMASK=%d\n", p->netmask);
    if (p->broadcast) fprintf(f, "BROADCAST=%s\n", p->broadcast);
    if (p->gateway) fprintf(f, "GATEWAY=%s\n", p->gateway);
    if (p->metric) fprintf(f, "METRIC=%d\n", p->metric);
    if (p->domain) fprintf(f, "DOMAIN=%s\n", p->domain);
  } else if (!p->peerdns)
    fprintf(f, "PEERDNS=no\n");

  if (t && *t && (p->ipaddr || p->peerdns)) {
    i=0;
    while (*t)
      fprintf(f, "NAMESERVER%d=%s\n", ++i, *t++);
  }

  if (n->type == WIRELESS) {
    fprintf(f, "ESSID=%s\n", p->essid);
    if (p->mode == ADHOC) fprintf(f, "MODE=Ad-hoc\n");
    else if (p->mode == MONITOR) fprintf(f, "MODE=Monitor\n");
    else if (p->mode == MANAGED) fprintf(f, "MODE=Managed\n");
    else fprintf(f, "MODE=Auto\n");

	if (p->nickname) fprintf(f, "NICKNAME=%s\n", p->nickname);
    if (p->rts) fprintf(f, "RTS=%d\n", p->rts);
    if (p->frag) fprintf(f, "FRAG=%d\n", p->frag);
    if (p->freq) fprintf(f, "FREQ=%d\n", p->freq);
    if (p->nwid) fprintf(f, "NWID=%s\n", p->nwid);
    if (p->sens) fprintf(f, "SENS=%d\n", p->sens);
    if (p->rate > 0) fprintf(f, "RATE=%d\n", p->rate);
    else if (p->rate == -1) fprintf(f, "RATE=auto\n");
    if (p->channel) fprintf(f, "CHANNEL=%d\n", p->channel);
    t = p->keys;
    if (p->key_mode != OFF && t) {
      i=0;
      while (*t)
        fprintf(f, "KEY%d=%s\n", ++i, *t++);
    }
    if (p->key_mode == OPEN)
      fprintf(f, "KEY_MODE=open\n");
    else if (p->key_mode == RESTRICTED)
      fprintf(f, "KEY_MODE=restricted\n");
    else if (p->key_mode == WPA)
      fprintf(f, "KEY_MODE=wpa\n");
    if ( (p->key_mode == OPEN || p->key_mode == RESTRICTED ) && p->sendkey && p->sendkey <= i) fprintf(f, "SENDKEY=%d\n", p->sendkey);
    if (p->smtp_host) {
      fprintf(f, "SMTP_HOST=%s\n", p->smtp_host);
      if (p->smtp_port != 25)
        fprintf(f, "SMTP_HOST=%d\n", p->smtp_port);
      if (p->smtp_user) fprintf(f, "SMTP_USER=%s\n", p->smtp_user);
      if (p->smtp_pass) fprintf(f, "SMTP_PASS=%s\n", p->smtp_pass);
    }
  }
  fclose(f);
  char *param = malloc(17 + strlen(n->iface) + strlen(fic));
  sprintf(param, "profile put %s '%s'", n->iface, fic);
  launch(param, 1);
  free(param);
  free(fic);
  return 1;
}

profile *profile_new()
{
  profile *p = malloc(sizeof(profile));
  memset(p, 0, sizeof(profile));
  p->sendkey = 1;
  p->peerdns = 1;
  p->smtp_port = 25;
  return p;
}

profile *netswitch_get(netswitch *n, char *profilename)
{
  if (!profilename) return NULL;
  profile *p = profile_new();
  char *pr = malloc(16 + strlen(n->iface) + strlen(profilename));
  sprintf(pr, "profile get %s '%s'", n->iface, profilename);
  FILE *f = launch_read(pr, 1);
  if (!f) {
    free(pr);
    return NULL;
  }
  char *ligne = malloc(100);
  int numkey = 1, numns = 1;
  char *idkey = malloc(6);
  sprintf(idkey, "KEY%d=", numkey);
  char *egal,*idns = malloc(13);
  sprintf(idns, "NAMESERVER%d=", numns);
  while (fgets(ligne, 100, f)) {
    ligne[strlen(ligne)-1]='\0';
    egal = strchr(ligne, '=');
    if (!egal || egal - ligne == strlen(ligne) - 1)
      continue;
	    
    if (!memcmp(ligne, "IPADDR=", 7)) {
      p->ipaddr = malloc(strlen(ligne) - 6);
      strcpy(p->ipaddr, ligne + 7);
    }else if (!memcmp(ligne, "HARDWARE=", 9)) {
      p->hwaddr = malloc(strlen(ligne) - 8);
      strcpy(p->hwaddr, ligne + 9);
    } else if (!memcmp(ligne, "NETMASK=", 8)) {
      p->netmask = atoi(ligne + 8);
    } else if (!memcmp(ligne, "METRIC=", 7)) {
      p->metric = atoi(ligne + 7);
    } else if (!memcmp(ligne, "BROADCAST=", 10)) {
      p->broadcast = malloc(strlen(ligne) - 9);
      strcpy(p->broadcast, ligne + 10);
    } else if (!memcmp(ligne, "GATEWAY=", 8)) {
      p->gateway = malloc(strlen(ligne) - 7);
      strcpy(p->gateway, ligne + 8);
    } else if (!memcmp(ligne, "DOMAIN=", 7)) {
      p->domain = malloc(strlen(ligne) - 6);
      strcpy(p->domain, ligne + 7);
      printf("%s\n", p->domain);
    } else if (!memcmp(ligne, "NICKNAME=", 9)) {
      p->nickname = malloc(strlen(ligne) - 8);
      strcpy(p->nickname, ligne + 9);
    } else if (!memcmp(ligne, "ESSID=", 6)) {
      p->essid = malloc(strlen(ligne) - 5);
      strcpy(p->essid, ligne + 6);
    } else if (!memcmp(ligne, "MODE=", 5)) {
      if (!strcmp(ligne + 5, "Managed"))
        p->mode = MANAGED;
      else if (!strcmp(ligne + 5, "Ad-hoc"))
        p->mode = ADHOC;
      else if (!strcmp(ligne + 5, "Monitor"))
        p->mode = MONITOR;
      else
        p->mode = AUTO;
    } else if (!memcmp(ligne, "CHANNEL=", 8)) {
      p->channel = atoi(ligne + 8);
    } else if (!memcmp(ligne, "NWID=", 5)) {
      p->nwid = malloc(strlen(ligne) - 5);
      strcpy(p->nwid, ligne + 5);
    } else if (!memcmp(ligne, "SENS=", 5)) {
      p->sens = atoi(ligne + 5);
    } else if (!memcmp(ligne, "RTS=", 4)) {
      p->rts = atoi(ligne + 4);
    } else if (!memcmp(ligne, "FRAG=", 5)) {
      p->frag = atoi(ligne + 5);
    } else if (!memcmp(ligne, "FREQ=", 5)) {
      p->freq = atoi(ligne + 5);
    } else if (!memcmp(ligne, "RATE=", 5)) {
      p->rate = atoi(ligne + 5);
    } else if (!memcmp(ligne, "PEERDNS=", 8)) {
      if (!strcmp(ligne + 8, "no")) p->peerdns = 0;
    } else if (!memcmp(ligne, idns, strlen(idns))) {
      if (p->nameservers) {
        p->nameservers = realloc(p->nameservers, numns * sizeof(char*));
      } else {
        p->nameservers = malloc(sizeof(char*));
      }
      p->nameservers[numns-1] = malloc(1 + strlen(ligne) - strlen(idns));
      strcpy(p->nameservers[numns-1], ligne + strlen(idns));
      sprintf(idns, "NAMESERVER%d=", ++numns);
    } else if (!memcmp(ligne, idkey, strlen(idkey))) {
      if (p->keys) {
        p->keys = realloc(p->keys, numkey * sizeof(char*));
      } else {
        p->keys = malloc(sizeof(char*));
	if (p->key_mode == OFF) p->key_mode = OPEN;
      }
      p->keys[numkey-1] = malloc(1 + strlen(ligne) - strlen(idkey));
      strcpy(p->keys[numkey-1], ligne + strlen(idkey));
      sprintf(idkey, "KEY%d=", ++numkey);
    } else if (!memcmp(ligne, "KEY_MODE=", 9)) {
      if (!strcmp(ligne + 9, "open"))
        p->key_mode = OPEN;
      else if (!strcmp(ligne + 9, "restricted"))
        p->key_mode = RESTRICTED;
      else if (!strcmp(ligne + 9, "wpa"))
        p->key_mode = WPA;
    } else if (!memcmp(ligne, "SENDKEY=", 8)) {
      p->sendkey = atoi(ligne + 8);
    } else if (!memcmp(ligne, "SMTP_HOST=", 10)) {
      p->smtp_host = malloc(strlen(ligne));
      strcpy(p->smtp_host, ligne + 10);
    } else if (!memcmp(ligne, "SMTP_DOMAIN=", 12)) {
      p->smtp_domain = malloc(strlen(ligne) - 11);
      strcpy(p->smtp_domain, ligne + 12);
    } else if (!memcmp(ligne, "SMTP_PORT=", 10)) {
      p->smtp_port = atoi(ligne + 10);
    } else if (!memcmp(ligne, "SMTP_USER=", 10)) {
      p->smtp_user = malloc(strlen(ligne));
      strcpy(p->smtp_user, ligne + 10);
    } else if (!memcmp(ligne, "SMTP_PASS=", 10)) {
      p->smtp_pass = malloc(strlen(ligne));
      strcpy(p->smtp_pass, ligne + 10);
    } /*else
      printf("Ignored argument: %s\n", ligne);*/
  }
  if (p->keys) {
    p->keys = realloc(p->keys, numkey * sizeof(char*));
    p->keys[numkey-1] = NULL;
  }
  if (p->nameservers) {
    p->nameservers = realloc(p->nameservers, numns * sizeof(char*));
    p->nameservers[numns-1] = NULL;
  }
  pclose(f);
  free(ligne);
  free(pr);
  free(idkey);
  free(idns);
  return p;
}

void profile_free(profile *p)
{
  if (p->ipaddr) free(p->ipaddr);
  if (p->broadcast) free(p->broadcast);
  if (p->gateway) free(p->gateway);
  if (p->essid) free(p->essid);
  if (p->nwid) free(p->nwid);
  if (p->nickname) free(p->nickname);
  char **keys = p->keys;
  if (keys) {
    while (*keys) {
      free(*keys);
      keys++;
    }
    free(p->keys);
  }
  keys = p->nameservers;
  if (keys) {
    while (keys && *keys) {
      free(*keys);
      keys++;
    }
    free(p->nameservers);
  }
  free(p);
}

int netswitch_number()
{
  FILE *p = launch_read("number", 0);
  char *tmp = malloc(6);
  int res = 0;
  if (fgets(tmp, 6, p))
    res = atoi(tmp);
  pclose(p);
  free(tmp);
  return res;
}

netlist *netswitch_scan()
{
  FILE *p = launch_read("scan", 1);
  netlist *q = NULL, *t=NULL;
  if (p) {
    char *buf = malloc(80);
    while (fgets(buf, 80, p)) {
      buf[strlen(buf)-1]='\0';
      while (buf[strlen(buf)-1] == ' ')
        buf[strlen(buf)-1]='\0';
      if (!memcmp(buf, "PROTO=", 6)) {
	q->protocol = malloc(strlen(buf) - 5);
	strcpy(q->protocol, buf + 6);
      } else if (!memcmp(buf, "ESSID=", 6)) {
        if (strcmp(buf + 6, "<hidden>")) {
	  q->essid = malloc(strlen(buf) - 5);
	  strcpy(q->essid, buf + 6);
	} else q->essid = NULL;
      } else if (!memcmp(buf, "BSSID=", 6)) {
        if (q) {
          q->next = malloc(sizeof(netlist));
	  q->next->next = t;
	  q = q->next;
	} else {
          q = malloc(sizeof(netlist));
	  q->next = q;
	  t = q;
	}
	q->bssid = malloc(strlen(buf) - 5);
	q->link = 0;
	q->quality = 0;
	strcpy(q->bssid, buf + 6);
      } else if (!memcmp(buf, "MODE=", 5)) {
        if (!strcmp(buf + 5, "Master"))
          q->mode = MANAGED;
        else if (!strcmp(buf + 5, "Ad-hoc"))
          q->mode = ADHOC;
      } else if (!memcmp(buf, "RATE=", 5)) {
	q->rate = atoi(buf + 5);
      } else if (!memcmp(buf, "SIGNAL=", 7)) {
	q->link = atoi(buf + 7);
      } else if (!memcmp(buf, "QUALITY=", 8)) {
	q->quality = atoi(buf + 8);
      } else if (!memcmp(buf, "CH=", 3)) {
	q->channel = atoi(buf + 3);
      } else if (!memcmp(buf, "ENC=", 4)) {
        q->encryption = !strcmp(buf + 4, "on");
      } /*else {
        printf(_("Ignored argument: %s\n"), buf);
      }*/
    }
    pclose(p);
    free(buf);
  }
  return t;
}

void netlist_free(netlist *p)
{
  netlist *r,*q = p;
  do {
    if (q->bssid) free(q->bssid);
    if (q->essid) free(q->essid);
    if (q->protocol) free(q->protocol);
    r = q->next;
    free(q);
    q = r;
  } while (q != p);
  p = NULL;
}

/*iface_info *netswitch_info(netswitch *n)
{
  char *pr = malloc(6 + strlen(n->iface));
  sprintf(pr, "info %s", n->iface);
  FILE *p = launch_read(pr, 0);
  free(pr);
  if (p) {
    iface_info *info = malloc(sizeof(iface_info));
    info->hwaddr = NULL;
    info->ipnet = NULL;
    info->ipaddr = NULL;
    info->ipbroadcast = NULL;
    info->ipgateway = NULL;
    info->wifilink = 0;
    info->wifiessid = NULL;
    char *egal,*buf = malloc(80);
    while (fgets(buf, 80, p)) {
      buf[strlen(buf)-1]='\0';
      egal = strchr(buf, '=');
      if (!egal || egal - buf == strlen(buf) - 1)
        continue;
      if (!memcmp(buf, "HWADDR=", 7)) {
        info->hwaddr = malloc(strlen(buf) - 6);
        strcpy(info->hwaddr, buf + 7);
      } else if (!memcmp(buf, "IPADDR=", 7)) {
        info->ipaddr = malloc(strlen(buf) - 6);
        strcpy(info->ipaddr, buf + 7);
      } else if (!memcmp(buf, "IPBRD=", 6)) {
        info->ipbroadcast = malloc(strlen(buf) - 5);
        strcpy(info->ipbroadcast, buf + 6);
      } else if (!memcmp(buf, "IPGW=", 5)) {
        info->ipgateway = malloc(strlen(buf) - 4);
        strcpy(info->ipgateway, buf + 5);
      } else if (!memcmp(buf, "IPNET=", 6)) {
        info->ipnet = malloc(strlen(buf) - 5);
        strcpy(info->ipnet, buf + 6);
      } else if (!memcmp(buf, "WFID=", 5)) {
        info->wifiessid = malloc(strlen(buf) - 4);
        strcpy(info->wifiessid, buf + 5);
      } else if (!memcmp(buf, "WFLK=", 5)) {
        info->wifilink = atoi(buf + 5);
      } else if (!memcmp(buf, "STATUS=", 7)) {
        if (!strcmp(buf + 7, "on"))
          info->status = 1;
        else
          info->status = 0;
      }
    }
    pclose(p);
    free(buf);
    return info;
  }
  return NULL;
}*/

static char*
format_ipv4(guint32 ip)
{
  char str[INET_ADDRSTRLEN];
  inet_ntop(AF_INET, &ip, str, sizeof str);
  char *tmp = malloc(1 + sizeof(str));
  strcpy(tmp, str);
  return tmp;
}


static char*
format_ipv6(const guint8 ip[16])
{
  char str[INET6_ADDRSTRLEN];
  inet_ntop(AF_INET6, ip, str, sizeof str);
  char *tmp = malloc(1 + sizeof(str));
  strcpy(tmp, str);
  return tmp;
}

iface_info *netswitch_info(netswitch *n)
{
  glibtop_netload netload;
  glibtop_get_netload(&netload, n->iface);
  iface_info *info = malloc(sizeof(iface_info));
  info->hwaddr = NULL;
  info->ipnet = NULL;
  info->ipaddr = NULL;
  info->ipbroadcast = NULL; // Keep NULL
  info->ipgateway = NULL; // keep NULL
  info->wifilink = 0;
  info->wifiessid = NULL;

  info->status = (netload.if_flags & (1L << GLIBTOP_IF_FLAGS_UP) ? 1 : 0);
  //info->status = (netload.if_flags & (1L << GLIBTOP_IF_FLAGS_RUNNING) ? 1: 0);
  info->ipaddr = format_ipv4(netload.address);
  if (!strcmp(info->ipaddr, "0.0.0.0")) {
    info->status = 0;
    free(info->ipaddr);
    info->ipaddr = NULL;
  } else
    info->ipnet = format_ipv4(netload.subnet);
  //info->ipv6 = format_ipv6(netload.address6);

  guint8 *hw = netload.hwaddress;
  info->hwaddr = malloc(18);
  sprintf(info->hwaddr, "%02X:%02X:%02X:%02X:%02X:%02X",
		hw[0], hw[1], hw[2], hw[3],
		hw[4], hw[5]);
  char *buf = malloc(100), *pos;
  if (n->type == WIRELESS && NETSWITCH_HAS_WIFI) {
    char *cmd = malloc(40);
    int size;
    sprintf(cmd, "LC_ALL=C /sbin/iwconfig %s", n->iface);
    FILE *iwconfig = popen(cmd, "r");
    while (fgets(buf, 100, iwconfig)) {
      if (!memcmp(buf, n->iface, strlen(n->iface))) {
        pos = strstr(buf, "ESSID:") + 7;
        size = strchr(pos, '\"') - pos;
        info->wifiessid = malloc(1 + size);
        memcpy(info->wifiessid, pos, size);
        info->wifiessid[size] = '\0';
      } else if (strstr(buf, "Quality=")) {
        info->wifilink = atoi(strstr(buf, "Quality=") + 8);
      }
    }
    pclose(iwconfig);
    free(cmd);
  }
  FILE *route = fopen("/proc/net/route", "r");
  char *iface = malloc(strlen(n->iface) +1);
  char *dest = malloc(9);
  char *gateway = malloc(9);
  if (route) {
    while (fgets(buf, 100, route)) {
      if (memcmp(buf, n->iface, strlen(n->iface))) continue;
      sscanf(buf, "%s %s %s", iface, dest, gateway);
      if (!strcmp(dest, "00000000")) {
        info->ipgateway = malloc(16);
        sscanf(gateway, "%2X%2X%2X%2X",
		hw, hw+1, hw+2, hw+3);
        sprintf(info->ipgateway, "%d.%d.%d.%d",
		hw[3], hw[2], hw[1], hw[0]);
	break;
      }
    }
    fclose(route);
  }
  free(buf);
  free(iface);
  free(dest);
  free(gateway);
  return info;
}

void iface_info_free(iface_info *p)
{
  if (p->hwaddr) free(p->hwaddr);
  if (p->ipnet) free(p->ipnet);
  if (p->ipaddr) free(p->ipaddr);
  if (p->ipbroadcast) free(p->ipbroadcast);
  if (p->ipgateway) free(p->ipgateway);
  if (p->wifiessid) free(p->wifiessid);
  free(p);
}

int netswitch_autoconnect(netswitch *n)
{
  return launch("auto", 1);
}
