/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 2010 Oracle and/or its affiliates. All rights reserved.
 *
 * Oracle and Java are registered trademarks of Oracle and/or its affiliates.
 * Other names may be trademarks of their respective owners.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 *
 * Contributor(s):
 *
 * Portions Copyrighted 2008 Sun Microsystems, Inc.
 */

package org.netbeans.modules.parsing.impl.indexing.lucene;

import java.io.File;
import java.io.IOException;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;
import org.netbeans.api.annotations.common.CheckForNull;
import org.netbeans.api.annotations.common.NonNull;
import org.netbeans.modules.parsing.impl.indexing.ClusteredIndexables;
import org.netbeans.modules.parsing.impl.indexing.PathRegistry;
import org.netbeans.modules.parsing.lucene.support.DocumentIndex2;
import org.netbeans.modules.parsing.lucene.support.DocumentIndexCache;
import org.netbeans.modules.parsing.lucene.support.IndexManager;
import org.openide.util.Exceptions;
import org.openide.util.Pair;
import org.openide.util.Parameters;
import org.openide.util.Utilities;

/**
 *
 * @author Tomas Zezula
 */
public final class DocumentBasedIndexManager {

    private static DocumentBasedIndexManager instance;

    //@GuardedBy("this")
    @org.netbeans.api.annotations.common.SuppressWarnings(
    value="DMI_COLLECTION_OF_URLS",
    justification="URLs have never host part")
    private final Map<URL, Pair<DocumentIndex2.Transactional, DocumentIndexCache>> indexes =
            new HashMap<URL, Pair<DocumentIndex2.Transactional, DocumentIndexCache>> ();
    //@GuardedBy("this")
    private boolean closed;

    private DocumentBasedIndexManager() {}


    public static enum Mode {
        OPENED,
        CREATE,
        IF_EXIST;
    }


    public static synchronized DocumentBasedIndexManager getDefault () {
        if (instance == null) {
            instance = new DocumentBasedIndexManager();
        }
        return instance;
    }

   @CheckForNull
   @org.netbeans.api.annotations.common.SuppressWarnings(
    value="DMI_COLLECTION_OF_URLS",
    justification="URLs have never host part")
    public synchronized DocumentIndex2.Transactional getIndex (final URL root, final Mode mode) throws IOException {
        assert root != null;
        assert PathRegistry.noHostPart(root) : root;
        if (closed) {
            return null;
        }
        Pair<DocumentIndex2.Transactional, DocumentIndexCache> li = indexes.get(root);
        if (li == null) {
            try {
                switch (mode) {
                    case CREATE:
                    {
                        final File file = Utilities.toFile(root.toURI());
                        file.mkdir();
                        final DocumentIndexCache cache = ClusteredIndexables.createDocumentIndexCache();
                        final DocumentIndex2.Transactional index = (DocumentIndex2.Transactional) IndexManager.createTransactionalDocumentIndex(file, cache);
                        li = Pair.<DocumentIndex2.Transactional, DocumentIndexCache>of(index, cache);

                        indexes.put(root,li);
                        break;
                    }
                    case IF_EXIST:
                    {
                        final File file = Utilities.toFile(root.toURI());
                        String[] children;
                        if (file.isDirectory() && (children=file.list())!= null && children.length > 0) {
                            final DocumentIndexCache cache = ClusteredIndexables.createDocumentIndexCache();
                            final DocumentIndex2.Transactional index = (DocumentIndex2.Transactional) IndexManager.createTransactionalDocumentIndex(file, cache);
                            li = Pair.<DocumentIndex2.Transactional, DocumentIndexCache>of(index, cache);
                            indexes.put(root,li);
                        }
                        break;
                    }
                }
            } catch (URISyntaxException e) {
                throw new IOException(e);
            }
        }
        return li == null ? null : li.first();
    }

   @CheckForNull
   public synchronized DocumentIndexCache getCache(@NonNull final URL root) {
       final Pair<DocumentIndex2.Transactional, DocumentIndexCache> entry = indexes.get(root);
       return entry == null ? null : entry.second();
   }

   @CheckForNull
   public synchronized DocumentIndex2.Transactional getIndex(@NonNull final DocumentIndexCache cache) {
       Parameters.notNull("cache", cache);  //NOI18N
       for (Pair<DocumentIndex2.Transactional,DocumentIndexCache> e : indexes.values()) {
           if (cache.equals(e.second())) {
               return e.first();
           }
       }
       return null;
   }
   
   public synchronized void close() {
       if (closed) {
           return;
       }
       closed = true;
       for (Pair<DocumentIndex2.Transactional, DocumentIndexCache> index : indexes.values()) {
           try {
            index.first().close();
           } catch (IOException ioe) {
               Exceptions.printStackTrace(ioe);
           }
       }
   }

}
