/*
 * Copyright (c) 2003-2005 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
 
using Nemerle.Collections;
using Nemerle.Compiler.Parsetree;
using Nemerle.Assertions;

namespace Nemerle.Compiler
{
  enum OpContext {
    | Prefix
    | Postfix
  }

  [System.Flags]
  enum TokenStoppers {
    | None   = 0x0000
    | Pipe   = 0x0001 // |
    | Equal  = 0x0002 // =
    | Braces = 0x0004 // {}
    | Arrow  = 0x0008 // =>
    | Colon  = 0x0010 // :
    | Dollar = 0x0020 // $
    | Where  = 0x0040 // where
    | All = (Pipe | Equal | Braces | Arrow | Where)
  }

  /** Parser class, which transforms preparsed token tree into Nemerle parse trees.

      It also contains some helper methods to perform operation needed to parse
      string as expression or entire file given [LexerBase] instance.
   */
  public partial class MainParser
  {
    /// Reverse Polish Notation token. It is used on the stack created during translation
    /// of infix operator expression into RPN on the way to create appropriate PExpr.
    [Record]
    internal class OperatorInfo {
      public Name : string;
      public LeftPriority : int;
      public RightPriority : int;

      public static RoundInfo : OperatorInfo = OperatorInfo ("(", 290,291);
      public static SquareInfo : OperatorInfo = OperatorInfo ("[", 290, 291);
      public static ProductInfo : OperatorInfo = OperatorInfo ("*", 261, 260);
      public static UnMinus : OperatorInfo = UnaryOperatorInfo ("-", 281, 280);
      public static UnPlus : OperatorInfo = UnaryOperatorInfo ("+", 281, 280);
    }

    internal class UnaryOperatorInfo : OperatorInfo {
      public this (name : string, left : int, right : int) {
        base (name, left, right);
      }
    }

    internal class BinaryOperatorInfo : OperatorInfo {
      public this (name : string, left : int, right : int) {
        base (name, left, right);
      }
    }

    class NaryOperatorInfo : OperatorInfo {
      public this (copy : OperatorInfo) {
        base (copy.Name, copy.LeftPriority, copy.RightPriority);
      }

      public mutable ExprsBegin : list [PExpr];
    }

    mutable env : GlobalEnv;
    mutable in_variant : bool;
    mutable stream : Token;
    mutable last_tok : Token;

    mutable streams_stack : Stack [Token] = Stack (20);

    /** Creates the instance of parsing engine with given environment
        (set of imported namespaces and entered ones)

        Parser instance is created with syntax extensions from Nemerle.Core namespace
        (which is always open by default).
     */
    public this (env : GlobalEnv) {
      this.env = env;
    }

    /** Parse given string as expression, given context in which this
        expression is situated.
     */
    public static ParseExpr ([NotNull] env : GlobalEnv, 
                             [NotNull] expr : string, 
                             allow_empty = false) : PExpr
    {
      // we must prevent lexer from bailing out on last token at end of input
      def lexer = LexerString (expr + " ", Location_stack.top ());
      ParseExpr (env, lexer, allow_empty)
    }

    /** Parse expression given lexer producing its content and context in which this
        expression is situated.
     */
    public static ParseExpr ([NotNull] env : GlobalEnv,
                             [NotNull] lexer : LexerBase,
                             allow_empty = false) : PExpr
    {
      def preparser = PreParser (lexer);
      def tokens = preparser.PreParse ();
      def parser = MainParser (env);
       
      if (tokens.Child == null)
        if (allow_empty)
          null
        else 
          Message.FatalError ("expression without content is not allowed here");
      else {
        if (tokens.Child.Next == null)
          parser.push_stream ((tokens.Child :> Token.LooseGroup).Child);
        else
          parser.push_stream (tokens);
        parser.parse_expr (TokenStoppers.None);
      }
    }
    
    /** Parse well formed source code of toplevel program, using the given instance
        of [LexerBase] for obtaining tokens.
     */
    public static Parse (lex : LexerBase) : list [TopDeclaration]
    {
      def preparser =
        if (Options.IndentationSyntax)
          PreParserIndent (lex)
        else
          PreParser (lex);
      def parser = MainParser (GlobalEnv.Core);
      
      mutable result = [];
      mutable topstream = preparser.ParseTopLevel ().Child :> Token.LooseGroup;

      def iter () {
        when (topstream != null) {
          parser.eat_assembly_attributes (topstream);
          
          if (IsTopLevel (topstream)) {
            result = parser.ParseTopLevel (topstream, result);
            topstream = topstream.Next :> Token.LooseGroup;
            iter ();
          }
          else {
            def seq = parser.parse_expr_sequence (topstream);
            def decl = Util.locate (topstream.Location, <[ decl:
              module $(parser.mkname ("_N_AutoModule") : name) {
                Main () : void {
                  ..$seq
                }
              }
            ]>.td);
            result = decl :: result;
          }
        }
      }
      iter ();
      
      List.Rev (result);
    }

    static IsTopLevel (tok : Token.LooseGroup) : bool
    {
      match (tok.Child) {
        | Token.Keyword (k) =>
          match (k) {
            | "public" | "private" | "static" | "internal"
            | "abstract" | "sealed" | "partial" | "class" | "module" | "variant"
            | "enum" | "interface" | "struct" | "delegate" | "type" | "macro"
              => true
            | _ => false
          }
          
        | Token.SquareGroup
        | Token.Namespace | Token.Using | null => true
          
        | _ => false
      }
    }
    
    #region Utility functions operating on current token stream
    push_stream (newstream : Token) : void {
      streams_stack.Push (stream);
      stream = newstream;
    }

    pop_stream () : void {
      stream = streams_stack.Pop ();
      when (streams_stack.IsEmpty)
        stream = null;
      // else previous stack was stored before incrementing pointer
    }

    pop_stream (finish_entity : string) : void {
      expect_empty (finish_entity);
      pop_stream ();
    }

    get_token () : Token {
      if (stream != null) {
        last_tok = stream;
        stream = stream.Next;
        last_tok
      }
      else {
        Error (last_tok, "unexpected end of token sequence");
        Token.EndOfGroup (last_tok.Location)
      }
    } 

    peek_token () : Token {
      if (stream != null) stream
      else Token.EndOfGroup (last_tok.Location)
    }

    peek_second_token () : Token {
      if (stream != null)
        if (stream.Next != null) stream.Next
        else Token.EndOfGroup (stream.Location)
      else Token.EndOfGroup (last_tok.Location)
    }

    peek_sibling_token () : Token {
      if (stream != null) stream
      else {
        def str = streams_stack.Peek ();
        if (str != null && str.Next != null)
          (str.Next :> Token.LooseGroup).Child
        else
          Token.EndOfGroup (last_tok.Location)
      }
    }

    jump_to_sibling () : void {
      if (stream == null) {
        def top_next = streams_stack.Pop ().Next;
        streams_stack.Push (top_next);
        def parent = top_next :> Token.LooseGroup;
        if (parent == null)
          Error (last_tok, "unexpected end of token sequence")
        else
          stream = parent.Child;
      }
      else
        Util.ice ("cannot jump to sibling, some tokens are still here")
    }
    
    shift_sibling () : void {
      if (stream == null) {
        jump_to_sibling ();
        shift ()
      }
      else
        shift ();
    }

    new_group_beginning () : void {
      when (stream != null) {
        def newgroup = Token.LooseGroup (stream);
        newgroup.Next = streams_stack.Pop ();
        streams_stack.Push (newgroup);
        stream = null;
      }
    }
    
    push_back () : void {
      stream = last_tok;
      last_tok = null;
    }
    
    shift () : void {
      if (stream != null) {
        last_tok = stream;
        stream = stream.Next;
      }
      else Error (last_tok, "unexpected end of token sequence");
    }
    
    expect_empty (msg : string) : void {
      when (stream != null)
        Error (stream, "unexpected token after " + msg)
    }

    expect_operator (op : string) : void {
      match (get_token ()) {
        | Token.Operator (o) when op == o => ()
        | Token.Operator (o) =>
          match (peek_token ()) {
            | Token.Operator (o') when op == o + o' => shift (); ()
            | x => Error (x, "expecting operator `" + op + "'");
          }
        | x => Error (x, "expecting operator `" + op + "'");
      }
    }

    flag_keyword (kw : string) : bool {
      match (peek_token ()) {
        | Token.Keyword (n) when n == kw => shift (); true
        | _ => false
      }
    }

    flag_sibling_keyword (kw : string) : bool {
      match (peek_sibling_token ()) {
        | Token.Keyword (k) when kw == k => shift_sibling (); true
        | _ => false
      }
    }

    should_stop (stop : TokenStoppers) : bool {
      if (stream == null) true
      else {
        match (stream) {
          | Token.Semicolon => true
          | Token.LooseGroup (Token.Operator (str))
          | Token.Operator (str) =>
            match (str) {
              | "|" when stop %&& TokenStoppers.Pipe 
              | "=" when stop %&& TokenStoppers.Equal 
              | "=>" when stop %&& TokenStoppers.Arrow
              | "$" when stop %&& TokenStoppers.Dollar
              | ":" when stop %&& TokenStoppers.Colon => true
              | _ => false
            }
          | Token.Keyword ("where") => stop %&& TokenStoppers.Where
          | _ => false
        }
      }
    }

    comma_separated_list ['a] (f : void -> 'a) : list ['a] {
      def loop (acc) {
        match (peek_token ()) {
          | Token.Comma => shift (); loop (f () :: acc)
          | _ => List.Rev (acc)
        }
      };
      loop ([f ()])
    }

    process_groups ['a] (first : Token, name : string, f : void -> 'a) : list ['a]
      where 'a : class
    {
      push_stream (first);
      mutable members = [];
      while (stream != null) {
        def child = f ();
        when (child != null) members = child :: members;
      }
      pop_stream (name);
      List.Rev (members)
    }
    #endregion Utility functions operating on current token stream

    static fatal_error (tok : Token, msg : string) : PExpr {
      Error (tok, msg);
      PExpr.Error (tok.Location);
    }

    static fatal_error (loc : Location, msg : string) : PExpr {
      Message.Error (loc, msg);
      PExpr.Error (loc);
    }
    
    static Error (tok : Token, msg : string) : void {
      Message.Error (tok.Location, "parse error near " + tok.ToString (true) + ": " + msg)
    }

    mkname (id : string) : Name {
      Name.NameInCurrentColor (id, env)
    }
    
    TokenMap ['a] (tokens : Token, f : void -> 'a) : list ['a] {
      mutable result = [];
      foreach (tok is Token.LooseGroup in tokens) {
        push_stream (tok.Child);
        result = f () :: result;
        pop_stream ("group of tokens");
      }
      List.Rev (result);
    }
    
    get_splicable_id () : Splicable {
      assert (env != null);
      def tok = get_token ();
      def loc = tok.Location;
      match (tok) {
        | Token.Identifier (n) => Splicable.Name (loc, mkname (n))
        
        | Token.IdentifierToComplete (pref) => Splicable.HalfId (loc, mkname (pref))
          
        | Token.Operator ("$") =>
          def second = get_token ();
          def loc = loc + second.Location;          
          match (second) {
            | Token.Identifier (id)  =>
              Splicable.Expression (loc, PExpr.Ref (loc, mkname (id)))

            | Token.RoundGroup (group) =>
              push_stream (group);
              def expr =
                match (parse_expr_sequence (TokenStoppers.Colon)) {
                  | [x] => x
                  | _ =>
                    fatal_error (loc, "only single element in `$(...)' is allowed");
                }
              def result = 
              match (peek_token ()) {
                | Token.LooseGroup (Token.Operator (":") as colon) =>
                  shift ();
                  push_stream (colon.Next);
                  def tok = get_token ();
                  pop_stream ("spliced expression");
                  match (tok) {
                    | Token.Identifier (ty) =>
                      Splicable.Expression (loc, PExpr.TypeEnforcement
                        (loc, expr, PExpr.Ref (loc, mkname (ty))))
                        
                    | x =>
                      Splicable.Expression (fatal_error (x, "expecting identifier specifying splicing type"))
                  }

                | Token.EndOfGroup => Splicable.Expression (loc, expr)
                | x => Splicable.Expression (fatal_error (x, "expecting end of spliced expression"))
              }
              pop_stream ("spliced expression");
              result

            | Token.Keyword ("_") => Splicable.Expression (loc, PExpr.Wildcard (loc))

            | x => Splicable.Expression (fatal_error (x, "expecting expression after `$' operator"))
          }
              
        | Token.Keyword ("_") => Splicable.Name (loc, mkname (Util.tmpname ("u")))
          
        | x => Splicable.Expression (fatal_error (x, "expecting identifier"))
      }
    }

   
    /** This function parses top level group updating global environment
        when entering into new namespace, using directive, class, etc.
     */
    ParseTopLevel (tok : Token.LooseGroup, mutable acc : list [TopDeclaration]) : list [TopDeclaration]
    {
      eat_assembly_attributes (tok);
      def tokens = tok.Child;
      match (tokens) {
        | Token.Namespace (new_env, _) =>
          def begin_env = env;

          // update current environment
          env = new_env;
          
          // parse elements of namespace with new environment enabled
          foreach (child :> Token.LooseGroup in tokens)
            acc = ParseTopLevel (child, acc);

          // bring env from outside of namespace
          env = begin_env;
          acc

        | Token.Using (new_env) =>
          env = new_env;
          acc

        | Token.SquareGroup (null) => Error (tok, "empty custom attribute"); acc
        
        | null => acc

        | _ => ParseTypeDeclaration (tokens) :: acc
      }
    }

    // assembly attribute
    eat_assembly_attributes (tok : Token.LooseGroup) : void {
      match (tok.Child) {
        | Token.SquareGroup (LooseGroup (Identifier ("assembly"))) as square =>
          mutable assembly_custom = [square];
          def mods = Modifiers (NemerleAttributes.None, []);
          take_attributes_out (ref assembly_custom, System.AttributeTargets.Assembly,
                               true, mods);
          foreach (cust in mods.custom_attrs)
            AttributeCompiler.AddAssemblyAttribute (env, cust);
            
          tok.Child = square.Next;
          eat_assembly_attributes (tok);
            
        | _ => ()
      }
    }
    
    /** Parses toplevel type (like class, interface, etc.).
        Expects [toks] to be first token in type declaration.
     */
    ParseTypeDeclaration (toks : Token) : TopDeclaration
    {
      push_stream (toks);
      mutable customs_token = get_customs ();
      def mods = Modifiers (get_modifiers (), []);
      take_attributes_out (ref customs_token, System.AttributeTargets.Class, true, mods);

      def result = ParseTypeDeclaration (mods);
      pop_stream ("type declaration");

      result;
    }

    /** Parses toplevel type (like class, interface, etc.).
        We are inside stream with modifiers and custom attributes already read.
     */
    ParseTypeDeclaration (mods : Modifiers) : TopDeclaration
    {
      def tok = get_token ();
      def res =
        match (tok) {
          | Token.Keyword (key) =>
            match (key) {
              | "type" | "class" | "struct" | "module" | "interface" | "enum"
              | "variant" =>
                /// first get name of this declartion
                def name = get_splicable_id ();
                // now generic type parameters
                def tyvars = parse_tyvars ();                  
                def t_extends =
                  match (peek_token ()) {
                    | Token.Operator (":") =>
                      shift ();
                      match (maybe_parse_ellipsis ()) {
                        | Some (e) => [e]
                        | _ =>
                          // parse , separated sequence as one expression
                          comma_separated_list (fun () { parse_expr (TokenStoppers.All) });
                      }
                    | _ => []
                  };
                // where constraints for specified generic type parameters
                def typarms = parse_where_constraints (tyvars);
                parse_top_extensions (mods, MacroTargets.Class);

                mutable members = null;
                mutable end_loc = name.Location;
                unless (key == "type")
                  match (get_token ()) {
                    | Token.BracesGroup (children) as tok =>
                      end_loc = tok.Location;
                      push_stream (children);
                      match (maybe_parse_ellipsis ()) {
                        | Some(e) =>
                          pop_stream ();
                          members = [ClassMember.Field (e.loc, null, null, e)]; // special encoding for <[ decl: class A { ..$mems } ]>
                          
                        | _ =>
                          pop_stream ();
                          when (key == "variant") in_variant = true;
                          members = process_groups (children, "type member", ParseClassMember);
                          in_variant = false
                      }

                    | x => Error (x, "expecting type body");
                  }

                def td =
                  match (key) {
                    | "class" => TopDeclaration.Class (typarms, t_extends, members)
                    | "struct" =>
                      mods.mods |= NemerleAttributes.Struct;
                      TopDeclaration.Class (typarms, t_extends, members)

                    | "module" =>
                      mods.mods |= NemerleAttributes.Static;
                      TopDeclaration.Class (typarms, t_extends, members)

                    | "type" =>
                      expect_operator ("=");
                      def t = parse_expr ();
                      expect_empty ("type alias declaration");
                      TopDeclaration.Alias (typarms, t)

                    | "interface" => TopDeclaration.Interface (typarms, t_extends, members)
                    | "variant" => TopDeclaration.Variant (typarms, t_extends, members)
                    | "enum" => TopDeclaration.Enum (t_extends, members)

                    | _ => Util.ice ()
                  };
                td.name = name;
                td.loc = tok.Location + end_loc;
                td

              | "delegate" =>
                def h = parse_fun_header (null);
                def td = TopDeclaration.Delegate (h);
                expect_empty ("delegate declaraion");
                td.loc = tok.Location + h.Location;
                td.name = h.name;                
                td

              | "macro" => 
                def header = parse_fun_header (null);
                def synt = 
                  match (peek_token ()) {
                    | Token.Keyword ("syntax") =>
                      shift ();
                      match (get_token ()) {
                        | Token.RoundGroup (groups) => parse_expr_sequence (groups)
                        | t => Error (t, "expecting comma sperated list of syntax specifiers in `()'"); []
                      }
                    | _ => []
                  }
                def expr = parse_block ([]);
                def res = TopDeclaration.Macro (header, synt, expr);
                res.name = header.name;
                res.loc = tok.Location + header.Location;
                res

              | _ => Error (tok, "expecting type declaration"); TopDeclaration.Delegate (null);
            }
          | x => Error (x, "expecting type declaration"); TopDeclaration.Delegate (null);
        };
      res.modifiers = mods;
      res
    }

    
    ParseClassMember () : ClassMember
    {
      match (get_token ()) {
        | Token.LooseGroup (toks) =>
          push_stream (toks);
          mutable customs = get_customs ();
          def mods = Modifiers (get_modifiers (), []);

          parse_top_extensions (mods, MacroTargets.All & ~MacroTargets.Parameter);

          def tok = peek_token ();
          def result = 
            match (tok) {
              | Token.Operator ("$") | Token.Identifier =>
                def id = get_splicable_id ();
                mutable is_property = false;
                def (tyvars, parms) =
                  match ((peek_token (), peek_second_token ())) {
                    | (Token.SquareGroup, Token.SquareGroup) =>
                      is_property = true;
                      (parse_tyvars (), parse_parameters ());

                    | (Token.SquareGroup, Token.RoundGroup) =>
                      (parse_tyvars (), parse_parameters ());            

                    | (Token.SquareGroup, _) =>
                      is_property = true;                      
                      ((null, null), parse_parameters ());                        

                    | (Token.RoundGroup, _) =>
                      ((null, null), parse_parameters ());

                    | _ =>
                      is_property = true; // most probably it is a field or property
                      ((null, null), [])
                  }

                def ret_type = parse_return_type (true);
                def typarms = parse_where_constraints (tyvars);

                if (is_property)
                  parse_property (mods, customs, id, ret_type, parms, typarms)
                else {
                  def header = Fun_header (tok.Location + ret_type.Location,
                                           name = id,
                                           ret_type = ret_type, 
                                           parms = parms, 
                                           typarms = typarms);
                  take_attributes_out (ref customs, System.AttributeTargets.Method, true, mods);
                  parse_method (header, mods);
                }


              | Token.Keyword ("this") =>
                shift ();
                def parms = parse_parameters ();                
                def header = Fun_header (tok.Location,
                                         name = Splicable.Name (mkname (".ctor")),
                                         ret_type = PExpr.Void (),
                                         parms = parms,
                                         typarms = Typarms.Empty);
                take_attributes_out (ref customs, System.AttributeTargets.Method, true, mods);
                parse_method (header, mods);

              | Token.Keyword ("event") =>
                shift ();
                parse_event (tok.Location, mods, customs)

              | Token.Operator ("|") =>
                shift ();
                def target =
                  if (in_variant) System.AttributeTargets.Class else System.AttributeTargets.Field;
                take_attributes_out (ref customs, target, true, mods);
                    
                unless (mods.Attributes == NemerleAttributes.None)
                  Error (tok, "modifiers not allowed on options"); 
                parse_option (mods)

              | _ =>
                take_attributes_out (ref customs, System.AttributeTargets.Class, true, mods);
                def td = ParseTypeDeclaration (mods);
                ClassMember.TypeDeclaration (loc = td.loc, 
                                             name = td.name, 
                                             modifiers = td.modifiers, 
                                             td = td)
            }
          pop_stream ("class member");
          result
            
        | _ => null
      }
    }
    
    parse_method (h : Fun_header, modifiers : Modifiers) : ClassMember
    {
      mutable impl = [];
      when (flag_keyword ("implements"))
        impl = comma_separated_list (fun () { parse_expr (TokenStoppers.All) });

      parse_top_extensions (modifiers, MacroTargets.Method);
        
      def body =
        match (peek_token ()) {
          | Token.BracesGroup as brac => 
            shift ();
            // we skip body parsing in completion mode,
            // to save time and ignore errors
            if (Nemerle.Completion.Engine.IsInCompletionMode)
              FunBody.Parsed (PExpr.Sequence (brac.Location, []))
            else
              FunBody.Parsed (parse_block (brac, h.parms))

          | Token.EndOfGroup => FunBody.Abstract ()
          | Token.Operator ("$") =>
            def result = FunBody.Parsed (parse_spliced_expr ());
            new_group_beginning ();
            result

          | x => Error (x, "expecting method body"); FunBody.Abstract ();
        };

      ClassMember.Function (header = h, name = h.name, modifiers = modifiers, 
                            loc = h.loc, body = body, kind = FunKind.Method (impl))
    }

    parse_option (modifiers : Modifiers) : ClassMember
    {
      def make_option (id : Splicable, children) {
        if (in_variant) {
          def members =
            if (children != null)
              process_groups (children, "elements of variant option", ParseClassMember);
            else [];
          def td = TopDeclaration.VariantOption (name = id, loc = id.Location,
                                                 modifiers = modifiers,
                                                 decls = members,
                                                 typarms = null);
          ClassMember.TypeDeclaration (loc = id.loc, name = null, modifiers = null, td = td)
        }
        else {
          when (children != null)
            Error (children, "only variant types can hold options with members");
          ClassMember.EnumOption (id.Location, id, modifiers, None ());
        }
      }
      
      def id = get_splicable_id ();
      match (peek_token ()) {
        | Token.BracesGroup (children) =>
          shift ();
          make_option (id, children)

        | Token.EndOfGroup => make_option (id, null)
        | Token.Operator ("=") =>
          shift ();
          def val = Some (parse_expr (TokenStoppers.Pipe));
          new_group_beginning ();
          ClassMember.EnumOption (id.Location, id, Modifiers (NemerleAttributes.None, []), val);

        | _ =>
          new_group_beginning ();
          make_option (id, null);
      }
    }

    parse_accessor_body (p : list [Fun_parm], implemented : list [PExpr]) : FunKind * FunBody
    {
      match (peek_token ()) {
        | Token.EndOfGroup => (FunKind.Method (implemented), FunBody.Abstract ())
        | _ =>
          def bl = parse_block (p);
          (FunKind.Method (implemented), FunBody.Parsed (bl))
      };
    }
    
    parse_property (mods : Modifiers, mutable customs : list [Token.SquareGroup],
                    id : Splicable, ret_type : PExpr, parms : list [Fun_parm],
                    typarms : Typarms) : ClassMember
    {
      def loc = id.Location + ret_type.Location;
      mutable impl = [];
      when (flag_keyword ("implements"))
        impl = comma_separated_list (fun () { parse_expr (TokenStoppers.All) });

      match (peek_token ()) {
        | Token.BracesGroup as group =>
          shift ();
          take_attributes_out (ref customs, System.AttributeTargets.Property, true, mods);
          mutable setter = None ();
          mutable getter = None ();

          def plain_name = match (id) { | Splicable.Name (n) => n.Id | _ => "" };
          
          def prop_ty = Util.locate (loc, 
            match (parms) {
              | [parm] => <[ $(parm.ty) -> $ret_type ]>
              | [] => ret_type
              | _ =>
                def input_parms = List.Map (parms, fun (x : Fun_parm) { x.ty });
                <[ @* (..$input_parms) -> $ret_type ]>
            }
          );

          foreach (Token.LooseGroup (toks) in group) {
            push_stream (toks);
            mutable mycustoms = get_customs ();
            def mymods = get_modifiers ();

            match (get_token ()) {
              | Token.Identifier (i) as nametok =>
                def loc = nametok.Location;
                Util.locate (loc, 
                  match (i) {
                    | "get" =>
                      unless (getter.IsNone)
                        Message.Error ("property cannot have multiple getters");

                      def method_atts = Modifiers (mymods, []);                        
                      take_attributes_out (ref mycustoms, System.AttributeTargets.Method, true, method_atts);
                      parse_top_extensions (method_atts, MacroTargets.Method);

                      def (kind, bd) = parse_accessor_body ([], impl);
                      def name = Splicable.Name (mkname ("get_" + plain_name));
                      def fh = Fun_header (loc, name, ret_type, parms);
                      getter = Some (ClassMember.Function (fh.name, method_atts, fh, kind, bd));

                    | "set" =>
                      unless (setter.IsNone)
                        Message.Error ("property cannot have multiple setters");

                      def method_atts = Modifiers (mymods, []);                        
                      take_attributes_out (ref mycustoms, System.AttributeTargets.Method, false, method_atts);
                      parse_top_extensions (method_atts, MacroTargets.Method);                      

                      def par_atts = Modifiers (NemerleAttributes.None, []);
                      take_attributes_out (ref mycustoms, System.AttributeTargets.Parameter, true, par_atts);
                      
                      def val_n = Splicable.Name (mkname ("value"));
                      def setter_parms = parms + [Fun_parm (val_n, ret_type, par_atts)];

                      def (kind, bd) = parse_accessor_body (setter_parms, impl);
                      def name = Splicable.Name (mkname ("set_" + plain_name));
                      def fh = Fun_header (loc, typarms, name, PExpr.Void (), setter_parms);
                      setter = Some (ClassMember.Function (fh.name, method_atts, fh, kind, bd));

                    | _ =>
                      def nm = Splicable.Name (mkname (i));
                      def t = parse_return_type (false);
                      def attrs = Modifiers (mymods, []);
                      take_attributes_out (ref mycustoms,
                                           System.AttributeTargets.Field, true, attrs);
                      
                      def embed = PExpr.Quoted (SyntaxElement.ClassMember (
                                    ClassMember.Field (loc + t.Location, nm, attrs, t)));
                      mods.AddCustomAttribute (<[
                        $(mkname ("Nemerle") : name).InternalMacros.PropertyEmbeddedField ($embed)
                      ]>);
                  }
                );

              | x => Error (x, "expecting property accessor"); 
            }
            pop_stream ("property member");
          }
          ClassMember.Property (loc, id, mods, ret_type, prop_ty, parms, getter, setter)


        | Token.Operator ("=") =>
          shift ();
          def val = parse_expr ();
          Util.locate (val.Location,
            mods.AddCustomAttribute (<[
              $(mkname ("Nemerle") : name).InternalMacros.StaticInitializer ($val)
            ]>)
          );
          take_attributes_out (ref customs, System.AttributeTargets.Field, true, mods);
          ClassMember.Field (loc, id, mods, ret_type)
          
        | Token.EndOfGroup =>
          take_attributes_out (ref customs, System.AttributeTargets.Field, true, mods);
          ClassMember.Field (loc, id, mods, ret_type)
          
        | t => Error (t, "expecting `;' or `{ }' in field / property declaration"); null
      }
    }

    /** Parse event definition from the place after 'event' keyword
        name {
          add {  }
          remove { }
          potiantially many field definitions
        }
     */
    parse_event (mutable loc : Location, mods : Modifiers,
                 mutable customs : list [Token.SquareGroup]) : ClassMember
    {
      def id = get_splicable_id ();
      def ret_type = parse_return_type (false);
      loc += ret_type.Location;
      def plain_name = match (id) { | Splicable.Name (n) => n.Id | _ => "" };      
      def val_n = Splicable.Name (loc, mkname ("value"));
      mutable remove = null;
      mutable add = null;
      
      match (peek_token ()) {
        | Token.BracesGroup as group =>
          shift ();
          take_attributes_out (ref customs, System.AttributeTargets.Event, true, mods);

          foreach (Token.LooseGroup (toks) in group) {
            push_stream (toks);
            mutable mycustoms = get_customs ();
            def mymods = get_modifiers ();

            match (get_token ()) {
              | Token.Identifier (i) as nametok =>
                def mloc = nametok.Location;
                match (i) {
                  | "remove" | "add" =>
                    if (i == "remove")
                      unless (remove == null)
                        Message.Error (mloc, "event cannot have multiple remove methods");
                    else
                      unless (add == null)
                        Message.Error (mloc, "event cannot have multiple add methods");

                    def method_atts = Modifiers (mymods, []);                           
                    take_attributes_out (ref mycustoms, System.AttributeTargets.Method, false, method_atts);
                    parse_top_extensions (method_atts, MacroTargets.Method);                    

                    def par_atts = Modifiers (NemerleAttributes.None, []);
                    take_attributes_out (ref mycustoms, System.AttributeTargets.Parameter, true, par_atts);
                    def method_parms = [Fun_parm (val_n, ret_type, par_atts)];
                    
                    def (kind, bd) = parse_accessor_body (method_parms, []);
                    def name = Splicable.Name (mloc, mkname (i + "_" + plain_name));
                    def fh = Fun_header (mloc, name, PExpr.Void (mloc), method_parms);
                    def method = ClassMember.Function (loc, name, method_atts, fh, kind, bd);
                    if (i == "remove")
                      remove = method;
                    else
                      add = method;

                  | _ =>
                    def nm = Splicable.Name (loc, mkname (i));
                    def t = parse_return_type (false);
                    def attrs = Modifiers (mymods, []);
                    take_attributes_out (ref mycustoms, System.AttributeTargets.Field, true, attrs);
                    def f = ClassMember.Field (mloc + t.Location, nm, attrs, t);
                    def embed = PExpr.Quoted (mloc, SyntaxElement.ClassMember (f));
                    mods.AddCustomAttribute (<[
                      $(mkname ("Nemerle") : name).InternalMacros.EventEmbeddedField ($embed)
                    ]>);
                }

              | x => Error (x, "expecting event accessor"); 
            }
            pop_stream ("property member");
          }
          when (add == null || remove == null)
            Message.Error (loc, "both of event accessors `add' and 'remove' must be specified");
          
          ClassMember.Event (loc, id, mods, ret_type, null, add, remove)
          
        | Token.EndOfGroup =>
          // first take out event attributes (those without target also get here)
          take_attributes_out (ref customs, System.AttributeTargets.Event, false, mods);
          
          /// auto-generated field
          def fmods = NemerleAttributes.Private %| NemerleAttributes.Mutable;
          def field_name = mkname (Util.tmpname (plain_name));
          def field_attrs = Modifiers (fmods, []);
          take_attributes_out (ref customs, System.AttributeTargets.Field, false, field_attrs);
          def field = ClassMember.Field (loc, Splicable.Name (field_name), field_attrs, ret_type);

          def method_atts = Modifiers (mods.Attributes, []);                                         
          take_attributes_out (ref customs, System.AttributeTargets.Method, true, method_atts);

          def method_parms = [Fun_parm (val_n, ret_type, Modifiers.Empty)];

          def name = Splicable.Name (loc, mkname ("add_" + plain_name));
          def fh = Fun_header (loc, name, PExpr.Void (loc), method_parms);
          // funbody is filled during typing
          add = ClassMember.Function (loc, name, method_atts, fh, FunKind.Method ([]), null);

          def name = Splicable.Name (loc, mkname ("remove_" + plain_name));
          def fh = Fun_header (loc, name, PExpr.Void (loc), method_parms);
          remove = ClassMember.Function (loc, name, method_atts, fh, FunKind.Method ([]), null);

          ClassMember.Event (loc, id, mods, ret_type, field, add, remove)
          
        | t => Error (t, "expecting `;' or `{ }' in event declaration"); null
      }
    }
    
    maybe_parse_ellipsis () : option [PExpr]
    {
      match (peek_token ()) {
        | Token.LooseGroup (Token.Operator ("..") as tok) =>
          shift ();
          push_stream (tok);
          shift ();
          def spliced = parse_spliced_expr ();
          pop_stream ("ellipsis splicing");
          Some (PExpr.Ellipsis (tok.Location + spliced.Location, spliced))
          
        | Token.Operator ("..") as tok =>
          shift ();
          def spliced = parse_spliced_expr ();
          Some (PExpr.Ellipsis (tok.Location + spliced.Location, spliced))
          
        | _ => None ()
      }
    } 
    
    get_customs () : list [Token.SquareGroup] {
      def loop (acc) {
        match (peek_token ()) {
          | Token.SquareGroup (ch) as gr =>
            shift ();
            if (ch != null)
              loop (gr :: acc)
            else {
              Error (gr, "custom attributes brackets cannot be empty");
              loop (acc)
            }

          | _ => List.Rev (acc)
        }
      }
      match (peek_token ()) {
        | Token.Operator ("..") as begin =>
          mutable end = null;
          repeat (3) end = get_token ();
          end.Next = null;
          [Token.SquareGroup (begin.Location, Token.LooseGroup (begin.Location, begin))]

        | _ => loop ([])
      }
    }

    /** allowed targets are:
          assembly: assembly 
          module:  module   (not supprted currently)
          class, struct, interface, enum: type 
          delegate: type, return
          method: method, return
          parameter: param 
          field: field 
          property indexer: property 
          property get accessor: method, return
          property set accessor: method, param, return
          event field: event, field, method
          event property: event, property (what is this?)
          event add: method, param
          event remove: method, param
    */
    take_attributes_out (from : ref list [Token.SquareGroup], 
                         what : System.AttributeTargets,
                         comply_on_other : bool,
                         mods : Modifiers) : void
    {
      from = List.RevFilter (from, fun (x : Token.SquareGroup) {
        mutable leave = true;
        match (x.Child) {
          | Token.LooseGroup (first) =>
            push_stream (first);
            def target = 
              match (first) {
                | Token.Identifier ("assembly") => System.AttributeTargets.Assembly
                | Token.Identifier ("field") => System.AttributeTargets.Field
                | Token.Keyword ("event") => System.AttributeTargets.Event
                | Token.Identifier ("method") => System.AttributeTargets.Method
                | Token.Keyword ("module") => System.AttributeTargets.Module
                | Token.Identifier ("param") => System.AttributeTargets.Parameter
                | Token.Identifier ("property") => System.AttributeTargets.Property
                | Token.Identifier ("return") => System.AttributeTargets.ReturnValue
                | Token.Keyword ("type") => System.AttributeTargets.Class
                | _ => System.AttributeTargets.All
              }
            when (target != System.AttributeTargets.All) {
              shift (); // ignore target token
              expect_operator (":");
            }
            if (target %&& what || target == System.AttributeTargets.All) {
              leave = false;
              // parse body of first element in this [ , , ]
              mods.AddCustomAttribute (parse_expr ());

              // parse all remaining in current bracket group [ , , , ]
              when (x.Child.Next != null) {
                x.Child = x.Child.Next;
                mods.custom_attrs = parse_expr_sequence (x.Child, mods.custom_attrs);
              }
              pop_stream ("custom attribute")
            }
            else {
              when (comply_on_other)
                Error (first, $"unexpected attribute target `$(first)'");
              // there are still some unpased tokens here, ignore them
              pop_stream ();                
            }

          | _ => Util.ice ("broken brackets in attribute")
        }
        leave
      });
    }
    
    get_modifiers () : NemerleAttributes
    {
      def loop (acc) {
        def tok = peek_token ();
        match (tok) {
          | Token.Keyword (key) =>
            def add_and_loop (attr : NemerleAttributes) {
              shift ();              
              when (attr %&& acc)
                Error (tok, "attribute '" + key + "' specified more than once");
              loop (attr %| acc)
            }
            match (key) {
              | "mutable" => add_and_loop (NemerleAttributes.Mutable)
              | "public" => add_and_loop (NemerleAttributes.Public)
              | "private" => add_and_loop (NemerleAttributes.Private)
              | "static" => add_and_loop (NemerleAttributes.Static)
              | "new" => add_and_loop (NemerleAttributes.New)
              | "protected" => add_and_loop (NemerleAttributes.Protected)
              | "internal" => add_and_loop (NemerleAttributes.Internal)
              | "abstract" => add_and_loop (NemerleAttributes.Abstract)
              | "sealed" => add_and_loop (NemerleAttributes.Sealed)
              | "override" => add_and_loop (NemerleAttributes.Override)
              | "virtual" => add_and_loop (NemerleAttributes.Virtual)
              | "volatile" => add_and_loop (NemerleAttributes.Volatile)
              | "partial" => add_and_loop (NemerleAttributes.Partial)
              | "extern" => add_and_loop (NemerleAttributes.Extern)
              | _ => acc
            }
          | _ =>
            // perform some sanity checks on the declared attributes
            when ((acc %&& NemerleAttributes.Virtual)
                  && (acc %&& NemerleAttributes.Override))
              Message.Warning (tok.Location, "the `virtual' attribute is "
                               "redundant, `override' implies `virtual'");
            acc
        }
      }
      loop (NemerleAttributes.None)
    }


    get_tyvar () : string * int {
      match (peek_token ()) {
        | Token.Identifier (tv) =>
          shift ();
          (tv, MacroColorizer.Color)
        | t => Error (t, "expecting type variable"); ("error", 0)
      }
    }
    
    parse_tyvars () : list [string * int] * PExpr
    {
      match (peek_token ()) {
        | Token.SquareGroup (null) as t =>
          shift ();
          Error (t, "expecting type variables"); (null, null)
          
        | Token.SquareGroup (children) as group =>
          push_stream ((children :> Token.LooseGroup).Child);          
          match (maybe_parse_ellipsis ()) {
            | Some (e) =>
              pop_stream ("type variables");
              shift ();              
              ([], e)

            | _ =>
              pop_stream ();
              shift (); // now we are after whole '[..]' group
              (TokenMap (group, get_tyvar), PExpr.Void ())
          }
        | _ => (null, null)
      };
    }

    /** `splicing_type' is for noting, that there are spliced tyvars
     * if yes, it's PType.Spliced with expression describing their list
     * else it's PType.Void ()
     */
    parse_where_constraints (tyvars : list [string * int], splicing_type : PExpr) 
    : Typarms 
    {
      def loop (acc) {
        if (flag_keyword ("where"))
          match (maybe_parse_ellipsis ()) {
            | Some (e) => (List.Rev (acc), e)
            | _ =>
              def tv = get_tyvar ();
              expect_operator (":");
              def types = comma_separated_list (fun () {
                match (peek_token ()) {
                  | Token.Keyword ("enum" as key) as tok                  
                  | Token.Keyword ("class" as key) as tok
                  | Token.Keyword ("struct" as key) as tok =>
                    shift ();
                    PExpr.Ref (tok.Location, mkname (key))

                  | Token.Keyword ("new") as tok =>
                    shift ();
                    match (get_token ()) {
                      | Token.RoundGroup (null) => ()
                      | t => Error (t, "expected `()' after `new' in generic constraint")
                    }
                    PExpr.Ref (tok.Location, mkname ("new"))

                  | _ => parse_expr (TokenStoppers.Braces | TokenStoppers.Where | TokenStoppers.Equal)
                }
              });
              def acc = List.FoldLeft (types, acc, fun (t, acc) { 
                Constraint (tv, t) :: acc 
              });
              loop (acc)
          }
        else (List.Rev (acc), PExpr.Void ())
      };
      if (tyvars != null) {
        def (where_cts, where_spl_t) = loop ([]);
        match ((splicing_type, where_spl_t)) {
          | (PExpr.Void, PExpr.Void) => Typarms (tyvars, where_cts)
          | _ =>
            // we have spliced type variables
            Typarms (tyvars, Constraint (("", 0), <[ ($splicing_type, $where_spl_t) ]>) 
                     :: where_cts)
        }
      }
      else Typarms.Empty
    }

    /**
     *  Parse plain functional header (with optional '[..]' generic parameters
     *                                 and '(..)' fun parameters)
     */
    parse_fun_header (mutable name : Splicable, allow_patterns = false) : Fun_header
    {
      when (name == null) name = get_splicable_id ();
      def tyvars = parse_tyvars ();      
      def parms = parse_parameters (allow_patterns);
      def ret_type = parse_return_type (true);
      def typarms = parse_where_constraints (tyvars);

      Fun_header (name.Location + ret_type.Location,
                  name = name,
                  ret_type = ret_type, 
                  parms = parms, 
                  typarms = typarms)
    }
   

    parse_parameter () : Fun_parm
    {
      parse_parameter (false)
    }

    /** Parse function parameter definition from given LooseGroup */
    parse_parameter (allow_patterns : bool) : Fun_parm {
      mutable customs_token = get_customs ();
      def mods = Modifiers (get_modifiers (), []);
      take_attributes_out (ref customs_token, System.AttributeTargets.Parameter, true, mods);      

      parse_top_extensions (mods, MacroTargets.Parameter);

      mutable pattern = null;
     
      def id =
        match (peek_token ()) {
          | Token.Keyword ("params") =>
            shift ();
            mods.custom_attrs = <[ System.ParamArrayAttribute ]> :: mods.custom_attrs;
            get_splicable_id ()
            
          | Token.RoundGroup as tok when allow_patterns =>
            pattern = parse_expr ();
            Splicable.Name (tok.Location, Macros.NewSymbol ("pat"));
            
          | _ =>
            get_splicable_id ()
        }
        
      def t =
        match (peek_token ()) {
          | Token.Operator (":") => shift (); parse_expr (TokenStoppers.Equal)
          | _ => PExpr.Wildcard (id.Location)
        }

      match (peek_token ()) {
        | Token.Operator ("=") => 
          shift ();
          def e = parse_expr ();
          mods.custom_attrs = <[ System.ComponentModel.DefaultValueAttribute ($e)
                              ]> :: mods.custom_attrs;
        | _ => ()
      }
      
      Fun_parm (loc = id.loc, name = id, ty = t,
                modifiers = mods, pattern_hack = pattern)
    }

    /// parameters of (a:foo,b,c) or [a,b:foo,c] kind
    parse_parameters (allow_patterns = false) : list [Fun_parm] {
      def group = get_token ();
      match (group) {
        | Token.RoundGroup (null) | Token.SquareGroup (null) => []
        | Token.RoundGroup (child)          
        | Token.SquareGroup (child) =>
          push_stream (child);
          match (maybe_parse_ellipsis ()) {
            | Some (e) =>
              pop_stream ("ellipsis spliced expression");
              [Fun_parm (e.loc, Splicable.Name (mkname ("")), PExpr.Void (e.loc), 
                         Modifiers (NemerleAttributes.None, [e]))]
                         
            | _ => pop_stream (); TokenMap (group, fun () { parse_parameter (allow_patterns) });
          }
        | _ => Error (group, "expecting function parameters"); []
      }
    }

    parse_return_type (allow_inference : bool) : PExpr {
      match (peek_token ()) {
        | Token.Operator (":") =>
          shift ();
          parse_expr (TokenStoppers.Braces | TokenStoppers.Dollar 
                      | TokenStoppers.Equal | TokenStoppers.Where)
          
        | _ when allow_inference => PExpr.Wildcard (last_tok.Location)
        | t => Error (t, "expecting `:' and type specifier for class member");
          PExpr.Wildcard (last_tok.Location)
      };
    }
       
    parse_block (tok : Token.BracesGroup, parms : list [Fun_parm]) : PExpr
    {
      def loc = tok.Location;
      mutable res =
        match (tok.Child) {
          | null => PExpr.Sequence (loc, [])

          // function body with embedded matching { | pat => bd | ... }
          | Token.LooseGroup (Token.Operator ("|")) as group =>
            // convert function's parameters to tuple to be matched
            def parms_to_tupl (prs : list [Fun_parm], acc) {
              match (prs) {
                | [] => PExpr.Tuple (group.Location, List.Rev (acc))
                | (Fun_parm where (name = Splicable.Name (x), loc = l)) :: xs => 
                  parms_to_tupl (xs, PExpr.Ref (l, x) :: acc)
                | x :: _  => fatal_error (x.Location, "illegal spliced parameter?");
              }
            };
            def cases = process_groups (group, "function body", parse_match_case);
            def to_be_matched =
              match (parms) {
                | [Fun_parm where (name = Splicable.Name (x), loc = l)] => PExpr.Ref (l, x)
                | _::_::_ => parms_to_tupl (parms, [])

                // syntax encountered is normally incorrect, but we allow it, so macros
                // and syntax extensions can reshape it
                | [] => null
                
                | x :: _ => fatal_error (x.Location, "illegal spliced parameter?");
            }
            PExpr.Match (loc, to_be_matched, cases)
            
          // standard body enclosed by { ... }
          | groups => 
            PExpr.Sequence (loc, parse_expr_sequence (groups));
        }
        
      foreach (p in parms)
        when (p.pattern_hack != null) {
          res = <[
            def $(p.pattern_hack) = $(p.name.GetName () : name); 
            $res 
          ]>;
          p.pattern_hack = null;
        }

      res
    }

    /// parse {...} or $body - usually it is a body of method
    parse_block (parms : list [Fun_parm]) : PExpr
    {
      def tok = get_token ();
      match (tok) {
        // entire function body may be spliced
        | Token.Operator ("$") =>
          push_back (); parse_spliced_expr ()

        | Token.BracesGroup as brac => parse_block (brac, parms)
        | x => fatal_error (x, "expected `{' at the beginning of function body"); 
      }
    }

    make_operator_call (name : string, e1 : PExpr, e2 : PExpr) : PExpr
    {
      def loc = e1.Location + e2.Location;
      def make_splicable (e) {
        | PExpr.Ref (n) => Splicable.Name (e.Location, n)
        | PExpr.ToComplete (n) => Splicable.HalfId (e.Location, n)
        | PExpr.Spliced (s) => Splicable.Expression (e.Location, s)
        | _ =>
          Message.Error (e2.Location, "expecting identifier after `.'");
          null
      }
      match (name) {
        | "." =>
          match (e2) {
            | PExpr.ListLiteral (args) => PExpr.GenericSpecifier (loc, e1, args)
            | _ => PExpr.Member (loc, e1, make_splicable (e2))
          }

        | "=" => PExpr.Assign (loc, e1, e2)
        | ":" => PExpr.TypeEnforcement (loc, e1, e2)
        | ":>" => PExpr.TypeConversion (loc, e1, e2)
        | "::" => Util.locate (loc, <[ list.Cons ($e1, $e2) ]>)
        | "as" => PExpr.As (loc, e1, make_splicable (e2))
        | "where" => PExpr.Where (loc, e1, e2)
        | "is" => PExpr.Is (loc, e1, e2)

        | "matches" =>
          Util.locate (loc, <[ match ($e1) { | $e2 => true | _ => false } ]>)
          
        | _ => PExpr.Call (loc, PExpr.Ref (loc, mkname (name)), [e1, e2])
      }
    }

    make_operator_call (name : string, e : PExpr) : PExpr
    {
      PExpr.Call (e.Location, PExpr.Ref (e.Location, mkname (name)), [e])
    }
    
    make_operator_call (name : string, parms : list [PExpr]) : PExpr
    {
      def head = List.Head (parms);
      def loc = head.Location;
      match (name) {
        | "(" => PExpr.Call (loc, head, List.Tail (parms))
        | "[" => PExpr.Indexer (loc, head, List.Tail (parms))
        | _ => PExpr.Call (loc, PExpr.Ref (loc, mkname (name)), parms)
      }
    }
    
    static roll_exprs (exprs : ref list [PExpr], begin : list [PExpr]) : list [PExpr]
    {
      mutable result = [];
      while ((exprs : object) != begin) {
        match (exprs) {
          | x :: xs =>
            result = x :: result;
            exprs = xs;
          | _ => Util.ice ("invalidated expressions stack")
        }
      }
      result
    } 
    
    roll_operators (exprs : ref list [PExpr], ops : ref list [OperatorInfo], priority : int) : void
    {
      def loop () {
        match (ops) {
          | [] => ()
          | x :: xs =>
            // a * b + c --- *'s right is higher, than +'s left, so a*b rolls
            // when + occurs
            when (x.RightPriority >= priority) {
              // we take all expressions, which appeared after this operator
              // appeared on stack (and sometimes the one from top at that time)
              def expr = 
                match (x) {
                  | nary is NaryOperatorInfo =>
                    def parms = roll_exprs (ref exprs, nary.ExprsBegin);
                    make_operator_call (x.Name, parms)
                    
                  | _ is BinaryOperatorInfo =>
                    match (exprs) {
                      | e1 :: e2 :: rest =>
                        exprs = rest;
                        make_operator_call (x.Name, e2, e1)
                      | _ => Util.ice ("not enough parms for binary")
                    }
                  | _ =>
                    def e = List.Head (exprs);
                    exprs = List.Tail (exprs);
                    make_operator_call (x.Name, e);
                }
              
              // put new expression on the stack
              exprs = expr :: exprs;
              ops = xs;
              loop ();
            }
        }
      }
      loop ();
    }

    parse_expr () : PExpr { parse_expr (TokenStoppers.None); }
    
    /** Parse expression from current stream of tokens.
        Reading is terminated before every token from [stop].

        This is the top expression parsing method, it uses infix to RPN method
        to parse operators and [parse_embedded_expr] to parse expression in between.
        */
    parse_expr (stop : TokenStoppers) : PExpr
    {
      mutable _expr_stack = [];
      mutable rpn_op_stack = [];
      mutable _op_context = OpContext.Prefix;

      def loop () {
        if (_op_context == OpContext.Prefix)
          match (peek_token ()) {
            | Token.Operator ("..") =>
              shift ();
              def expr = parse_spliced_expr ();
              _expr_stack = PExpr.Ellipsis (expr.Location, expr) :: _expr_stack;
              expect_empty ("ellipsis expression");
              _op_context = OpContext.Postfix;
              loop ()

            | Token.Keyword (str)
            | Token.Operator (str) when str != "$" =>
              match (env.SyntaxKeywords.Find (str)) {
                | Some (syntax_definition) when syntax_definition.Target %&& ((1 << 15) :> MacroTargets) =>
                  shift ();
                  _expr_stack = parse_syntax_definition (syntax_definition, stop) :: _expr_stack;
                  // after raw expression we are in postfix context
                  _op_context = OpContext.Postfix;
                  loop ()

                | _ =>
                  def info =
                    match (str) {
                      | "-" => OperatorInfo.UnMinus
                      | "+" => OperatorInfo.UnPlus
                      | _ => env.LookupOperator (str);
                    }
                  
                  if (info is UnaryOperatorInfo) {
                    shift ();
                    rpn_op_stack = info :: rpn_op_stack;
                    // we leave prefix context here (for things like `!!expr')
                  }
                  else {
                    def expr = parse_embedded_expr (stop);
                    _expr_stack = expr :: _expr_stack;
                    // after raw expression we are in postfix context
                    _op_context = OpContext.Postfix;
                  }
                  loop ();
              }

            | _ =>
              def expr = parse_embedded_expr (stop);
              _expr_stack = expr :: _expr_stack;
              // after raw expression we are in postfix context
              _op_context = OpContext.Postfix;
              loop ()
          }
        else {
          mutable group = null : Token;
          def info = 
            match (peek_token ()) {
              | Token.Operator ("*") =>
                NaryOperatorInfo (OperatorInfo.ProductInfo);

              | Token.Operator (str) =>
                if (should_stop (stop))
                  null
                else
                  env.FetchOperator (str)
                  
              | Token.Keyword (k) =>
                if (should_stop (stop))
                  null
                else
                  env.LookupOperator (k)

              | Token.SquareGroup (g)  =>
                group = g;                
                NaryOperatorInfo (OperatorInfo.SquareInfo);

              | Token.RoundGroup (g) =>
                group = g;
                NaryOperatorInfo (OperatorInfo.RoundInfo);

              | _ => null
            }
          if (info == null) {
            // this is the end of parsing, roll all operators
            roll_operators (ref _expr_stack, ref rpn_op_stack, 0);
            // there should be only one expr on the stack, return it
            match (_expr_stack) {
              | [x] => x
              | _ => Util.ice ("something different than one expression after rolling")
            }
          }
          else {
            shift ();
            roll_operators (ref _expr_stack, ref rpn_op_stack, info.LeftPriority);

            match (info) {
              | x is NaryOperatorInfo => x.ExprsBegin = List.Tail (_expr_stack);
              | _ => ()
            }
            
            // special case for x * y * z
            if (info.Name == "*") {
              _op_context = OpContext.Prefix;
              match (rpn_op_stack) {
                | x :: _ when x.Name == "*" => ()
                | _ => rpn_op_stack = info :: rpn_op_stack;
              }
            }
            else
              rpn_op_stack = info :: rpn_op_stack;

            // parse inner elements of bracket-like operators: `( ... )'
            _expr_stack = parse_expr_sequence (group, _expr_stack);

            when (info is BinaryOperatorInfo)
              _op_context = OpContext.Prefix;
            
            loop ();
          }
        }
      }
      loop ();
    }
    
    parse_literal () : PExpr
    {
      def tok = peek_token ();
      mutable cast_to = null;
      def lit = 
        match (tok) {
          | Token.Keyword ("null") => Literal.Null ()
          | Token.Keyword ("true") => Literal.Bool (true)
          | Token.Keyword ("false") => Literal.Bool (false)
          | Token.RoundGroup (null) => Literal.Void ()
          | Token.StringLiteral (s) => assert (s != null); Literal.String (s)
          
          | Token.FloatLiteral (n) => Literal.Float (n)
          | Token.DoubleLiteral (n) => Literal.Double (n)
          | Token.DecimalLiteral (n) => Literal.Decimal (n)
          | Token.CharLiteral (c) => Literal.Char (c)
          | Token.IntegerLiteral (l, cast) =>
            cast_to = cast; 
            if (cast == null) l.WithProperType () else l

          | _ => null
        }
      if (lit != null) {
        shift ();
        def res = PExpr.Literal (tok.Location, lit);
        if (cast_to == null) res
        else PExpr.TypeEnforcement (tok.Location, res, cast_to);
      }
      else null
    }

    parse_spliced_expr () : PExpr.Spliced {
      def id = get_splicable_id ();
      match (id) {
        | Splicable.Expression (e) => PExpr.Spliced (id.Location, e)
        | _ =>
          Message.FatalError (id.Location, "expected spliced expression");
      }
    }

    parse_match_case () : MatchCase {
      def loop2 (acc) {
        def res = parse_expr (TokenStoppers.Pipe | TokenStoppers.Arrow);
        match (get_token ()) {
          | Token.Operator ("=>") => List.Rev (res :: acc)
          | Token.Operator ("|") => loop2 (res :: acc)
          | x => Error (x, "found junk after pattern"); List.Rev (res :: acc)
        }
      };
      def case_group = (stream :> Token.LooseGroup);
      push_stream (case_group.Child);
      
      // eat pattern part and '=>'
      def pats = 
        match (peek_token ()) {
          | Token.Operator ("|") =>
            shift ();
            match (maybe_parse_ellipsis ()) {
              | Some (e) => expect_operator ("=>");
                match (peek_token ()) {
                  | Token.EndOfGroup | Token.Operator ("|") =>
                    Error (last_tok, "expecting body of match case")
                  | _ => ()
                }
                [e]
              | None => loop2 ([])
            }
          | _ => loop2 ([])
        };
      // modify beginning of current group to point at match case body  
      case_group.Child = stream; 
      pop_stream ();

      def expr =
        match (parse_expr_sequence (TokenStoppers.Pipe)) {
          | [x] => x
          | [] => fatal_error (last_tok, "match case body cannot be empty")
          | l => PExpr.Sequence (case_group.Child.Location, l)
        };

      MatchCase (pats, expr)
    }

    // parsing of expressions beginning (mostly) with unique keyword
    // we use this method to parse expressions occuring between operators, which
    // are parsed in [parse_expr] using infix to RPN method
    parse_embedded_expr (stop : TokenStoppers) : PExpr
    {
      def lit = parse_literal ();
      if (lit != null) lit
      else {
        def tok = get_token ();
        def loc = tok.Location;

        match (tok) {
          | Token.Keyword ("void") => PExpr.Void (loc)
          | Token.Keyword ("this") => PExpr.This (loc)
          | Token.Keyword ("base") => PExpr.Base (loc)
          | Token.Keyword ("_") => PExpr.Wildcard (loc)
          | Token.Keyword ("typeof") =>
            match (get_token ()) {
              | Token.RoundGroup (child) =>
                match (parse_expr_sequence (child, [])) {
                  | [x] => PExpr.Typeof (loc, x)
                  | _ => fatal_error (child.Location, "expecting single type in `typeof (...)'");
                }
              | x => fatal_error (x, "expecting `(..)' after `typeof'");
            }

          | Token.Keyword ("array") =>
            def rank =
              match (peek_token ()) {            
                | Token.Operator (".") =>
                  shift ();
                  match (get_token ()) {
                    | Token.SquareGroup (null) => <[ 1 ]>
                    | Token.SquareGroup (child) =>
                      match (parse_expr_sequence (child, [])) {
                        | [x] => x
                        | _ => Error (child, "expecting `[rank]' after `array.'"); <[ 1 ]>
                      }
                    | x => Error (x, "expecting `[rank]' after `array.'"); <[ 1 ]>
                  }
                | _ => <[ 1 ]>
              }

            def body = peek_token ();              
            match (body) {
              | Token.SquareGroup | Token.Operator ("$") =>
                PExpr.Array (loc, rank, parse_embedded_expr (TokenStoppers.All))

              | Token.RoundGroup (groups) =>
                shift ();
                def exprs = parse_expr_sequence (groups);
                PExpr.EmptyArray (loc + body.Location, exprs)

              | x =>
                fatal_error (x, "expected [ ..elements of array.. ], ( .."
                             "sizes of empty array.. )  or `array .[rank] [..]' in array creation");
            }
            
          // tuples
          | Token.RoundGroup (groups) =>
            match (parse_expr_sequence (groups)) {
              | [PExpr.Ellipsis] as exprs => PExpr.Tuple (loc, exprs)
              | [e] => e
              | exprs => PExpr.Tuple (loc, exprs)
            }

          | Token.SquareGroup (groups) =>
            PExpr.ListLiteral (loc, parse_expr_sequence (groups));

          | Token.BracesGroup as brac when !(stop %&& TokenStoppers.Braces) =>
            parse_block (brac, [])

          | Token.QuoteGroup (group) =>
            push_stream (group);
            def expr = parse_quotation ();
            pop_stream ("code quotation");
            expr
            
          | Token.Keyword ("ref") =>
            def refexpr = parse_expr (TokenStoppers.All);
            PExpr.ParmByRef (loc + refexpr.loc, refexpr)

          | Token.Keyword ("out") =>
            def outexpr = parse_expr (TokenStoppers.All);
            PExpr.ParmOut (loc + outexpr.loc, outexpr)

          | Token.Keyword ("throw") =>
            if (stream != null) {
              def exn = parse_expr (stop);
              PExpr.Throw (loc + exn.loc, exn)
            }
            else
              PExpr.Throw (loc, null)
            
          | Token.Keyword ("mutable") =>
            mutable ids = [];
            mutable vals = [];

            def loop () {
              def id = parse_expr (TokenStoppers.All);
              ids ::= id;
              match (peek_token ()) {
                | Token.Operator ("=") =>
                  shift ();
                  def val = parse_expr (stop);
                  vals ::= val;
                
                | _ => vals ::= null;
              }
              when (peek_token () is Token.Comma) {
                shift ();
                loop ();
              }
            }
            loop ();

            match ((ids, vals)) {
              | ([id], [val]) => PExpr.DefMutable (if (val != null) loc + val.loc else loc, id, val)
              | _ => PExpr.DefMutable (loc, PExpr.Tuple (loc, ids.Reverse ()),
                                       PExpr.Tuple (loc, vals.Reverse ()))
            }

          | Token.Keyword ("fun") =>
            def tyvars = parse_tyvars ();      
            def parms = parse_parameters (allow_patterns = true);
            def ret_type = parse_return_type (true);
            def typarms = parse_where_constraints (tyvars);

            def h = Fun_header (loc + ret_type.Location,
                                name = Splicable.Name (mkname ("")),
                                ret_type = ret_type, 
                                parms = parms, 
                                typarms = typarms);
            
            def expr = parse_block (parms);
            PExpr.Lambda (h.loc, Function_decl (h, expr))

          | Token.Operator ("$") =>
            match (peek_token ()) {
              | Token.StringLiteral (str) as stok =>
                shift ();
                // the sprint macro must get current location and global context...
                Util.locate (stok.Location, <[ $(mkname ("Nemerle") : name).IO.sprint ($(str : string)) ]>)

              | Token.SquareGroup (group) as grp =>
                shift ();
                push_stream (group);
                def expr = parse_expr_sequence (TokenStoppers.Pipe);
                match (stream) {
                  | Token.LooseGroup (Child = Token.Operator ("|")) as t =>
                    t.Child = t.Child.Next;
                  | _ => {}
                }
                def conds = parse_expr_sequence (TokenStoppers.None);
                def all = expr + conds;
                pop_stream ("list comprehension");
                Util.locate (grp.Location, 
                             <[ $(mkname ("Nemerle") : name).Utility.
                                ExpandListComprehension (..$all) ]>)

              | _ =>
                push_back ();
                parse_spliced_expr ();
            }
            
          | Token.Keyword ("match") =>
            def expr = 
              match (get_token ()) {
                | Token.RoundGroup (group) when group != null =>
                  match (parse_expr_sequence (group, [])) {
                    | [x] => x
                    | _ => fatal_error (group.Location, "expecting single expression in `match (..)'");
                  }
                | x => fatal_error (x, "expecting non-empty `(..)' after `match'");
              }
            match (get_token ()) {
              | Token.BracesGroup (Token.LooseGroup (Token.Operator ("..")) as group) =>
                push_stream (group);
                match (maybe_parse_ellipsis ()) {
                  | Some (e) =>
                    pop_stream ("match body");
                    PExpr.Match (loc, expr, [MatchCase ([], e)]);
                  | _        => Util.ice ()
                }
              | Token.BracesGroup (group) =>
                PExpr.Match (loc, expr, process_groups (group, "match body", parse_match_case))
                
              | x => fatal_error (x, "expecting '{' after 'match (e)'");
            }

          | Token.Keyword ("try") =>
            def try_body = parse_block ([]);
            mutable body = try_body; // resulting expression

            // exception handlers
            when (flag_sibling_keyword ("catch")) {
              def mktry (h : MatchCase, body) {
                match (h.patterns) {
                  | [PExpr.TypeEnforcement (PExpr.Wildcard, t)] =>
                    Message.Warning (602, loc, "using ``:'' as a type tests is "
                                     "deprecated, please use ``is'' instead");
                    PExpr.TryWith (loc, body, Splicable.Name (loc, mkname (Util.tmpname ("u"))), t, h.body)
                  
                  | [PExpr.Is (PExpr.Wildcard, t)] =>
                    PExpr.TryWith (loc, body, Splicable.Name (loc, mkname (Util.tmpname ("u"))), t, h.body)

                  | [PExpr.TypeEnforcement (PExpr.Ref (id), t)] =>
                    Message.Warning (602, loc, "using ``:'' as a type tests is "
                                     "deprecated, please use ``is'' instead");
                    PExpr.TryWith (loc, body, Splicable.Name (id), t, h.body)

                  | [PExpr.Is (PExpr.Ref (id), t)] =>
                    PExpr.TryWith (loc, body, Splicable.Name (loc, id), t, h.body)

                  | [PExpr.Ref (id)] =>
                    PExpr.TryWith (loc, body, Splicable.Name (loc, id), <[ System.Exception ]>, h.body)

                  | [PExpr.Wildcard] =>
                    PExpr.TryWith (loc, body, Splicable.Name (loc, mkname (Util.tmpname ("u"))), 
                                   <[ System.Exception ]>, h.body)

                  | [PExpr.TypeEnforcement (PExpr.Spliced (id), t)] =>
                    Message.Warning (602, loc, "using ``:'' as a type tests is "
                                     "deprecated, please use ``is'' instead");
                    PExpr.TryWith (loc, body, Splicable.Expression (id), t, h.body)

                  | [PExpr.Is (PExpr.Spliced (id), t)] =>
                    PExpr.TryWith (loc, body, Splicable.Expression (loc, id), t, h.body)

                  | _ =>
                    Message.Error (h.body.Location, "exception catch pattern must"
                                   " be in form of `| e is ExceptionType => handler' or"
                                   "`| e => handler' for System.Exception"); body
                }
              };
              match (get_token ()) {
                | Token.BracesGroup (group) =>
                  body = List.FoldLeft (process_groups (group, "exception handlers", parse_match_case),
                                        try_body, mktry);

                | x => Error (x, "expecting handlers of exceptions")
              }
            }
            
            if (flag_sibling_keyword ("finally")) {
              def handler = parse_block ([]);
              PExpr.TryFinally (loc, body, handler)
            }
            else {
              if (body == (try_body : object)) {
                Message.Error (loc, "expecting `catch' or `finally'");
                try_body
              }
              else body
            }

          | Token.Keyword ("def") =>
            def parse_define () {
              def pat = parse_expr (stop | TokenStoppers.Equal);
              expect_operator ("=");
              def expr = parse_expr (stop);
              PExpr.Define (loc + expr.loc, pat, expr)
            };
            def parse_funs (acc, idopt) {
              def h = parse_fun_header (idopt, allow_patterns = true);
              def fd = Function_decl (h, parse_block (h.parms));
              if (flag_sibling_keyword ("and")) {
                def id = get_splicable_id ();
                parse_funs (fd :: acc, id)
              }
              else    
                PExpr.DefFunctions (loc + h.loc, List.Rev (fd :: acc))
            };

            match (maybe_parse_ellipsis ()) {
              | Some (e) => PExpr.DefFunctions (loc, [Function_decl (null, e)])
              | _ =>
                match (peek_token ()) {
                  | Token.Keyword ("_")
                  | Token.Operator ("$")
                  | Token.Identifier =>
                    def id = get_splicable_id ();
                    match (peek_token ()) {
                      | Token.SquareGroup
                      | Token.RoundGroup => parse_funs ([], id)
                      | Token.Operator (":") with
                        ty = ({ shift (); def e = parse_expr (stop | TokenStoppers.Equal);
                               expect_operator ("="); e }) 
                      | Token.Operator ("=") with ty = null =>
                        def expr =
                          if (ty == null) {
                            shift ();
                            parse_expr (stop)
                          }
                          else
                            PExpr.TypeEnforcement (loc, parse_expr (stop), ty);
                      
                        def define = 
                          match (id) {
                            | Splicable.Name (name) => PExpr.Ref (loc, name)
                            | Splicable.Expression (e) => PExpr.Spliced (loc, e)
                            | Splicable.HalfId (e) => PExpr.ToComplete (loc, e)
                          }
                        PExpr.Define (loc + expr.loc, define, expr)

                      | x => fatal_error (x, "expecting `def x = expr' or `def f () { .. }'");
                    }
                  | _ => parse_define ()
                }
            }

          | Token.Identifier (i) => PExpr.Ref (loc, mkname (i))
          
          | Token.IdentifierToComplete (i) => PExpr.ToComplete (loc, mkname (i))
            
          | Token.Keyword (k) =>
            // here the expression syntax extensions are triggered, by first keyword
            // when it has some syntax definition attached
            match (env.SyntaxKeywords.Find (k)) {
              | Some (syntax_definition) when syntax_definition.Target %&& ((1 << 15) :> MacroTargets) =>
                parse_syntax_definition (syntax_definition, stop)                
              | _ => fatal_error (tok, "unexpected keyword in expression context");
            }

          | _ => fatal_error (tok, "expecting expression");
        }
      }
    }


    parse_quotation () : PExpr {
      match (peek_token ()) {
        | Token.LooseGroup (Token.Identifier (first) as tok) as group
          when tok.Next != null && (tok.Next is Token.Operator (":")) =>

          group.Child = tok.Next.Next;
          
          def push ['a] (describe, f : void -> 'a) {
            push_stream (group.Child);
            def entity = f ();
            pop_stream (describe);
            shift ();
            entity
          }
        
          def element =
            match (first) {
              | "ttype" =>
                match (parse_expr_sequence (TokenStoppers.None)) {
                  | [x] => SyntaxElement.TType (x)
                  | _ =>
                    Error (group, "expecting single type in quotated code");
                    SyntaxElement.TType (null)
                }
              | "fundecl" =>
                def decl = push ("quoted function", fun () {
                  def h = parse_fun_header (null);
                  Function_decl (h, parse_block (h.parms));
                });
                SyntaxElement.Function (decl)

              | "case"  => SyntaxElement.MatchCase (parse_match_case ())
              | "parameter" => SyntaxElement.Parameter (push ("quoted parameter", parse_parameter))
              | "decl" => SyntaxElement.ClassMember (ParseClassMember ())
              | _ => SyntaxElement.Expression (fatal_error (tok, "bad quotation type"))
            }
          PExpr.Quoted (tok.Location, element)

        | x =>
          def expr =
            match (parse_expr_sequence (TokenStoppers.None)) {
              | [x] => x
              | l => PExpr.Sequence (x.Location, l)
            };
          PExpr.Quoted (expr.Location, SyntaxElement.Expression (expr))
      }
    }
    

    parse_expr_sequence (stop : TokenStoppers) : list [PExpr]
    {
      List.Rev (parse_expr_sequence (stop, []))
    }
    
    parse_expr_sequence (group : Token) : list [PExpr]
    {
      List.Rev (parse_expr_sequence (group, []))
    }

    parse_expr_sequence (group : Token, acc : list [PExpr]) : list [PExpr]
    {
      push_stream (group);
      def result = parse_expr_sequence (TokenStoppers.None, acc);
      pop_stream ("group of expressions");
      result
    }

    
    /** special function to parse sequence in a way allowing other functions
        switching to sibling LooseGroup of tokens. Like:
          LooseGroup (..a b c..); LooseGroup (d..)

        It assumes that the token in current [stream] is a LooseGroup, and
        reads this and all following LooseGroups as expressions.
     */
    parse_expr_sequence (stop : TokenStoppers, mutable result : list [PExpr]) : list [PExpr]
    {
      def loop () {
        unless (should_stop (stop)) {
          def group_token = stream :> Token.LooseGroup;
          push_stream (group_token.Child);
          result = parse_expr (stop) :: result;
          
          if (stream == null || !should_stop (stop)) {
            // special message for obvious lack of ; in sequence
            when (stream != null)
              Message.Error (stream.Location, "expected `;'");
              
            pop_stream ("expression in sequence");
            shift (); // shift group in which we reached the end
            loop ();
          }
          else {
            def next = stream;
            pop_stream ();
            (stream :> Token.LooseGroup).Child = next;
          }
        }
      }
      loop ();
      result
    }
  } // end MainParser
} // end namespace
