/*
 * Copyright (c) 2003-2005 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using Nemerle.Compiler;
using Nemerle.Compiler.Typedtree;
using Nemerle;
using System.Text.RegularExpressions;

using Nemerle.Collections;
using Nemerle.Utility;

using PT = Nemerle.Compiler.Parsetree;

[assembly: Nemerle.Internal.OperatorAttribute ("Nemerle.Core", "is", false, 210, 211)]
[assembly: Nemerle.Internal.OperatorAttribute ("Nemerle.Core", ".", false, 285, 301)]
[assembly: Nemerle.Internal.OperatorAttribute ("Nemerle.Core", "++", true, 283, 284)]
[assembly: Nemerle.Internal.OperatorAttribute ("Nemerle.Core", "?", true, 283, 284)]
[assembly: Nemerle.Internal.OperatorAttribute ("Nemerle.Core", "--", true, 283, 284)]            
[assembly: Nemerle.Internal.OperatorAttribute ("Nemerle.Core", "where", false, 284, 300)]
[assembly: Nemerle.Internal.OperatorAttribute ("Nemerle.Core", "/", false, 260, 261)]
[assembly: Nemerle.Internal.OperatorAttribute ("Nemerle.Core", "%", false, 260, 261)]
[assembly: Nemerle.Internal.OperatorAttribute ("Nemerle.Core", "->", false, 251, 250)]      
[assembly: Nemerle.Internal.OperatorAttribute ("Nemerle.Core", ":", false, 270, 246)]
[assembly: Nemerle.Internal.OperatorAttribute ("Nemerle.Core", ":>", false, 270, 246)]
[assembly: Nemerle.Internal.OperatorAttribute ("Nemerle.Core", "+", false, 240, 241)]
[assembly: Nemerle.Internal.OperatorAttribute ("Nemerle.Core", "-", false, 240, 241)]
[assembly: Nemerle.Internal.OperatorAttribute ("Nemerle.Core", "<<", false, 230, 231)]
[assembly: Nemerle.Internal.OperatorAttribute ("Nemerle.Core", ">>", false, 230, 231)]
[assembly: Nemerle.Internal.OperatorAttribute ("Nemerle.Core", "::", false, 221, 220)]
[assembly: Nemerle.Internal.OperatorAttribute ("Nemerle.Core", "in", false, 120, 121)]
[assembly: Nemerle.Internal.OperatorAttribute ("Nemerle.Core", "..", false, 230, 231)]
[assembly: Nemerle.Internal.OperatorAttribute ("Nemerle.Core", "as", false, 215, 301)]            
[assembly: Nemerle.Internal.OperatorAttribute ("Nemerle.Core", "<", false, 210, 211)]
[assembly: Nemerle.Internal.OperatorAttribute ("Nemerle.Core", ">", false, 210, 211)]
[assembly: Nemerle.Internal.OperatorAttribute ("Nemerle.Core", "<=", false, 210, 211)]
[assembly: Nemerle.Internal.OperatorAttribute ("Nemerle.Core", ">=", false, 210, 211)]
[assembly: Nemerle.Internal.OperatorAttribute ("Nemerle.Core", "&", false, 190, 191)]
[assembly: Nemerle.Internal.OperatorAttribute ("Nemerle.Core", "%&", false, 190, 191)]
[assembly: Nemerle.Internal.OperatorAttribute ("Nemerle.Core", "%&&", false, 190, 191)]
[assembly: Nemerle.Internal.OperatorAttribute ("Nemerle.Core", "^", false, 180, 181)]
[assembly: Nemerle.Internal.OperatorAttribute ("Nemerle.Core", "%^", false, 180, 181)]
[assembly: Nemerle.Internal.OperatorAttribute ("Nemerle.Core", "|", false, 170, 171)]
[assembly: Nemerle.Internal.OperatorAttribute ("Nemerle.Core", "%|", false, 170, 171)]
[assembly: Nemerle.Internal.OperatorAttribute ("Nemerle.Core", "==", false, 165, 166)]
[assembly: Nemerle.Internal.OperatorAttribute ("Nemerle.Core", "!=", false, 165, 166)]
[assembly: Nemerle.Internal.OperatorAttribute ("Nemerle.Core", "&&", false, 160, 161)]
[assembly: Nemerle.Internal.OperatorAttribute ("Nemerle.Core", "||", false, 150, 151)]
[assembly: Nemerle.Internal.OperatorAttribute ("Nemerle.Core", "=", false, 141, 140)]
[assembly: Nemerle.Internal.OperatorAttribute ("Nemerle.Core", "*=", false, 141, 140)]
[assembly: Nemerle.Internal.OperatorAttribute ("Nemerle.Core", "/=", false, 141, 140)]
[assembly: Nemerle.Internal.OperatorAttribute ("Nemerle.Core", "%=", false, 141, 140)]
[assembly: Nemerle.Internal.OperatorAttribute ("Nemerle.Core", "+=", false, 141, 140)]      
[assembly: Nemerle.Internal.OperatorAttribute ("Nemerle.Core", "-=", false, 141, 140)]
[assembly: Nemerle.Internal.OperatorAttribute ("Nemerle.Core", "<<=", false, 141, 140)]
[assembly: Nemerle.Internal.OperatorAttribute ("Nemerle.Core", ">>=", false, 141, 140)]
[assembly: Nemerle.Internal.OperatorAttribute ("Nemerle.Core", "&=", false, 141, 140)]
[assembly: Nemerle.Internal.OperatorAttribute ("Nemerle.Core", "^=", false, 141, 140)]
[assembly: Nemerle.Internal.OperatorAttribute ("Nemerle.Core", "|=", false, 141, 140)]      
[assembly: Nemerle.Internal.OperatorAttribute ("Nemerle.Core", "when", false, 130, 131)]
[assembly: Nemerle.Internal.OperatorAttribute ("Nemerle.Core", "with", false, 130, 131)]
[assembly: Nemerle.Internal.OperatorAttribute ("Nemerle.Core", "=>", false, 145, 120)] // lambda expression

[assembly: Nemerle.Internal.OperatorAttribute ("Nemerle.Core", "!", true, 281, 280)]
[assembly: Nemerle.Internal.OperatorAttribute ("Nemerle.Core", "~", true, 281, 280)]


namespace Nemerle.Imperative
{
  macro Return (expr = null)
    syntax ("return", Optional (expr))
  {
    if (expr == null || expr is <[ () ]>)
      <[ $(PT.Name.Global ("_N_return") : name) () ]>
    else
      <[ $(PT.Name.Global ("_N_return") : name) ($expr) ]>
  }
  
  macro Break ()
    syntax ("break")
  {
    <[ $(PT.Name.Global ("_N_break") : name) () ]>
  }
  
  macro Continue ()
    syntax ("continue")
  {
    <[ $(PT.Name.Global ("_N_continue") : name) () ]>
  }
}


namespace Nemerle.Core
{
  /** MACROS EXTENDING SYNTAX OF LANGUAGE */

  /**
   * The 'unchecked' macro, it is only a syntax-extension place holder.
   * All the real work is done inside the typer.
   */
  macro @unchecked (expr)
  syntax ("unchecked", expr)
  {
    <[ dont_use_me ($expr) ]>
  }

  /**
   * The 'checked' macro, it is only a syntax-extension place holder.
   * All the real work is done inside the typer.
   */
  macro @checked (expr)
  syntax ("checked", expr)
  {
    <[ dont_use_me ($expr) ]>
  }
  
  
  /// Yet another syntax extender.
  macro @yield (expr = null)
    syntax ("yield", expr)
  {
    <[ dont_use_me ($expr) ]>
  }
  
  /** specialized macro for [if] condition with good warning messages,
      it performs typing of given expressions to check their type
      correctness
   */
  macro @if (cond, e1, e2)
  syntax ("if", "(", cond, ")", e1, Optional (";"), "else", e2) 
  {
    <[ 
      match ($cond) { 
        | true => $e1 
        | _ => $e2
      } 
    ]>
  }

  /** this macro provides convenient way to write a simple while loop,
      which performs execution of [body] parameter as long as
      [condition] is true
    */
  macro @while (cond, body)
  syntax ("while", "(", cond, ")", body) 
  {
    def loop = Nemerle.Macros.Symbol (Util.tmpname ("while_"));

    <[ 
      $(PT.Name.Global ("_N_break") : name) : {
        def $(loop : name) () : void {
          when ($cond) {
            $(PT.Name.Global ("_N_continue") : name) : {
              $body 
            } 
            $(loop : name) ()
          }
        } 
        $(loop : name) (); 
      }
    ]>
  }


  macro repeatmacro (times, body)
  syntax ("repeat", "(", times, ")", body)
  {
    <[ for(mutable t = $times; t > 0;--t) $body ]>
  }


  /** shortcut for [if (cond) body else ()] */
  macro whenmacro (cond, body)
  syntax ("when", "(", cond, ")", body) 
  {
    <[ match ($cond) { | true => $body : void | _ => () } ]>
  }

  macro @for (init, cond, change, body)
  syntax ("for", "(", Optional (init), ";", Optional (cond), ";",
          Optional (change), ")", body) 
  {
    def init = if (init != null) init else <[ () ]>;
    def cond = if (cond != null) cond else <[ true ]>;
    def change = if (change != null) change else <[ () ]>;    

    def loop = Nemerle.Macros.Symbol (Util.tmpname ("for_"));
    
    <[ 
      $init;
      $(PT.Name.Global ("_N_break") : name) : {
        def $(loop : name) () : void {
          when ($cond) {
            $(PT.Name.Global ("_N_continue") : name) : {
              $body
            }
            $change; 
            $(loop : name) ()
          }
        }
        $(loop : name) ();
      }
    ]>
  }

  /** shortcut for [if (cond) () else body] */
  macro @unless (cond, body)
  syntax ("unless", "(", cond, ")", body) 
  {
    <[ match ($cond) { | false => $body : void | _ => () } ]>
  }

  /** macro providing C# 'using' functionality
      http://msdn.microsoft.com/library/default.asp?url=/library/en-us/csspec/html/vclrfcsharpspec_8.asp  
      http://msdn.microsoft.com/library/default.asp?url=/library/en-us/csref/html/vclrfusingstatement.asp
   */
  macro @using (body, params args : array [expr])
  syntax ("using", "(", args, ")", body) 
  {
    def len = args.Length;
    def store_exprs (i, preexprs, postexprs) {
      if (i < len) {
        match (args[i]) {
          | <[ mutable $(str : name) = $e ]> =>
            store_exprs (i + 1, <[ mutable $(str : name) = $e ]> :: preexprs, 
                         <[ maybe_valuetype_dispose ($(str : name)) ]> :: postexprs)
                         
          | <[ def $(str : name) = $e ]> 
          | <[ $(str : name) = $e ]> =>
            store_exprs (i + 1, <[ def $(str : name) = $e ]> :: preexprs, 
                         <[ maybe_valuetype_dispose ($(str : name)) ]> :: postexprs)
          | e =>
            def x = Macros.NewSymbol ();            
            store_exprs (i + 1, <[ def $(x : name) = $e ]> :: preexprs,
                         <[ maybe_valuetype_dispose ($(x : name)) ]> :: postexprs)
        }
      }
      else
        (preexprs, postexprs)
    };

    def (preexps, postexps) = store_exprs (0, [], []);
    List.FoldLeft2 (preexps, postexps, <[ $body ]>, fun (pre, post, acc) {
      <[ $pre; try { $acc } finally { $post } ]>
    });
  }

  // disposes given value, if it is reference type then check nullness before disposing
  macro maybe_valuetype_dispose (val)
  {
    // this should also work
    <[
      def disp = $val : System.IDisposable;
      when (disp != null)
        disp.Dispose ();
    ]>
    
    /*
    def tval = Tyexpr.ty_expr (Nemerle.Macros.ImplicitCTX (), val);
    def tc = Tyutil.GetTypeTypeInfo (Tyexpr.type_of (tval));
    if (tc != null && tc.IsValueType)
      <[ ($(tval : typed) :> System.IDisposable).Dispose () ]>
    else
      <[
        when ($(tval : typed) != null)
          ($(tval : typed) :> System.IDisposable).Dispose ()
      ]>
    */
  }

  macro @lock (x, body)
  syntax ("lock", "(", x, ")", body)
  {
    def typer = Macros.ImplicitCTX ();
    def tx = typer.TypeExpr (x);
    typer.DelayMacro (fun (fail_loud) {
      match (tx.Type.Hint) {
        | Some (Class (tc, _)) when tc.IsValueType =>
          when (fail_loud) 
            Message.Error (x.Location, $"`$tc' is not a reference type as required by the lock expression");
          None ()
          
        | None =>
          when (fail_loud) 
            Message.Error (x.Location, "compiler was unable to analyze type of locked object, but it must verify that it is reference type");
          None ()
        
        | _ =>
          def result = 
            <[ 
              def to_lock = $(tx : typed);
              System.Threading.Monitor.Enter (to_lock);
              try {
                $body
              } finally {
                System.Threading.Monitor.Exit (to_lock);
              }
            ]>;
          Some (result)
      }
    });
  }

  macro dowhile (cond, body) 
  syntax ("do", body, "while", "(", cond, ")") 
  {
    def loop = Nemerle.Macros.Symbol (Util.tmpname ("do_while_"));

    <[
      ($(PT.Name.Global ("_N_break") : name) : {
        def $(loop : name) () {
          $(PT.Name.Global ("_N_continue") : name) : {
            $body
          }
          when ($cond) $(loop : name) ();
        }
        $(loop : name) ()
      })
    ]>
  }

  /**
   * The 'foreach' macro introduces a construction equivalent
   * to C#'s 'foreach' keyword, iterating over a collection.
   */
  macro @foreach (inexpr, body)
  syntax ("foreach", "(", inexpr, ")", body)
  {
    match (ListComprehensionHelper.ExpandRange (inexpr, body)) {
      | Some (expr) => Nemerle.Imperative.Return (expr)
      | None => {}
    }

    def (iter, collection) =
      match (inexpr) {
        | <[ $i in $c ]> => (i, c)
        | e =>
          Message.FatalError ($ "the syntax is 'foreach (x in collection)', "
                                "got $e");
      }
      
    def typer = Macros.ImplicitCTX ();
    def tcollection = typer.TypeExpr (collection);

    // build the body of loop (may contain additional matching)
    def build_definition (val) {
      match (body) {
        | <[ match ($(null)) { ..$cases } ]> =>
          match (iter) {
            | <[ $(x : name) ]> when char.IsLower (x.Id[0]) | <[ (..$_) ]> => ()
            | _ => Message.FatalError ("only simple names available in pattern"
                                       " of foreach with direct matching")
          }

          <[ def $iter = $val; 
             match ($iter) { ..$cases } 
          ]>

        | _ =>
          def mat =
            match (iter) {
              | <[ $pat :> $ty ]> =>
                <[ match ($val :> $ty) { | $pat => $body; () | _ => () } ]>
              | _ =>
                <[ match ($val) { | $iter => $body; () | _ => () } ]>  
            }
          mat.cases.Iter (fun (x : PT.MatchCase) { x.disable_warnings = true });
          mat
      }
    }

    // here we choose if we want to use enumerator pattern
    // of access GetEnumerator through IEnumarable
    // http://www.jaggersoft.com/csharp_standard/15.8.4.htm
    def decide_enumerator_pattern (tyinfo) {
      def all = tyinfo.LookupMember ("GetEnumerator");
      
      def choosen = List.Exists (all, fun (mem : IMember) {
        | meth is IMethod when !meth.IsStatic && meth.GetParameters ().IsEmpty =>
          match (meth.ReturnType) {
            // FIXME: do additional conservative checks              
            | MType.Class (tc, _) when
              !tc.LookupMember ("MoveNext").IsEmpty &&
              !tc.LookupMember ("Current").IsEmpty => true
              
            | _ => false
          }
        | _ => false
      });
      if (choosen)
        <[ $(tcollection : typed).GetEnumerator () ]>
      else
        <[ ($(tcollection : typed) : System.Collections.IEnumerable).GetEnumerator () ]>
    }

    typer.DelayMacro (fun (fail_loudly) {
      match (tcollection.Type.Hint) {
        | Some (MType.Class (tc, args)) =>
          if (tc.SuperType (InternalType.Nemerle_list_tc).IsSome) {
            def arg = List.Head (args);
            def definition = build_definition (<[ x ]>);
            Some (<[
              // we explicitly set parameter type to list, because collection's type
              // can be more specific (list.Cons, etc.)
              ($(PT.Name.Global ("_N_break") : name) : {
                def foreach_loop (_ : list [$(arg : typed)]) {
                  | x :: xs =>
                    $(PT.Name.Global ("_N_continue") : name) : {
                      $definition;
                    }
                    foreach_loop (xs)
                  | _ => ()
                }
                foreach_loop ($(tcollection : typed))
              })
            ]>)
          }
          else {
            def init_body = decide_enumerator_pattern (tc);

            def is_disposable = 
              typer.JustTry (fun () {
                def expr = typer.TypeExpr (init_body);
                expr.Type.Require (<[ ttype: System.IDisposable ]>)
              });

            def finally_body = 
              if (is_disposable)
                <[ (enumerator : System.IDisposable).Dispose () ]>
              else
                <[
                  match (enumerator) {
                    | x is System.IDisposable => x.Dispose ();
                    | _ => ()
                  }
                ]>;

            def definition = build_definition (<[ enumerator.Current ]>);

            Some (<[ 
              def enumerator = $init_body;
              $(PT.Name.Global ("_N_break") : name) : {
                def loop () {
                  when (enumerator.MoveNext ()) {
                    $(PT.Name.Global ("_N_continue") : name) : {
                      $definition;
                    }
                    loop ();
                  }
                }
                try { loop () } 
                finally { $finally_body }
              }
            ]>)
          }

        | Some (MType.Array (_ , rank)) =>
          def indices  = array (rank);
          def lengths = array (rank);
          for (mutable i = 0; i < rank; ++i) {
            indices [i] = Macros.NewSymbol ();
            lengths [i] = Macros.NewSymbol ();
          }
          def indices_list = List.RevMap (List.FromArray (indices), fun (x) {
              <[ $(x : name) ]> 
          });
          def build_loops (depth)  {
            /// build expression defining iteration symbols
            | 0 => build_definition ( <[ cached_collection [..$indices_list] ]> )
            | n => 
              def idx = indices [n - 1];
              <[ for (mutable $(idx : name) = 0; 
                      $(idx : name) < $(lengths [n - 1] : name);
                      ++ $(idx : name) ) 
                   $(build_loops (n - 1)) 
              ]>
          }
          mutable sequence = [ <[ $(build_loops (rank)) ]> ];
          if (rank == 1) 
            sequence = <[ def $(lengths [0] : name) = cached_collection.Length ]> :: sequence;
          else
            for (mutable i = rank - 1; i >= 0; --i)
              sequence = <[ def $(lengths [(rank - 1) - i] : name) = cached_collection.GetLength ($(i : int)) ]>
                         :: sequence;

          sequence = <[ def cached_collection = $(tcollection : typed) ]> :: sequence;
          Some (<[ { .. $sequence } ]>)

        | t =>
          when (fail_loudly) {
            def guess =
              match (t) {
                | Some (t) => $ "current best guess about the type is $t"
                | None => "the compiler has no idea what the type might be"
              }
            Message.Error ($ "collection in foreach must be an array or "
                             "type implementing enumerator pattern, $guess");
            Message.Hint ("try specifing the type directly using 'expr : SomeType'");
          }
          None ()
      }
    })
  }

  macro ignore (e) 
  {
    <[ def _ = $e; () ]>
  }

  macro abort (message = <[ "" ]>) 
  {
    <[ throw AssertionException ($(message.loc.File : string), 
                                 $(message.loc.Line : int), "", $message) ]>
  }

  /** MACROS EXTENDING TYPE SYSTEM OF LANGUAGE */


  [Nemerle.MacroUsage (Nemerle.MacroPhase.BeforeInheritance,
                       Nemerle.MacroTargets.Class,
                       Inherited = false, AllowMultiple = false)]
  macro Record (par : TypeBuilder, params _ : list [PExpr])
  {
      par.DisableImplicitConstructor ();
  }


  [Nemerle.MacroUsage (Nemerle.MacroPhase.WithTypedMembers,
                       Nemerle.MacroTargets.Class,
                       Inherited = false, AllowMultiple = false)]
  macro Record (par : TypeBuilder, params options : list [PExpr])
  {
    def instance_flags = BindingFlags.Instance %| BindingFlags.Public %| 
      BindingFlags.NonPublic %| BindingFlags.DeclaredOnly;

    def inclusion_regexs = MacrosHelper.AnalyseNameInclusionPatterns (options);
      
    def make_ctor (is_value_type, base_ctor : IMethod) {
      def (ctor_parms, base_call) =
        if (base_ctor == null)
          ([], if (is_value_type) <[ () ]> else <[ base () ]>)
        else {
          def pp = base_ctor.GetParameters ();
          def callparms = List.Map (pp, fun (fp : Fun_parm) {
            <[ $(fp.name : usesite) ]>
          });
          (List.Map (pp, fun (fp : Fun_parm) {
             <[ parameter: $(fp.name : usesite) : $(fp.ty : typed) ]>
           }),
           <[ base (..$callparms) ]>)
        };

      def flds = par.GetFields (instance_flags);

      def collect (mem : IField, acc) {
        if (MacrosHelper.NameMatchesPatterns (mem.Name, inclusion_regexs)) {
          def n = Macros.UseSiteSymbol (mem.Name);
          def fp = <[ parameter: $(n : name) : $(mem.GetMemType () : typed) ]>;
          def ex = <[ this.$(n : name) = $(n : name) ]>;
          def (es, ps) = acc;
          (ex :: es, fp :: ps)
        }
        else 
          acc
      };
      def (assigns, parms) = List.FoldLeft (flds, ([], []), collect);
      
      def body = <[ { ..$(base_call :: assigns) } ]>;
      body.loc = par.Location;
      
      def attrs = Modifiers (mods = 
        match (par.GetTydecl ()) {
          | TypeDeclaration.Variant => NemerleAttributes.Protected
          | _ => NemerleAttributes.Public
        }, custom_attrs = []);
        
      def parms = List.Append (ctor_parms, List.Rev (parms));
      def meth = <[ decl: ..$attrs this (..$parms) $body ]>;
      meth.loc = par.Location;

      /// we do not try to add empty constructor if it exists
      if (parms is []) {
        def existing = par.GetConstructors (instance_flags);
        unless (List.Exists (existing, fun (x : IMethod) { 
            x.GetParameters ().IsEmpty
          }))
          par.DefineAndReturn (meth).HasBeenUsed = true;          
      }
      else
        par.DefineAndReturn (meth).HasBeenUsed = true;
    };
      
    match (par.SuperClass ()) {
      | Some (baseti) when baseti.FullName != "System.ValueType" =>
        def ctors = baseti.GetConstructors (instance_flags);
        foreach (x in ctors) make_ctor (false, x)

      // our parent is System.ValueType, so we will not generate any base ctor calls
      | Some => make_ctor (true, null)
        
      | _ => make_ctor (false, null)
    }
  }

  
  module MacrosHelper {
    public AnalyseNameInclusionPatterns (options : list [PT.PExpr]) : Regex * Regex
    {
      mutable inclusion = null;
      mutable exclusion = null;
      foreach (e in options) {
        | <[ Include = [..$names] ]> =>
          inclusion = Regex ("^" + names.ToString ("$|^") + "$")
        
        | <[ Include = $(regexp : string) ]> =>
          inclusion = Regex (regexp)
          
        | <[ Exclude = [..$names] ]> =>
          exclusion = Regex ("^" + names.ToString ("$|^") + "$")
        
        | <[ Exclude = $(regexp : string) ]> =>
          exclusion = Regex (regexp)
          
        | e =>
          Message.Error (e.Location, $"unsupported argument `$e' in macro, please specify 'Include/Exclude = [name1,name2]/pattern")
      }
      (inclusion, exclusion)
    }
    
    public NameMatchesPatterns (name : string, patterns : Regex * Regex) : bool
    {
      def (inclusion, exclusion) = patterns;
      (inclusion == null || inclusion.Match (name).Success) && 
      (exclusion == null || !exclusion.Match (name).Success)
    }
  }
  
  
  [Nemerle.MacroUsage (Nemerle.MacroPhase.BeforeInheritance,
                       Nemerle.MacroTargets.Class,
                       Inherited = false, AllowMultiple = false)]
  macro ExternallyVisibleOptions (t : TypeBuilder)
  {
    match (t.ParsedDeclaration) {
      | PT.TopDeclaration.Variant (_, decls) as v =>
        def tyvars = List.Map (v.typarms.tyvars, fun (nm, col) {
          <[ $(PT.Name (nm, col, null) : name) ]>
        });
        foreach (PT.ClassMember.TypeDeclaration (PT.TopDeclaration.VariantOption as vo) in decls) {
          def name = PT.Splicable.Name (PT.Name (vo.Name));
          def mods = Modifiers (t.Attributes, []);
          def tyname = NString.Split (t.FullName, array ['.']) + [vo.Name];
          def tyalias = <[ $(Util.ExprOfList (tyname)) [ ..$tyvars] ]>;
          def decl = PT.ClassMember.TypeDeclaration 
            (t.Location, null, null, PT.TopDeclaration.Alias (t.Location, name, 
                                                              mods, v.typarms, tyalias));
          match (t.DeclaringType) {
            | null => _ = t.ParsedName.context.Define (decl);            
            | parent => _ = (parent :> TypeBuilder).DefineNestedType (decl);
          }
        }
      | _ =>
        Message.FatalError ("ExternallyVisibleOptions attribute can only be applied to variant types.")
    }
 
  }


  [Nemerle.MacroUsage (Nemerle.MacroPhase.WithTypedMembers,
                       Nemerle.MacroTargets.Parameter)]
   macro _N_ExtensionMethodOnThisParameter (tb : TypeBuilder, mb : MethodBuilder, p : ParameterBuilder)
     syntax ("this")
   {
     def add_attr (m) {
       unless (m.GetCustomAttributes ().Exists (
         _ is <[ Nemerle.Internal.Extension () ]>))
         m.AddCustomAttribute (<[ Nemerle.Internal.Extension () ]>)
     }

     if (p : object != mb.GetParameters ().Head)
       Message.Error ("'this' modifier (for extension method) can only be "
                      "used on the first parameter");
     else if (!((mb.Attributes %&& NemerleAttributes.Public) &&
                (mb.Attributes %&& NemerleAttributes.Static)))
       Message.Error ("extension methods need to be marked public static")
     else {
       add_attr (tb.GetModifiers ());
       add_attr (mb.GetModifiers ());
       LibraryReferenceManager.AddExtensionMethod (mb);
     }
   }
}

namespace Nemerle.Macros
{
  [Nemerle.MacroUsage (Nemerle.MacroPhase.BeforeInheritance,
                       Nemerle.MacroTargets.Method,
                       Inherited = true, AllowMultiple = false)]
  macro Hygienic (_ : TypeBuilder, m : ParsedMethod) {
    m.Body = <[
      MacroColorizer.PushNewColor (MacroColorizer.UseColor,
                                   MacroColorizer.UseContext);
      def result = $(m.Body);
      MacroColorizer.PopColor ();
      result
    ]>;
  }


  macro DefineCTX (ctx) {
    <[ def $(Macros.GetImplicitCTXName () : name) = $ctx ]>
  }

  macro ImplicitCTX () {
    <[ $(MacroClasses.GetImplicitCTXName () : name) ]>
  }  
  
  /** creates new symbol with given id and current global context */
  macro Symbol (id)
  {
    def env = Macros.ImplicitCTX().Env;
    def nr = env.GetMacroContext ();
    <[ PT.Name.NameInCurrentColor ($id, $("_N_MacroContexts" : dyn).Get ($(nr : int))) ]>
  }

  macro pretty_print_expr (exp, expand : bool)
  {
    def ctx = if (expand) Some (Macros.ImplicitCTX()) else None ();
    PrettyPrint.PrintExpr (ctx, exp);
    <[ () ]>
  }
}

namespace Nemerle.Extensions {
  /**
   * The `lambda' macro
   */
  macro @lambda (parm : parameter, body)
  syntax ("lambda", parm, "->", body)
  {
    match (parm) {
      | <[ parameter: $(iname : name) : $ty ]> =>
        <[ fun ($(iname : name) : $ty) { $body } ]>
      | _ =>
        Message.FatalError ("expected a single parameter for the lambda abstraction");
    }
  } 

  
  [Nemerle.MacroUsage (Nemerle.MacroPhase.WithTypedMembers,
                       Nemerle.MacroTargets.Class,
                       Inherited = false, AllowMultiple = false)]
  macro TupleToString (t : TypeBuilder) {
    def flds = t.GetFields (BindingFlags.Public %| BindingFlags.NonPublic %|          
                            BindingFlags.Instance %| BindingFlags.DeclaredOnly);
    def appends = 
      List.RevMap (flds, fun (x : IField) { 
        <[ ignore (str.Append ($(x.Name : usesite).ToString ())); ]>
      });
    def body_seq =
      List.FoldLeft (List.Tail (appends), List.Head (appends) ::
                     [<[ ignore (str.Append (")")); str.ToString () ]>], fun (x, acc) {
        x :: <[ ignore (str.Append (", ")) ]> :: acc
      });
    def body_seq = <[ def str = System.Text.StringBuilder ("(") ]> :: body_seq;
    t.Define (<[ decl:
      public override ToString () : string { ..$body_seq }
    ]>);
  }

  [Nemerle.MacroUsage (Nemerle.MacroPhase.BeforeInheritance,
                       Nemerle.MacroTargets.Class,
                       Inherited = false, AllowMultiple = false)]
  macro DisableImplicitConstructor (t : TypeBuilder) {
    t.DisableImplicitConstructor ();
  }

  [Nemerle.MacroUsage (Nemerle.MacroPhase.WithTypedMembers,
                       Nemerle.MacroTargets.Class,
                       Inherited = false, AllowMultiple = false)]
  macro StructuralEquality (t : TypeBuilder) {
    def flds = t.GetFields (BindingFlags.Public %| BindingFlags.NonPublic %|
                            BindingFlags.Instance %| BindingFlags.DeclaredOnly);
    def body = 
      List.FoldLeft (flds, <[ true ]>, fun (x : IField, acc) { 
        def nm = Macros.UseSiteSymbol (x.Name);
        <[ $acc && $(nm : name).Equals (tup.$(nm : name)) ]>
      });
    def tname = t.ParsedTypeName;
    def full = <[ if (o.GetType ().Equals (this.GetType ())){
                    def tup = (o :> $tname);
                    $body
                  } else false ]>;
    t.Define (<[ decl:
      public override Equals (o : System.Object) : bool { $full }
    ]>);
  }

  [Nemerle.MacroUsage (Nemerle.MacroPhase.WithTypedMembers,
                       Nemerle.MacroTargets.Class,
                       Inherited = false, AllowMultiple = false)]
  macro LexicographicCompareTo (t : TypeBuilder) {
    def tname = t.ParsedName;
    def flds = t.GetFields (BindingFlags.Public %| BindingFlags.NonPublic %|
                            BindingFlags.Instance %| BindingFlags.DeclaredOnly);
    def compareField (fld : IField)
    {
        def nm = Macros.UseSiteSymbol (fld.Name);
        def hasLess = match (fld.GetMemType ()) {
            | MType.Class (ti, _) =>
              def name = ti.FullName;
              match (name) {
                  | "Nemerle.Core.string"
                  | "System.String"
                  | "Nemerle.Core.int"
                  | "System.Int32"
                  | "System.UInt32"
                  | "Nemerle.Core.float"
                  | "System.Single"
                  | "Nemerle.Core.double"
                  | "System.Double"
                  | "Nemerle.Core.char"
                  | "System.Char" =>
                    true
                  | _ =>
                    false
              }
            | _ => 
              false
        }
        if (hasLess)
        {
            <[ 
              if (this.$(nm : name) < other.$(nm : name)) 
              {
                  -1
              }
              else if (this.$(nm : name) > other.$(nm : name)) 
              {
                  1
              }
              else
              {
                  0
              }
           ]>
       }
       else
       {
           <[ this.$(nm : name).CompareTo (other.$(nm : name)); ]>
       }
    }
    def body = 
      List.FoldRight (flds, <[ 0 ]>, fun (x : IField, acc) { 
        <[ 
           def cmp = $(compareField (x));
           if (cmp == 0) 
           {
               $(acc)
           }
           else
           {
               cmp
           }
        ]>
      });
    def full = if (t.IsValueType) 
    {
        body 
    }
    else
    {
        <[ 
           if (object.ReferenceEquals (this, other)) 
           {
               0
           }
           else if (object.ReferenceEquals (other, null))
           {
               1
           }
           else
           {
               $(body)
           }
        ]>
    }

    t.Define (<[ decl:
      public CompareTo (other : $(tname : name)) : int { $full }
    ]>);
    t.Define (<[ decl:
      public CompareTo (Oother : object) : int 
      { 
        try 
        {
          def other = Oother :> $(tname : name);
          this.CompareTo (other)
        } 
        catch 
        {
          | _ is System.InvalidCastException => throw System.ArgumentException ()
        }
      }
    ]>);
  }

  [Nemerle.MacroUsage (Nemerle.MacroPhase.WithTypedMembers,
                       Nemerle.MacroTargets.Class,
                       Inherited = false, AllowMultiple = false)]
  macro StructuralHashCode (t : TypeBuilder) {
    def flds = t.GetFields (BindingFlags.Public %| BindingFlags.NonPublic %|          
                            BindingFlags.Instance %| BindingFlags.DeclaredOnly);

    def body = 
      List.FoldLeft (flds, <[ 0 ]>, fun (x : IField, acc) {
        <[ $acc %^ $(x.Name : usesite).GetHashCode () ]>
      });
    t.Define (<[ decl:
      public override GetHashCode () : int {
        $body
      }
    ]>);
  }
  
  /** MACROS, WHICH WE CAN DO BETTER THAN HASKELL */

  macro SelectFromTuple (k : int, n : int, tupl)
  {
    def symb = Macros.NewSymbol ();
    mutable tup = [];
    for (mutable i = n; i > 0; --i)
      if (i == k)
        tup = <[ $(symb : name) ]> :: tup
      else
        tup = <[ _ ]> :: tup;

    <[ def (.. $tup) = $tupl; $(symb : name) ]>
  }

  macro TupleMap (f, tup)
  {
    match (tup) {
      | <[ (.. $elms) ]> =>
        def mapped = List.Map (elms, fun (e) { <[ $f ($e) ]> });
        <[ (.. $mapped) ]>
      | _ => 
        Message.FatalError ("'TupleMap' macro expects function and tuple")
      }
  }

  macro PrintTuple (tup, size : int)
  {
    def symbols = array (size);
    mutable pvars = [];
    for (mutable i = size - 1; i >= 0; --i) {
      symbols[i] = Macros.NewSymbol ();
      pvars = <[ $(symbols[i] : name) ]> :: pvars;
    };
    mutable exps = [];
    for (mutable i = size - 1; i >= 0; --i)
      exps = <[ System.Console.WriteLine ($(symbols[i] : name)) ]> :: exps;

    exps = <[ def (.. $pvars) = $tup ]> :: exps;
    <[ {.. $exps } ]>
  }

  macro PrintTupleTyped (tup)
  {
    def tup' = Macros.ImplicitCTX().TypeExpr (tup);
    match (tup'.Type.Hint) {
      | Some (MType.Tuple (args)) =>
        def size = args.Length;
        def symbols = array (size);
        mutable pvars = [];
        for (mutable i = size - 1; i >= 0; --i) {
          symbols[i] = Macros.NewSymbol ();
          pvars = <[ $(symbols[i] : name) ]> :: pvars;
        };
        mutable exps = [];
        for (mutable i = size - 1; i >= 0; --i)
          exps = <[ System.Console.WriteLine ($(symbols[i] : name)) ]> :: exps;
          
        exps = <[ def (.. $pvars) = $tup ]> :: exps;
        <[ {.. $exps } ]>
      | _ => 
        Message.FatalError ("expected tuple")
    }
  }
  
  macro DefaultValue (ty) {
    def tty = Macros.ImplicitCTX().MonoBindType (ty);
    <[ $(Macros.DefaultValueOfType (tty)) : $ty ]>
  }
} // end ns

namespace Nemerle.Diagnostics {

  /**
     Insert given expression before every expression in every sequence 
     of method's body. 
     Implicitly visible variables are [_line : int], [_file : string], 
     [_method : string], [_expr : string]
     Meaning current line number, filename, method's name, next expression
     which will be executed converted to human readable string
   */
  [Nemerle.MacroUsage (Nemerle.MacroPhase.WithTypedMembers,
                       Nemerle.MacroTargets.Method)]
  macro Trace (_ : TypeBuilder, m : MethodBuilder,
               tracecall = <[ 
                    Nemerle.IO.printf ("Trace: %s:%d: %s --> %s\n", 
                                       $("_file" : usesite),
                                       $("_line" : usesite),
                                       $("_method" : usesite),
                                       $("_expr" : usesite)) 
                            ]>) {
    def exps_strings = Stack ();
    def add (in_pattern, is_post, x) {
      if (in_pattern)
        x
      else if (is_post) 
        match (x) {
          | <[ {.. $seq } ]> =>
            // we must process seqence in reversed order, because
            // stack was filled this way  {push 1; 2; 3; push last 4}
            def nseq = 
            List.Flatten (List.RevMap (List.Rev (seq), fun (e : Parsetree.PExpr) { 
              [<[
                 def $("_line" : usesite) = $(e.loc.Line : int);
                 def $("_expr" : usesite) = $(exps_strings.Pop () : string);
                 $tracecall;
               ]>, e]
            }));
            <[ { ..$nseq } ]>
          | _ => x
        }
      else {
        match (x) {
          | <[ { .. $seq } ]> =>
            foreach (e in seq) 
              exps_strings.Push (PrettyPrint.SprintExpr (None (), e));
          | _ => ()
        }
        x
      }
    };
    def bod = Macros.TraverseExpr (None (), m.Body, false, add);
    m.Body = <[ 
      def $("_file" : usesite) = $(m.Body.loc.File : string);
      def $("_method" : usesite) = $(m.Name : string);
      $bod;
    ]>
  }


  [Nemerle.MacroUsage (Nemerle.MacroPhase.WithTypedMembers,
                       Nemerle.MacroTargets.Class)]
  macro Trace (t : TypeBuilder, tracecall) {
    def meths = t.GetMethods (BindingFlags.Public %| BindingFlags.NonPublic %|          
                              BindingFlags.Instance %| BindingFlags.Static %| BindingFlags.DeclaredOnly);
    foreach (x :> MethodBuilder in meths) {
      x.AddMacroAttribute (<[ $("Nemerle" : usesite).Diagnostics.Trace ($tracecall) ]>);
    }
  }

  
  macro @time (code)
  syntax ("time", code)
  {
    def loc = code.Location.ToString ();
    <[ 
       def begin = System.DateTime.Now;
       $code;
       def end = System.DateTime.Now;
       System.Console.WriteLine ($(loc : string) + ": execution took " + (end - begin).ToString ());
    ]>
  }
}


namespace Nemerle {

  [Nemerle.MacroUsage (Nemerle.MacroPhase.BeforeInheritance,
                       Nemerle.MacroTargets.Method)]
  macro NotImplemented (t : TypeBuilder, m : ParsedMethod) {
    def ignores = List.Map (m.header.parms, fun (x : PT.Fun_parm) {
      <[ _ = $(x.name.GetName () : name) ]>
    });
    def message = "Method `" + m.name.GetName ().Id + "' in type `" + 
                  t.FullName + "' is not implemented yet.";
    m.Body = <[ 
      { ..$ignores }; 
      throw System.NotImplementedException ($(message : string)) 
    ]>
  }
  
  [Nemerle.MacroUsage (Nemerle.MacroPhase.BeforeInheritance,
                       Nemerle.MacroTargets.Method)]
  macro OverrideObjectEquals (t : TypeBuilder, m : ParsedMethod)
  {
    match (m.header.parms) {
      | [x] =>
        t.Define (<[ decl:
          public override Equals (x : object) : bool
          {
            if (typeof ($(x.ty)).IsInstanceOfType (x))
              this.$(m.header.name.GetName () : name) (x :> $(x.ty))
            else
              false
          }
        ]>)
      | _ =>
        Message.Error ("Equals()-like method shall have a single argument")
    }
  }

  [Nemerle.MacroUsage (Nemerle.MacroPhase.BeforeInheritance,
                       Nemerle.MacroTargets.Method)]
  macro ForwardThis (t : TypeBuilder, m : ParsedMethod, this_expr)
  {
    def pref = $ "_FT_$(m.header.name.GetName ().Id)_";
    def meth = Nemerle.Macros.Symbol (Util.tmpname (pref));

    t.Define (<[ decl:
      private $(meth : name) (.. $(m.header.parms)) : $(m.header.ret_type)
      {
        $(m.Body)
      }
    ]>);
    
    def parms = List.Map (m.header.parms, fun (x : PT.Fun_parm) {
      <[ $(x.name.GetName () : name) ]>
    });
    
    m.Body = <[ 
      $this_expr . $(meth : name) ( .. $parms )
    ]>
  }

}
