/*
 * Copyright (c) 2003, 2004 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

namespace Nemerle.Core
{
  /**
   * The optional value variant.
   */
  public variant option ['a]
  {
    | None
    | Some { val : 'a; }

    /**
     * Pretty prints the optional value.
     */
    public override ToString () : string
    {
      match (this){
        | Some (x) => "Some (" + x.ToString () + ")"
        | None     => "None"
      }
    }

    /**
     * Typesafe equality checks.
     */
    [Nemerle.OverrideObjectEquals]
    public Equals (o : option ['a]) : bool
    {
      Nemerle.Core.Option.Equals (this, o)
    }

    
    /**
     *  Structural HashCode provider
     */
    public override GetHashCode () : int
    {
      match (this) {
        | None => 0
        | Some (x) => x.GetHashCode ()
      }
    }


    public IsSome : bool
    {
      get { this is Some (_) }
    }


    public IsNone : bool
    {
      get { this is None () }
    }

    public WithDefault (val : 'a) : 'a
    {
      match (this) {
        | Some (v) => v
        | None => val
      }
    }

    public HasValue : bool
    {
      get { this is Some (_) }
    }

    public Value : 'a
    {
      get { 
        match (this) {
          | Some (x) => x
          | None =>
            throw System.ArgumentException ("option.Value")
        }
      }
    }
  }


  /**
   * Operations on optional values
   */
  public module Option
  {
    /**
     * Safe equality check
     */
    public Equals ['a] (x : option ['a], y : option ['a]) : bool
    {
      | (Some (a), Some (b)) => a.Equals (b)
      | (None, None)         => true
      | _                    => false
    }

    /**
     * Safe optional value mapping.
     */
    public Map ['a, 'b] (x : option ['a], f : 'a -> 'b) : option ['b]
    {
      match (x) {
        | Some (x) => Some (f (x))
        | None => None ()
      }
    }    

    /**
     * Same as ignore (Map (x, f)).
     */
    public Iter ['a] (x : option ['a], f : 'a -> void) : void
    {
      match (x) {
        | Some (x) => f (x)
        | None => ()
      }
    }    

    /**
     * Checks if the optional value is present.
     */
    public IsSome ['a] (x : option ['a]) : bool
    {
      match (x) {
        | Some => true
        | _ => false
      }
    }

    /**
     * Returns `true' if the optional value is not present.
     */
    public IsNone['a] (x : option ['a]) : bool
    {
      match (x) {
        | None => true
        | _ => false
      }
    }  

    /**
     * Unwraps an optional value; throws an argument exception
     * if the value is not present.
     */
    public UnSome ['a] (x : option ['a]) : 'a
    {
      match (x) {
        | Some (x) => x
        | None =>
          throw System.ArgumentException ("Option.UnSome")
      }
    }
  }
}
