/*
 * Copyright (c) 2005 Kamil Skalski <nazgul at nemerle.org>
 * Copyright (c) 2005 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */


using System;
using System.IO;
using System.Collections;
using System.Reflection;

using Microsoft.Build.Framework;
using Microsoft.Build.Tasks;
using Microsoft.Build.Utilities;

using Nemerle.Utility;

namespace Nemerle.Tools.MSBuildTask
{
  public class Ncc : ManagedCompiler
  {
    protected override ToolName : string
    {
      get { "ncc.exe"; }
    }
    
    protected override GenerateFullPathToTool () : string {
      def my_file = typeof(Ncc).Assembly.Location;
      def ncc_file = Path.Combine (Path.GetDirectoryName(my_file), ToolName);
      ncc_file
    } 
  
    protected override AddResponseFileCommands(commandLine : CommandLineBuilderExtension) : void
    {
      commandLine.AppendSwitch ("/no-color");
      commandLine.AppendSwitchIfNotNull("/lib:", base.AdditionalLibPaths, ",");
      commandLine.AppendSwitchIfNotNull("/nowarn:", this.DisabledWarnings, ",");
      when (NoStandardLib)
        commandLine.AppendSwitch("/nostdlib");
      when (WarningLevel != 4)
        commandLine.AppendSwitchIfNotNull("/warn:", WarningLevel.ToString());
      commandLine.AppendSwitchIfNotNull("/doc:", this.DocumentationFile);
      commandLine.AppendSwitchUnquotedIfNotNull("/define:", base.DefineConstants);

      Bag ["Optimize"] = null; // prevent standard handling by ManagedCompiler      
      
      base.AddResponseFileCommands(commandLine);

      when (base.ResponseFiles != null)
      {
        def items = base.ResponseFiles;
        foreach (it in items)
          commandLine.AppendSwitchIfNotNull("/fromfile:", it.ItemSpec);
      }
      
      when (base.References != null)
      {
        def items = base.References;
        foreach (it in items)
          commandLine.AppendSwitchIfNotNull("/ref:", it.ItemSpec);
      }
      
    }
#if !RUNTIME_MONO
    protected override LogEventsFromTextOutput(singleLine : string, messageImportance : MessageImportance) : void
    {
      def get_location (before) {
        def str = singleLine.Substring (0, singleLine.IndexOf (before));
        if (string.IsNullOrEmpty (str)) (null, 0,0,0,0)
        else {
          def parts = Nemerle.Collections.List.FromArray (str.Split (':'));
          match (parts) {
            | [file, _] => (Path.GetFullPath(file), 0, 0, 0, 0)
            | [file, line, col, _] => (Path.GetFullPath(file), int.Parse (line), int.Parse (col), int.Parse (line), int.Parse (col))
            | [file, line1, col1, line2, col2, _] => (Path.GetFullPath(file), int.Parse (line1), int.Parse (col1), int.Parse (line2), int.Parse (col2))
            | _ => (null, 0,0,0,0)
          }
        }
      }
    
      if (singleLine.IndexOf ("error:") != -1) {
        def (file, l1, c1, l2, c2) = get_location ("error:");
        Log.LogError (null, null, null, file, l1, c1, l2, c2, singleLine.Substring (singleLine.IndexOf ("error:") + 7));
      }
      else if (singleLine.IndexOf ("warning:") != -1) {
        def (file, l1, c1, l2, c2) = get_location ("warning:");
        Log.LogWarning (null, null, null, file, l1, c1, l2, c2, singleLine.Substring (singleLine.IndexOf ("warning:") + 9));
      }
      else if (singleLine.IndexOf ("debug:") != -1) {
        def (file, l1, c1, l2, c2) = get_location ("debug:");
        Log.LogMessage (null, null, null, file, l1, c1, l2, c2, singleLine.Substring (singleLine.IndexOf ("debug:") + 7));
      }
      else _ = Log.LogMessageFromText(singleLine, messageImportance);
    }
#endif   
    protected override GetResponseFileSwitch(responseFilePath : string) : string
    {
      "/from-file:\"" + responseFilePath + "\"";
    }
          
    [Accessor (flags = WantSetter)]
    mutable _disabled_warnings : array [string];

    [Accessor (flags = WantSetter)]
    mutable _documentation_file : string;

    [Accessor (flags = WantSetter)]
    mutable _no_standard_lib : bool = false;

    [Accessor (flags = WantSetter)]
    mutable _warning_level : int = 4;
        
    [Accessor (flags = WantSetter)]
    mutable _warnings_as_errors : bool;
  }
}