// Copyright (c) 2003-2005 The University of Wroclaw.
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
//    1. Redistributions of source code must retain the above copyright
//       notice, this list of conditions and the following disclaimer.
//    2. Redistributions in binary form must reproduce the above copyright
//       notice, this list of conditions and the following disclaimer in the
//       documentation and/or other materials provided with the distribution.
//    3. The name of the University may not be used to endorse or promote
//       products derived from this software without specific prior
//       written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
// IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
// OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
// NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
// TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
// PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
// LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
// NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//

using System;
using System.IO;
using antlr;
using Nemerle.Utility;
using Nemerle.IO;

namespace Nemerle.CSharp
{
    public class CMain
    {
        enum InputKind {
          | CSharp
          | ASPX
          | ASMX
        }


        static get_input_kind (in_file : string) : InputKind
        {
            if (in_file.EndsWith (".asmx"))
              InputKind.ASMX
            else if (in_file.EndsWith (".aspx"))
              InputKind.ASPX
            else
              InputKind.CSharp
        }

        static make_lexer (in_file : string) : CSharpLexer
        {
            def uncomment (s) {
              "/*ASPX-UNCOMMENT:" + s.TrimEnd () + "*/"
            }

            def asmx_filter (s) {
              if (s.TrimStart ().StartsWith ("<%"))
                uncomment (s) + "\n"
              else s
            }

            mutable in_aspx_code = false;
            def aspx_filter (s) {
              def s' = s.Trim ().ToLower ();
              if (s' == "<script runat=\"server\">") {
                in_aspx_code = true;
                uncomment (s) + "class AspxRemoveMe {\n"
              } else if (s' == "</script>") {
                in_aspx_code = false;
                "/*ASPX-REMOVE-BRACE*/}" + uncomment (s) + "\n"
              } else if (in_aspx_code)
                s
              else
                uncomment (s) + "\n"
            }

            match (get_input_kind (in_file)) {
              | CSharp =>
                CSharpLexer (FileStream (in_file, FileMode.Open, FileAccess.Read))
              | ASPX =>
                CSharpLexer (PipeReader (StreamReader (in_file), aspx_filter))
              | ASMX =>
                CSharpLexer (PipeReader (StreamReader (in_file), asmx_filter))
            }
        }

        public static Main() : void
        {
	    mutable files = [];

	    def make_conversion (in_file, out_file)
	    {
		Message.InFile = in_file;

		Emit.Initialize (out_file,
                                 need_output_filter = get_input_kind (in_file) != InputKind.CSharp);

		ExtendedToken.ClearWhitespaces ();

                def lexer = make_lexer (in_file);
		lexer.setTokenObjectClass("Nemerle.CSharp.ExtendedToken");
              
		def parser = CSharpParser (lexer);
             
		parser.compilation_unit ();
              
		Emit.End ();

		when (Message.WasError)
		    throw ErrorException ();
	    }
	    
	    def print_version () 
	    {
		System.Console.Error.Write ("C# to Nemerle translator (cs2n) version 0.9.2 (release)\n"
					    "(c) 2003-2005 University of Wroclaw,"
					    " All rights reserved.\n");
		System.Environment.Exit (0);
	    };
	    
	    mutable help_opts = [];        
	    def print_help () {
		System.Console.WriteLine (Getopt.Usage (help_opts));
		System.Environment.Exit (0);
	    };
	    
	    def opts = Options.GetCommonOptions () + [
		Getopt.CliOption.Flag (name = "-version",
                     aliases = ["-V"],
                     help = "Output version information and exit",
                     handler = print_version),
		Getopt.CliOption.Flag (name = "-help",
                     aliases = ["-h"],
                     help = "Display this usage message and exit",
                     handler = print_help),
		Getopt.CliOption.NonOption (name = "",
                          help = "Specify file to compile",
                          handler = fun (s) { files = s :: files })
	    ];	    
	    help_opts = opts;

	    Getopt.Parse (opts);

	    match (files) 
	    {
		| [] =>
		    Getopt.Error ("need one file to translate\n" + Getopt.Usage (opts))
		| [file] =>
		    try
		    {
			match (Options.OutputFileName)
			{
			    | None =>
				if(file.EndsWith(".cs"))
				    make_conversion ( file, file.Substring(0,file.Length - 2) + "n" );
				else
				    make_conversion ( file, file + ".n" );
			    | Some (s) =>
				make_conversion ( file, s);
			}
		    }
		    catch
		    {			
			| _ is Nemerle.CSharp.ErrorException =>	
			    Message.Error ("There were some errors in translation.");
			    System.Environment.Exit (2);

			| e is System.IO.FileNotFoundException => 			    
			    Message.Error (e.Message);
			    System.Environment.Exit (1);

		    }
		| _ =>
		    Getopt.Error ("need one file to translate\n" + Getopt.Usage (opts))
	    }          
	}
    }
}
