/*
 * Copyright (c) 2003, 2004 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */ 

namespace Sioux.Fit
{
  using Nemerle.Collections;
  using Nemerle.Xml;
  using Nemerle.IO;

  using System.Xml;
  using System.Xml.Xsl;

  using Sioux;


  /**
   * The submissions class
   */
  class Submission 
  {
    values : Hashtable [string, string];

    internal val (name : string) : string
    {
      Fit.get_val (values, name)
    }

    generate_id () : void
    {
      def sb = System.Text.StringBuilder (17);
      for (mutable i = 0; i < 16; i = i + 1) {
        ignore (sb.Append (((random.Next (24) + ('a' :> int)) :> char)))
      };
      values.Set ("id", sb.ToString ());
    }

    internal store () : void
    {
      lock (Fit.submissions) {
        match (Fit.submissions.Get (val ("id"))) {
          | Some (subm) => 
            def t = Fit.get_val (subm.values, "time_started");
            when (t != "")
              values.Set ("time_started", t);
          | None => ()
        };
        Fit.submissions.Set (val ("id"), this);
      }
    }

    internal read_post (pv : Hashtable [string, string], ignore_id : bool) : void
    {
      def old_id = val ("id");
      def read (name) { values.Set (name, Fit.get_val (pv, name).Trim ()) };
      iter_all (read);
      when (ignore_id || val ("id") == "")
        values.Set ("id", old_id);
        
      values.Set ("time_last_edit", System.DateTime.Now.ToString ());

      def id = val ("id");
      def loop (i) {
        if (i >= id.Length) true
        else {
          def ch = (id[i] :> int);
          if (ch >= ('a' :> int) && ch <= ('z' :> int)) loop (i + 1)
          else false
        }
      };
      if (id.Length != 16 || !loop (0)) {
        throw DieException ()
      } else {
        when (Option.IsNone (validate ()))
          store ()
      }
    }

    internal serialize (doc : XmlDocument) : XmlNode
    {
      def n = doc.CreateElement ("submission");
      def add_field (name) {
        def n' = doc.CreateElement (name);
        ignore (n.AppendChild (n'));
        ignore (n'.AppendChild (doc.CreateTextNode (val (name))));
      };
      iter_all (add_field);
      add_field ("time_started");
      add_field ("time_last_edit");
      n
    }

    internal unserialize (n : XmlNode) : void
    {
      def add_field (n : XmlNode) {
        when (n != null) {
          when (n.NodeType == XmlNodeType.Element) {
            values.Set (n.Name, n.InnerText)
          };
          add_field (n.NextSibling)
        }
      };
      add_field (n.FirstChild)
    }

    iter_all (f : string -> void) : void
    {
      List.Iter (text_fields, f);
      List.Iter (bool_fields, f);
      List.Iter (edit_fields, f);
      f ("id");
    }

    internal dump () : string
    {
      def res = System.Text.StringBuilder ();
      def read (name) { ignore (res.Append (sprintf ("%s='%s'\n", name, val (name)))) };
      iter_all (read);
      res.ToString ()
    }

    internal this ()
    {
      values = Hashtable ();
      values.Set ("time_started", System.DateTime.Now.ToString ());
      values.Set ("time_last_edit", System.DateTime.Now.ToString ());
      generate_id ();
    }
    
    internal get_submission_form (t : XmlTemplate) : XmlTemplate
    {
      List.Iter ("id" :: text_fields, fun (name) {
                  t.SetText (name + "/value", val (name))
                });
      t.SetText ("id2/value", val ("id"));
      List.Iter (bool_fields, fun (name) {
                  when (val (name) != "")
                    t.SetText (name + "/checked", "checked")
                });
      List.Iter (edit_fields, fun (name) {
                  t.SetText (name, val (name))
                });
      
      t
    }

    internal send_kill_email (admin_mode : bool) : void
    {
      def m = System.Web.Mail.MailMessage ();
      m.BodyEncoding = System.Text.Encoding.GetEncoding ("iso-8859-2");
      m.Subject = "Potwierdzenie usunięcia zgłoszenia na FIT";
      def fit_email = "FIT XVIII <fit18@ii.uni.wroc.pl>";
      def real_name = sprintf ("%s %s <%s>", val ("first_name"), 
                               val ("last_name"), val ("email"));
      if (admin_mode)
        m.To = fit_email
      else {
        m.To = real_name;
        m.Cc = fit_email;
      };
      m.From = fit_email;
      m.Body = "Usunięto zgłoszenie.\n" +
                "Dane zgłoszenia:\n" + dump ();
      System.Web.Mail.SmtpMail.Send (m)
    }

    internal send_confirmation_email (admin_mode : bool, edit_mode : bool) : void
    {
      def m = System.Web.Mail.MailMessage ();
      m.BodyEncoding = System.Text.Encoding.GetEncoding ("iso-8859-2");
      m.Subject = 
        if (edit_mode)
          "Potwierdzenie edycji zgłoszenia uczestnictwa w FIT XVIII"
        else
          "Potwierdzenie zgłoszenia uczestnictwa w FIT XVIII";
      def fit_email = "FIT XVIII <fit18@ii.uni.wroc.pl>";
      if (admin_mode)
        m.To = fit_email
      else {
        m.To = sprintf ("%s %s <%s>", val ("first_name"), 
                          val ("last_name"), val ("email"));
        m.Cc = fit_email;
      };
      m.From = fit_email;
      m.Body = 
        (if (edit_mode) 
          "Przyjęto edycję zgłoszenie na FIT XVII.\n\n"
        else 
          "Przyjęto zgłoszenie na FIT XVII.\n\n") +
        "Edycji zgłoszenia można dokonać pod adresem:\n" +
        "  http://lilith.ii.uni.wroc.pl:8000/edit.xml?id=" + val ("id") + "\n\n" +
        "Dane zgłoszenia:\n" + dump ();
      System.Web.Mail.SmtpMail.Send (m)
    }

    internal validate () : option [XmlTemplate]
    {
      def broken (n) { val (n).Trim () == "" };
      if (List.Exists (["email", "first_name", "last_name"], broken)) {
        def templ = XmlTemplate ("fit/error.xml");
        templ.SetText ("msg", "Nie wypełnione pole.");
        Some (templ)
      } else None ()
    }

    internal get_submission_confirm (t : XmlTemplate, 
                                     admin_mode : bool, 
                                     edit_mode : bool) : XmlTemplate
    {
      match (validate ()) {
        | Some (t) => t
        | None =>
          if (admin_mode)
            t.SetText ("email", "FITu")
          else
            t.SetText ("email", val ("email"));
          t.SetText ("dump_here", dump ());
          t.SetText ("edit/href", (if (admin_mode) Fit.secret_password else "") 
                                  + "/edit.xml?id=" + val ("id"));
          send_confirmation_email (admin_mode, edit_mode);
          
          t
      }
    }

    static text_fields : list [string];
    static bool_fields : list [string];
    static edit_fields : list [string];
    static random : System.Random;

    static this ()
    {
      text_fields = [
        "first_name",
        "last_name",
        "email",
        "acc_name",
        "acc_address",
        "acc_nip",
        "university",
        "ref_authors",
        "ref_title"
      ];
      bool_fields = [
        "vega",
        "treking",
        "single_room",
        "bus"
      ];
      edit_fields = ["remarks"];
      random = System.Random ();
    }
  }
}
