/*
 * Copyright (c) 2003-2005 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using Nemerle.Compiler;
using Nemerle.Utility;
using Nemerle.Compiler.Typedtree;

namespace Nemerle.Compiler.Typedtree
{
  public variant TypeDeclaration
  {
    | Class
    | Alias         { ty : MType; }
    | Interface
    | Variant       { members : list [TypeInfo]; }
    | VariantOption
    | Enum
  }

  public class Fun_parm : Located
  {
    public name : string;
    public color : int;
    public mutable ty : TyVar;
    public modifiers : Modifiers;
    public kind : ParmKind;

    public mutable default_value : option [Literal] = None ();
    // for local functions only
    mutable local_default_value : option [TExpr] = None ();

    public mutable decl : LocalValue;
    public mutable builder : System.Reflection.Emit.ParameterBuilder;

    public this (loc : Location, name : string, color : int, ty : TyVar, 
                 modifiers : Modifiers, kind : ParmKind)
    {
      base (loc);

      this.name = name;
      this.color = color;
      this.ty = ty;
      this.modifiers = modifiers;
      this.kind = kind;
    }

    public this (name : string, color : int, ty : TyVar, 
                 modifiers : Modifiers, kind : ParmKind)
    {
      this.name = name;
      this.color = color;
      this.ty = ty;
      this.modifiers = modifiers;
      this.kind = kind;
    }

    public SystemType : System.Type
    {
      get {
        ty.Fix ().SystemType;
      }
    }

    public Name : string
    {
      get { name }
    }

    public HasDefaultValue : bool
    {
      get {
        default_value.IsSome || local_default_value.IsSome
      }
    }

    public DefaultValueAsTExpr () : TExpr
    {
      match (local_default_value) {
        | Some (e) => e
        | None =>
          def lit = Option.UnSome (default_value);
          TExpr.Literal (Typer.TypeOfLiteral (lit), lit)
      }
    }

    internal GetLocalDefaultValueFromModifiers (par : Typer) : void
    {
      // store default value attribute
      foreach (<[ System.ComponentModel.DefaultValueAttribute ($e) ]>
               in modifiers.custom_attrs)
      {
        when (local_default_value.IsSome)
          Message.Error (e.loc, 
                         $ "default value specified twice for parameter "
                           "`$name'");
        local_default_value = Some (par.TypeExpr (e, Typer.AtLeast (ty)));
      }
    }
    
    internal GetDefaultValueFromModifiers (par : TypeBuilder) : void
    {
      // store default value attribute
      foreach (<[ System.ComponentModel.DefaultValueAttribute ($e) ]>
               in modifiers.custom_attrs)
      {
        match (ConstantFolder.FoldConstants (par.GlobalEnv, e)) {
          | Parsetree.PExpr.Literal (lit) =>
            when (lit is Literal.Decimal)
              Message.Error (e.loc,
                             $ "only `null' is allowed for default value of "
                               "type `decimal' (parameter `$name')");
            when (lit is Literal.Void)
              Message.Error (e.loc,
                             $ "the void literal is not allowed for default "
                              "value of parameter `$name'");
            when (default_value.IsSome)
              Message.Error (e.loc, 
                             $ "default value specified twice for parameter "
                               "`$name'");
            when (ty == null && lit is Literal.Null)
              Message.Error (e.loc, 
                             $ "type inference not supported when default value is null (parameter "
                               "`$name')");

            def ok =
              if (ty == null) {
                ty = lit.GetInternalType ();
                true
              } else if (ty.IsFixed)
                Typer.LiteralConversionPossible (lit, ty.FixedValue)
              else
                ty.Provide (lit.GetInternalType ());
            when (!ok)
              Message.Error (e.loc, 
                             $ "invalid type for default value of "
                               "parameter `$name', needed $ty, got $lit");

            def lit =
              if (ok && ty.IsFixed)
                match (lit.WithType (ty.FixedValue)) {
                  | Some (lit) => lit
                  | None => 
                    // ice?
                    Message.Error (e.loc, $ "cannot convert $lit to $ty");
                    lit
                }
              else lit;
              
            default_value = Some (lit);
            
          | _ =>
            Message.Error (e.loc,
                           $ "invalid default value for parameter `$name': "
                             "`$e', only literals are supported");
        }
      }
      
      // and discard it
      modifiers.custom_attrs =
        modifiers.custom_attrs.Filter (fun (_) {
          | <[ System.ComponentModel.DefaultValueAttribute ($_) ]> => false
          | _ => true
        });
    }


    public override ToString () : string
    {
      $ "(parm: $name : $ty)"
    }
  }

  public enum FunctionUsage 
  {
    | NotUsed
    | UsedJustOnce
    | Used
    | UsedAsFirstClass
  }
  
  public class Fun_header : Located, Nemerle.IComparable [Fun_header]
  {
    public mutable name : string;
    public id : int;
    public mutable ret_type : TyVar;
    public mutable parms : list [Fun_parm];
    public mutable typarms : list [StaticTyVar];
    public tenv : TyVarEnv;

    public mutable usage : FunctionUsage;
    public mutable body : FunBody;

    /// local values defined in this function, which are closurised (used by children) 
    public mutable closure_vars : list [LocalValue] = [];
    /// local value, which is declaring this function (valid for local funs)
    public mutable decl : LocalValue;
    public mutable all_locals : list [LocalValue] = [];
    internal mutable children_funs : list [Fun_header] = [];
    internal mutable uses_try_block : bool;
    internal mutable yield_type : TyVar;
    internal mutable is_in_closure_of : int;

    // FIXME: move to hashtables in t3
    internal mutable used_closures : list [Fun_header];
    internal mutable static_method : IMethod;
    internal mutable typarms_to_pass : list [MType]; 
    internal mutable closure_type : MType.Class;
    // end

    public CompareTo (other : Fun_header) : int
    {
      id - other.id
    }

    public this (loc : Location, 
                 name : string, 
                 ret_type : TyVar, 
                 parms : list [Fun_parm], 
                 typarms : list [StaticTyVar], 
                 tenv : TyVarEnv)
    {
      base (loc);
      this.name = name;
      this.ret_type = ret_type;
      this.parms = parms;
      this.typarms = typarms;
      this.tenv = tenv;

      id = Util.next_id ();
      usage = FunctionUsage.NotUsed;
    }
    
    /**
     *  Retrieves functions in which current function is nested. Empty list if
     *  this is a plain method of some class.
     */
    public GetParents () : list [Fun_header]
    {
      if (decl != null)
        match (decl.ValKind) {
          | LocalValue.Kind.Function (_, pars) => pars
          | _ => Util.ice ("invalid value kind for function")
        }
      else []
    }
    
    public override ToString () : string 
    {
      name
    }
  }

  
  [Record]
  public variant Pattern : Located
  {
    | Wildcard
    | As          { pat : Pattern; decl : LocalValue; }
    | HasType     { tycon : TypeInfo; }
    | Tuple       { args : list [Pattern]; }
    | Record      { args : list [IMember * Pattern]; }
    | Application { name : TypeInfo; arg : Pattern; }
    | Enum        { fld : IField; val : Nemerle.Compiler.Literal; }
    | Literal     { lit : Nemerle.Compiler.Literal; }
    | Error

    public mutable ty : TyVar;
    
    public this () { }

    public SystemType : System.Type
    {
      get { 
        Util.cassert (ty != null);
        ty.SystemType
      }
    }

    public Type : TyVar
    {
      get {
        Util.cassert (ty != null);
        ty
      }
    }

    public StripEnums () : Pattern
    {
      Walk (fun (_) {
        | Enum (_, l) => Literal (l)
        | _ => null
      })
    }


    /** Walk [this] with the function [f].

        The function is applied to each and every pattern node.
        
        If [f] returns a value different than [null], the value is
        returned, from this [Walk] function.

        If it returns [null], the walking function descends into tree
        elements in a map-like fashion.

        If the type of pattern returned was to be [null], it is set to
        the type of the original pattern.

        This function implements the visitor pattern.  */
    public Walk (f : Pattern -> Pattern) : Pattern
    {
      def do_walk (pat) {
        | Enum 
        | Pattern.Literal
        | Error
        | HasType
        | Wildcard => pat
        
        | As (pat, decl) =>
          As (walk (pat), decl)

        | Tuple (args) =>
          Tuple (args.Map (walk))
          
        | Record (args) =>
          Record (args.Map (fun (f, p) { (f, walk (p)) }))
          
        | Application (c, a) =>
          Application (c, walk (a))
      } and walk (pat) {
        def res = Util.locate (pat.loc, {
          def res = f (pat);
          if (res == null) do_walk (pat) else res;
        });
        when (res.ty == null)
          res.ty = pat.ty;
        res
      }

      walk (this)
    }


    public override ToString () : string
    {
      PrettyPrint.SprintPattern (this)
    }
  }

  [Record]
  public class Match_case
  {
    public mutable patterns : list [Pattern * TExpr * list [LocalValue * TExpr]];
    public mutable body : TExpr;
    public mutable disable_warnings : bool;
  }

  public variant ConversionKind
  {
    | MethodCall { sr : TExpr.StaticRef; }
    | IL { is_checked : bool; }
    | UpCast
    | DownCast
    | Implicit
    | IgnoreValue // generates a warning in Typer2
    | Unspecified // like DownCast, but avoid warnings

    // will disappear after generics bootstrap
    | GenericSim

    // Boxing and Nop are to be unused when cgexpr is killed
    | Boxing
    | Nop
  }

  [System.Flags]
  enum TExprFlags
  {
    | IsAssigned           = 0x0001
    | Visited              = 0x0002
    
    | Throws               = 0x0004
    | ThrowsComputed       = 0x0008

    | NeedAddress          = 0x0010

    | Addressable          = 0x0020
    | AddressableComputed  = 0x0040
    
    | JumpTarget           = 0x0080

    | Constrained          = 0x0100

    | GenerateTail         = 0x0200

    | SkipWriteCheck       = 0x0400
  }

  public variant TExpr : Located
  { 
    // always valid
    | StaticRef             { from : MType.Class; mem : IMember; type_parms : list [TyVar]; }
    | LocalRef              { decl : LocalValue; }
    | ImplicitValueTypeCtor
    | FieldMember           { obj : TExpr; fld : IField; }
    | MethodRef             { obj : TExpr; meth : IMethod; type_parms : list [TyVar]; notvirtual : bool; }
    | Call                  { mutable func : TExpr; mutable parms : list [Parm]; mutable is_tail : bool; }
    | Assign                { target : TExpr; source : TExpr; }
    | DefValIn              { name : LocalValue; val : TExpr; mutable body : TExpr; }
    | Throw                 { exn : TExpr; }
    | TryFault              { body : TExpr; handler : TExpr; }
    | TryWith               { body : TExpr; exn : LocalValue; handler : TExpr; }
    | TryFinally            { body : TExpr; handler : TExpr; }
    | Literal               { val : Nemerle.Compiler.Literal; }
    | This
    | Base                  { base_ctor : IMethod; }
    | TypeConversion        { mutable expr : TExpr; target_type : TyVar; kind : ConversionKind; }
    | Sequence              { mutable e1 : TExpr; mutable e2 : TExpr; }
    | Tuple                 { args : list [TExpr]; }
    | Array                 { args : list [TExpr]; dimensions : list [TExpr]; }
    | TypeOf                { target_type : TyVar; }
    | ArrayIndexer          { obj : TExpr; args : list [TExpr]; }
    | TupleIndexer          { obj : TExpr; pos : int; len : int; } // 0-based
    | OpCode                { name : string; }

    // invalid after T2
    | PropertyMember        { obj : TExpr; prop : IProperty; }
    | StaticPropertyRef     { from : MType.Class; prop : IProperty; }
    | EventMember           { obj : TExpr; ev : IEvent; }
    | StaticEventRef        { from : MType.Class; ev : IEvent; }
    | ConstantObjectRef     { from : MType.Class; mem : IField; }
    | Block                 { jump_out : LocalValue; body : TExpr; }
    | Delayed               { susp : Typer.DelayedTyping; }
    | Error

    // invalid after T3
    | DefFunctionsIn        { funs : list [Fun_header]; mutable body : TExpr; }
    | Match                 { expr : TExpr; cases : list [Match_case]; }
    | SelfTailCall          { parms : list [Parm]; }
    | LocalFunRef           { decl : LocalValue; type_parms : list [TyVar]; }
    
    // new CG-related stuff
    | MethodAddress         { from : TyVar; meth : IMethod; is_virt : bool;
                              type_parms : list [TyVar]; }
    | MultipleAssign        { assigns : list [LocalValue * TExpr]; }
    // we cannot directly reference Label from Goto, because Walk() would
    // destroy it
    | Label                 { id : int; body : TExpr; }
    // The try_block exact value has meaning only in T4, outside a value 
    // different than zero means that we should use [leave] and not [br].
    // This is set in T4, if you create nodes of this type earlier, set 
    // it to 1.
    | Goto                  { target : int; mutable try_block : int; }
    | DefaultValue
    | If                    { cond : TExpr; e_then : TExpr; e_else : TExpr; }
    | HasType               { expr : TExpr; test_ty : MType; }
    | NotNull               { expr : TExpr; }
    | Switch                { indexing_expr : TExpr;
                              default : option [TExpr]; 
                              cases : list [int * TExpr]; }
    
    public mutable ty : TyVar;
    
    [FlagAccessor (IsAssigned, flags = WantSetter | Internal)]
    [FlagAccessor (JumpTarget, flags = WantSetter | Internal)]
    [FlagAccessor (Visited, flags = WantSetter | Internal)]
    [FlagAccessor (GenerateTail, flags = WantSetter | Internal)]
    [FlagAccessor (SkipWriteCheck, flags = WantSetter | Internal)]
    mutable flags : TExprFlags;

    #region Typer4 and ILGenerator stuff
    internal Throws : bool
    {
      get {
        Util.cassert (flags %&& TExprFlags.ThrowsComputed,
                      $ "throws not computed for $this");
        flags %&& TExprFlags.Throws
      }

      set {
        Util.cassert (! (flags %&& TExprFlags.ThrowsComputed),
                      $ "throws already computed for $this");
        if (value)
          flags |= TExprFlags.Throws
        else
          flags &= ~TExprFlags.Throws;
        flags |= TExprFlags.ThrowsComputed;
      }
    }

    internal NeedsConstrained : bool
    {
      get { flags %&& TExprFlags.Constrained  }
      set { when (value) flags |= TExprFlags.Constrained }
    }

    internal IsAddressable : bool
    {
      get {
        unless (flags %&& TExprFlags.AddressableComputed)
          match (this) {
            | StaticRef
            | LocalRef
            | ArrayIndexer
            | This =>
              IsAddressable = true;
            | FieldMember (obj, fld) =>
              if (fld.DeclaringType.IsValueType)
                IsAddressable = obj.IsAddressable;
              else
                IsAddressable = true;
            | _ =>
              IsAddressable = false;
          }

        flags %&& TExprFlags.Addressable
      }

      set {
        Util.cassert (! (flags %&& TExprFlags.AddressableComputed));
        if (value)
          flags |= TExprFlags.Addressable
        else
          flags &= ~TExprFlags.Addressable;
        flags |= TExprFlags.AddressableComputed;
      }
    }


    internal NeedAddress : bool
    {
      get { flags %&& TExprFlags.NeedAddress }

      set {
        Util.cassert (value == true);
        Util.cassert (! NeedAddress);

        match (this) {
          | StaticRef
          | LocalRef
          | ArrayIndexer
          | This =>
            flags |= TExprFlags.NeedAddress
            
          | FieldMember (obj, fld) =>
            flags |= TExprFlags.NeedAddress;
            when (fld.DeclaringType.IsValueType && !obj.NeedAddress)
              obj.NeedAddress = true;
              
          | _ => {}
        }
      }
    }
    #endregion

    public Type : TyVar
    {
      get {
        when (ty == null) {
          match (this) {
            | Throw
            | Goto
            | Error =>
              ty = Solver.FreshTyVar ()
            | TExpr.Literal (l) =>
              ty = l.GetInternalType ()
            | Assign =>
              ty = InternalType.Void
            | HasType =>
              ty = InternalType.Boolean
            | _ =>
              Util.ice ($ "type is null for $loc, $this");
          }
        }

        Util.cassert (ty != null, $ "still null, $loc, $this");

        ty
      }
    }

    public MType : MType
    {
      get { Type.Fix () }
    }
    
    public SystemType : System.Type
    {
      get {
        Type.SystemType
      }
    }

    public this () { }
    
    public this (ty : TyVar)
    {
      this.ty = ty;
    }
    
    public this (loc : Location, ty : TyVar)
    {
      base (loc);
      this.ty = ty;
    }
    

    public override ToString () : string {
      PrettyPrint.SprintTyExpr (this)
    }


    // it does not copy calls
    public Copy () : TExpr
    {
      Walk (do_copy)
    }


    static do_copy (expr : TExpr) : TExpr
    {
      match (expr) {
        | This => This ()
        | LocalRef (decl) => LocalRef (decl)
        | StaticRef (from, mem, parms) => StaticRef (from, mem, parms)
        | OpCode (o) => OpCode (o)
        | TypeOf (t) => TypeOf (t)
        | TExpr.Literal (l) => TExpr.Literal (l)
        | Base (ctor) => Base (ctor)
        | ImplicitValueTypeCtor => ImplicitValueTypeCtor ()
        | MethodAddress (from, meth, v, typarms) => MethodAddress (from, meth, v, typarms)
        | Goto (id, t) => Goto (id, t)
        | DefaultValue => DefaultValue ()
        | ConstantObjectRef (from, mem) => ConstantObjectRef (from, mem)

        | _ => null
      }
    }


    static walk (f : TExpr -> TExpr, expr : TExpr) : TExpr
    {
      def res = Util.locate (expr.loc, {
        def res = f (expr);
        def res =
          if (res == null) do_walk (f, expr) else res;
        if (res == null) expr else res
      });
      when (res.ty == null)
        res.ty = expr.ty;
      //Message.Debug ($"walk: $expr -> $res");
      res
    }


    static null_walk (f : TExpr -> TExpr, expr : TExpr) : TExpr
    {
      def res = Util.locate (expr.loc, {
        def res = f (expr);
        if (res == null) do_walk (f, expr) else res;
      });
      when (res != null && res.ty == null)
        res.ty = expr.ty;
      //Message.Debug ($"null_walk: $expr -> $res");
      res
    }


    static walks (f : TExpr -> TExpr, exprs : list [TExpr]) : list [TExpr]
    {
      mutable cnt = 0;
      mutable total = 0;
      mutable first_expr = null;
      
      foreach (expr in exprs) {
        when (first_expr == null) {
          def expr' = null_walk (f, expr);
          when (expr' != null)
            first_expr = expr';
          cnt++;
        }
        total++;
      }

      if (first_expr == null) exprs
      else {
        def loop (res, n, l) {
            match (l) {
              | x :: xs =>
                if (n == 0)
                  xs.RevAppend (first_expr :: res)
                else
                  loop (walk (f, x) :: res, n - 1, xs)
              | [] => Util.ice ()
            }
        }
        loop ([], total - cnt, exprs.Rev ())
      }
    }

    static do_walk (f : TExpr -> TExpr, expr : TExpr) : TExpr
    {
      match (expr) {
        | Delayed
        | PropertyMember
        | StaticPropertyRef
        | EventMember
        | Block
        | StaticEventRef =>
          // this is supposed to be run after Typer2
          Message.Warning (expr.loc, 
                           $ "invalid expr in walk: $(expr.GetType()): $expr");
          assert (false)
          

        | This
        | LocalRef
        | LocalFunRef
        | StaticRef 
        | Error
        | OpCode 
        | TypeOf
        | TExpr.Literal
        | Base
        | ImplicitValueTypeCtor
        | MethodAddress
        | Goto
        | DefaultValue
        | ConstantObjectRef =>
          null
          

        | DefFunctionsIn (funs, body) =>
          foreach (fn in funs)
            match (fn.body) {
              | FunBody.Typed (e) =>
                fn.body = FunBody.Typed (walk (f, e))
              | _ => {}
            }
          DefFunctionsIn (funs, walk (f, body))
          
          
        | MethodRef (obj, meth, tp, notvirt) =>
          def obj = null_walk (f, obj);
          if (obj == null) null
          else
            MethodRef (obj, meth, tp, notvirt)


        | FieldMember (obj, fld) =>
          def obj = null_walk (f, obj);
          if (obj == null) null
          else
            FieldMember (obj, fld)
          
          
        | Call (func, parms, is_tail) =>
          def func = walk (f, func);
          mutable rev_parms = [];
          foreach (p in parms)
            rev_parms = Parm (p.kind, walk (f, p.expr), p.name, p.required_type) :: rev_parms;
          Call (func, rev_parms.Rev (), is_tail)

        
        | SelfTailCall (parms) =>
          foreach (p in parms)
            p.expr = walk (f, p.expr);
          null
        

        | Assign (target, source) =>
          def target' = walk (f, target);
          def source' = walk (f, source);
          if (target' : object == target && source' : object == source)
            null
          else
            Assign (target', source')


        | MultipleAssign (assigns) =>
          MultipleAssign (assigns.Map (fun (s, t) { (s, walk (f, t)) }))
          
   
        | DefValIn (name, val, body) =>
          def val' = walk (f, val);
          def body' = walk (f, body);
          if (val' : object == val && body' : object == body)
            null
          else
            DefValIn (name, val', body')
          
          
        | Match (matched_value, cases) =>
          foreach (case in cases) {
            mutable pats = [];
            foreach ((pat, expr, assigns) in case.patterns) {
              def assigns =
                assigns.Map (fun (v, e) { (v, walk (f, e)) });
              pats = (pat, walk (f, expr), assigns) :: pats
            }
            case.patterns = pats.Rev ();
            case.body = walk (f, case.body);
          }
          def matched_value = null_walk (f, matched_value);
          if (matched_value == null) null
          else
            Match (matched_value, cases)


        | If (cond, e1, e2) =>
          def cond' = walk (f, cond);
          def e1' = walk (f, e1);
          def e2' = walk (f, e2);
          if (cond' : object == cond &&
              e1' : object == e1 &&
              e2' : object == e2)
            null
          else
            If (cond', e1', e2')


        | Switch (ind, defl, cases) =>
          Switch (walk (f, ind),
                  match (defl) {
                    | Some (d) => Some (walk (f, d))
                    | None => defl
                  },
                  cases.Map (fun (n, e) { (n, walk (f, e)) }))


        | HasType (e, t) =>
          def e = null_walk (f, e);
          if (e == null) null
          else
            HasType (e, t)
          

        | NotNull (e) =>
          def e = null_walk (f, e);
          if (e == null) null
          else
            NotNull (walk (f, e))
          

        | Throw (exn) =>
          // exception can be null for `throw;' rethrow expression
          if (exn != null) {
            def exn = null_walk (f, exn);
            if (exn == null) null
            else Throw (exn)
          } else
            null

        | TryWith (body, exn, handler) =>
          TryWith (walk (f, body), exn, walk (f, handler))
          
          
        | TryFinally (body, handler) =>
          TryFinally (walk (f, body), walk (f, handler))
          

        | TryFault (body, handler) =>
          TryFault (walk (f, body), walk (f, handler))
          

        | TypeConversion (expr, t, kind) =>
          def expr = null_walk (f, expr);
          if (expr == null) null
          else
            TypeConversion (expr, t, kind)
          
          
        | Sequence (e1, e2) =>
          def e1' = walk (f, e1);
          def e2' = walk (f, e2);
          if (e1' : object == e1 && e2' : object == e2)
            null
          else
            Sequence (e1', e2')
          
          
        | Tuple (args) =>
          Tuple (walks (f, args))


        | TupleIndexer (obj, k, n) =>
          def obj = null_walk (f, obj);
          if (obj == null) null
          else
            TupleIndexer (obj, k, n)
          
          
        | Array (args, dimensions) =>
          Array (walks (f, args), walks (f, dimensions))
          
          
        | ArrayIndexer (obj, args) =>
          ArrayIndexer (walk (f, obj), walks (f, args))


        | Label (id, body) =>
          def body = null_walk (f, body);
          if (body == null) null
          else
            Label (id, body)
      }
    }

    /** Walk [this] with the function [f].

        The function is applied to each and every expression node.
        
        If [f] returns a value different than [null], the value is
        returned, from this [Walk] function.

        If it returns [null], the walking function descends into tree
        elements in a map-like fashion.

        If the type of expression returned was to be [null], it is set
        to the type of the original expression.

        This function implements the visitor pattern.  */
    public Walk (f : TExpr -> TExpr) : TExpr
    {
      walk (f, this)
    }


    static mutable true_literal : TExpr;
    static mutable false_literal : TExpr;
    
    public static BoolLiteral (val : bool) : TExpr
    {
      when (true_literal == null) {
        false_literal = 
          TExpr.Literal (InternalType.Boolean,
                         Nemerle.Compiler.Literal.Bool (false));
        true_literal = 
          TExpr.Literal (InternalType.Boolean,
                         Nemerle.Compiler.Literal.Bool (true));
      }

      if (val) true_literal else false_literal
    }

    static this ()
    {
      Passes.OnInit += fun () {
        true_literal = null;
        false_literal = null;
      }
    }

    public static TrueLiteral : TExpr { get { BoolLiteral (true) } }
    public static FalseLiteral : TExpr { get { BoolLiteral (false) } }
  }

  public enum ParmKind {
    | Normal
    | Ref
    | Out
  }

  [Record]
  public class Parm
  {
    public kind : ParmKind;
    public mutable expr : TExpr;
    public name : string;  // "" if no name given
    public mutable required_type : TyVar;

    public this (expr : TExpr)
    {
      kind = ParmKind.Normal;
      this.expr = expr;
      name = "";
    }
  }
} // Nemerle.Compiler.Typedtree
