/*
 * Copyright (c) 2004, 2005 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using Nemerle.Compiler.SolverMacros;
using Nemerle.Collections;
using Nemerle.Utility;

namespace Nemerle.Compiler 
{
  /** Represent top-level type constructor for a given type. */
  public variant MType : TyVar
  {
    #region Options
    | Class {
        tycon : TypeInfo;
        args : list [TyVar];
      }
    | TyVarRef {
        tyvar : StaticTyVar;
      }
    | Fun {
        from : TyVar;
        to : TyVar;
      }
    | Tuple {
        args : list [TyVar];
      }
    | Array {
        t : TyVar;
        rank : int;
      }
    | Ref {
        t : TyVar;
      }
    | Out {
        t : TyVar;
      }
    | Void

      /* Used when given value is required to have all the listed types.

         Invariant 1: the types listed cannot be in the subtyping relation
         pairwise.

         Invariant 2: there can be only Class() objects inside.
         
         This type is not expressible in the .NET type system directly,
         it can be however expressed with type variable bounds in some
         cases. */
    | Intersection {
        types : list [MType];
      }
    #endregion


    /** Check for type equality, taking intersection types
        into account.  */
    public Equals (t : MType) : bool
    {
      if (this : object == t : object)
        true
      else
        match ((t, this)) {
          | (Class (ti1, a1), Class (ti2, a2)) =>
            ti1.Equals (ti2) && a1.Equals (a2)
                          
          | (TyVarRef (tv1), TyVarRef (tv2)) =>
            tv1.Equals (tv2)
            
          | (Fun (f1, t1), Fun (f2, t2)) =>
            f1.Equals (f2) && t1.Equals (t2)
            
          | (Tuple (a1), Tuple (a2)) =>
            a1.Equals (a2)

          | (Out (t1), Out (t2))
          | (Ref (t1), Ref (t2))
          | (Array (t1, r1), Array (t2, r2)) when (r1 == r2) =>
            t1.Equals (t2)

          | (Void, Void) => true

          | (Intersection (l1), Intersection (l2)) =>
            if (l1.Length == l2.Length) {
              def h = Hashtable ();
              foreach (MType.Class (ti, _) as t in l1)
                h [ti] = t;
              mutable same = true;
              foreach (MType.Class (ti, _) as t in l2)
                if (h.Contains (ti))
                  same = same && h [ti].Equals (t)
                else
                  same = false;
              same
            } else false

          | _ => false
      }
    }
    
    #region System.Type conversion
    public override NonVoidSystemType : System.Type
    {
      get {
        match (this) {
          | Void => SystemType.Object
          | _ => SystemType
        }
      }
    }
    
    public override SystemType : System.Type
    {
      get {
        match (this) {
          | Void => SystemType.Void

          | Tuple =>
            TupleType.Make (this).SystemType

          | Fun =>
            FunctionType.Make (this).SystemType

          | TyVarRef (tv) => tv.SystemType

          | Class (ti, []) => ti.SystemType;

          | Class (ti, args) =>
            def tconstructor = ti.SystemType;
            
            def typedargs = array (args.Length);
            mutable idx = 0;
            mutable formals = ti.Typarms;

            def errs = Message.ErrorCount;
            foreach (arg in args) {
              match (formals) {
                | f :: fs =>
                  formals = fs;
                  f.CheckConstraints (arg, this);
                | [] => Util.ice ()
              }
              typedargs [idx] = arg.SystemType;
              ++idx;
            }
            
            //tconstructor.BindGenericParameters (typedargs);  
            //Message.Debug ($"bgp: $this");
            if (errs == Message.ErrorCount)
              tconstructor.GetGenericTypeDefinition ().MakeGenericType (typedargs)
            else
              SystemType.Object

          | Ref (t)
          | Out (t) => t.SystemType.MakeByRefType ()

          | Array (et, rank) =>
            if (rank == 1)
              et.SystemType.MakeArrayType ()
            else            
              et.SystemType.MakeArrayType (rank)

          | Intersection =>
            // we cannot generate system type for intersection. maybe we could use
            // some approximation? the obvious idea would be to prohibit this kind
            // of stuff to occur after typing is done.
            assert (false)
        }
      }
    }
    #endregion


    #region Unification stuff
    /** Check for type equality, taking intersection types
        into account. If it's possible that types are equal -- enforce 
        that.  Assume non-seperated types. */
    public TryEnforcingEquality (t : MType) : bool
    {
      assert (!this.IsSeparated);
      assert (!t.IsSeparated);

      match ((this, t)) {
        | (Class (tc1, args1), Class (tc2, args2))
          when tc1.Equals (tc2) =>
          List.ForAll2 (args1, args2, fun (t1 : TyVar, t2 : TyVar) {
            t1.Unify (t2)
          })
          
        | (Intersection (l1), Intersection (l2)) 
          when List.Length (l1) == List.Length (l2) =>
          def ht = Hashtable ();
          foreach ((Class (tc, _)) as t in l1)
            ht [tc] = t;
          mutable cnt = 0;
          foreach (Class (tc, _) in l2)
            when (ht.Contains (tc))
              ++cnt;
          if (cnt == List.Length (l1)) {
            mutable failed = false;
            foreach ((Class (tc, _)) as t in l2)
              failed = failed || !ht [tc].Unify (t);
            failed
          } else false

        | (Fun (f1, t1), Fun (f2, t2)) =>
          f1.Unify (f2) && t1.Unify (t2)
          
        | (Tuple (l1), Tuple (l2)) 
          when List.Length (l1) == List.Length (l2) =>
          List.ForAll2 (l1, l2, fun (x : TyVar, y : TyVar) { x.Unify (y) })

        | (Array (t1, rank1), Array (t2, rank2)) when rank1 == rank2 =>
          t1.Unify (t2)
        
        | (TyVarRef (tv1), TyVarRef (tv2)) =>
          tv1.Equals (tv2)
          
        | _ => false
      }
    }


    /** Enforce [this] to be subtype of [t]. */
    public override Require (t : MType) : bool
    {
      def s = Passes.Solver;

      match ((this, t)) {
        | (_, Class (tc, _)) 
          when tc.Equals (InternalType.Object_tc) => true
          
        | (Class (tc1, args1), Class (tc2, args2)) =>
          //Message.Debug ($"Require $this $t");
          if (tc1.Equals (tc2) && args1.Equals (args2))
            true
          else
            match (tc1.SuperType (tc2)) {
              | Some (args) =>
                //Message.Debug ($"args $args"); 
                def subst = tc1.MakeSubst (args1);
                List.ForAll2 (args, args2, 
                             fun (t : MType, tv : TyVar) { 
                               tv.Unify (subst.Apply (t)) 
                             })
                
              | None =>
                SaveError (s.CurrentMessenger, 
                           $ "$tc1 is not a subtype of $tc2 [simple require]");
                false
            }
          
        | (Fun (f1, t1), Fun (f2, t2)) =>
          // f2.Require (f1) && t1.Require (t2)
          f1.Unify (f2) && t1.Unify (t2)
          
        | (Tuple (l1), Tuple (l2)) 
          when List.Length (l1) == List.Length (l2) =>
          List.ForAll2 (l1, l2, fun (x : TyVar, y : TyVar) { x.Unify (y) })
          
        | (Array (t1, rank1), Array (t2, rank2))
          when rank1 == rank2 =>
          // XXX we don't allow array covariance here! we may want to change it
          t1.Unify (t2)

        | (Ref (t1), Ref (t2))
        | (Out (t1), Out (t2)) =>
          t1.Unify (t2)
        
        // XXX these two shouldn't be needed
        //| (Void, Class (tc, _)) when tc.Equals (InternalType.Void_tc)
        //| (Class (tc, _), Void) when tc.Equals (InternalType.Void_tc)
        | (Void, Void) => true

        | (Array, Class (tc, _)) 
          when tc.Equals (InternalType.Array_tc) => true

        | (TyVarRef (tv1), TyVarRef (tv2))
          when tv1.Equals (tv2) => true

        | (TyVarRef (tv1), t2) =>
          tv1.LowerBound.Require (t2)

        | (Intersection (lst), (Class (tc, _)) as t2) =>
          def loop (_) {
            | (Class (tc', _) as t1) :: xs =>
              if (tc'.SuperType (tc).IsSome)
                t1.Require (t2)
              else
                loop (xs)
            | [] =>
              SaveError (s.CurrentMessenger, 
                         $ "$(this) is not a subtype of $t [simple "
                           "require, intersection]");
              false
            | _ => assert (false)
          }
          loop (lst)
        
        | (t1, Intersection (lst)) =>
          List.ForAll (lst, fun (t2) { t1.Require (t2) })

        | _ =>
          SaveError (s.CurrentMessenger,
                     $ "$(this) is not a subtype of $t [simple require]");
          false
      }
    }

    
    /** Enforce [this] to be equal [t]. */
    public override Unify (t : MType) : bool
    {
      match ((this, t)) {
        | (Class, Class) when TryEnforcingEquality (t)
        | (TyVarRef, TyVarRef) when TryEnforcingEquality (t)
        | (Fun, Fun) when TryEnforcingEquality (t)
        | (Array, Array) when TryEnforcingEquality (t)
        | (Tuple, Tuple) when TryEnforcingEquality (t)
        | (Intersection, Intersection) when TryEnforcingEquality (t) =>
          true
        
        | (Ref (t1), Ref (t2))
        | (Out (t1), Out (t2)) =>
          t1.Unify (t2)
        
        | (Void, Void) => true

        | _ =>
          SaveError (Passes.Solver.CurrentMessenger,
                     $ "the types $(this) and $t are not compatible "
                       "[simple unify]");
          false
      }
    }
    #endregion


    #region Pretty printing
    public override ToString () : string
    {
      match (this) {
        | Class (tc, []) =>
          def trim = tc.FullName.Replace ("Nemerle.Core.", "");
          match (trim) {
            | "System.String" => "string"
            | "System.Int32" => "int"
            | "System.Single" => "float"
            | "System.Char" => "char"
            | "System.Boolean" => "bool"
            | "System.Void" => "void"
            | _ => trim
          }

        | Class (tc, args) => 
          def trim = tc.FullName.Replace ("Nemerle.Core.", "");
          trim + args.ToString ()
          
        | TyVarRef (s) => s.ToString ()
        
        | Fun (t1, t2) => $ "$t1 -> $t2"
          
        | Tuple (lst) => "(" + lst.ToString (" * ") + ")"

        | Ref (t) => $ "ref $t"
          
        | Out (t) => $ "out $t"
          
        | Array (t, 1) => $ "array [$t]"
          
        | Array (t, n) => $ "array.$n [$t]"
          
        | Void => "void"
        
        | Intersection (lst) => "(" + lst.ToString (" AND ") + ")"
      }
    }
    #endregion
    
  
    #region Public properties
    /** Check if given type cannot be supertyped by a plain type
        constructor. */
    public IsSeparated : bool
    {
      get {
        match (this) {
          | Class => false

          | TyVarRef
          | Fun
          | Tuple
          | Array => false

          | Ref
          | Out
          | Void => true
          
          | Intersection (lst) =>
            foreach (x in lst)
              assert (!x.IsSeparated);
            false
        }
      }
    }


    public override CanBeNull : bool
    {
      get {
        match (this) {
          | Class (ti, _) => !ti.IsValueType
          
          | Ref
          | Out
          | Tuple
          | Void => false
          
          | TyVarRef (s) =>
            s.SpecialConstraints %&& 
            System.Reflection.GenericParameterAttributes.ReferenceTypeConstraint ||
            found: {
              foreach (ctr in s.Constraints)
                when (ctr.CanBeNull) found (true);
              false
            }
          
          | Fun
          | Array
          | Intersection => true
        }
      }
    }


    internal override NeedNoSubst : bool
    {
      get {
        match (this) {
          | Void
          | Class (_, []) => true
          
          | _ => false
        }
      }
    }


    public IsInterface : bool
    {
      get {
        match (this) {
          | Class (tc, _) => tc.IsInterface
          // XXX hmm..
          // | Intersection (lst) =>
          //   List.ForAll (lst, fun (x : MType) { x.IsInterface })
          | _ => false
        }
      }
    }
    
    
    public IsSystemObject : bool
    {
      get {
        match (this) {
          | Class (tc, []) => tc.Equals (InternalType.Object_tc)
          | _ => false
        }
      }
    }


    public IsPrimitive : bool
    {
      get {
        match (this) {
          | Class (tc, []) =>
            tc.IsEnum ||
            tc.SystemType.IsPrimitive
          | _ => false
        }
      }
    }
    
    
    public IsValueType : bool
    {
      get {
        match (this) {
          | Class (tc, _) => tc.IsValueType
          | Tuple ([_, _])
          | Tuple ([_, _, _]) => true
          | _ => false
        }
      }
    }


    public IsEnum : bool
    {
      get {
        match (this) {
          | Class (tc, _) => tc.IsEnum
          | _ => false
        }
      }
    }


    public TypeInfo : TypeInfo
    {
      get {
        match (this) {
          | Class (tc, _) => tc
          | _ => null
        }
      }
    }
    #endregion
    
    
    #region Public helper functions
    static public ConstructFunctionType (parms : list [MType], res : MType) : MType
    {
      ConstructFunctionType (Solver.MonoTypes (parms), res)
    }

    
    static public ConstructFunctionType (parms : list [TyVar], res : TyVar) : MType
    {
      def from =
        match (parms) {
          | [x] => x
          | [] => InternalType.Void
          | lst => MType.Tuple (lst)
        }

      MType.Fun (from, res)
    }
    

    static public ConstructFunctionType (header : Typedtree.Fun_header) : MType
    {
      def parms = List.Map (header.parms, 
                            fun (fp : Typedtree.Fun_parm) { fp.ty });
      
      ConstructFunctionType (parms, header.ret_type)
    }

    public GetFunctionArguments () : list [MType]
    {
      match (this) {
        | Void => []
        | Tuple (lst) =>
          List.Map (lst, fun (x : TyVar) { x.Fix () })
        | x => [x]
      }
    }


    public GetUnfixedFunctionArguments () : list [TyVar]
    {
      match (this) {
        | Void => []
        | Tuple (lst) => lst
        | x => [x]
      }
    }


    public static AccessibilityIntersect (a1 : Accessibility, 
                                          a2 : Accessibility) : Accessibility
    {
      match ((a1, a2)) {
        | (Accessibility.Private, _)
        | (_, Accessibility.Private) => Accessibility.Private
        | (Accessibility.Internal, Accessibility.Protected)
        | (Accessibility.Protected, Accessibility.Internal)
        | (_, Accessibility.ProtectedAndInternal)
        | (Accessibility.ProtectedAndInternal, _) => Accessibility.ProtectedAndInternal
        | (_, Accessibility.Protected)
        | (Accessibility.Protected, _) => Accessibility.Protected
        | (_, Accessibility.Internal)
        | (Accessibility.Internal, _) => Accessibility.Internal
        | (_, Accessibility.ProtectedOrInternal)
        | (Accessibility.ProtectedOrInternal, _) => Accessibility.ProtectedOrInternal
        | (Accessibility.Public, Accessibility.Public) => Accessibility.Public
      }
    }


    /** Check if [access] doesn't grant more access than any of tycons in
        [this].  The [what] parameter is used only for error messages.  */
    public CheckAccessibility (what : IMember, access : Accessibility) : void
    {
      match (this) {
        | MType.Class (tc, parms) =>
          def maybe_me = what.DeclaringType;
          
          if (AccessibilityIntersect (tc.Accessibility, access) != access &&
              (maybe_me == null || !maybe_me.Equals (tc)))
            Message.Error ($ "$what is more accessible than `$(tc)'")
          else
            foreach (t in parms)
              t.Fix ().CheckAccessibility (what, access)
            
        | MType.TyVarRef | MType.Void => {}
        
        | MType.Ref (t)
        | MType.Out (t)
        | MType.Array (t, _) =>
          t.Fix ().CheckAccessibility (what, access)
          
        | MType.Fun (t1, t2) =>
          t1.Fix ().CheckAccessibility (what, access);
          t2.Fix ().CheckAccessibility (what, access)
          
        | MType.Tuple (parms) =>
          foreach (t in parms)
            t.Fix ().CheckAccessibility (what, access)

        | MType.Intersection (lst) =>
          foreach (elem in lst)
            elem.CheckAccessibility (what, access)
      }
    }


    /** Get type of member when referenced on value of the current type,
        which has to be fixed.  */
    public TypeOfMember (member : IMember) : TyVar
    {
      match (this) {
        | Class (ti, args) =>
          def s1 = ti.SubtypingSubst (member.DeclaringType);
          def s2 = ti.MakeSubst (args);
          s2.Apply (s1.Apply (member.GetMemType ()).Fix ())
        | Array =>
          InternalType.Array.TypeOfMember (member)
          
        | TyVarRef (tyvar) =>
          foreach (t is MType.Class in tyvar.Constraints) // FIXME: what to do with "where 'a : 'b"?
            when (t.tycon.Equals (member.DeclaringType))
              Nemerle.Imperative.Return (t.TypeOfMember (member));
          foreach (t is MType.Class in tyvar.Constraints)
            when (t.tycon.SuperType (member.DeclaringType).IsSome)
              Nemerle.Imperative.Return (t.TypeOfMember (member));
          Util.ice ($"not found for member $(member) with dt $(member.DeclaringType) -- am in $(this)")
        
        | _ =>
          Util.ice ($ "unsupported type: for member $(member) with dt $(member.DeclaringType) -- am in $(this)")
      }
    }
    

    public ConvertTypeFromTypeInfo (ty : MType, from : TypeInfo) : TyVar
    {
      match (this) {
        | Class (ti, args) =>
          def s1 = ti.SubtypingSubst (from);
          def s2 = ti.MakeSubst (args);
          s2.Apply (s1.Apply (ty).Fix ())

        | _ =>
          Util.ice ($"unsupported type: $this");
      }
    }
      
    public TypeOfMethodWithTyparms (method : IMethod) : TyVar * list [TyVar]
    {
      match (this) {
        | Class (ti, args) =>
          def s1 = ti.SubtypingSubst (method.DeclaringType);
          def s2 = ti.MakeSubst (args);
          def (s3, vars) = Subst.Fresh (method.GetHeader ().typarms);
          s2.AddSubst (s3);
          (s2.Apply (s1.Apply (method.GetMemType ()).Fix ()), vars)
        | _ =>
          assert (false, $ "unsupported type: $(this)")
      }
    }



    public TypeOfMethod (method : IMethod) : TyVar
    {
      match (this) {
        | Class (ti, args) =>
          def s1 = ti.SubtypingSubst (method.DeclaringType);
          def s2 = ti.MakeSubst (args);
          def (s3, _vars) = Subst.Fresh (method.GetHeader ().typarms);
          s2.AddSubst (s3);
          // FIXME we use it for method implementation
          // Util.cassert (method.GetHeader ().typarms.IsEmpty,
          //              $ "TypeOfMethod used for $method");
          s2.Apply (s1.Apply (method.GetMemType ()).Fix ())
        | _ =>
          assert (false, $ "unsupported type: $(this)")
      }
    }


    /** This is a hack used in external/Codec.n to expand type aliases
        in imported types.  NTE won't generate aliases there, so it shouldn't
        be needed later.  */
    public Expand () : MType
    {
      match (this) {
        | Class (ti, args) =>
          def tydecl = ti.GetTydecl ();
          if (tydecl == null) this
          else
            match (tydecl) {
              | Typedtree.TypeDeclaration.Alias (t) =>
                def subst = ti.MakeSubst (args);
                subst.MonoApply (t)
              | _ => this
            }
        | _ => this
      }
    }


    /** Get the exact instantiation upon which [this] implements 
        [super_type]. 
        
        For example for:
          class Foo[T] : Bar [int, list[T]]
        the call:
          Foo[string].GetInstantiatedSuperType (Bar)
        will return:
          Bar [int, list[string]]
       */
    public GetInstantiatedSuperType (super_type : TypeInfo) : MType.Class
    {
//      Message.Debug ($"this $this - super $super_type");
      if (super_type.Equals (InternalType.Object_tc))
        InternalType.Object
      else
        match (this) {
          | Class (sub_type, _) as res when sub_type.Equals (super_type) =>
            res

          | Class (sub_type, parms) =>
            def sub = sub_type.MakeSubst (parms);
            def parms' = Option.UnSome (sub_type.SuperType (super_type));
            Class (super_type, parms'.Map (sub.Apply))

          | TyVarRef (st) =>
/* CRASHES COMPILER            
            ret: {
              foreach (c :> MType.Class in st.Constraints) {
                def sub = c.tycon.MakeSubst (c.args);
                match (c.tycon.SuperType (super_type)) {
                  | Some (parms) =>
                    ret (Class (super_type, parms.Map (sub.Apply)))
                  | None => ()
                }
              }
              null // impossible
            }
*/
            mutable result = null;
              foreach (c :> MType.Class in st.Constraints) {
                def sub = c.tycon.MakeSubst (c.args);
                match (c.tycon.SuperType (super_type)) {
                  | Some (parms) =>
                    result = Class (super_type, parms.Map (sub.Apply));
                  | None => ()
                }
              }
            result
            
          | Fun =>
            FunctionType.Make (this).GetInstantiatedSuperType (super_type)

          | Array =>
            InternalType.Array.GetInstantiatedSuperType (super_type)

          | _ => Util.ice ($ "GIST: $this for $super_type")
        }
    }


    public FunReturnTypeAndParms () : option [list [TyVar] * TyVar]
    {
      match (this) {
        | Fun (from, to) =>
          def froms =
            match (from.Hint) {
              | Some (Void)
              | Some (Tuple) => from.Fix ().GetUnfixedFunctionArguments ()
              | _ => [from]
            }
          Some ((froms, to))
        | _ => None ()
      }
    }


    public FunReturnTypeAndParms (meth : IMethod) : list [TyVar] * TyVar
    {
      match (this) {
        | Fun (from, to) =>
          def froms =
            match (meth.GetParameters ()) {
              | [] => []
              | [_] => [from]
              | parms =>
                match (from.Fix ()) {
                  | Tuple (args) => args
                  | fixed => Util.ice ($ "$fixed <- $this $parms")
                }
            }
          (froms, to)
        | _ => Util.ice ()
      }
    }


    public SigRequire (other : MType) : bool
    {
      match ((FunReturnTypeAndParms (), other.FunReturnTypeAndParms ())) {
        | (Some ((f1, t1)), Some ((f2, t2))) when f1.Length == f2.Length =>
          t1.Require (t2) &&
          List.ForAll2 (f1, f2, fun (t1, t2) { t2.Require (t1) })
        | _ => false
      }
    }


    public TrySigRequire (other : MType) : bool
    {
      Passes.Solver.PushState ();
      def res = this.SigRequire (other);
      Passes.Solver.PopState ();
      res
    }


    public Iter (f : MType -> void) : void
    {
      f (this);

      match (this) {
        | Tuple (args)
        | Class (_, args) =>
          foreach (a in args)
            a.Fix ().Iter (f)

        | Fun (a, t) =>
          a.Fix ().Iter (f);
          t.Fix ().Iter (f);
          
        | Array (t, _)
        | Ref (t)
        | Out (t) =>
          t.Fix ().Iter (f)
          
        | Intersection (args) =>
          foreach (a in args)
            a.Iter (f)

        | Void
        | TyVarRef => {}
      }
    }
    #endregion


    #region Internal implementation
    internal Validate () : void
    {
      match (this) {
        | Intersection ([]) => assert (false)
        | Intersection ([_]) => assert (false)
        | Intersection (lst) =>
          
          def supers =
            List.FoldLeft (lst, Set (), fun (e, s : Set [TypeInfo]) {
              match (e) {
                | Class (tc, _) =>
                  def lst = List.Map (tc.GetSuperTypes (), 
                                      fun (x) { x.tycon });
                  s.ReplaceList (lst)
                | _ => assert (false)
              }
            });

          def lst = List.Map (lst, fun (_ : MType) { 
                                        | Class (tc, _) => tc
                                        | _ => assert (false)
                                      });
          assert (Set.FromList (lst).Intersect (supers).IsEmpty);

       | _ => ()
      }
    }
    #endregion


    #region Overridden stuff from TyVar
    public override Require (t : TyVar) : bool
    {
      if (t.IsFixed)
        Require (t.FixedValue)
      else
        t.Provide (this)
    }

    public override Provide (t : TyVar) : bool
    {
      if (t.IsFixed)
        Provide (t.FixedValue)
      else
        t.Require (this)
    }

    public override Provide (t : MType) : bool
    {
      t.Require (this)
    }

    public override Fix () : MType
    {
      this
    }

    public override FixedValue : MType
    {
      get { this }
    }

    public override IsAccessibleFrom (ti : TypeInfo) : bool
    {
      def can_access (x) { x.IsAccessibleFrom (ti) }
      
      match (this) {
        | Class (t, args) =>
          t.CanAccess (ti) && args.ForAll (can_access)

        | Void
        | TyVarRef => true

        | Tuple (args) =>
          args.ForAll (can_access)

        | Fun (from, to) =>
          from.IsAccessibleFrom (ti) && to.IsAccessibleFrom (ti)
          
        | Array (t, _)
        | Ref (t)
        | Out (t) => t.IsAccessibleFrom (ti)

        | Intersection (mtypes) =>
          mtypes.ForAll (can_access)
      }
    }

    
    internal this ()
    {
      lower_bound = this;
      upper_bound = this;
      serial = 1;
      flags = TyVar.Flags.IsMonoType;
    }
    #endregion
  }
}
